namespace Galago
{
	using System;
	using System.Collections;
	using System.Runtime.InteropServices;

	public class List
		: IDisposable, ICollection, IWrapper, ICloneable
	{
		private IntPtr rawList = IntPtr.Zero;
		private bool managed = false;
		private int length = -1;

		public List()
		{
		}

		public List(IntPtr list)
		{
			rawList = list;
		}

		~List()
		{
			Dispose(false);
		}

		public IntPtr Handle
		{
			get { return rawList; }
		}

		public bool Managed
		{
			set { managed = value; }
			get { return managed;  }
		}

		public IntPtr Raw
		{
			get { return rawList; }

			set
			{
				if (Managed && rawList != IntPtr.Zero)
					FreeList();

				rawList = value;
			}
		}

		[DllImport("libgalago.so")]
		static extern IntPtr galago_list_append(IntPtr list, IntPtr item);

		public void Append(IntPtr item)
		{
			rawList = galago_list_append(Handle, item);
		}

		[DllImport("libgalago.so")]
		static extern IntPtr galago_list_prepend(IntPtr list, IntPtr item);

		public void Prepend(IntPtr item)
		{
			rawList = galago_list_prepend(Handle, item);
		}

		[DllImport("libgalago.so")]
		static extern IntPtr galago_list_remove(IntPtr list, IntPtr item);

		public void Remove(IntPtr item)
		{
			rawList = galago_list_remove(Handle, item);
		}

		[DllImport("libgalago.so")]
		static extern int galago_list_get_count(IntPtr list);

		public int Count
		{
			get
			{
				if (length == -1)
					length = galago_list_get_count(Handle);

				return length;
			}
		}

		[DllImport("libgalago.so")]
		static extern IntPtr galago_list_get_next(IntPtr list);

		internal IntPtr Next(IntPtr cur)
		{
			return galago_list_get_next(cur);
		}

		[DllImport("libgalago.so")]
		static extern IntPtr galago_list_get_data(IntPtr list);

		internal IntPtr GetData(IntPtr cur)
		{
			return galago_list_get_data(cur);
		}

		public bool IsSynchronized
		{
			get { return false; }
		}

		public object SyncRoot
		{
			get { return null; }
		}

		public void CopyTo(Array array, int index)
		{
			object[] orig = new object[Count];
			int i = 0;

			foreach (object o in this)
				orig[i++] = o;

			orig.CopyTo(array, index);
		}

		internal object DataMarshal(IntPtr data)
		{
			object ret = null;

			// XXX
			if (true) // Object.IsObject(data))
				ret = Object.Wrap(data);

			return ret;
		}

		private class ListEnumerator
			: IEnumerator
		{
			private IntPtr current = IntPtr.Zero;
			private List list;

			public ListEnumerator(List list)
			{
				this.list = list;
			}

			public object Current
			{
				get
				{
					IntPtr data = list.GetData(current);
					object ret = null;
					ret = list.DataMarshal(data);
					return ret;
				}
			}

			public bool MoveNext()
			{
				if (current == IntPtr.Zero)
					current = list.rawList;
				else
					current = list.Next(current);

				return (current != IntPtr.Zero);
			}

			public void Reset()
			{
				current = IntPtr.Zero;
			}
		}

		// IEnumerator
		public IEnumerator GetEnumerator()
		{
			return new ListEnumerator(this);
		}

		public void Dispose()
		{
			Dispose(true);
			GC.SuppressFinalize(this);
		}

		protected virtual void Dispose(bool disposing)
		{
			if (!Managed)
				return;

			FreeList();
		}

		private void FreeList()
		{
			if (rawList != IntPtr.Zero)
				Free(rawList);

			rawList = IntPtr.Zero;
			length = -1;
		}

		[DllImport("libgalago.so")]
		static extern IntPtr galago_list_dup(IntPtr list);

		public object Clone()
		{
			return new List(galago_list_dup(Handle));
		}

		[DllImport("libgalago.so")]
		static extern void galago_list_destroy(IntPtr list);

		private void Free(IntPtr list)
		{
			if (list != IntPtr.Zero)
				galago_list_destroy(list);
		}
	}
}
