namespace Galago
{
	using System;
	using System.Collections;
	using System.Runtime.InteropServices;
	using System.Reflection;
	using System.Text;

	public class ObjectManager
	{
		private static Hashtable types = new Hashtable();

		[DllImport("libgalago.so")]
		static extern IntPtr galago_object_get_class(IntPtr raw);

		[DllImport("libgalago.so")]
		static extern IntPtr galago_class_get_name(IntPtr raw);

		public static Galago.Object CreateObject(IntPtr raw)
		{
			if (raw == IntPtr.Zero)
				return null;

			IntPtr rawClass = galago_object_get_class(raw);

			if (rawClass == IntPtr.Zero)
				return null;

			string typename = Marshal.PtrToStringAnsi(
				galago_class_get_name(rawClass));
			string mangled;

			if (types.ContainsKey(typename))
				mangled = (string)types[typename];
			else
				mangled = GetExpected(typename);

			Type t = LookupType(mangled);

			if (t == null)
				t = GetValidParentType(raw);

			if (t == null)
			{
				/*
				 * We should never reach this, since everything we wrap
				 * should be a GalagoObject, but ah well. Monkey see,
				 * monkey do.
				 */
				Console.WriteLine("*** Warning: No C# equivalent for class '" +
								  typename + "' found, returning null.");

				return null;
			}

			Galago.Object obj;

			try
			{
				obj = (Galago.Object)Activator.CreateInstance(t,
					BindingFlags.NonPublic | BindingFlags.Public |
					BindingFlags.Instance  | BindingFlags.CreateInstance,
					null, new object[]{raw}, null);
			}
			catch (MissingMethodException)
			{
				throw new Galago.MissingIntPtrCtorException(
					"Galago.Object subclass " + t + "must provide a " +
					"protected or public IntPtr ctor to support wrapping " +
					"of native object handles.");
			}

			return obj;
		}

		public static void RegisterType(string native_name, string managed_name,
										string assembly)
		{
			RegisterType(native_name, managed_name + "," + assembly);
		}

		public static void RegisterType(string native_name, string mangled)
		{
			types[native_name] = mangled;
		}

		static string GetExpected(string cname)
		{
			StringBuilder expected = new StringBuilder();
			string ns = "";
			bool needs_dot = true;

			for (int i = 0; i < cname.Length; i++)
			{
				if (needs_dot && i > 0 && Char.IsUpper(cname[i]))
				{
					ns = expected.ToString().ToLower();
					expected.Append('.');
					needs_dot = false;
				}

				expected.Append(cname[i]);
			}

			expected.AppendFormat(",{0}-sharp", ns);

			string expected_string = expected.ToString();
			RegisterType(cname, expected_string);

			return expected_string;
		}

		static Type LookupType(string mangled)
		{
			string[] toks = mangled.Split(',');

			if (toks.Length < 2)
			{
				throw new ArgumentException("mangled not properly formatted: " +
											mangled);
			}

			Assembly asm = Assembly.LoadWithPartialName(toks[1]);

			if (asm == null)
				return null;

			return asm.GetType(toks[0]);
		}

		static Type GetValidParentType(IntPtr raw)
		{
			IntPtr rawClass;
			string typename, mangled;
			Type t;

			// We will always end up at GalagoObject and will break this loop
			while (true)
			{
				rawClass = galago_object_get_class(raw);
				typename = Marshal.PtrToStringAnsi(
					galago_class_get_name(rawClass));

				if (types.ContainsKey(typename))
					mangled = (string)types[typename];
				else
					mangled = GetExpected(typename);

				t = LookupType(mangled);

				if (t != null)
					return t;
			}
		}
	}
}
