/*
 * Galaxium Messenger
 * Copyright (C) 2003-2007 Philippe Durand <draekz@gmail.com>
 * Copyright (C) 2007 Ben Motmans <ben.motmans@gmail.com>
 * 
 * License: GNU General Public License (GPL)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

using System;
using System.Collections.Generic;

using Gtk;
using Glade;

using Galaxium.Core;
using Galaxium.Gui;
using Galaxium.Gui.GtkGui;

namespace Galaxium.Client.GtkGui
{
	public class PreferencesDialog
	{
		[Widget ("PreferencesDialog")] Window _dialog;
		[Widget ("boxPanel")] VBox _boxPanel;
		[Widget ("boxList")] VBox _boxList;
		[Widget ("dialogImage")] Image _dialogImage;
		
		private Dictionary<string, PreferenceCategory> _categories;
		
		private PreferenceWidgetExtension _currentAddin;
		private IPreferenceWidget<Widget> _currentWidget;
		//private Dictionary<int, PreferencePlugin> _pageLookup;
		
		private PreferencePluginTreeView _treeView;
		
		private TreeCellDataFunc PixbufDataFunction = null;
		private TreeCellDataFunc TextDataFunction = null;
		
		public PreferencesDialog ()
		{
			XML gxml = new XML (GladeUtility.GetGladeResourceStream (typeof (PreferencesDialog).Assembly, "PreferencesDialog.glade"), null, null);
			gxml.Autoconnect (this);

			_dialog.Icon = IconUtility.GetIcon ("galaxium-preferences", IconSizes.Small);
			
			_dialogImage.FromPixbuf = IconUtility.GetIcon("galaxium-preferences", IconSizes.Other);
			_dialogImage.Show();
			
			_treeView = new PreferencePluginTreeView ();
			_treeView.Selection.Changed += new EventHandler(TreeViewSelectionChangedEvent);
			_treeView.Reorderable = false;
			
			ScrolledWindow scrolledWindow = new ScrolledWindow ();
			scrolledWindow.HscrollbarPolicy = PolicyType.Automatic;
			scrolledWindow.VscrollbarPolicy = PolicyType.Automatic;
			scrolledWindow.Add (_treeView);
			scrolledWindow.ShadowType = ShadowType.In;
			
			_boxList.PackStart (scrolledWindow, true, true, 0);
			
			_categories = new Dictionary<string, PreferenceCategory> ();
			
			foreach (PreferenceWidgetExtension node in AddinUtility.GetExtensionNodes("/Galaxium/Gui/Widgets/Preference"))
			{
				PreferenceCategory category = null;
					
				if (_categories.ContainsKey(node.Category))
					category = _categories[node.Category];
				else
				{
					category = new PreferenceCategory(node.Category);
					_categories.Add(category.Name, category);
				}
				
				category.AddPlugin(node);
			}
			
			foreach (PreferenceCategory category in _categories.Values)
				_treeView.AddNode(category);
			
			_treeView.ExpandAll ();
			_treeView.ShowAll();
			
			uint parent;
			if (_treeView.GetFirstChild (out parent)) {
				uint child;
				if (_treeView.GetFirstChild (parent, out child))
					_treeView.SelectNode (child);
			}
			
			_dialog.ShowAll ();
		}
 
		private void TreeViewSelectionChangedEvent (object sender, EventArgs args)
		{
			PreferenceWidgetExtension ext = null;
			
			uint node;
			if (_treeView.GetSelectedNode(out node))
			{
				object obj = _treeView.GetNodeObject(node);
				if (obj is PreferenceWidgetExtension)
					ext = ((PreferenceWidgetExtension)obj);
			}
			
			if (ext != null)
			{
				if (_currentWidget != null)
				{
					// As we remove the panel from view, shall we save settings here?
					_boxPanel.Remove (_currentWidget.NativeWidget);
				}
				
				IPreferenceWidget<Widget> widget = ext.CreateInstance() as IPreferenceWidget<Widget>;
				
				if (widget != null)
				{
					widget.Initialize();
					
					_currentAddin = ext;
					_currentWidget = widget;
					
					// Before we pack the panel in view, perhaps we load settings here?
					_boxPanel.PackStart (widget.NativeWidget, true, true, 0);
				}
				else
				{
					_currentAddin = null;
					_currentWidget = null;
				}
			}
		}
		
		private void OnCloseClicked (object sender, EventArgs args)
		{
			ConfigurationUtility.Save ();
			_dialog.Destroy ();
		}

		private void OnWindowDelete (object sender, DeleteEventArgs args)
		{
			ConfigurationUtility.Save ();
			_dialog.Destroy ();
			args.RetVal = false;
		}
	}
}