/*
 * Galaxium Messenger
 * Copyright (C) 2003-2007 Philippe Durand <draekz@gmail.com>
 * Copyright (C) 2007 Ben Motmans <ben.motmans@gmail.com>
 * Copyright (C) 2007 Paul Burton <paulburton89@gmail.com>
 * 
 * License: GNU General Public License (GPL)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

using System;
using System.Collections.Generic;

using Gtk;
using Notifications;

using Anculus.Core;
using Anculus.Gui;

using Galaxium.Core;
using Galaxium.Protocol;
using Galaxium.Protocol.Gui;
using Galaxium.Gui;
using Galaxium.Gui.GtkGui;

namespace Galaxium.Client.GtkGui
{
	public class GtkNotificationBackend : INotificationBackend, ITrayListener
	{
		private static GtkTrayWidget _trayWidget;
		private static bool _supress = false;
		private static List<IEntity> _statusEntities = new List<IEntity> ();
		
		public static bool Supress { get { return(_supress); } set { _supress = value; } }
		
		private uint _notifyTimer = 0;
		private Queue<Notification> _notifications = new Queue<Notification>();
		
		public void Initialize ()
		{
			_trayWidget = new GtkTrayWidget(this);
			
			// Listen to the activity utility's events.
			GtkActivityUtility.ActivityQueued += ActivityQueuedEvent;
			
			ShowIcon (IconUtility.GetIcon("galaxium-tray-offline"));
		}
		
		public void Unload ()
		{
			
		}

		private void ActivityQueuedEvent (object sender, EventArgs args)
		{
			UpdateFlashingIcon();
		}
		
		private void UpdateFlashingIcon ()
		{
			if (GtkActivityUtility.Pending)
			{
				IActivity activity = GtkActivityUtility.PeekQueueItem ();
				
				if(activity != null)
				{
					switch (activity.Type)
					{
						case ActivityTypes.Alert:
							ShowIcon (IconUtility.GetIcon("galaxium-tray-alert"));
							break;
						case ActivityTypes.Message:
							ShowIcon (IconUtility.GetIcon("galaxium-tray-unread"));
							break;
						case ActivityTypes.Transfer:
							ShowIcon (IconUtility.GetIcon("galaxium-tray-transfer"));
							break;
					}
					
					_trayWidget.Blinking = true;
				}
			}
			else
			{
				UpdateStatus (null);
				
				_trayWidget.Blinking = false;
			}
		}
		
		public void TrayActivated ()
		{
			if (GtkActivityUtility.Pending)
			{
				UpdateStatus (null);
				_trayWidget.Blinking = false;
				
				GtkActivityUtility.ProcessQueue ();
				
				UpdateFlashingIcon ();
			}
			else
				GalaxiumUtility.MainWindow.Visible = !GalaxiumUtility.MainWindow.Visible;
			
			if (!GtkActivityUtility.Pending)
			{
				UpdateStatus (null);
				
				_trayWidget.Blinking = false;
			}
		}
		
		public void TrayPopupActivated ()
		{
			Gtk.Menu menu = MenuUtility.CreateContextMenu("/Galaxium/Gui/Tray/Menu", new DefaultExtensionContext());
			
			_trayWidget.ShowMenu(menu);
		}
		
		public void ShowIcon (Gdk.Pixbuf pixbuf)
		{
			_trayWidget.Pixbuf = pixbuf;
		}
		
		private BasePresence EvaluateStatus ()
		{
			// Evaluating the status should show the lowest status before offline.
			IPresence statusPresence = Galaxium.Protocol.UnknownPresence.Instance;
			
			foreach (IEntity entity in _statusEntities)
				if (entity.Presence.BasePresence < statusPresence.BasePresence && entity.Presence.BasePresence != BasePresence.Offline)
					statusPresence = entity.Presence;
			
			return statusPresence.BasePresence;
		}
		
		public void UpdateStatus (IEntity entity)
		{
			if (entity != null)
				if (!_statusEntities.Contains(entity))
					_statusEntities.Add(entity);
			
			switch (EvaluateStatus ())
			{
				case BasePresence.Offline:
					ShowIcon(IconUtility.GetIcon("galaxium-tray-offline", IconSizes.Small));
					break;
				
				case BasePresence.Away:
					ShowIcon(IconUtility.GetIcon("galaxium-tray-away", IconSizes.Small));
					break;
				
				case BasePresence.Busy:
					ShowIcon(IconUtility.GetIcon("galaxium-tray-busy", IconSizes.Small));
					break;
				
				case BasePresence.Invisible:
					ShowIcon(IconUtility.GetIcon("galaxium-tray-invisible", IconSizes.Small));
					break;
				
				case BasePresence.Idle:
					ShowIcon(IconUtility.GetIcon("galaxium-tray-idle", IconSizes.Small));
					break;
				
				case BasePresence.Unknown:
					ShowIcon(IconUtility.GetIcon("galaxium-tray-offline", IconSizes.Small));
					break;
				
				default:
					ShowIcon(IconUtility.GetIcon("galaxium-tray-online", IconSizes.Small));
					break;
			}
		}
		
		public void ShowNotification (INotification notification)
		{
			ShowNotificationReal(notification, null);
		}
		
		public void ShowNotification (INotification notification, int displayTime)
		{
			ShowNotificationReal(notification, displayTime);
		}

		public void ShowNotificationReal (INotification notification, int? displayTime)
		{
			Notification notifyobject = new Notification(notification.Title, notification.Body);
			
			Gdk.Screen screen;
			Gdk.Rectangle area;
			_trayWidget.GetPosition(out screen, out area);
			
			notifyobject.SetGeometryHints(screen, area.X + (area.Width / 2), area.Y + (area.Height / 2));
			
			notifyobject.Urgency = Urgency.Low;
			try
			{
				notifyobject.Icon = new Gdk.Pixbuf (notification.Icon);
			}
			catch
			{
				Log.Warn ("Unable to generate pixbuf from byte array from notification object.");
			}
			
			if (displayTime.HasValue)
				notifyobject.Timeout = displayTime.Value;
			
			// If we are already showing a notification, there is really no sense in showing another
			// one right on top. We should wait until the timeout is over, and then display the next etc
			
			_notifications.Enqueue (notifyobject);
			
			ProcessNotifications();
		}
		
		private void ProcessNotifications ()
		{
			if (_notifications.Count < 1)
				return;
			
			if (_notifyTimer == 0)
			{
				Notification notification = _notifications.Dequeue ();
				notification.Show();
				
				// Timer is not on, lets start a new timer
				_notifyTimer = TimerUtility.RequestCallback (NotificationComplete, notification.Timeout);
			}
		}
		
		private void NotificationComplete ()
		{
			if (_notifyTimer != 0)
			{
				TimerUtility.RemoveCallback (_notifyTimer);
				_notifyTimer = 0;
			}
			
			ProcessNotifications ();
		}
	}
}