/*
 * Galaxium Messenger
 * Copyright (C) 2007 Ben Motmans <ben.motmans@gmail.com>
 * Copyright (C) 2007 Philippe Durand <draekz@gmail.com>
 * Copyright (C) 2007 Paul Burton <paulburton89@gmail.com>
 * 
 * License: GNU General Public License (GPL)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

using System;
using System.IO;

using Gdk;
using Gtk;
using Glade;

using Galaxium.Core;
using Galaxium.Gui;
using Galaxium.Gui.GtkGui;
using Galaxium.Protocol;

using Anculus.Core;

namespace Galaxium.Client.GtkGui
{
	public class SoundPreferenceWidget: IPreferenceWidget<Widget>
	{
		[Widget("vbSets")] VBox _vbSets;
		[Widget("btnPlay")] ToolButton _btnPlay;
		[Widget("tvSounds")] TreeView _tvSounds;
		[Widget("lblAuthor")] Label _lblAuthor;
		[Widget("lblDescription")] Label _lblDescription;
		
		Widget _nativeWidget;
		Widget _parentLayoutWidget;
		ImageComboBox<ISoundSet> _cbSets;
		ISoundSet _noneSet;
		ListStore _tvSoundsStore;
		
		public void Initialize ()
		{
			_nativeWidget = GladeUtility.ExtractWidget<Widget> (GladeUtility.GetGladeResourceStream (typeof (GtkPreferenceWidget).Assembly, "SoundPreferenceWidget.glade"), "widget", this);
			
			_tvSounds.AppendColumn("Event", new CellRendererText(), "text", 0);
			_tvSounds.AppendColumn("File", new CellRendererText(), "text", 1);
			
			_tvSounds.Columns[1].SetCellDataFunc(_tvSounds.Columns[1].CellRenderers[0], new TreeCellDataFunc (RenderSoundFilename));
 
			_tvSoundsStore = new ListStore(typeof(string), typeof(string));
			_tvSounds.Model = _tvSoundsStore;
			
			_tvSounds.Selection.Changed += tvSoundsSelectionChanged;
			
			_cbSets = new ImageComboBox<ISoundSet>(new ImageComboTextLookup<ISoundSet>(SetTextLookup), new ImageComboPixbufLookup<ISoundSet>(SetImageLookup));
			_vbSets.PackStart(_cbSets, false, true, 0);
			
			_noneSet = new AbstractSoundSet("None", string.Empty, string.Empty);
			_cbSets.Append(_noneSet);
			
			foreach (ISoundSet soundSet in SoundSetUtility.Sets)
				_cbSets.Append(soundSet);
			
			_cbSets.Changed += SetChanged;
			
			if (SoundSetUtility.ActiveSet != null)
				_cbSets.Select(SoundSetUtility.ActiveSet);
			else
				_cbSets.Select(_noneSet);
			
			_nativeWidget.ShowAll ();
		}
		
		string SetTextLookup (ISoundSet item)
		{
			return item.Name;
		}

		Gdk.Pixbuf SetImageLookup (ISoundSet item)
		{
			return IconUtility.GetIcon ("galaxium-preferences-themes", IconSizes.Small);
		}
		
		void SetChanged(object sender, EventArgs args)
		{
			ISoundSet soundSet = _cbSets.GetSelectedItem();
			
			SoundSetUtility.ActiveSet = soundSet == _noneSet ? null : soundSet;
			
			_lblAuthor.Text = soundSet.Creator;
			_lblDescription.Text = soundSet.Description;
			
			_tvSoundsStore.Clear();
			
			if (soundSet != _noneSet)
			{
				foreach (ISound sound in soundSet.Sounds)
					_tvSoundsStore.AppendValues(sound.Sound.ToString(), sound.Filename);
			}
		}
		
		void btnPlayClicked(object sender, EventArgs args)
		{
			TreeIter iter;
			_tvSounds.Selection.GetSelected(out iter);
			
			if (iter.Equals(TreeIter.Zero))
				return;
			
			string filename = _tvSoundsStore.GetValue(iter, 1) as string;

			SoundUtility.Play (filename);
		}
		
		private void RenderSoundFilename(TreeViewColumn column, CellRenderer cell, TreeModel model, TreeIter iter)
		{
			string filename = (string)model.GetValue(iter, 1);
			(cell as CellRendererText).Text = Path.GetFileName(filename);
		}
		
		void tvSoundsSelectionChanged(object sender, EventArgs args)
		{
			_btnPlay.Sensitive = true;
		}
		
		public Widget NativeWidget
		{
			get { return _nativeWidget; }
		}
	}
}
