/*
 * Galaxium Messenger
 * 
 * Copyright (C) 2008 Philippe Durand <draekz@gmail.com>
 * 
 * License: GNU General Public License (GPL)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

using System;
using System.Web;
using System.Collections.Generic;

using Gtk;
using Glade;

using Anculus.Core;
using Anculus.Gui;

using Galaxium.Core;
using Galaxium.Protocol;
using Galaxium.Protocol.Gui;
using Galaxium.Gui;

namespace Galaxium.Gui.GtkGui
{
	public abstract class BasicChatWidget : AbstractChatWidget
	{
		const int _snapSize = 5;
		
		protected static Dictionary<ISession, EventHandler> _fontChangedHandlers = new Dictionary<ISession, EventHandler> ();
		
		protected Label _status_label = null;
		protected Image _activityImage = null;
		protected Toolbar _entryToolbar = null;
		protected Label _personalLabel = null;
		protected Image _personalImage = null;
		protected ImageView _own_image = null;
		protected ImageView _contact_image = null;
		protected ScrolledWindow _messageEntryWindow = null;
		protected MessageDisplayWidget _messageDisplay;
		protected MessageEntryWidget _messageEntry;
		protected VBox _contactListBox;
		
		protected bool _historyComplete = false;
		protected List<ITextMessage> _incomingMessages = new List<ITextMessage> ();
		
		private IConfigurationSection _config = Configuration.Conversation.Section;
		
		public MessageEntryWidget MessageEntry { get { return _messageEntry; } }
		public MessageDisplayWidget MessageDisplay { get { return _messageDisplay; } }
		
		public BasicChatWidget(IContainerWindow<Widget> window, IConversation conversation) : base (window, conversation)
		{
			// Register and setup the font changed handler.
			if (!_fontChangedHandlers.ContainsKey (_conversation.Session))
				_fontChangedHandlers.Add (_conversation.Session, null);
			
			_fontChangedHandlers[_conversation.Session] += delegate { LoadFont (); };
			
			_messageDisplay = new MessageDisplayWidget (_conversation);
			
			_messageEntry = new MessageEntryWidget (_conversation);
			_messageEntry.DragReceived += OnDragReceived;
			
			_messageEntryWindow = new ScrolledWindow ();
			_messageEntryWindow.ShadowType = ShadowType.None;
			_messageEntryWindow.HscrollbarPolicy = PolicyType.Never;
			_messageEntryWindow.VscrollbarPolicy = PolicyType.Automatic;
			_messageEntryWindow.Add (_messageEntry);
			_messageEntryWindow.ShowAll ();
			
			_personalLabel = new Gtk.Label ();
			_personalLabel.Xalign = 0.0f;
			_personalLabel.UseMarkup = true;
			_personalLabel.Ellipsize = Pango.EllipsizeMode.End;
			
			_personalImage = new Image ();
			_personalImage.FromPixbuf = IconUtility.GetIcon ("galaxium-status-online", IconSizes.Small);
			
			_entryToolbar = new Gtk.Toolbar ();
			_entryToolbar.ToolbarStyle = ToolbarStyle.BothHoriz;
			_entryToolbar.Orientation = Orientation.Horizontal;
			_entryToolbar.IconSize = IconSize.SmallToolbar;
			
			_activityImage = new Gtk.Image ();
			
			_status_label = new Label ();
			_status_label.UseMarkup = true;
			_status_label.Xalign = 0.0f;
			
			_own_image = new ImageView ();
			_own_image.ButtonPressEvent += OwnImageButtonPressEvent;
			
			_contact_image = new ImageView ();
		}
		
		public override void Initialize ()
		{
			// You must call this before anything else.
			base.Initialize ();
			
			HBox personalBox = new HBox ();
			personalBox.PackStart (_personalImage, false, false, 0);
			personalBox.PackStart (_personalLabel, true, true, 0);
			personalBox.Spacing = 5;
			personalBox.BorderWidth = 1;
			
			SetChatWidget (ChatWidgetPositions.LeftDisplayPane, _messageDisplay);
			SetChatWidget (ChatWidgetPositions.MessageEntry, _messageEntryWindow);
			SetChatWidget (ChatWidgetPositions.EntryToolbar, _entryToolbar, false, false, 0);
			SetChatWidget (ChatWidgetPositions.Identification, personalBox, true, true, 0);
			SetChatWidget (ChatWidgetPositions.LeftStatus, _activityImage, false, true, 0);
			SetChatWidget (ChatWidgetPositions.CenterStatus, _status_label);
			
			// Because we have to remove the widget before destroying the window.
			_nativeWidget.Unrealized += delegate
			{
				RemoveChatWidget (ChatWidgetPositions.LeftDisplayPane);
			};
			
			Viewport own_viewport = new Viewport ();
			own_viewport.Add (_own_image);
			own_viewport.ShowAll ();
			own_viewport.ShadowType = ShadowType.In;
			SetChatWidget (ChatWidgetPositions.LeftInput, own_viewport, false, false, 0);
			
			_own_image.SwitchTo (GenerateOwnImage ());
			
			// Create your contact's image view for the right hand side of the input box.
			Viewport contact_viewport = new Viewport ();
			contact_viewport.Add (_contact_image);
			contact_viewport.ShowAll ();
			contact_viewport.ShadowType = ShadowType.In;
			
			SetChatWidget (ChatWidgetPositions.RightInput, contact_viewport, false, false, 0);
			
			// Initialize must be called in order for the user interface to work.
			_messageEntry.KeyPressEvent += KeyPressEvent;
			_messageEntry.KeyReleaseEvent += KeyReleaseEvent;
			
			_activityImage.FromPixbuf = IconUtility.GetIcon("galaxium-typing_blank");
			
			_vertical_pane.Position = 600;
			
			// We aren't ready until the GdkWindows for the widgets are created
			// otherwise WinkDisplay hits a null reference trying to find it's
			// position and size
			_nativeWidget.ExposeEvent += BecomeReady;
			
			LoadFont ();
			
			_messageEntry.GrabFocus ();
		}
		
		protected virtual void BecomeReady (object sender, EventArgs args)
		{
			//_conversation.ListenerReady = true;
			_nativeWidget.ExposeEvent -= BecomeReady;
			
			/*Log.Debug ("{0} {1} {2} {3} {4} {5} {6} {7} {8} {9} {10} {11} {12} {13} {14}",
			           _inputBox.HeightRequest, _leftInputBox.HeightRequest, _rightInputBox.HeightRequest,
			           _mainFrame.HeightRequest, _entryBox.HeightRequest, _entryToolbarBox.HeightRequest, _messageEntryBox.HeightRequest,
			           _ownImage.HeightRequest, _contactImage.HeightRequest, _messageEntry.HeightRequest, _entryToolbar.HeightRequest,
			           _ownImage.Children[0].HeightRequest, _contactImage.Children[0].HeightRequest,
			           (_ownImage.Children[0] as EventBox).Child.HeightRequest, (_contactImage.Children[0] as EventBox).Child.HeightRequest);*/
			
			if (Configuration.Logging.Section.GetBool (Configuration.Logging.EnableLogging.Name, Configuration.Logging.EnableLogging.Default))
			{
				if (EnableLogging)
					_messageDisplay.AddSystemMessage ("This conversation is being logged.");
				else
					_messageDisplay.AddSystemMessage ("This conversation is not being logged.");
			}
			else
				_messageDisplay.AddSystemMessage ("Logging is currently disabled in the preferences.");
		}
		
		public override void Focus ()
		{
			_messageEntry.GrabFocus();
		}
		
		public override void Update ()
		{
			_window.Toolbar.Visible = UseDefaultView ? _config.GetBool (Configuration.Conversation.ShowActivityToolbar.Name, Configuration.Conversation.ShowActivityToolbar.Default) : ShowActionToolbar;
			_entryToolbar.Visible = UseDefaultView ? _config.GetBool (Configuration.Conversation.ShowInputToolbar.Name, Configuration.Conversation.ShowInputToolbar.Default) : ShowInputToolbar;
			_leftInputBox.Visible = UseDefaultView ? _config.GetBool (Configuration.Conversation.ShowAccountImage.Name, Configuration.Conversation.ShowAccountImage.Default) : ShowAccountImage;
			_rightInputBox.Visible = UseDefaultView ? _config.GetBool (Configuration.Conversation.ShowContactImage.Name, Configuration.Conversation.ShowContactImage.Default) : ShowContactImage;
			_identificationSpacer.Visible = UseDefaultView ? _config.GetBool (Configuration.Conversation.ShowIdentification.Name, Configuration.Conversation.ShowIdentification.Default) : ShowPersonalMessage;
		}
		
		public abstract void LoadFont ();
		public abstract void SaveFont ();
		public abstract void SendFile (string filename);
		
		[GLib.ConnectBefore	()]
		protected virtual void KeyReleaseEvent (object sender, KeyReleaseEventArgs args)
		{
			
		}
		
		[GLib.ConnectBefore ()]
		protected virtual void KeyPressEvent (object sender, KeyPressEventArgs args)
		{
			
		}
		
		protected virtual void OwnImageButtonPressEvent (object sender, ButtonPressEventArgs args)
		{
			
		}
		
		private void OnDragReceived (object sender, DragReceivedEventArgs args)
		{
			if (args.UriList != null && args.UriList.Length > 0)
			{
				if (args.UriList.Length > 3)
				{
					string error = "You may only send a maximum of 3 files to an individual at any one time.";
					MessageDialog errordialog = new MessageDialog (null, DialogFlags.Modal, MessageType.Error, ButtonsType.Ok, error);
					errordialog.Run ();
					errordialog.Destroy ();
				}
				
				for (int i = 0; i < ((args.UriList.Length < 3) ? args.UriList.Length : 3); i++)
					SendFile (EncodingUtility.UrlDecode (args.UriList[i].Substring(7)));
			}
		}
		
		protected void SwitchContactImage (IContact contact)
		{
			IIconSize iconSize = IconSizes.Huge;
			
			Gdk.Pixbuf pixbuf = null;
			Gdk.PixbufAnimation anim = null;
			
			try
			{
				if (contact.DisplayImage != null && !contact.SupressImage)
				{
					if ((contact.DisplayImage.ImageBuffer != null) && (contact.DisplayImage.ImageBuffer.Length > 0))
					{
						Gdk.PixbufLoader loader = new Gdk.PixbufLoader (contact.DisplayImage.ImageBuffer);
						
						pixbuf = PixbufUtility.GetScaledPixbuf (loader.Pixbuf, 96, 96);
						
						if (!loader.Animation.IsStaticImage)
							anim = loader.Animation;
					}
				}
			}
			catch { }
			
			if (pixbuf == null)
				pixbuf = PixbufUtility.GetScaledPixbuf (IconUtility.GetIcon ("galaxium-displayimage", iconSize), 96, 96);
			
			_contact_image.FadeTo (pixbuf, delegate
			{
				if (anim != null)
					_contact_image.SwitchTo (anim);
			});
		}
		
		protected Gdk.Pixbuf GenerateOwnImage()
		{
			IIconSize iconSize = IconSizes.Huge;
			
			Gdk.Pixbuf pixbuf = null;
			
			if (Conversation.Session.Account.DisplayImage != null)
				if(Conversation.Session.Account.DisplayImage.ImageBuffer != null)
					if (Conversation.Session.Account.DisplayImage.ImageBuffer.Length > 0)
						pixbuf = PixbufUtility.GetScaledPixbuf (new Gdk.Pixbuf(Conversation.Session.Account.DisplayImage.ImageBuffer), 96, 96);
			
			if (pixbuf == null)
				pixbuf = PixbufUtility.GetScaledPixbuf (IconUtility.GetIcon ("galaxium-displayimage", iconSize), 96, 96);
			
			return pixbuf;
		}
	}
}
