# -*- Mode: Python; coding: utf-8; indent-tabs-mode: nil; tab-width: 4 -*-
# Copyright 2013 Canonical
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.

from time import sleep


class GalleryUtils(object):
    """An emulator class that makes it easy to interact with
       general components of the gallery app."""

    retry_delay = 0.2

    def __init__(self, app):
        self.app = app

    def select_single_retry(self, object_type, **kwargs):
        """Returns the item that is searched for with app.select_single
        In case of the item was not found (not created yet) a second attempt is
        taken 1 second later."""
        item = self.app.select_single(object_type, **kwargs)
        tries = 10
        while item is None and tries > 0:
            sleep(self.retry_delay)
            item = self.app.select_single(object_type, **kwargs)
            tries = tries - 1
        return item

    def select_many_retry(self, object_type, **kwargs):
        """Returns the item that is searched for with app.select_many
        In case of no item was not found (not created yet) a second attempt is
        taken 1 second later"""
        items = self.app.select_many(object_type, **kwargs)
        tries = 10
        while len(items) < 1 and tries > 0:
            sleep(self.retry_delay)
            items = self.app.select_many(object_type, **kwargs)
            tries = tries - 1
        return items

    def get_qml_view(self):
        """Get the main QML view"""
        return self.app.select_single("QQuickView")

    def get_main_photo_viewer_loader(self):
        """Returns the loader item for the PhotoViewer."""
        return self.app.select_single("QQuickLoader",
                                      objectName="photoViewerLoader")

    def get_main_photo_viewer(self):
        """Returns the MediaListView."""
        return self.select_single_retry("MediaListView",
                                        objectName="mediaListView")

    def get_albums_viewer_loader(self):
        """Returns the loader item for the AlbumsOverview."""
        return self.app.select_single("QQuickLoader",
                                      objectName="albumsCheckerboardLoader")

    def get_delete_dialog(self):
        """Returns the delete dialog in the events view."""
        return self.select_single_retry("DeleteDialog",
                                        objectName="deleteDialog")

    def get_delete_dialog_delete_button(self):
        """Returns the delete button of the delete popover."""
        return self.app.select_single("Button", objectName="deleteDialogYes",
                                      visible=True)

    def get_delete_dialog_cancel_button(self):
        """Returns the cancel button of the delete popover."""
        return self.app.select_single("Button", objectName="deleteDialogNo",
                                      visible=True)

    def get_first_event(self):
        """Returns the first event in the event view"""
        return self.app.select_single("OrganicMediaList",
                                      objectName="organicEventItem0")

    def number_of_events(self):
        """Returns the number of events in the event view (might differ to the
        total number in the model, because of the listview"""
        return len(self.app.select_many("OrganicMediaList"))

    def number_of_photos_in_events(self):
        """Returns the number of events"""
        photo_delegates = self.app.select_many("QQuickItem",
                                               objectName="eventPhoto")
        return len(photo_delegates)

    def get_first_image_in_event_view(self):
        """Returns the first photo of the gallery."""
        event = self.get_first_event()
        return event.select_many("OrganicItemInteraction")[1]

    def get_all_albums(self):
        """Returns all albums in the albums view"""
        albums = self.select_many_retry("CheckerboardDelegate",
                                        objectName="checkerboardDelegate")
        return albums

    def get_first_album(self):
        """Returns the first album in the albums view"""
        albums = self.select_many_retry("CheckerboardDelegate",
                                        objectName="checkerboardDelegate")
        return albums[-1]

    def get_edit_album_button(self):
        """Returns the edit album button in the album popover"""
        return self.app.select_single("Standard",
                                      objectName="editAlbumListItem")

    def get_cover_menu_item(self, idx):
        """Returns the item of the cover menu with index idx"""
        return self.app.select_many("Standard",
                                    objectName="albumCoverMenuItem")[idx]
