/******************************************
 *
 * $GAMGI/src/chem/gamgi_chem_property.c
 *
 * Copyright (C) 2008 Carlos Pereira
 *
 * Distributed under the terms of the GNU
 * General Public License: $GAMGI/LICENSE
 *
 */

#include "gamgi_engine.h"
#include "gamgi_chem.h"

/**********************************************************************
 * This list contains all naturally occurring isotopes, with their    *
 * relative abundances, plus all the isotopes with a half-life longer *
 * than one year (all elements until Cf except At, Rn, Fr), or, when  *
 * these do not exist, one day (Rn, plus Es, Fm, Md), one hour (At,   *
 * Lr, Rf, Db), one minute (Fr, No, Sg) or one second (Bh, Hs, Mt,    *
 * Ds, Rg). Some isotopes have both a natural abundance and a half    *
 * life decay, necessarily very long. Some of these isotopes          *
 * correspond to excited states (Rh, Ag, Sn, Ta, Re, Ir, Bi, Am).     *
 *                                                                    *
 * From: http://www.wikipedia.org/, by Carlos Pereira                 *
 **********************************************************************/

const int gamgi_chem_property_mass_driver[GAMGI_CHEM_ATOM_MAX + 1] = {

/***************************************
 * number of positions in double array *
 ***************************************/

GAMGI_CHEM_MASS_ARRAY,

/****************************************************
 * starting offset for each element in double array *
 ****************************************************/

0, 8, 13, 18, 24, 29, 37, 42, 49, 52, 59, 65, 72, 78, 88, 91, 100, 108, 
121, 130, 152, 155, 169, 176, 187, 193, 210, 216, 235, 240, 251, 256, 
269, 272, 290, 295, 316, 323, 335, 338, 356, 368, 390, 400, 417, 426, 
444, 452, 479, 486, 513, 521, 546, 552, 573, 585, 604, 614, 629, 632, 
647, 657, 678, 689, 708, 717, 737, 743, 760, 766, 785, 798, 819, 829, 
850, 860, 884, 892, 912, 915, 935, 943, 962, 976, 983, 1000, 1005, 1033, 
1040, 1045, 1059, 1064, 1086, 1093, 1115, 1125, 1147, 1154, 1167, 1180, 
1187, 1194, 1204, 1208, 1212, 1216, 1220, 1233, 1240, 1244, 1251,
};

const double gamgi_chem_property_mass[GAMGI_CHEM_MASS_ARRAY]= {

/*******
 * H 1 *
 *******/

1.00795,
1.007825, 99.9885,
2.014102, 0.0115,
3.016049, 12.32, GAMGI_CHEM_MASS_YEAR,

/********
 * He 2 *
 ********/

4.002602,
3.016029, 0.000137,
4.002603, 99.999863,

/********
 * 3 Li *
 ********/

6.941,
6.015123, 7.59,
7.016005, 92.41,
     
/********
 * 4 Be *
 ********/

9.012182,
9.012182, 100,
10.013534, 1.51E6, GAMGI_CHEM_MASS_YEAR,
        
/*******
 * 5 B *
 *******/

10.812,
10.012937, 19.9,
11.009305, 80.1,   
      
/*******
 * 6 C *
 *******/

12.0107,
12.000000, 98.93,
13.003355, 1.07,
14.003242, 5.73E3, GAMGI_CHEM_MASS_YEAR,
       
/*******
 * 7 N *
 *******/

14.0067,
14.003074, 99.632,
15.000109, 0.368,
                                                             
/*******
 * 8 O *
 *******/

15.9994,
15.994915, 99.757,
16.999132, 0.038,
17.999161, 0.205,

/*******
 * 9 F *
 *******/

18.998403,
18.998403, 100,
         
/*********
 * 10 Ne *
 *********/

20.1798,
19.992440, 90.48,
20.993847, 0.27,
21.991385, 9.25,

/*********
 * 11 Na *
 *********/

22.989769,
21.994436, 2.6027, GAMGI_CHEM_MASS_YEAR,
22.989769, 100,
        
/*********
 * 12 Mg *
 *********/

24.3051,
23.985042, 78.99,
24.985837, 10.00,
25.982593, 11.01,
      
/*********
 * 13 Al *
 *********/

26.981539,
25.986892, 7.17E5, GAMGI_CHEM_MASS_YEAR,
26.981539, 100,

/*********
 * 14 Si *
 *********/

28.0855,
27.976927, 92.2296,
28.976495, 4.6832,
29.973770, 3.0872,
31.974148, 170, GAMGI_CHEM_MASS_YEAR,

/********
 * 15 P *
 ********/

30.973762,
30.973762, 100,

/********
 * 16 S *
 ********/

32.065,
31.972071, 94.93,
32.971459, 0.76,
33.967867, 4.29,
35.967081, 0.02,

/*********
 * 17 Cl *
 *********/

35.453,
34.968853, 75.78,
35.968307, 3.01E5, GAMGI_CHEM_MASS_YEAR,
36.965903, 24.22,

/*********
 * 18 Ar *
 *********/

39.948,
35.967545, 0.3365,
37.962732, 0.0632,
38.964314, 269, GAMGI_CHEM_MASS_YEAR,
39.962383, 99.6003,
41.963047, 32.9, GAMGI_CHEM_MASS_YEAR,

/********
 * 19 K *
 ********/

39.0983,
38.963707, 93.2581,
39.963998, 0.0117, 1.277E9, GAMGI_CHEM_MASS_YEAR,
40.961826, 6.7302,

/*********
 * 20 Ca *
 *********/

40.078,
39.962591, 96.941, 5.9E21, GAMGI_CHEM_MASS_YEAR,
40.962278, 1.03E5, GAMGI_CHEM_MASS_YEAR,
41.958618, 0.647,
42.958767, 0.135,
43.955482, 2.086,
45.953693, 0.004, 2.5E15, GAMGI_CHEM_MASS_YEAR,
47.952534, 0.187, 4.3E19, GAMGI_CHEM_MASS_YEAR,

/*********
 * 21 Sc *
 *********/

44.955912,
44.955912, 100,

/*********
 * 22 Ti *
 *********/

47.867,
43.959690, 60.0, GAMGI_CHEM_MASS_YEAR,
45.952632, 8.25,
46.951763, 7.44,
47.947946, 73.72,
48.947870, 5.41,
49.944791, 5.18,

/********
 * 23 V *
 ********/

50.9415,
49.947159, 0.250, 1.5E17, GAMGI_CHEM_MASS_YEAR,
50.943960, 99.750,

/*********
 * 24 Cr *
 *********/

51.9961,
49.946044, 4.345, 1.8E17, GAMGI_CHEM_MASS_YEAR,
51.940508, 83.789,
52.940649, 9.501,
53.938880, 2.365,

/*********
 * 25 Mn *
 *********/

54.938046,
52.941290, 3.74E6, GAMGI_CHEM_MASS_YEAR,
54.938045, 100,

/*********
 * 26 Fe *
 *********/

55.845,
53.939611, 5.845, 3.1E22, GAMGI_CHEM_MASS_YEAR,
54.938293, 2.737, GAMGI_CHEM_MASS_YEAR,
55.934938, 91.754,
56.935394, 2.119,
57.933276, 0.282,
59.934072, 1.5E6, GAMGI_CHEM_MASS_YEAR,
      
/*********
 * 27 Co *
 *********/

58.933195,
58.933195, 100,
59.933817, 5.2714, GAMGI_CHEM_MASS_YEAR,
       
/*********
 * 28 Ni *
 *********/

58.6934,
57.935343, 68.0769, 7.00E20, GAMGI_CHEM_MASS_YEAR,
58.934347, 7.6E4, GAMGI_CHEM_MASS_YEAR,
59.930786, 26.2231,
60.931056, 1.1399,
61.928345, 3.6345,
62.929669, 100.1, GAMGI_CHEM_MASS_YEAR,
63.927966, 0.9256,
                                                  
/*********
 * 29 Cu *
 *********/

63.546,
62.929598, 69.17,
64.927790, 30.83,

/*********
 * 30 Zn *
 *********/

65.409,
63.929142, 48.63,
65.926033, 27.90,
66.927127, 4.10,
67.924844, 18.75,
69.925319, 0.62,
     
/*********
 * 31 Ga *
 *********/

69.723,
68.925574, 60.108,
70.924701, 39.892,
      
/*********
 * 32 Ge *
 *********/

72.64,
69.924247, 20.84,
71.922076, 27.54,  
72.923459, 7.73,
73.921178, 36.28,
75.921403, 7.61, 1.78E21, GAMGI_CHEM_MASS_YEAR,
     
/*********
 * 33 As *
 *********/

74.921597,
74.921597, 100,
       
/*********
 * 34 Se *
 *********/

78.96,
73.922476, 0.89,
75.919214, 9.37,
76.919914, 7.63,
77.917309, 23.77,
78.918499, 2.95E5, GAMGI_CHEM_MASS_YEAR,
79.916521, 49.61,
81.916699, 8.73, 1.08E20, GAMGI_CHEM_MASS_YEAR,
     
/*********
 * 35 Br *
 *********/

79.904,
78.918337, 50.69,
80.916291, 49.31,
      
/*********
 * 36 Kr *
 *********/

83.798,
77.920365, 0.35,   2.3E20, GAMGI_CHEM_MASS_YEAR,
79.916379, 2.28,
80.916592, 2.29E5, GAMGI_CHEM_MASS_YEAR,
81.913484, 11.58,
82.914136, 11.49,
83.911507, 57.00,
84.912527, 10.776, GAMGI_CHEM_MASS_YEAR,
85.910611, 17.30,
    
/*********
 * 37 Rb *
 *********/

85.4678,
84.911790, 72.17,
86.909181, 27.83, 4.88E10, GAMGI_CHEM_MASS_YEAR,

/*********
 * 38 Sr *
 *********/

87.62,
83.913425, 0.56,
85.909260, 9.86,
86.908877, 7.00,
87.905612, 82.58,
89.907738, 28.90, GAMGI_CHEM_MASS_YEAR,

/********
 * 39 Y *
 ********/

88.905848,
88.905848, 100,

/*********
 * 40 Zr *
 *********/

91.224,
89.904704, 51.45,
90.905646, 11.22,
91.905041, 17.15,
92.906476, 1.53E6, GAMGI_CHEM_MASS_YEAR,
93.906315, 17.38, 1.1E17, GAMGI_CHEM_MASS_YEAR,
95.908273, 2.80, 20E18, GAMGI_CHEM_MASS_YEAR,

/*********
 * 41 Nb *
 *********/

92.90638,
90.906996, 680, GAMGI_CHEM_MASS_YEAR,
91.907194, 3.47E7, GAMGI_CHEM_MASS_YEAR,
92.906378, 100,
93.907284, 2.03E4, GAMGI_CHEM_MASS_YEAR,

/*********
 * 42 Mo *
 *********/

95.94,
91.906811, 14.84, 1.90E20, GAMGI_CHEM_MASS_YEAR,
92.906813, 4.0E3, GAMGI_CHEM_MASS_YEAR,
93.905088, 9.25,
94.905842, 15.92,
95.904680, 16.68,
96.906022, 9.55,
97.905408, 24.13,
99.907478, 9.63, 7.8E18, GAMGI_CHEM_MASS_YEAR,

/*********
 * 43 Tc *
 *********/

97.907216,
96.906366, 4.21E6, GAMGI_CHEM_MASS_YEAR,
97.907216, 4.2E6, GAMGI_CHEM_MASS_YEAR,
98.906255, 2.111E5, GAMGI_CHEM_MASS_YEAR,

/*********
 * 44 Ru *
 *********/

101.07,
 95.907599, 5.54, 6.7E15, GAMGI_CHEM_MASS_YEAR,
 97.905288, 1.87,
 98.905939, 12.76,
 99.904220, 12.60,
100.905582, 17.06,
101.904349, 31.55,
103.905433, 18.62,

/*********
 * 45 Rh *
 *********/

102.905504,
100.906164, 3.3, GAMGI_CHEM_MASS_YEAR,
-101.906844, 3.742, GAMGI_CHEM_MASS_YEAR,
102.905504, 100,

/*********
 * 46 Pd *
 *********/

106.42,
101.905609, 1.02,
103.904036, 11.14,
104.905085, 22.33,
105.903486, 27.33,
106.905133, 6.5E6, GAMGI_CHEM_MASS_YEAR,
107.903892, 26.46,
109.905153, 11.72, 6.00E17, GAMGI_CHEM_MASS_YEAR,

/*********
 * 47 Ag *
 *********/

107.8682,
106.905098, 51.839,
-107.905957, 418, GAMGI_CHEM_MASS_YEAR,
108.904752, 48.161,

/*********
 * 48 Cd *
 *********/

112.412,
105.906460, 1.25, 4.10E20, GAMGI_CHEM_MASS_YEAR,
107.904185, 0.89, 4.10E17, GAMGI_CHEM_MASS_YEAR,
109.903002, 12.49,
110.904178, 12.80,
111.902758, 24.13,
112.904402, 12.22, 7.7E15, GAMGI_CHEM_MASS_YEAR,
113.903359, 28.73, 6.4E18, GAMGI_CHEM_MASS_YEAR,
115.904756, 7.49, 3.1E19, GAMGI_CHEM_MASS_YEAR,
       
/*********
 * 49 In *
 *********/

114.818,
112.904058, 4.29,
114.903879, 95.71, 4.41E14, GAMGI_CHEM_MASS_YEAR,
       
/*********
 * 50 Sn *
 *********/

118.710,
111.904819, 0.97,
113.902779, 0.66,
114.903342, 0.34,
115.901741, 14.54,
116.902952, 7.68,
117.901603, 24.22,
118.903308, 8.59,
119.902195, 32.58,
-120.904236, 44, GAMGI_CHEM_MASS_YEAR,
121.903439, 4.63,
123.905274, 5.79,
125.907653,  2.30E5, GAMGI_CHEM_MASS_YEAR,
      
/*********
 * 51 Sb *
 *********/

121.760,
120.903816, 57.21,
122.904214, 42.79,
124.905254, 2.75856, GAMGI_CHEM_MASS_YEAR,

      
/*********
 * 52 Te *
 *********/

127.60,
119.90402, 0.09, 2.2E16, GAMGI_CHEM_MASS_YEAR,
121.903044, 2.55,
122.904270, 0.89, 6.00E14, GAMGI_CHEM_MASS_YEAR,
123.902818, 4.74,
124.904431, 7.07,
125.903312, 18.84,
127.904463, 31.74,  2.2E24, GAMGI_CHEM_MASS_YEAR,
129.906224, 34.08, 7.90E20, GAMGI_CHEM_MASS_YEAR,

/********
 * 53 I *
 ********/

126.90447,
126.904473, 100,
128.904988, 1.57E7, GAMGI_CHEM_MASS_YEAR,
         
/*********
 * 54 Xe *
 *********/

131.294,
123.905893, 0.09, 4.8E16, GAMGI_CHEM_MASS_YEAR,
125.904274, 0.09,
127.903531, 1.92,
128.904780, 26.44,
129.903508, 4.08,
130.905082, 21.18,
131.904154, 26.89,
133.905395, 10.44,
135.907220, 8.87,
       
/*********
 * 55 Cs *
 *********/

132.905452,
132.905452, 100,
133.906718, 2.0652, GAMGI_CHEM_MASS_YEAR,
134.905977, 2.3E6, GAMGI_CHEM_MASS_YEAR,
136.907090, 30.1671, GAMGI_CHEM_MASS_YEAR,
         
/*********
 * 56 Ba *
 *********/

137.328,
129.906321, 0.106, 4.0E21, GAMGI_CHEM_MASS_YEAR,
131.905061, 0.101, 3.00E18, GAMGI_CHEM_MASS_YEAR,
133.904508, 2.417,
134.905689, 6.592,
135.904576, 7.854,
136.905827, 11.232,
137.905247, 71.698,

/*********
 * 57 La *
 *********/

138.90548,
136.906494, 6E4, GAMGI_CHEM_MASS_YEAR,
137.907112, 0.090, 1.02E11, GAMGI_CHEM_MASS_YEAR,
138.906353, 99.910,

/*********
 * 58 Ce *
 *********/

140.116,
135.907172, 0.185, 3.8E16, GAMGI_CHEM_MASS_YEAR,
137.905991, 0.251, 1.50E14, GAMGI_CHEM_MASS_YEAR,
139.905439, 88.450,
141.909244, 11.114, 5.0E16, GAMGI_CHEM_MASS_YEAR,

/*********
 * 59 Pr *
 *********/

140.907653,
140.907653, 100,

/*********
 * 60 Nd *
 *********/

144.242,
141.907723, 27.2,
142.909814, 12.2,
143.910087, 23.8,
144.912574, 8.3,
145.913117, 17.2,
147.916893, 5.7,
149.920891, 5.6,

/*********
 * 61 Pm *
 *********/

144.912749,
144.912749, 17.7, GAMGI_CHEM_MASS_YEAR,
145.914697, 5.53, GAMGI_CHEM_MASS_YEAR,
146.915139, 2.6234, GAMGI_CHEM_MASS_YEAR,

/*********
 * 62 Sm *
 *********/

150.36,
143.911999, 3.07,
145.913041, 1.03E8, GAMGI_CHEM_MASS_YEAR,
146.914898, 14.99,
147.914823, 11.24,
148.917185, 13.82,
149.917276, 7.38,
150.919932, 91, GAMGI_CHEM_MASS_YEAR,
151.919732, 26.75,
153.922209, 22.75,

/*********
 * 63 Eu *
 *********/

151.964,
149.919702, 36.9, GAMGI_CHEM_MASS_YEAR,
150.919850, 47.81,
151.921745, 13.537, GAMGI_CHEM_MASS_YEAR,
152.921230, 52.19,

/*********
 * 64 Gd *
 *********/

157.25,
151.919791, 0.20, 1.08E14, GAMGI_CHEM_MASS_YEAR,
153.920866, 2.18,
154.922622, 14.80,
155.922123, 20.47,
156.923960, 15.65,
157.924104, 24.84,
159.927054, 21.86, 3.1E+19, GAMGI_CHEM_MASS_YEAR,
 
/*********
 * 65 Tb *
 *********/

158.925347,
156.924025, 71, GAMGI_CHEM_MASS_YEAR,
157.925413, 180, GAMGI_CHEM_MASS_YEAR,
158.925347, 100,

/*********
 * 66 Dy *
 *********/

162.500,
153.924425, 3.0E6, GAMGI_CHEM_MASS_YEAR,
155.924284, 0.06, 1E18, GAMGI_CHEM_MASS_YEAR,
157.924409, 0.10,
159.925198, 2.34,
160.926933, 18.91,
161.926798, 25.51,
162.928731, 24.90,
163.929175, 28.18,

/*********
 * 67 Ho *
 *********/

164.930322,
162.928734, 4.570E3, GAMGI_CHEM_MASS_YEAR,
164.930322, 100,

/*********
 * 68 Er *
 *********/

167.259,
161.928778, 0.14, 1.40E14, GAMGI_CHEM_MASS_YEAR,
163.929200, 1.61,
165.930293, 33.61,
166.932048, 22.93,
167.932370, 26.78,
169.935464, 14.93,  3.20E17, GAMGI_CHEM_MASS_YEAR,

/*********
 * 69 Tm *
 *********/

168.934213,
168.934213, 100,
170.936429, 1.92, GAMGI_CHEM_MASS_YEAR,

/*********
 * 70 Yb *
 *********/

173.04,
167.933898, 0.13, 1.30E14, GAMGI_CHEM_MASS_YEAR,
169.934762, 3.04,
170.936326, 14.28,
171.936382, 21.83,
172.938211, 16.13,
173.938862, 31.83,
175.942572, 12.76, 1.60E17, GAMGI_CHEM_MASS_YEAR,

/*********
 * 71 Lu *
 *********/

174.967,
172.938931, 1.37, GAMGI_CHEM_MASS_YEAR,
173.940338, 3.31, GAMGI_CHEM_MASS_YEAR,
174.940772, 97.41,
175.942686, 2.59, 3.85E10, GAMGI_CHEM_MASS_YEAR,

/*********
 * 72 Hf *
 *********/

178.49,
171.939448, 1.87, GAMGI_CHEM_MASS_YEAR,
173.940046, 0.16, 2.0E15, GAMGI_CHEM_MASS_YEAR,
175.941409, 5.26,
176.943221, 18.60,
177.943699, 27.28,
178.945816, 13.62,
179.946550, 35.08,
181.950554, 8.90E6, GAMGI_CHEM_MASS_YEAR,

/*********
 * 73 Ta *
 *********/

180.94788,
178.945930, 1.82, GAMGI_CHEM_MASS_YEAR,
-179.947465, 0.012, 1.2E15, GAMGI_CHEM_MASS_YEAR,
180.947996, 99.988,
        
/********
 * 74 W *
 ********/

183.84,
179.946704, 0.12, 7.00E17, GAMGI_CHEM_MASS_YEAR,
181.948204, 26.50, 7.0E19, GAMGI_CHEM_MASS_YEAR,
182.950223, 14.31, 8.0E19, GAMGI_CHEM_MASS_YEAR,
183.950931, 30.64, 1.80E20, GAMGI_CHEM_MASS_YEAR,
185.954364, 28.43, 4.1E18, GAMGI_CHEM_MASS_YEAR,
      
/*********
 * 75 Re *
 *********/

186.207,
184.952955, 37.40,
-185.954986, 2.0E5, GAMGI_CHEM_MASS_YEAR,
186.955753, 62.60, 4.12E10, GAMGI_CHEM_MASS_YEAR,

/*********
 * 76 Os *
 *********/

190.23,
183.952489, 0.02, 5.6E13, GAMGI_CHEM_MASS_YEAR,
185.953838, 1.59, 2.0E15, GAMGI_CHEM_MASS_YEAR,
186.955751, 1.96,
187.955838, 13.24,
188.958148, 16.15,
189.958447, 26.26,
191.961481, 40.78, 9.8E12, GAMGI_CHEM_MASS_YEAR,
193.965182, 6.0, GAMGI_CHEM_MASS_YEAR,

/*********
 * 77 Ir *
 *********/

192.217,
190.960594, 37.3,
-191.962605, 241, GAMGI_CHEM_MASS_YEAR,
192.962926, 62.7,

/*********
 * 78 Pt *
 *********/

195.084,
189.959933, 0.014, 6.5E11, GAMGI_CHEM_MASS_YEAR,
191.961038, 0.782,
192.962987, 50, GAMGI_CHEM_MASS_YEAR,
193.962680, 32.967,
194.964791, 33.832,
195.964952, 25.242,
197.967893, 7.163, 3.20E14, GAMGI_CHEM_MASS_YEAR,

/*********
 * 79 Au *
 *********/

196.966569,
196.966569, 100,

/*********
 * 80 Hg *
 *********/

200.59,
193.965439, 444, GAMGI_CHEM_MASS_YEAR,
195.965833, 0.15, 2.5E18, GAMGI_CHEM_MASS_YEAR,
197.966769, 9.97,
198.968280, 16.87,
199.968326, 23.10,
200.970302, 13.18,
201.970643, 29.86,
203.973494, 6.87,

/*********
 * 81 Tl *
 *********/

204.3833,
202.972344, 29.524,
203.973864, 3.78, GAMGI_CHEM_MASS_YEAR,
204.974428, 70.476,

/*********
 * 82 Pb *
 *********/

207.2,
203.973044, 1.4, 1.40E17, GAMGI_CHEM_MASS_YEAR,
204.974482, 1.53E7, GAMGI_CHEM_MASS_YEAR,
205.974465, 24.1,
206.975897, 22.1,
207.976652, 52.4, 2E19, GAMGI_CHEM_MASS_YEAR,
209.984189, 22.20, GAMGI_CHEM_MASS_YEAR,

/*********
 * 83 Bi *
 *********/

208.980399,
206.978471, 32.9, GAMGI_CHEM_MASS_YEAR,
207.979742, 3.68E5, GAMGI_CHEM_MASS_YEAR,
208.980399, 100, 1.9E19, GAMGI_CHEM_MASS_YEAR,
-209.984120, 3.04E6, GAMGI_CHEM_MASS_YEAR,

/*********
 * 84 Po *
 *********/

208.982430,
207.981246, 2.898, GAMGI_CHEM_MASS_YEAR,
208.982430, 102, GAMGI_CHEM_MASS_YEAR,

/*********
 * 85 At *
 *********/

209.987149,
206.985784, 1.80, GAMGI_CHEM_MASS_HOUR,
207.986590, 1.63, GAMGI_CHEM_MASS_HOUR,
208.986174, 5.42, GAMGI_CHEM_MASS_HOUR,
209.987149, 100, 8.1, GAMGI_CHEM_MASS_HOUR,
210.987496, 7.215, GAMGI_CHEM_MASS_HOUR,

/*********
 * 86 Rn *
 *********/

222.017578,
222.017578, 100, 3.8235, GAMGI_CHEM_MASS_DAY,

/*********
 * 87 Fr *
 *********/

223.019736,
209.996408, 3.19, GAMGI_CHEM_MASS_MINUTE,
210.995537, 3.10, GAMGI_CHEM_MASS_MINUTE,
211.996202, 20.1, GAMGI_CHEM_MASS_MINUTE,
221.014256, 4.9, GAMGI_CHEM_MASS_MINUTE,
222.017552, 14.2, GAMGI_CHEM_MASS_MINUTE,
223.019736, 22.01, GAMGI_CHEM_MASS_MINUTE,
224.02326, 3.33, GAMGI_CHEM_MASS_MINUTE,
225.02557, 4.0, GAMGI_CHEM_MASS_MINUTE,
227.03184, 2.47, GAMGI_CHEM_MASS_MINUTE,

/*********
 * 88 Ra *
 *********/

226.025410,
226.025410, 1.601E3, GAMGI_CHEM_MASS_YEAR,
228.031070, 5.75, GAMGI_CHEM_MASS_YEAR,

/*********
 * 89 Ac *
 *********/

227.027752,
227.027752, 100, 21.772, GAMGI_CHEM_MASS_YEAR,

/*********
 * 90 Th *
 *********/

232.038055,
228.028741, 1.9116, GAMGI_CHEM_MASS_YEAR,
229.031762, 7.34E3, GAMGI_CHEM_MASS_YEAR,
230.033139, 7.538E4, GAMGI_CHEM_MASS_YEAR,
232.038055, 100, 1.406E10, GAMGI_CHEM_MASS_YEAR,

/*********
 * 91 Pa *
 *********/

231.035884,
231.035884, 100, 3.276E4, GAMGI_CHEM_MASS_YEAR,

/********
 * 92 U *
 ********/

238.02891,
232.037156, 68.9, GAMGI_CHEM_MASS_YEAR,
233.039635, 1.592E5, GAMGI_CHEM_MASS_YEAR,
234.040952, 0.0055, 2.456E5, GAMGI_CHEM_MASS_YEAR,
235.043930, 0.7200, 7.04E8, GAMGI_CHEM_MASS_YEAR,
236.045568, 2.342E7, GAMGI_CHEM_MASS_YEAR,
238.050788, 99.2745, 4.468E9, GAMGI_CHEM_MASS_YEAR,

/*********
 * 93 Np *
 *********/

237.048173,
236.04658, 1.55E5, GAMGI_CHEM_MASS_YEAR,
237.048173, 2.145E6, GAMGI_CHEM_MASS_YEAR,

/*********
 * 94 Pu *
 *********/

244.064204,
236.046058, 2.859, GAMGI_CHEM_MASS_YEAR,
238.049560, 87.7, GAMGI_CHEM_MASS_YEAR,
239.052163, 2.411E4, GAMGI_CHEM_MASS_YEAR,
240.053814, 6.562E3, GAMGI_CHEM_MASS_YEAR,
241.056852, 14.291, GAMGI_CHEM_MASS_YEAR,
242.058743, 3.75E+5, GAMGI_CHEM_MASS_YEAR,
244.064205, 8.01E+7, GAMGI_CHEM_MASS_YEAR,

/*********
 * 95 Am *
 *********/

243.061381,
241.056829, 432.3, GAMGI_CHEM_MASS_YEAR,
-242.059549, 141, GAMGI_CHEM_MASS_YEAR,
243.061381, 7.37E3, GAMGI_CHEM_MASS_YEAR,

/*********
 * 96 Cm *
 *********/

247.070355,
243.061389, 29.1, GAMGI_CHEM_MASS_YEAR,
244.062753, 18.10, GAMGI_CHEM_MASS_YEAR,
245.065491, 8.5E3, GAMGI_CHEM_MASS_YEAR,
246.067224, 4.76E3, GAMGI_CHEM_MASS_YEAR,
247.070355, 1.57E7, GAMGI_CHEM_MASS_YEAR,
248.072350, 3.49E5, GAMGI_CHEM_MASS_YEAR,
250.078357, 9.0E3, GAMGI_CHEM_MASS_YEAR,

/*********
 * 97 Bk *
 *********/

247.070308,
247.070308, 1380, GAMGI_CHEM_MASS_YEAR,
248.07310, 9, GAMGI_CHEM_MASS_YEAR,

/*********
 * 98 Cf *
 *********/

251.079588,
249.074854, 351, GAMGI_CHEM_MASS_YEAR,
250.076406, 13.09, GAMGI_CHEM_MASS_YEAR,
251.079588, 900, GAMGI_CHEM_MASS_YEAR,
252.081627, 2.646, GAMGI_CHEM_MASS_YEAR,

/*********
 * 99 Es *
 *********/

252.08299,
252.08299, 471.7, GAMGI_CHEM_MASS_DAY,
253.084825, 20.47, GAMGI_CHEM_MASS_DAY,
254.088023, 275.8, GAMGI_CHEM_MASS_DAY,
255.090273, 39.8, GAMGI_CHEM_MASS_DAY,

/**********
 * 100 Fm *
 **********/

257.095106,
253.085185, 3.00, GAMGI_CHEM_MASS_DAY,
257.095106, 100.5, GAMGI_CHEM_MASS_DAY,

/**********
 * 101 Md *
 **********/

258.098432,
258.098432, 51.5, GAMGI_CHEM_MASS_DAY,
260.10365, 31.9, GAMGI_CHEM_MASS_DAY,

/**********
 * 102 No *
 **********/

259.10103,
253.09068, 1.62, GAMGI_CHEM_MASS_MINUTE,
255.093241, 3.1, GAMGI_CHEM_MASS_MINUTE,
259.10103, 59, GAMGI_CHEM_MASS_MINUTE,

/**********
 * 103 Lr *
 **********/

262.10963,
262.10963, 4, GAMGI_CHEM_MASS_HOUR,

/**********
 * 104 Rf *
 **********/

265.11670,
265.11670, 13, GAMGI_CHEM_MASS_HOUR,

/**********
 * 105 Db *
 **********/

268.12546,
268.12546, 32, GAMGI_CHEM_MASS_HOUR,

/**********
 * 106 Sg *
 **********/

271.13348,
271.13348, 2.4, GAMGI_CHEM_MASS_MINUTE,

/**********
 * 107 Bh *
 **********/

267.12765,
264.1246, 1.4, GAMGI_CHEM_MASS_SECOND,
266.12694,  5, GAMGI_CHEM_MASS_SECOND,
267.12765, 22, GAMGI_CHEM_MASS_SECOND,
272.13804, 10, GAMGI_CHEM_MASS_SECOND,

/**********
 * 108 Hs *
 **********/

269.13406,
269.13406, 27, GAMGI_CHEM_MASS_SECOND,
270.13465, 3.6, GAMGI_CHEM_MASS_SECOND,

/**********
 * 109 Mt *
 **********/

276.15117,
276.15117, 0.72, GAMGI_CHEM_MASS_SECOND,

/**********
 * 110 Ds *
 **********/

280.15981,
280.15981, 12, GAMGI_CHEM_MASS_SECOND,
281.16207, 9.6, GAMGI_CHEM_MASS_SECOND,

/**********
 * 111 Rg *
 **********/

280.16448,
280.16448, 3.6, GAMGI_CHEM_MASS_SECOND,
};

/************************************************************************
 * 1) Half distance between atoms in its element natural state,         *
 * (most from L.E. Sutton (Ed.), Table of interatomic distances         *
 * and configuration in molecules and ions, Supplement 1956-1959,       *
 * Special publication No. 18, Chemical Society, London, UK,            *
 * 1965.). Available up to Cf (98), except Pm, At, Rn, Fr.              *
 *                                                                      *
 * 2) Effective atomic (from J.C. Slater, J. Chem. Phys. 1964, 39,      *
 * 3199), empirically derived by comparison of bond lengths in over     *
 * 1200 bond types in ionic, metallic, and covalent crystals and        *
 * molecules. Available up to Am (95) except He, Ne, Kr, Xe, At,        *
 * Rn, Fr.                                                              *
 *                                                                      *
 * 3) Calculated atomic (from E. Clementi, D.L.Raimondi, and W.P.       *
 * Reinhardt, J. Chem. Phys. 1963, 38, 2686), obtained from SCF         *
 * ab-initio calculations. Available up to Rn (86) except La, Ce.       *
 *                                                                      *
 * 4) Effective covalent (including from R.T. Sanderson in Chemical     *
 * Periodicity, Reinhold, New York, USA, 1962.), empirically obtained   *
 * by comparing distances between single-bonded equal atoms. Available  *
 * for all elements up to La (57), plus Lu (71) to Bi (83) plus Rn.     *
 *                                                                      *
 * 5) Calculated covalent, (from Beatriz Cordero et al, in "Covalent    *
 * radii revisited", Dalton Trans., 2008), arguably more consistent     *
 * than the effective covalent radius. Available up to Cm (96). For C,  *
 * there are radius available for sp3, sp2 and sp hybridization. For    *
 * Mn, Fe, Co there are radius available for low (LS) and high (HS)     *
 * spin configurations.                                                 *
 *                                                                      *
 * 6) Van der Waals (mainly from A. Bondi, J. Phys. Chem., 1964, 68,    *
 * 441.), established from contact distances between non-bonding atoms  *
 * in touching molecules or atoms.                                      *
 *                                                                      *
 * From: http://www.webelements.com/, by Carlos Pereira, Luis Veiros    *
 ************************************************************************/

const double gamgi_chem_property_radius[GAMGI_CHEM_RADIUS_ARRAY]= {

/*******
 * 1 H *
 *******/

0.3706, 0.2500, 0.5300, 0.3700, 0.3100, 1.2000,

/********
 * 2 He *
 ********/

1.5000, 0.0000, 0.3100, 0.3200, 0.2800, 1.4000,

/********
 * 3 Li *
 ********/

1.5195, 1.4500, 1.6700, 1.3400, 1.2800, 1.8200,

/********
 * 4 Be *
 ********/

1.1130, 1.0500, 1.1200, 0.9000, 0.9600, 0.0000,

/*******
 * 5 B *
 *******/

0.7945, 0.8500, 0.8700, 0.8200, 0.8400, 0.0000,

/*******
 * 6 C *
 *******/

0.7130, 0.7000, 0.6700, 0.7700, 0.7600, 1.7000,

/*******
 * 7 N *
 *******/

0.5488, 0.6500, 0.5600, 0.7500, 0.7100, 0.5500,

/*******
 * 8 O *
 *******/

0.6037, 0.6000, 0.4800, 0.7300, 0.6600, 1.5200,

/*******
 * 9 F *
 *******/

0.7090, 0.5000, 0.4200, 0.7100, 0.5700, 0.4700,

/*********
 * 10 Ne *
 *********/

1.5650, 0.0000, 0.3800, 0.6900, 0.5800, 1.5400,

/*********
 * 11 Na *
 *********/

1.8580, 1.8000, 1.9000, 1.5400, 1.6600, 2.2700,

/*********
 * 12 Mg *
 *********/

1.5985, 1.5000, 1.4500, 1.3000, 1.4100, 1.7300,

/*********
 * 13 Al *
 *********/

1.4315, 1.2500, 1.1800, 1.1800, 1.2100, 0.0000,

/*********
 * 14 Si *
 *********/

1.1760, 1.1000, 1.1100, 1.1100, 1.1100, 2.1000,

/********
 * 15 P *
 ********/

1.1050, 1.0000, 0.9800, 1.0600, 1.0700, 1.8000,

/********
 * 16 S *
 ********/

1.0250, 1.0000, 0.8800, 1.0200, 1.0500, 1.8000,

/*********
 * 17 Cl *
 *********/

0.9455, 1.0000, 0.7900, 0.9900, 1.0200, 1.7500,

/*********
 * 18 Ar *
 *********/

1.8600, 0.0000, 0.7100, 0.9700, 1.0600, 1.8800,

/********
 * 19 K *
 ********/

2.2720, 2.2000, 2.4300, 1.9600, 2.0300, 2.7500,

/*********
 * 20 Ca *
 *********/

1.9735, 1.8000, 1.9400, 1.7400, 1.7600, 0.0000,

/*********
 * 21 Sc *
 *********/

1.6060, 1.6000, 1.8400, 1.4400, 1.7000, 0.0000,

/*********
 * 22 Ti *
 *********/

1.4480, 1.4000, 1.7600, 1.3600, 1.6000, 0.0000,

/********
 * 23 V *
 ********/

1.3110, 1.3500, 1.7100, 1.2500, 1.5300, 0.0000,

/*********
 * 24 Cr *
 *********/

1.2490, 1.4000, 1.6600, 1.2700, 1.3900, 0.0000,

/*********
 * 25 Mn *
 *********/

1.3655, 1.4000, 1.6100, 1.3900, 1.3900, 0.0000,

/*********
 * 26 Fe *
 *********/

1.2410, 1.4000, 1.5600, 1.2500, 1.3200, 0.0000,

/*********
 * 27 Co *
 *********/

1.2530, 1.3500, 1.5200, 1.2600, 1.2600, 0.0000,

/*********
 * 28 Ni *
 *********/

1.2460, 1.3500, 1.4900, 1.2100, 1.2400, 1.6300,

/*********
 * 29 Cu *
 *********/

1.2780, 1.3500, 1.4500, 1.3800, 1.3200, 1.4000,

/*********
 * 30 Zn *
 *********/

1.3325, 1.3500, 1.4200, 1.3100, 1.2200, 1.3900,

/*********
 * 31 Ga *
 *********/

1.2210, 1.3000, 1.3600, 1.2600, 1.2200, 1.8700,

/*********
 * 32 Ge *
 *********/

1.2250, 1.2500, 1.2500, 1.2200, 1.2000, 0.0000,

/*********
 * 33 As *
 *********/

1.2450, 1.1500, 1.1400, 1.1900, 1.1900, 1.8500,

/*********
 * 34 Se *
 *********/

1.1605, 1.1500, 1.0300, 1.1600, 1.2000, 1.9000,

/*********
 * 35 Br *
 *********/

1.1420, 1.1500, 0.9400, 1.1400, 1.2000, 1.8500,

/*********
 * 36 Kr *
 *********/

2.0200, 0.0000, 0.8800, 1.1000, 1.1600, 2.0200,

/*********
 * 37 Rb *
 *********/

2.4750, 2.3500, 2.6500, 2.1100, 2.2000, 0.0000,

/*********
 * 38 Sr *
 *********/

2.1515, 2.0000, 2.1900, 1.9200, 1.9500, 0.0000,

/********
 * 39 Y *
 ********/

1.7755, 1.8000, 2.1200, 1.6200, 1.9000, 0.0000,

/*********
 * 40 Zr *
 *********/

1.5895, 1.5500, 2.0600, 1.4800, 1.7500, 0.0000,

/*********
 * 41 Nb *
 *********/

1.4290, 1.4500, 1.9800, 1.3700, 1.6400, 0.0000,

/*********
 * 42 Mo *
 *********/

1.3625, 1.4500, 1.9000, 1.4500, 1.5400, 0.0000,

/*********
 * 43 Tc *
 *********/

1.3515, 1.3500, 1.8300, 1.5600, 1.4700, 0.0000,

/*********
 * 44 Ru *
 *********/

1.3250, 1.3000, 1.7800, 1.2600, 1.4600, 0.0000,

/*********
 * 45 Rh *
 *********/

1.3450, 1.3500, 1.7300, 1.3500, 1.4200, 0.0000,

/*********
 * 46 Pd *
 *********/

1.3755, 1.4000, 1.6900, 1.3100, 1.3900, 1.6300,

/*********
 * 47 Ag *
 *********/

1.4445, 1.6000, 1.6500, 1.5300, 1.4500, 1.7200,

/*********
 * 48 Cd *
 *********/

1.4895, 1.5500, 1.6100, 1.4800, 1.4400, 1.5800,

/*********
 * 49 In *
 *********/

1.6255, 1.5500, 1.5600, 1.4400, 1.4200, 1.9300,

/*********
 * 50 Sn *
 *********/

1.4050, 1.4500, 1.4500, 1.4100, 1.3900, 2.1700,

/*********
 * 51 Sb *
 *********/

1.4500, 1.4500, 1.3300, 1.3800, 1.3900, 0.0000,

/*********
 * 52 Te *
 *********/

1.4320, 1.4000, 1.2300, 1.3500, 1.3800, 2.0600,

/********
 * 53 I *
 ********/

1.3330, 1.4000, 1.1500, 1.3300, 1.3900, 1.9800,

/*********
 * 54 Xe *
 *********/

2.1950, 0.0000, 1.0800, 1.3000, 1.4000, 2.1600,

/*********
 * 55 Cs *
 *********/

2.6545, 2.6000, 2.9800, 2.2500, 2.4400, 0.0000,

/*********
 * 56 Ba *
 *********/

2.1735, 2.1500, 2.5300, 1.9800, 2.1500, 0.0000,

/*********
 * 57 La *
 *********/

1.8695, 1.9500, 0.0000, 1.6900, 2.0700, 0.0000,

/*********
 * 58 Ce *
 *********/

1.8250, 1.8500, 0.0000, 0.0000, 2.0400, 0.0000,

/*********
 * 59 Pr *
 *********/

1.8200, 1.8500, 2.4700, 0.0000, 2.0300, 0.0000,

/*********
 * 60 Nd *
 *********/

1.8140, 1.8500, 2.0600, 0.0000, 2.0100, 0.0000,

/*********
 * 61 Pm *
 *********/

0.0000, 1.8500, 2.0500, 0.0000, 1.9900, 0.0000,

/*********
 * 62 Sm *
 *********/

1.7895, 1.8500, 2.3800, 0.0000, 1.9800, 0.0000,

/*********
 * 63 Eu *
 *********/

1.9945, 1.8500, 2.3100, 0.0000, 1.9800, 0.0000,

/*********
 * 64 Gd *
 *********/

1.7865, 1.8000, 2.3300, 0.0000, 1.9600, 0.0000,

/*********
 * 65 Tb *
 *********/

1.7625, 1.7500, 2.2500, 0.0000, 1.9400, 0.0000,

/*********
 * 66 Dy *
 *********/

1.7515, 1.7500, 2.2800, 0.0000, 1.9200, 0.0000,

/*********
 * 67 Ho *
 *********/

1.7430, 1.7500, 2.2600, 0.0000, 1.9200, 0.0000,

/*********
 * 68 Er *
 *********/

1.7340, 1.7500, 2.2600, 0.0000, 1.8900, 0.0000,

/*********
 * 69 Tm *
 *********/

1.7235, 1.7500, 2.2200, 0.0000, 1.9000, 0.0000,

/*********
 * 70 Yb *
 *********/

1.9400, 1.7500, 2.2200, 0.0000, 1.8700, 0.0000,

/*********
 * 71 Lu *
 *********/

1.7175, 1.7500, 2.1700, 1.6000, 1.8700, 0.0000,

/*********
 * 72 Hf *
 *********/

1.5635, 1.5500, 2.0800, 1.5000, 1.7500, 0.0000,

/*********
 * 73 Ta *
 *********/

1.4300, 1.4500, 2.0000, 1.3800, 1.7000, 0.0000,

/********
 * 74 W *
 ********/

1.3705, 1.3500, 1.9300, 1.4600, 1.6200, 0.0000,

/*********
 * 75 Re *
 *********/

1.3705, 1.3500, 1.8800, 1.5900, 1.5100, 0.0000,

/*********
 * 76 Os *
 *********/

1.3375, 1.3000, 1.8500, 1.2800, 1.4400, 0.0000,

/*********
 * 77 Ir *
 *********/

1.3570, 1.3500, 1.8000, 1.3700, 1.4100, 0.0000,

/*********
 * 78 Pt *
 *********/

1.3875, 1.3500, 1.7700, 1.2800, 1.3600, 1.7500,

/*********
 * 79 Au *
 *********/

1.4420, 1.3500, 1.7400, 1.4400, 1.3600, 1.6600,

/*********
 * 80 Hg *
 *********/

1.5025, 1.5000, 1.7100, 1.4900, 1.3200, 1.5500,

/*********
 * 81 Tl *
 *********/

1.7040, 1.9000, 1.5600, 1.4800, 1.4500, 1.9600,

/*********
 * 82 Pb *
 *********/

1.7500, 1.8000, 1.5400, 1.4700, 1.4600, 2.0200,

/*********
 * 83 Bi *
 *********/

1.5450, 1.6000, 1.4300, 1.4600, 1.4800, 0.0000,

/*********
 * 84 Po *
 *********/

1.6725, 1.9000, 1.3500, 0.0000, 1.4000, 0.0000,

/*********
 * 85 At *
 *********/

0.0000, 0.0000, 1.2700, 0.0000, 1.5000, 0.0000,

/*********
 * 86 Rn *
 *********/

0.0000, 0.0000, 1.2000, 1.4500, 1.5000, 0.0000,

/*********
 * 87 Fr *
 *********/

0.0000, 0.0000, 0.0000, 0.0000, 2.6000, 0.0000,

/*********
 * 88 Ra *
 *********/

2.2290, 2.1500, 0.0000, 0.0000, 2.2100, 0.0000,

/*********
 * 89 Ac *
 *********/

1.8780, 1.9500, 0.0000, 0.0000, 2.1500, 0.0000,

/*********
 * 90 Th *
 *********/

1.7975, 1.8000, 0.0000, 0.0000, 2.0600, 0.0000,

/*********
 * 91 Pa *
 *********/

1.6060, 1.8000, 0.0000, 0.0000, 2.0000, 0.0000,

/********
 * 92 U *
 ********/

1.3850, 1.7500, 0.0000, 0.0000, 1.9600, 1.8600,

/*********
 * 93 Np *
 *********/

1.3100, 1.7500, 0.0000, 0.0000, 1.9000, 0.0000,

/*********
 * 94 Pu *
 *********/

1.5130, 1.7500, 0.0000, 0.0000, 1.8700, 0.0000,

/*********
 * 95 Am *
 *********/

1.7250, 1.7500, 0.0000, 0.0000, 1.8000, 0.0000,

/*********
 * 96 Cm *
 *********/

1.7385, 0.0000, 0.0000, 0.0000, 1.6900, 0.0000,

/*********
 * 97 Bk *
 *********/

1.6990, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000,

/*********
 * 98 Cf *
 *********/

1.6885, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000,

/*********
 * 99 Es *
 *********/

0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000,

/**********
 * 100 Fm *
 **********/

0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000,

/**********
 * 101 Md *
 **********/

0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000,

/**********
 * 102 No *
 **********/

0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000,

/**********
 * 103 Lr *
 **********/

0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000,

/**********
 * 104 Rf *
 **********/

0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000,

/**********
 * 105 Db *
 **********/

0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000,

/**********
 * 106 Sg *
 **********/

0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000,

/**********
 * 107 Bh *
 **********/

0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000,

/**********
 * 108 Hs *
 **********/

0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000,

/**********
 * 109 Mt *
 **********/

0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000,

/**********
 * 110 Ds *
 **********/

0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000,

/**********
 * 111 Rg *
 **********/

0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000,

/************************************************************************
 * special cases (the first value is         6  C  76 sp3  73 sp2 69 sp *
 * stored in the normal covalent calculated  25 Mn 139  LS 161  HS      *
 * positions, the remaining are stored here  26 Fe 132  LS 152  HS      *
 * and handled as special cases:             27 Co 126  LS 150  HS      *
 ************************************************************************/

0.7300, 0.6900, 1.6100, 1.5200, 1.5000,
};

/***************************************************************************
 * Ionic effective radius from R.D. Shannon, Acta Cryst. A32, 751, 
   (1976), empirically derived from about 1000 distances, taken mainly 
   from oxide and fluoride structures, plus a range of correlations. 
   These radius are a function of valence, coordination, mass (for H) 
   and low (LS) and high * (HS) electronic spin (for Cr, Mn, Fe, Co, 
   Ni). Available for all        *
   elements up to Cf (98) expect He, Ne, Ar, Kr, Rn.                       *
 *                                                                         *
 * To get the so-called ionic crystalline radius, suggested by Fumi        *
 * and Tosi and published also by Shannon in the same paper, just sum      *
   0.14 to the cation and subtract 0.14 to the anion, so the cation-anion  *
   distance remain unchanged. According to Shannon, it is felt that these  *
   crystal radii correspond more closely to the physical size of ions in   *
   a solid. However, they might less efective in predicting the cation     *
 * coordination by using Pauling's first rule.                             *
 ***************************************************************************/

const int gamgi_chem_property_ionic_driver[GAMGI_CHEM_IONIC_DRIVER_ARRAY]= {

/************************************
 * number of positions in int array *
 ************************************/

GAMGI_CHEM_IONIC_DRIVER_ARRAY,

/*************************************************
 * starting offset for each element in int array *
 *************************************************/

112, 118, 119, 124, 129, 134, 139, 147, 154, 162, 163, 172, 178, 183, 187,
194, 199, 207, 208, 217, 225, 229, 239, 252, 269, 296, 318, 334, 347, 359,
365, 370, 376, 382, 390, 400, 401, 411, 419, 425, 433, 444, 457, 465, 476,
483, 493, 508, 516, 521, 528, 531, 541, 550, 554, 562, 571, 579, 592, 600,
609, 614, 625, 637, 643, 652, 662, 668, 674, 682, 692, 697, 703, 712, 721,
731, 743, 750, 758, 766, 775, 784, 799, 806, 812, 815, 816, 819, 823, 826,
834, 844, 862, 876, 886, 897, 903, 909, 915, 916, 917, 918, 921, 922, 923,
924, 925, 926, 927, 928, 929,

/*******
 * H 1 *
 *******/

0,
+1,
1000,
2000,
2000, GAMGI_CHEM_IONIC_D,

/********
 * He 2 *
 ********/

-1,

/********
 * 3 Li *
 ********/

3,
+1,
4000,
6000,
8000,

/********
 * 4 Be *
 ********/

6,
+2,
3000,
4000,
6000,

/*******
 * 5 B *
 *******/

9,
+3,
3000,
4000,
6000,

/*******
 * 6 C *
 *******/

12,
+4,
3000,
4000,
6000,

/*******
 * 7 N *
 *******/

15,
-3,
4000,
+3,
6000,
+5,
3000,
6000,

/*******
 * 8 O *
 *******/

19,
-2,
2000,
3000,
4000,
6000,
8000,

/*******
 * 9 F *
 *******/
         
24,
-1,
2000,
3000,
4000,
6000,
+7,
6000,

/*********
 * 10 Ne *
 *********/

-1,

/*********
 * 11 Na *
 *********/

29,
+1,
4000,
5000,
6000,
7000,
8000,
9000,
12000,

/*********
 * 12 Mg *
 *********/

36,
+2,
4000,
5000,
6000,
8000,

/*********
 * 13 Al *
 *********/

40,
+3,
4000,
5000,
6000,

/*********
 * 14 Si *
 *********/

43,
+4,
4000,
6000,

/********
 * 15 P *
 ********/

45,
+3,
6000,
+5,
4000,
5000,
6000,

/********
 * 16 S *
 ********/

49,
+3,
4001,
5000,
6000,

/*********
 * 17 Cl *
 *********/

52,
-1,
6000,
+5,
3001,
+7,
4000,
6000,

/*********
 * 18 Ar *
 *********/

-1,

/********
 * 19 K *
 ********/

56,
+1,
4000, 
6000, 
7000, 
8000, 
9000, 
10000,
12000,

/*********
 * 20 Ca *
 *********/

63,
+2,
6000,
7000,
8000,
9000,
10000,
12000,

/*********
 * 21 Sc *
 *********/

69,
+3,
6000,
8000,

/*********
 * 22 Ti *
 *********/

71,
+2,
6000,
+3,
6000,
+4,
4000,
5000,
6000,
8000,

/********
 * 23 V *
 ********/

77,
+2,
6000,
+3,
6000,
+4,
5000,
6000,
8000,
+5,
4000,
5000,
6000,

/*********
 * 24 Cr *
 *********/

85,
+2,
6000, GAMGI_CHEM_IONIC_LS, GAMGI_CHEM_IONIC_HS,
+3,
6000,
+4,
4000,
6000,
+5,
4000,
6000,
8000,
+6,
4000,
6000,

/*********
 * 25 Mn *
 *********/

95,
+2,
4000, GAMGI_CHEM_IONIC_HS,
5000, GAMGI_CHEM_IONIC_HS,
6000, GAMGI_CHEM_IONIC_LS, GAMGI_CHEM_IONIC_HS,
7000, GAMGI_CHEM_IONIC_HS,
8000,
+3,
5000,
6000, GAMGI_CHEM_IONIC_LS, GAMGI_CHEM_IONIC_HS,
+4,
4000,
6000,
+5,
4000,
+6,
4000,
+7,
4000,
6000,

/*********
 * 26 Fe *
 *********/

110,
+2,
4000,
4002,
6000, GAMGI_CHEM_IONIC_LS, GAMGI_CHEM_IONIC_HS,
8000, GAMGI_CHEM_IONIC_HS,
+3,
4000, GAMGI_CHEM_IONIC_HS,
5000,
6000, GAMGI_CHEM_IONIC_LS, GAMGI_CHEM_IONIC_HS,
8000, GAMGI_CHEM_IONIC_HS,
+4,
6000,
+6,
4000,

/*********
 * 27 Co *
 *********/

122,
+2,
4000, GAMGI_CHEM_IONIC_HS,
5000,
6000, GAMGI_CHEM_IONIC_LS, GAMGI_CHEM_IONIC_HS,
8000,
+3,
6000, GAMGI_CHEM_IONIC_LS, GAMGI_CHEM_IONIC_HS,
+4,
4000,
6000,

/*********
 * 28 Ni *
 *********/

131,
+2,
4000,
4002,
5000,
6000,
+3,
6000, GAMGI_CHEM_IONIC_LS, GAMGI_CHEM_IONIC_HS,
+4,
6000, GAMGI_CHEM_IONIC_LS,

/*********
 * 29 Cu *
 *********/

138,
+1,
2000,
4000,
6000,
+2,
4000,
4002,
5000,
6000,
+3,
6000,

/*********
 * 30 Zn *
 *********/
     
146,
+2,
4000,
5000,
6000,
8000,

/*********
 * 31 Ga *
 *********/

150,
+3,
4000,
5000,
6000,

/*********
 * 32 Ge *
 *********/

153,
+2,
6000,
+4,
4000,
6000,

/*********
 * 33 As *
 *********/
 
156,
+3,
6000,
+5,
4000,
6000,

/*********
 * 34 Se *
 *********/
     
159,
-2,
6000,
+4,
6000,
+6,
4000,
6000,

/*********
 * 35 Br *
 *********/

163,
-1,
6000,
+3,
4002,
+5,
3001,
+7,
4000,
6000,

/*********
 * 36 Kr *
 *********/

-1,

/*********
 * 37 Rb *
 *********/

168,
+1,
6000,
7000,
8000,
9000,
10000,
11000,
12000,
14000,

/*********
 * 38 Sr *
 *********/

176,
+2,
6000,
7000,
8000,
9000,
10000,
12000,

/********
 * 39 Y *
 ********/

182,
+3,
6000,
7000,
8000,
9000,

/*********
 * 40 Zr *
 *********/

186,
+4,
4000,
5000,
6000,
7000,
8000,
9000,

/*********
 * 41 Nb *
 *********/

192,
+3,
6000,
+4,
6000,
8000,
+5,
4000,
6000,
7000,
8000,

/*********
 * 42 Mo *
 *********/

199,
+3,
6000,
+4,
6000,
+5,
4000,
6000,
+6,
4000,
5000,
6000,
7000,

/*********
 * 43 Tc *
 *********/

207,
+4,
6000,
+5,
6000,
+7,
4000,
6000,

/*********
 * 44 Ru *
 *********/

211,
+3,
6000,
+4,
6000,
+5,
6000,
+7,
4000,
+8,
4000,

/*********
 * 45 Rh *
 *********/

216,
+3,
6000,
+4,
6000,
+5,
6000,

/*********
 * 46 Pd *
 *********/

219,
+1,
2000,
+2,
4002,
6000,
+3,
6000,
+4,
6000,

/*********
 * 47 Ag *
 *********/

224,
+1,
2000,
4000,
4002,
5000,
6000,
7000,
8000,
+2,
4002,
6000,
+3,
4002,
6000,

/*********
 * 48 Cd *
 *********/

235,
+2,
4000,
5000,
6000,
7000,
8000,
12000,

/*********
 * 49 In *
 *********/

241,
+3,
4000,
6000,
8000,

/*********
 * 50 Sn *
 *********/

244,
+4,
4000,
5000,
6000,
7000,
8000,

/*********
 * 51 Sb *
 *********/
      
249,
+5,
6000,

/*********
 * 52 Te *
 *********/

250,
-2,
6000,
+4,
3000,
4000,
6000,
+6,
4000,
6000,

/********
 * 53 I *
 ********/

256,
-1,
6000,
+5,
3001,
6000,
+7,
4000,
6000,

/*********
 * 54 Xe *
 *********/

261,
+8,
4000,
6000,

/*********
 * 55 Cs *
 *********/
         
263,
+1,
6000,
8000,
9000,
10000,
11000,
12000,

/*********
 * 56 Ba *
 *********/

269,
+2,
6000,
7000,
8000,
9000,
10000,
11000,
12000,

/*********
 * 57 La *
 *********/

276,
+3,
6000,
7000,
8000,
9000,
10000,
12000,

/*********
 * 58 Ce *
 *********/

282,
+1,
6000,
7000,
8000,
9000,
10000,
12000,
+4,
6000,
8000,
10000,
12000,

/*********
 * 59 Pr *
 *********/

292,
+3,
6000,
8000,
9000,
+4,
6000,
8000,

/*********
 * 60 Nd *
 *********/

297,
+2,
8000,
9000,
+3,
6000,
8000,
9000,
12000,

/*********
 * 61 Pm *
 *********/

303,
+3,
6000,
8000,
9000,

/*********
 * 62 Sm *
 *********/

306,
+2,
7000,
8000,
9000,
+3,
6000,
7000,
8000,
9000,
12000,

/*********
 * 63 Eu *
 *********/

314,
+2,
6000,
7000,
8000,
9000,
10000,
+3,
6000,
7000,
8000,
9000,

/*********
 * 64 Gd *
 *********/
 
323,
+3,
6000,
7000,
8000,
9000,

/*********
 * 65 Tb *
 *********/

327,
+3,
6000,
7000,
8000,
9000,
+4,
6000,
8000,

/*********
 * 66 Dy *
 *********/

333,
+2,
6000,
7000,
8000,
+3,
6000,
7000,
8000,
9000,

/*********
 * 67 Ho *
 *********/

340,
+3,
6000,
8000,
9000,
10000,

/*********
 * 68 Er *
 *********/

344,
+3,
6000,
7000,
8000,
9000,

/*********
 * 69 Tm *
 *********/

348,
+2,
6000,
7000,
+3,
6000,
8000,
9000,

/*********
 * 70 Yb *
 *********/

353,
+2,
6000,
7000,
8000,
+3,
6000,
7000,
8000,
9000,

/*********
 * 71 Lu *
 *********/

360,
+3,
6000,
8000,
9000,

/*********
 * 72 Hf *
 *********/

363,
+4,
4000,
6000,
7000,
8000,

/*********
 * 73 Ta *
 *********/

367,
+3,
6000,
+4,
6000,
+5,
6000,
7000,
8000,

/********
 * 74 W *
 ********/
      
372,
+4,
6000,
+5,
6000,
+6,
4000,
5000,
6000,

/*********
 * 75 Re *
 *********/

377,
+4,
6000,
+5,
6000,
+6,
6000,
+7,
4000,
6000,

/*********
 * 76 Os *
 *********/

382,
+4,
6000,
+5,
6000,
+6,
5000,
6000,
+7,
6000,
+8,
4000,

/*********
 * 77 Ir *
 *********/

388,
+3,
6000,
+4,
6000,
+5,
6000,

/*********
 * 78 Pt *
 *********/

391,
+2,
4002,
6000,
+4,
6000,
+5,
6000,

/*********
 * 79 Au *
 *********/

395,
+1,
6000, 
+3,
4002,
6000,
+5,
6000,

/*********
 * 80 Hg *
 *********/

399,
+1,
3000,
6000,
+2,
2000,
4000,
6000,
8000,

/*********
 * 81 Tl *
 *********/

405,
+1,
6000,
8000,
12000,
+3,
4000,
6000,
8000,

/*********
 * 82 Pb *
 *********/

411,
+2,
4001,
6000,
7000,
8000,
9000,
10000,
11000,
12000,
+4,
4000,
5000,
6000,
8000,

/*********
 * 83 Bi *
 *********/

423,
+3,
5000,
6000,
8000,
+5,
6000,

/*********
 * 84 Po *
 *********/

427,
+4,
6000,
8000,
+6,
6000,

/*********
 * 85 At *
 *********/

430,
+7,
6000,

/*********
 * 86 Rn *
 *********/

-1,

/*********
 * 87 Fr *
 *********/

431,
+1,
6000,

/*********
 * 88 Ra *
 *********/

432,
+2,
8000,
12000,

/*********
 * 89 Ac *
 *********/

434,
+3,
6000,

/*********
 * 90 Th *
 *********/

435,
+4,
6000,
8000,
9000,
10000,
11000,
12000,

/*********
 * 91 Pa *
 *********/

441,
+3,
6000,
+4,
6000,
8000,
+5,
6000,
8000,
9000,

/********
 * 92 U *
 ********/

447,
+3,
6000,
+4,
6000,
7000,
8000,
9000,
12000,
+5,
6000,
7000,
+6,
2000,
4000,
6000,
7000,
8000,

/*********
 * 93 Np *
 *********/

460,
+2,
6000,
+3,
6000,
+4,
6000,
8000,
+5,
6000,
+6,
6000,
+7,
6000,

/*********
 * 94 Pu *
 *********/

467,
+3,
6000,
+4,
6000,
8000,
+5,
6000,
+6,
6000,

/*********
 * 95 Am *
 *********/

472,
+2,
7000,
8000,
9000,
+3,
6000,
8000,
+4,
6000,
8000,

/*********
 * 96 Cm *
 *********/

479,
+3,
6000,
+4,
6000,
8000,

/*********
 * 97 Bk *
 *********/

482,
+3,
6000,
+4,
6000,
8000,

/*********
 * 98 Cf *
 *********/

485,
+3,
6000,
+4,
6000,
8000,

/*********
 * 99 Es *
 *********/

-1,

/**********
 * 100 Fm *
 **********/

-1,

/**********
 * 101 Md *
 **********/

-1,

/**********
 * 102 No *
 **********/

488,
+2,
6000,

/**********
 * 103 Lr *
 **********/

-1,

/**********
 * 104 Rf *
 **********/

-1,

/**********
 * 105 Db *
 **********/

-1,

/**********
 * 106 Sg *
 **********/

-1,

/**********
 * 107 Bh *
 **********/

-1,

/**********
 * 108 Hs *
 **********/

-1,

/**********
 * 109 Mt *
 **********/

-1,

/**********
 * 110 Ds *
 **********/

-1,

/**********
 * 111 Rg *
 **********/

-1,

};

const double gamgi_chem_property_ionic[GAMGI_CHEM_IONIC_ARRAY]= {

/*******
 * H 1 *
 *******/

-0.38,
-0.18,
-0.10,

/********
 * He 2 *
 ********/

/********
 * 3 Li *
 ********/

0.590,
0.76,
0.92,
     
/********
 * 4 Be *
 ********/

0.16,
0.27,
0.45,
        
/*******
 * 5 B *
 *******/

0.01,
0.11,
0.27,
      
/*******
 * 6 C *
 *******/

-0.08,
0.15,
0.16,

/*******
 * 7 N *
 *******/

1.46,
0.16,
-0.104,
0.13,
                                                             
/*******
 * 8 O *
 *******/

1.35,
1.36,
1.38,
1.40,
1.42,

/*******
 * 9 F *
 *******/
         
1.285,
1.30,
1.31,
1.33,
0.08,

/*********
 * 10 Ne *
 *********/

/*********
 * 11 Na *
 *********/

0.99,
1.00,
1.02,
1.12,
1.18,
1.24,
1.39,
        
/*********
 * 12 Mg *
 *********/

0.57,
0.66,
0.720,
0.89,

/*********
 * 13 Al *
 *********/

0.39,
0.48,
0.535,

/*********
 * 14 Si *
 *********/

0.26,
0.400,

/********
 * 15 P *
 ********/

0.44,
0.17,
0.29,
0.38,

/********
 * 16 S *
 ********/

0.76,
0.80,
0.76,

/*********
 * 17 Cl *
 *********/

1.81,
0.12,
0.08,
0.27,

/*********
 * 18 Ar *
 *********/

/********
 * 19 K *
 ********/

1.37,
1.38,
1.46,
1.51,
1.55,
1.59,
1.64,

/*********
 * 20 Ca *
 *********/

1.00,
1.06,
1.12,
1.18,
1.23,
1.34,

/*********
 * 21 Sc *
 *********/

0.745,
0.870,

/*********
 * 22 Ti *
 *********/

0.86,
0.670,
0.42,
0.51,
0.605,
0.74,

/********
 * 23 V *
 ********/

0.79,
0.640,
0.53,
0.58,
0.72,
0.355,
0.46,
0.54,

/*********
 * 24 Cr *
 *********/

0.73, 
0.80,
0.615,
0.41,
0.55,
0.345,
0.49,
0.57,
0.26,
0.44,

/*********
 * 25 Mn *
 *********/

0.66,
0.75,
0.67,
0.830,
0.90,
0.96,
0.58,
0.58,
0.645,
0.39,
0.530,
0.33,
0.255,
0.25,
0.46,

/*********
 * 26 Fe *
 *********/

0.63,
0.64,
0.61,
0.780,
0.92,
0.49,
0.58,
0.55,
0.645,
0.78,
0.585,
0.25,

/*********
 * 27 Co *
 *********/

0.58,
0.67,
0.65,
0.745,
0.90,
0.545,
0.61,
0.40,
0.53,
       
/*********
 * 28 Ni *
 *********/

0.55,
0.49,
0.63,
0.690,
0.56,
0.60,
0.48,

/*********
 * 29 Cu *
 *********/

0.46,
0.60,
0.77,
0.57,
0.57,
0.65,
0.73,
0.54,

/*********
 * 30 Zn *
 *********/
     
0.60,
0.68,
0.740,
0.90,

/*********
 * 31 Ga *
 *********/

0.47,
0.55,
0.620,
      
/*********
 * 32 Ge *
 *********/

0.73,
0.390,
0.530,
     
/*********
 * 33 As *
 *********/
 
0.58,
0.335,
0.46,

/*********
 * 34 Se *
 *********/
     
1.98,
0.50,
0.28,
0.42,

/*********
 * 35 Br *
 *********/

1.96,
0.59,
0.31,
0.25,
0.39,
      
/*********
 * 36 Kr *
 *********/

/*********
 * 37 Rb *
 *********/

1.52,
1.56,
1.61,
1.63,
1.66,
1.69,
1.72,
1.83,

/*********
 * 38 Sr *
 *********/

1.18,
1.21,
1.26,
1.31,
1.36,
1.44,

/********
 * 39 Y *
 ********/

0.900,
0.96,
1.019,
1.075,

/*********
 * 40 Zr *
 *********/

0.59,
0.66,
0.72,
0.78,
0.84,
0.89,

/*********
 * 41 Nb *
 *********/

0.72,
0.68,
0.79,
0.48,
0.64,
0.69,
0.74,

/*********
 * 42 Mo *
 *********/

0.69,
0.650,
0.46,
0.61,
0.41,
0.50,
0.59,
0.73,

/*********
 * 43 Tc *
 *********/

0.645,
0.60,
0.37,
0.56,

/*********
 * 44 Ru *
 *********/

0.68,
0.620,
0.565,
0.38,
0.36,

/*********
 * 45 Rh *
 *********/

0.665,
0.60,
0.55,

/*********
 * 46 Pd *
 *********/

0.59,
0.64,
0.86,
0.76,
0.615,

/*********
 * 47 Ag *
 *********/

0.67,
1.00,
1.02,
1.09,
1.15,
1.22,
1.28,
0.79,
0.94,
0.67,
0.75,

/*********
 * 48 Cd *
 *********/

0.78,
0.87,
0.95,
1.03,
1.10,
1.31,
       
/*********
 * 49 In *
 *********/

0.62,
0.800,
0.92,
       
/*********
 * 50 Sn *
 *********/

0.55,
0.62,
0.690,
0.75,
0.81,

/*********
 * 51 Sb *
 *********/
      
0.60,

/*********
 * 52 Te *
 *********/

2.21,
0.52,
0.66,
0.97,
0.43,
0.56,

/********
 * 53 I *
 ********/

2.20,
0.44,
0.95,
0.42,
0.53,
         
/*********
 * 54 Xe *
 *********/

0.40,
0.48,

/*********
 * 55 Cs *
 *********/
         
1.67,
1.74,
1.78,
1.81,
1.85,
1.88,

/*********
 * 56 Ba *
 *********/

1.35,
1.38,
1.42,
1.47,
1.52,
1.57,
1.61,

/*********
 * 57 La *
 *********/

1.032,
1.10,
1.160,
1.216,
1.27,
1.36,

/*********
 * 58 Ce *
 *********/

1.01,
1.07,
1.143,
1.196,
1.25,
1.34,
0.87,
0.97,
1.07,
1.14,

/*********
 * 59 Pr *
 *********/

0.99,
1.126,
1.179,
0.85,
0.96,

/*********
 * 60 Nd *
 *********/

1.29,
1.35,
0.983,
1.109,
1.163,
1.27,

/*********
 * 61 Pm *
 *********/

0.97,
1.093,
1.144,

/*********
 * 62 Sm *
 *********/

1.22,
1.27,
1.32,
0.958,
1.02,
1.079,
1.132,
1.24,

/*********
 * 63 Eu *
 *********/

1.17,
1.20,
1.25,
1.30,
1.35,
0.947,
1.01,
1.066,
1.120,

/*********
 * 64 Gd *
 *********/
 
0.938,
1.00,
1.053,
1.107,

/*********
 * 65 Tb *
 *********/

0.923,
0.98,
1.040,
1.095,
0.76,
0.88,

/*********
 * 66 Dy *
 *********/

1.07,
1.13,
1.19,
0.912,
0.97,
1.027,
1.083,

/*********
 * 67 Ho *
 *********/

0.901,
1.015,
1.072,
1.12,

/*********
 * 68 Er *
 *********/

0.890,
0.945,
1.004,
1.062,

/*********
 * 69 Tm *
 *********/

1.03,
1.09,
0.880,
0.994,
1.052,

/*********
 * 70 Yb *
 *********/

1.02,
1.08,
1.14,
0.868,
0.925,
0.985,
1.042,

/*********
 * 71 Lu *
 *********/

0.861,
0.977,
1.032,

/*********
 * 72 Hf *
 *********/

0.58,
0.71,
0.76,
0.83,

/*********
 * 73 Ta *
 *********/

0.72,
0.68,
0.64,
0.69,
0.74,

/********
 * 74 W *
 ********/
      
0.66,
0.62,
0.42,
0.51,
0.60,

/*********
 * 75 Re *
 *********/

0.63,
0.58,
0.55,
0.38,
0.53,

/*********
 * 76 Os *
 *********/

0.630,
0.575,
0.49,
0.545,
0.525,
0.39,

/*********
 * 77 Ir *
 *********/

0.68,
0.625,
0.57,

/*********
 * 78 Pt *
 *********/

0.60,
0.80,
0.625,
0.57,

/*********
 * 79 Au *
 *********/

1.37,
0.68,
0.85,
0.57,

/*********
 * 80 Hg *
 *********/

0.97,
1.19,
0.69,
0.96,
1.02,
1.14,

/*********
 * 81 Tl *
 *********/

1.50,
1.59,
1.70,
0.75,
0.885,
0.98,

/*********
 * 82 Pb *
 *********/

0.98,
1.19,
1.23,
1.29,
1.35,
1.40,
1.45,
1.49,
0.65,
0.73,
0.775,
0.94,

/*********
 * 83 Bi *
 *********/

0.96,
1.03,
1.17,
0.76,

/*********
 * 84 Po *
 *********/

0.94,
1.08,
0.67,

/*********
 * 85 At *
 *********/

0.62,

/*********
 * 86 Rn *
 *********/

/*********
 * 87 Fr *
 *********/

1.80,

/*********
 * 88 Ra *
 *********/

1.48,
1.70,

/*********
 * 89 Ac *
 *********/

1.12,

/*********
 * 90 Th *
 *********/

0.94,
1.05,
1.09,
1.13,
1.18,
1.21,

/*********
 * 91 Pa *
 *********/

1.04,
0.90,
1.01,
0.78,
0.91,
0.95,

/********
 * 92 U *
 ********/

1.025,
0.89,
0.95,
1.00,
1.05,
1.17,
0.76,
0.84,
0.45,
0.52,
0.73,
0.81,
0.86,

/*********
 * 93 Np *
 *********/

1.10,
1.01,
0.87,
0.98,
0.75,
0.72,
0.71,

/*********
 * 94 Pu *
 *********/

1.00,
0.86,
0.96,
0.74,
0.71,

/*********
 * 95 Am *
 *********/

1.21,
1.26,
1.31,
0.975,
1.09,
0.85,
0.95,

/*********
 * 96 Cm *
 *********/

0.97,
0.85,
0.95,

/*********
 * 97 Bk *
 *********/

0.96,
0.83,
0.93,

/*********
 * 98 Cf *
 *********/

0.95,
0.821,
0.92,

/*********
 * 99 Es *
 *********/

/**********
 * 100 Fm *
 **********/

/**********
 * 101 Md *
 **********/

/**********
 * 102 No *
 **********/

1.1,

/**********
 * 103 Lr *
 **********/

/**********
 * 104 Rf *
 **********/

/**********
 * 105 Db *
 **********/

/**********
 * 106 Sg *
 **********/

/**********
 * 107 Bh *
 **********/

/**********
 * 108 Hs *
 **********/

/**********
 * 109 Mt *
 **********/

/**********
 * 110 Ds *
 **********/

/**********
 * 111 Rg *
 **********/

};

/*******************************************
 * Format: [0, 1] for each R, G, B channel *
 * From: http://gabedit.sourceforge.net/,  *
 * with InsightII modifications (C, Si)    *
 * By: Carlos Pereira                      *
 *******************************************/

const float gamgi_chem_property_color[GAMGI_CHEM_COLOR_ARRAY]= {

/*******
 * 1 H *
 *******/

1.000, 1.000, 1.000,

/********
 * 2 He *
 ********/

0.850, 1.000, 1.000,

/********
 * 3 Li *
 ********/

0.800, 0.500, 1.000,

/********
 * 4 Be *
 ********/

0.760, 1.000, 0.000,

/*******
 * 5 B *
 *******/

1.000, 0.710, 0.710,

/*******
 * 6 C *
 *******/

0.000, 0.650, 0.000,

/*******
 * 7 N *
 *******/

0.500, 0.500, 1.000,

/*******
 * 8 O *
 *******/

1.000, 0.000, 0.000,

/*******
 * 9 F *
 *******/

0.700, 1.000, 1.000,

/*********
 * 10 Ne *
 *********/

0.700, 0.890, 0.960,

/*********
 * 11 Na *
 *********/

0.000, 0.000, 1.000,

/*********
 * 12 Mg *
 *********/

0.540, 1.000, 0.000,

/*********
 * 13 Al *
 *********/

0.750, 0.650, 0.650,

/*********
 * 14 Si *
 *********/

1.000, 1.000, 0.000,

/********
 * 15 P *
 ********/

1.000, 0.500, 0.000,

/********
 * 16 S *
 ********/

1.000, 1.000, 0.650,

/*********
 * 17 Cl *
 *********/

0.120, 0.940, 0.120,

/*********
 * 18 Ar *
 *********/

0.500, 0.820, 0.890,

/********
 * 19 K *
 ********/

0.560, 0.250, 0.830,

/*********
 * 20 Ca *
 *********/

0.240, 1.000, 0.000,

/*********
 * 21 Sc *
 *********/

0.900, 0.900, 0.900,

/*********
 * 22 Ti *
 *********/

0.750, 0.760, 0.780,

/********
 * 23 V *
 ********/

0.650, 0.650, 0.670,

/*********
 * 24 Cr *
 *********/

0.540, 0.600, 0.780,

/*********
 * 25 Mn *
 *********/

0.610, 0.480, 0.780,

/*********
 * 26 Fe *
 *********/

0.500, 0.480, 0.780,

/*********
 * 27 Co *
 *********/

0.440, 0.480, 0.780,

/*********
 * 28 Ni *
 *********/

0.360, 0.480, 0.760,

/*********
 * 29 Cu *
 *********/

1.000, 0.480, 0.380,

/*********
 * 30 Zn *
 *********/

0.490, 0.500, 0.690,

/*********
 * 31 Ga *
 *********/

0.760, 0.560, 0.560,

/*********
 * 32 Ge *
 *********/

0.400, 0.560, 0.560,

/*********
 * 33 As *
 *********/

0.740, 0.500, 0.890,

/*********
 * 34 Se *
 *********/

1.000, 0.630, 0.000,

/*********
 * 35 Br *
 *********/

0.650, 0.160, 0.160,

/*********
 * 36 Kr *
 *********/

0.360, 0.720, 0.820,

/*********
 * 37 Rb *
 *********/

0.440, 0.180, 0.690,

/*********
 * 38 Sr *
 *********/

0.000, 1.000, 0.000,

/********
 * 39 Y *
 ********/

0.580, 1.000, 1.000,

/*********
 * 40 Zr *
 *********/

0.580, 0.880, 0.880,

/*********
 * 41 Nb *
 *********/

0.450, 0.760, 0.790,

/*********
 * 42 Mo *
 *********/

0.330, 0.710, 0.710,

/*********
 * 43 Tc *
 *********/

0.230, 0.620, 0.620,

/*********
 * 44 Ru *
 *********/

0.140, 0.560, 0.560,

/*********
 * 45 Rh *
 *********/

0.040, 0.490, 0.550,

/*********
 * 46 Pd *
 *********/

0.000, 0.410, 0.520,

/*********
 * 47 Ag *
 *********/

0.880, 0.880, 1.000,

/*********
 * 48 Cd *
 *********/

1.000, 0.850, 0.560,

/*********
 * 49 In *
 *********/

0.650, 0.460, 0.450,

/*********
 * 50 Sn *
 *********/

0.400, 0.500, 0.500,

/*********
 * 51 Sb *
 *********/

0.620, 0.390, 0.710,

/*********
 * 52 Te *
 *********/

0.830, 0.480, 0.000,

/********
 * 53 I *
 ********/

0.580, 0.000, 0.580,

/*********
 * 54 Xe *
 *********/

0.260, 0.620, 0.690,

/*********
 * 55 Cs *
 *********/

0.340, 0.090, 0.560,

/*********
 * 56 Ba *
 *********/

0.000, 0.790, 0.000,

/*********
 * 57 La *
 *********/

0.440, 0.830, 1.000,

/*********
 * 58 Ce *
 *********/

1.000, 1.000, 0.780,

/*********
 * 59 Pr *
 *********/

0.850, 1.000, 0.780,

/*********
 * 60 Nd *
 *********/

0.780, 1.000, 0.780,

/*********
 * 61 Pm *
 *********/

0.640, 1.000, 0.780,

/*********
 * 62 Sm *
 *********/

0.560, 1.000, 0.780,

/*********
 * 63 Eu *
 *********/

0.380, 1.000, 0.780,

/*********
 * 64 Gd *
 *********/

0.270, 1.000, 0.780,

/*********
 * 65 Tb *
 *********/

0.190, 1.000, 0.780,

/*********
 * 66 Dy *
 *********/

0.120, 1.000, 0.780,

/*********
 * 67 Ho *
 *********/

0.000, 1.000, 0.610,

/*********
 * 68 Er *
 *********/

0.000, 0.900, 0.460,

/*********
 * 69 Tm *
 *********/

0.000, 0.830, 0.320,

/*********
 * 70 Yb *
 *********/

0.000, 0.750, 0.220,

/*********
 * 71 Lu *
 *********/

0.000, 0.670, 0.140,

/*********
 * 72 Hf *
 *********/

0.300, 0.760, 1.000,

/*********
 * 73 Ta *
 *********/

0.300, 0.650, 1.000,

/********
 * 74 W *
 ********/

0.130, 0.580, 0.840,

/*********
 * 75 Re *
 *********/

0.150, 0.490, 0.670,

/*********
 * 76 Os *
 *********/

0.150, 0.400, 0.590,

/*********
 * 77 Ir *
 *********/

0.090, 0.330, 0.530,

/*********
 * 78 Pt *
 *********/

0.960, 0.930, 0.820,

/*********
 * 79 Au *
 *********/

0.800, 0.820, 0.120,

/*********
 * 80 Hg *
 *********/

0.710, 0.710, 0.760,

/*********
 * 81 Tl *
 *********/

0.650, 0.330, 0.300,

/*********
 * 82 Pb *
 *********/

0.340, 0.350, 0.380,

/*********
 * 83 Bi *
 *********/

0.620, 0.310, 0.710,

/*********
 * 84 Po *
 *********/

0.670, 0.360, 0.000,

/*********
 * 85 At *
 *********/

0.460, 0.310, 0.270,

/*********
 * 86 Rn *
 *********/

0.260, 0.510, 0.590,

/*********
 * 87 Fr *
 *********/

0.260, 0.000, 0.400,

/*********
 * 88 Ra *
 *********/

0.000, 0.490, 0.000,

/*********
 * 89 Ac *
 *********/

0.440, 0.670, 0.980,

/*********
 * 90 Th *
 *********/

0.000, 0.730, 1.000,

/*********
 * 91 Pa *
 *********/

0.000, 0.630, 1.000,

/********
 * 92 U *
 ********/

0.000, 0.560, 1.000,

/*********
 * 93 Np *
 *********/

0.000, 0.500, 1.000,

/*********
 * 94 Pu *
 *********/

0.000, 0.420, 1.000,

/*********
 * 95 Am *
 *********/

0.330, 0.360, 0.950,

/*********
 * 96 Cm *
 *********/

0.470, 0.360, 0.890,

/*********
 * 97 Bk *
 *********/

0.540, 0.310, 0.890,

/*********
 * 98 Cf *
 *********/

0.630, 0.210, 0.830,

/*********
 * 99 Es *
 *********/

0.700, 0.120, 0.830,

/**********
 * 100 Fm *
 **********/

0.700, 0.120, 0.730,

/**********
 * 101 Md *
 **********/

0.700, 0.050, 0.650,

/**********
 * 102 No *
 **********/

0.740, 0.050, 0.530,

/**********
 * 103 Lr *
 **********/

0.780, 0.000, 0.400,

/**********
 * 104 Rf *
 **********/

0.800, 0.000, 0.350,

/**********
 * 105 Db *
 **********/

0.820, 0.000, 0.310,

/**********
 * 106 Sg *
 **********/

0.850, 0.000, 0.270,

/**********
 * 107 Bh *
 **********/

0.880, 0.000, 0.220,

/**********
 * 108 Hs *
 **********/

0.900, 0.000, 0.180,

/**********
 * 109 Mt *
 **********/

0.920, 0.000, 0.150,

/**********
 * 110 Ds *
 **********/

0.940, 0.000, 0.120,

/**********
 * 111 Rg *
 **********/

0.960, 0.000, 0.090,
};
