/**
 * File:          $RCSfile: jpeg_io.c,v $
 * Module:        JPEG format image file I/O functions
 * Part of:       Gandalf Library
 *
 * Revision:      $Revision: 1.8 $
 * Last edited:   $Date: 2004/01/06 00:49:02 $
 * Author:        $Author: pm $
 * Copyright:     (c) 2000 Imagineer Software Limited
 *
 * Notes:         Works with the libjpeg library
 */

/* This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdlib.h>
#include <stdio.h>
#include <gandalf/image/io/jpeg_io.h>
#include <gandalf/image/image_bit.h>
#include <gandalf/image/image_extract.h>
#include <gandalf/common/misc_error.h>

/* only compile if you have JPEG */
#ifdef HAVE_JPEG

#include <jpeglib.h>

#if BITS_IN_JSAMPLE != 8
#error Only 8-bit JPEG supported by Gandalf
#endif

#include <setjmp.h>

/**
 * \addtogroup ImagePackage
 * \{
 */

/**
 * \addtogroup ImageIO
 * \{
 */

struct my_error_mgr
{
   struct jpeg_error_mgr pub;
   jmp_buf setjmp_buffer;
};

typedef struct my_error_mgr *my_error_ptr;

METHODDEF(void)
my_error_exit (j_common_ptr cinfo)
{
   /* cinfo->err really points to a my_error_mgr struct, so coerce pointer */
   my_error_ptr myerr = (my_error_ptr) cinfo->err;

   /* Always display the message. */
   /* We could postpone this until after returning, if we chose. */
   (*cinfo->err->output_message) (cinfo);

   /* Return control to the setjmp point */
   longjmp(myerr->setjmp_buffer, 1);
}

/**
 * \brief Reads an image file in JPEG format from a file stream
 * \param infile The file stream to be read
 * \param image The image structure to read the image data into or \c NULL
 * \return Pointer to image structure, or \c NULL on failure.
 *
 * Reads the JPEG image from the file stream \a infile into the given \a image.
 * If \a image is passed as \c NULL a new image is dynamically allocated;
 * otherwise the already allocated \a image structure is reused.
 *
 * \sa gan_write_jpeg_image_stream().
 */
Gan_Image *
 gan_read_jpeg_image_stream(FILE *infile, Gan_Image *image)
{
   struct jpeg_decompress_struct cinfo;
   struct my_error_mgr jerr;
   Gan_ImageFormat format;
   JSAMPROW rowptr[1];
   
   cinfo.err = jpeg_std_error(&jerr.pub);
   jerr.pub.error_exit = my_error_exit;

   /* Establish the setjmp return context for my_error_exit to use. */
   if (setjmp(jerr.setjmp_buffer))
   {
      /* If we get here, the JPEG code has signaled an error.
       * We need to clean up the JPEG object, close the input file, and return.
       */
      jpeg_destroy_decompress(&cinfo);
      gan_err_flush_trace();
      gan_err_register ( "gan_read_jpeg_image_stream", GAN_ERROR_FAILURE, "" );
      return NULL;
   }

   jpeg_create_decompress(&cinfo);
   jpeg_stdio_src(&cinfo, infile);
   (void) jpeg_read_header(&cinfo, TRUE);

   if ( cinfo.jpeg_color_space == JCS_GRAYSCALE )
   {
      format = GAN_GREY_LEVEL_IMAGE;
      cinfo.out_color_space = JCS_GRAYSCALE;
   }
   else
   {
      format = GAN_RGB_COLOUR_IMAGE;
      cinfo.out_color_space = JCS_RGB;
   }

   jpeg_calc_output_dimensions(&cinfo);

   if ( cinfo.output_components != 1 && cinfo.output_components != 3 )
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_read_jpeg_image_stream", GAN_ERROR_INCOMPATIBLE,
                         "" );
      return NULL;
   }

   if(image == NULL)
   {
      /* Allocate memory for the new image */
      image = gan_image_alloc ( format, GAN_UCHAR,
                                (unsigned long) cinfo.output_height,
                                (unsigned long) cinfo.output_width );
      if ( image == NULL )
      {
         gan_err_register ( "gan_read_jpeg_image_stream", GAN_ERROR_FAILURE,
                            "" );
         return NULL;
      }
   }
   else
   {
      /*Use an already allocated image struct*/
      if ( gan_image_set_format_type_dims ( image, format, GAN_UCHAR,
                                          (unsigned long) cinfo.output_height,
                                          (unsigned long) cinfo.output_width )
           == NULL )
      {
         gan_err_register ( "gan_read_jpeg_image_stream", GAN_ERROR_FAILURE,
                            "" );
         return NULL;
      }
   }

   jpeg_start_decompress(&cinfo);

   switch ( image->format )
   {
      case GAN_GREY_LEVEL_IMAGE:
        while ( cinfo.output_scanline < cinfo.output_height )
        {
           rowptr[0] = (JSAMPROW) image->row_data.gl.uc[cinfo.output_scanline];
           (void) jpeg_read_scanlines ( &cinfo, rowptr, (JDIMENSION)1);
        }

        break;
        
      case GAN_RGB_COLOUR_IMAGE:
        while ( cinfo.output_scanline < cinfo.output_height )
        {
           rowptr[0] = (JSAMPROW)image->row_data.rgb.uc[cinfo.output_scanline];
           (void) jpeg_read_scanlines ( &cinfo, rowptr, (JDIMENSION)1 );
        }

        break;

      default:
        gan_err_flush_trace();
        gan_err_register ( "gan_read_jpeg_image_stream",
                           GAN_ERROR_ILLEGAL_TYPE,  "" );
        return NULL;
   }

   jpeg_finish_decompress(&cinfo);
   jpeg_destroy_decompress(&cinfo);
   return image;
}

/**
 * \brief Reads an image file in JPEG format.
 * \param filename The name of the image file
 * \param image The image structure to read the image data into or NULL
 * \return Pointer to image structure, or \c NULL on failure.
 *
 * Reads the JPEG image stored in the file \a filename into the given \a image.
 * If \a image is \c NULL a new image is dynamically allocated; otherwise the
 * already allocated image structure is reused.
 *
 * \sa gan_write_jpeg_image().
 */
Gan_Image *
 gan_read_jpeg_image ( const char *filename, Gan_Image *image )
{
   FILE *infile;
   Gan_Image *result;

   /* attempt to open file */
   infile = fopen ( filename, "rb" );
   if ( infile == NULL )
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_read_jpeg_image", GAN_ERROR_OPENING_FILE,
                         filename );
      return NULL;
   }

   result = gan_read_jpeg_image_stream ( infile, image );
   fclose(infile);
   return result;
}

/**
 * \brief Writes an image file to a stream in JPEG format.
 * \param outfile The file stream
 * \param image The image structure to write to the file
 * \param quality Quality value of compression in range 0...100
 * \return #GAN_TRUE on successful write operation, #GAN_FALSE on failure.
 *
 * Writes the given \a image into a JPEG file using a file stream \a outfile.
 *
 * \sa gan_read_jpeg_image_stream().
 */
Gan_Bool
 gan_write_jpeg_image_stream(FILE *outfile, Gan_Image *image, int quality)
{
   Gan_Image *converted_image=NULL;
   struct jpeg_compress_struct cinfo;
   struct jpeg_error_mgr error_mgr;
   JSAMPROW rowptr[1];

   cinfo.err = jpeg_std_error(&error_mgr);
   jpeg_create_compress ( &cinfo );
   jpeg_stdio_dest ( &cinfo, outfile );

   /* set JPEG image format */
   switch ( image->format )
   {
      case GAN_GREY_LEVEL_IMAGE:
        cinfo.input_components = 1;
        cinfo.in_color_space = JCS_GRAYSCALE;
        break;

      case GAN_RGB_COLOUR_IMAGE:
        cinfo.input_components = 3;
        cinfo.in_color_space = JCS_RGB;
        break;

      default:
        gan_err_flush_trace();
        gan_err_register ( "gan_write_jpeg_image_stream",
                           GAN_ERROR_ILLEGAL_TYPE, "" );
        return GAN_FALSE;
   }

   /* set JPEG image dimensions */
   cinfo.image_width  = image->width;
   cinfo.image_height = image->height;

   jpeg_set_defaults ( &cinfo );
   jpeg_set_quality ( &cinfo, quality, TRUE );
   jpeg_start_compress ( &cinfo, TRUE );

   /* convert image to 8-bit if necessary */
   if ( image->type == GAN_UCHAR )
      converted_image = image;
   else
   {
      converted_image = gan_image_convert_s ( image, image->format,
                                              GAN_UCHAR );
      if ( converted_image == NULL )
      {
        gan_err_register ( "gan_write_jpeg_image_stream",
                           GAN_ERROR_FAILURE, "" );
        return GAN_FALSE;
      }
   }

   switch ( image->format )
   {
      case GAN_GREY_LEVEL_IMAGE:
        while ( cinfo.next_scanline < cinfo.image_height )
        {
           rowptr[0] = (JSAMPROW)
                       converted_image->row_data.gl.uc[cinfo.next_scanline];
           (void) jpeg_write_scanlines(&cinfo, rowptr, (JDIMENSION) 1);
        }

        break;

      case GAN_RGB_COLOUR_IMAGE:
        while ( cinfo.next_scanline < cinfo.image_height )
        {
           rowptr[0] = (JSAMPROW)
                       converted_image->row_data.rgb.uc[cinfo.next_scanline];
           (void) jpeg_write_scanlines(&cinfo, rowptr, (JDIMENSION) 1);
        }

        break;

      default:
        gan_err_flush_trace();
        gan_err_register ( "gan_write_jpeg_image_stream",
                           GAN_ERROR_ILLEGAL_TYPE, "" );
        return GAN_FALSE;
   }
  
   /* that's it */
   if ( converted_image != image ) gan_image_free ( converted_image );
   jpeg_finish_compress(&cinfo);
   jpeg_destroy_compress(&cinfo);
   return GAN_TRUE;
}

/**
 * \brief Writes an image file in JPEG format.
 * \param filename The name of the image file
 * \param image The image structure to write to the file
 * \param quality Quality of JPEG compression in range 0...100
 * \return #GAN_TRUE on successful write operation, #GAN_FALSE on failure.
 *
 * Writes the given \a image into the JPEG file \a filename.
 *
 * \sa gan_read_jpeg_image().
 */
Gan_Bool
 gan_write_jpeg_image ( const char *filename, Gan_Image *image, int quality )
{
   FILE *outfile;
   Gan_Bool result;

   /* attempt to open file */
   outfile = fopen ( filename, "wb" );
   if ( outfile == NULL )
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_write_jpeg_image", GAN_ERROR_OPENING_FILE,
                         filename );
      return GAN_FALSE;
   }

   result = gan_write_jpeg_image_stream ( outfile, image, quality );
   fclose(outfile);
   return result;
}

/**
 * \}
 */

/**
 * \}
 */

#endif /* #ifdef HAVE_JPEG */
