/* Copyright (C) 2000-2002 Damir Zucic */

/*=============================================================================

				special_select.c

Purpose:
	Execute select command (the special version): try to interpret the
	selection string  as the special keyword.  Three selections  modes
	are available:  0 = overwrite,  1 = restrict,  2 = expand previous
	selection.  Special keywords  are:  ALL,  *,  POLAR,  HYDROPHOBIC,
	AROMATIC,  ALIPHATIC,  SMALL,  TINY,  POSITIVE,  NEGATIVE,  BASIC,
	ACIDIC, CHARGED, MAIN_CHAIN, SIDE_CHAINS, HETERO, CIS, TRANS, BAD,
	COMPLEMENT, SEQUENCE, SPHERE, ALTERNATE, MODEL, ABOVE,  BELOW  and
	TRIPLET.  Most  of these  keywords  may  be  abbreviated  to three
	characters. The only exception is keyword  HYDROPHOBIC:  it may be
	replaced by PHO or by HYDROPHO.

Input:
	(1) Pointer to MolComplexS structure, with macromol. complexes.
	(2) Number of macromolecular complexes.
	(3) Pointer to RuntimeS structure (the sequence buffer is there).
	(4) The selection string.
	(5) Selection mode index  (0 = overwr., 1 = restrict, 2 = expand).

Output:
	(1) The flag selectedF will be equal to one for selected atoms and
	    equal to zero for all other atoms.
	(2) Return value.

Return value:
	(1) The number of  selected atoms,  if selection  is done  in this
	    function.
	(2) -1 if selection string is not suitable for this function.
	(3) Some other negative value on failure.

========includes:============================================================*/

#include <stdio.h>

#include <string.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>
#include <X11/Xatom.h>

#include "defines.h"
#include "typedefs.h"

/*======function prototypes:=================================================*/

char		*ExtractToken_ (char *, int, char *, char *);
int		ExtractIndex_ (char *);
long		SelectHetero_ (MolComplexS *, int, int);
long		SelectCisTrans_ (MolComplexS *, int, int, int);
long		SelectComplement_ (MolComplexS *, int, int);
long		SelectSequence_ (MolComplexS *, int, RuntimeS *, int);
long		SelectSphere_ (MolComplexS *, int, RuntimeS *, int);
long		SelectAlternate_ (MolComplexS *, int, int);
long		SelectModel_ (MolComplexS *, int, int, int);
long		SelectAbove_ (MolComplexS *, int, int);
long		SelectBelow_ (MolComplexS *, int, int);
long		SelectTriplet_ (MolComplexS *, int, int);
long		SelectTM_ (MolComplexS *, int, int);
int		Include_ (SelectS *, char *);
int		Exclude_ (SelectS *, char *);
long		ApplySelection_ (MolComplexS *, int,
				 SelectS *, SelectS *, int);

/*======execute select command (special selection string):===================*/

long SpecialSelect_ (MolComplexS *mol_complexSP, int mol_complexesN,
		     RuntimeS *runtimeSP, char *stringP, int selection_modeI)
{
long		selected_atomsN;
char		tokenA[STRINGSIZE];
char		full_stringA[STRINGSIZE];
char		*P;
int		n;
int		slashesN = 0;
SelectS		include_selectS, exclude_selectS;
int		model_serialI;

/* Purify the selection string (remove the leading and trailing separators): */
if (!ExtractToken_ (tokenA, STRINGSIZE, stringP, " ,;\t\n")) return -1;

/* Check the selection string: */

/** ALL: **/
if ((strstr (tokenA, "ALL") == tokenA) || (strstr (tokenA, "*") == tokenA))
	strcpy (full_stringA, "*/*/*/*");

/** POLAR: **/
else if (strstr (tokenA, "POL") == tokenA)
	strcpy (full_stringA,
		"*/*/ARG,LYS,HIS,ASP,GLU,ASN,GLN,SER,THR,TRP,TYR/*");

/** HYDROPHOBIC: **/
else if ((strstr (tokenA, "HYDROPHO") == tokenA) ||
	 (strstr (tokenA, "PHO") == tokenA))
	strcpy (full_stringA,
		"*/*/ALA,ILE,LEU,MET,PHE,TRP,VAL,TYR,CYS,GLY,HIS/*");

/** AROMATIC: **/
else if (strstr (tokenA, "ARO") == tokenA)
	strcpy (full_stringA, "*/*/PHE,TYR,TRP,HIS/*");

/** ALIPHATIC: **/
else if (strstr (tokenA, "ALI") == tokenA)
	strcpy (full_stringA, "*/*/ILE,LEU,VAL/*");

/** SMALL: **/
else if (strstr (tokenA, "SMA") == tokenA)
	strcpy (full_stringA, "*/*/GLY,ALA,SER,THR,CYS,VAL,PRO,ASP,ASN/*");

/** TINY: **/
else if (strstr (tokenA, "TIN") == tokenA)
	strcpy (full_stringA, "*/*/GLY,ALA,SER/*");

/** POSITIVE: **/
else if (strstr (tokenA, "POS") == tokenA)
	strcpy (full_stringA, "*/*/ARG,LYS,HIS/*");

/** NEGATIVE: **/
else if (strstr (tokenA, "NEG") == tokenA)
	strcpy (full_stringA, "*/*/GLU,ASP/*");

/** BASIC: **/
else if (strstr (tokenA, "BAS") == tokenA)
	strcpy (full_stringA, "*/*/ARG,LYS,HIS/*");

/** ACIDIC: **/
else if (strstr (tokenA, "ACI") == tokenA)
	strcpy (full_stringA, "*/*/GLU,ASP/*");

/** CHARGED: **/
else if (strstr (tokenA, "CHA") == tokenA)
	strcpy (full_stringA, "*/*/ARG,LYS,HIS,GLU,ASP/*");

/** MAIN_CHAIN: **/
else if (strstr (tokenA, "MAI") == tokenA)
	strcpy (full_stringA, "*/*/*/CA,C,N,O,H");

/** SIDE_CHAINS: **/
else if (strstr (tokenA, "SID") == tokenA)
	strcpy (full_stringA, "*/*/*/* EXCEPT CA,C,N,O,H");

/** HETERO: **/
else if (strstr (tokenA, "HET") == tokenA)
	{
	selected_atomsN = SelectHetero_ (mol_complexSP, mol_complexesN,
					 selection_modeI);
	return selected_atomsN;
	}

/** CIS: **/
else if (strstr (tokenA, "CIS") == tokenA)
	{
	selected_atomsN = SelectCisTrans_ (mol_complexSP, mol_complexesN,
					   selection_modeI, 2);
	return selected_atomsN;
	}

/** TRANS: **/
else if (strstr (tokenA, "TRA") == tokenA)
	{
	selected_atomsN = SelectCisTrans_ (mol_complexSP, mol_complexesN,
					   selection_modeI, 1);
	return selected_atomsN;
	}

/** BAD: **/
else if (strstr (tokenA, "BAD") == tokenA)
	{
	selected_atomsN = SelectCisTrans_ (mol_complexSP, mol_complexesN,
					   selection_modeI, 0);
	return selected_atomsN;
	}

/** COMPLEMENT: **/
else if (strstr (tokenA, "COM") == tokenA)
	{
	selected_atomsN = SelectComplement_ (mol_complexSP, mol_complexesN,
					     selection_modeI);
	return selected_atomsN;
	}

/** SEQUENCE: **/
else if (strstr (tokenA, "SEQ") == tokenA)
	{
	selected_atomsN = SelectSequence_ (mol_complexSP, mol_complexesN,
					   runtimeSP, selection_modeI);
	return selected_atomsN;
	}

/** SPHERE: **/
else if (strstr (tokenA, "SPH") == tokenA)
	{
	selected_atomsN = SelectSphere_ (mol_complexSP, mol_complexesN,
					 runtimeSP, selection_modeI);
	return selected_atomsN;
	}

/** ALTERNATE: **/
else if (strstr (tokenA, "ALT") == tokenA)
	{
	selected_atomsN = SelectAlternate_ (mol_complexSP, mol_complexesN,
					    selection_modeI);
	return selected_atomsN;
	}

/** MODEL: **/
else if (strstr (tokenA, "MOD") == tokenA)
	{
	model_serialI = ExtractIndex_ (stringP);
	selected_atomsN = SelectModel_ (mol_complexSP, mol_complexesN,
					selection_modeI, model_serialI);
	return selected_atomsN;
	}

/** ABOVE: **/
else if (strstr (tokenA, "ABO") == tokenA)
	{
	selected_atomsN = SelectAbove_ (mol_complexSP, mol_complexesN,
					selection_modeI);
	return selected_atomsN;
	}

/** BELOW: **/
else if (strstr (tokenA, "BEL") == tokenA)
	{
	selected_atomsN = SelectBelow_ (mol_complexSP, mol_complexesN,
					selection_modeI);
	return selected_atomsN;
	}

/** TRIPLET: **/
else if (strstr (tokenA, "TRI") == tokenA)
	{
	selected_atomsN = SelectTriplet_ (mol_complexSP, mol_complexesN,
					  selection_modeI);
	return selected_atomsN;
	}

/** TM (transmembrane part): **/
else if (strstr (tokenA, "TM") == tokenA)
	{
	selected_atomsN = SelectTM_ (mol_complexSP, mol_complexesN,
				     selection_modeI);
	return selected_atomsN;
	}

/** String not recognized: **/
else return -1;		/* Must be -1! */

/* Count slashes: */
P = full_stringA;
while ((n = *P++) != '\0') if (n == '/') slashesN++;
if (slashesN != 3) return -1;

/* Identify chains,  residue ranges,  residue */
/* names and atoms which have to be included: */
if (Include_ (&include_selectS, full_stringA) < 0) return -2;

/* Identify chains,  residue ranges,  residue */
/* names and atoms which have to be excluded: */
if (Exclude_ (&exclude_selectS, full_stringA) < 0) return -3;

/* Do the selection: */
selected_atomsN = ApplySelection_ (mol_complexSP, mol_complexesN,
				   &include_selectS, &exclude_selectS,
				   selection_modeI);

/* Return the number of selected atoms: */
return selected_atomsN;
}

/*===========================================================================*/


