/* integrity.c
 *
 * Copyright (C) 1999,2000 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "integrity.h"
#include <config.h>
#include <gnome.h>

static GtkCTreeNode *integrity_check_1 (gASQL_Main_Config * conf,
					SqlMemFlink * link,
					GtkCTreeNode * node);
static GtkCTreeNode *integrity_check_2 (gASQL_Main_Config * conf,
					SqlMemFlink * link,
					GtkCTreeNode * node);
static GtkCTreeNode *integrity_check_3 (gASQL_Main_Config * conf,
					SqlMemFlink * link,
					GtkCTreeNode * node);


static void
popup_menu_click_cb (GtkWidget * wid, gchar * txt)
{
	GtkMenu *menu;
	menu = GTK_MENU (wid->parent);
	gtk_object_destroy (GTK_OBJECT (menu));
}

/* CB when a click is done in the results list */
static gint
button_press (GtkWidget * widget, GdkEvent * event)
{
	GtkWidget *wid, *menu;
	gint row, col;

	if (event->type == GDK_BUTTON_PRESS) {
		GdkEventButton *bevent = (GdkEventButton *) event;
		if (bevent->button == 3) {
			gtk_clist_get_selection_info (GTK_CLIST (widget),
						      bevent->x, bevent->y,
						      &row, &col);
			gtk_clist_select_row (GTK_CLIST (widget), row, col);
			menu = gtk_menu_new ();
			wid = gtk_menu_item_new_with_label ("Line one");
			gtk_signal_connect (GTK_OBJECT (wid), "activate",
					    GTK_SIGNAL_FUNC
					    (popup_menu_click_cb), "1");
			gtk_menu_append (GTK_MENU (menu), wid);
			gtk_widget_show (wid);
			wid = gtk_menu_item_new_with_label ("Line two");
			gtk_signal_connect (GTK_OBJECT (wid), "activate",
					    GTK_SIGNAL_FUNC
					    (popup_menu_click_cb), "2");
			gtk_menu_append (GTK_MENU (menu), wid);
			gtk_widget_show (wid);
			gtk_menu_popup (GTK_MENU (menu), NULL, NULL, NULL,
					NULL, bevent->button, bevent->time);
			/* Tell calling code that we have handled this event */
			return TRUE;
		}
	}

	/* Tell calling code that we have not handled this event; pass it on. */
	return FALSE;
}


/* callback when the DLG is closed */
static void integrity_check_clicked_cb (GtkWidget * dlg, gint btn,
					gASQL_Main_Config * conf);

/* main function call */
void
integrity_check_cb (GtkWidget * widget, gASQL_Main_Config * conf)
{
	gchar *str;
	GtkWidget *dlg, *table, *wid, *sw;
	gchar *titles[] =
		{ N_("Link and type of error"), N_("Table"), N_("Field"),
		N_("Error description")
	};
	gint i;

	if (conf->check_dlg)
		return;

	str = g_strdup_printf (_("Testing the integrity of DB '%s'"),
			       conf->db->name);
	dlg = gnome_dialog_new (str, _("Start"), _("Stop"),
				GNOME_STOCK_BUTTON_CLOSE, NULL);
	g_free (str);
	gnome_dialog_set_sensitive (GNOME_DIALOG (dlg), 1, FALSE);

	table = gtk_table_new (3, 2, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE (table), GNOME_PAD);
	gtk_table_set_col_spacings (GTK_TABLE (table), GNOME_PAD);
	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dlg)->vbox),
			    table, FALSE, FALSE, GNOME_PAD);
	gtk_widget_show (table);

	wid = gtk_label_new (_("Link:"));
	gtk_table_attach_defaults (GTK_TABLE (table), wid, 0, 1, 0, 1);
	gtk_widget_show (wid);

	wid = gtk_label_new ("");
	conf->check_link_name = wid;
	gtk_table_attach_defaults (GTK_TABLE (table), wid, 1, 2, 0, 1);
	gtk_widget_show (wid);

	wid = gtk_label_new (_("Done:"));
	gtk_table_attach_defaults (GTK_TABLE (table), wid, 0, 1, 1, 2);
	gtk_widget_show (wid);

	wid = gtk_progress_bar_new ();
	gtk_table_attach_defaults (GTK_TABLE (table), wid, 1, 2, 1, 2);
	conf->check_pbar = wid;
	gtk_widget_show (wid);

	wid = gtk_label_new (_("Errors:"));
	gtk_table_attach_defaults (GTK_TABLE (table), wid, 0, 1, 2, 3);
	gtk_widget_show (wid);

	sw = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
					GTK_POLICY_ALWAYS, GTK_POLICY_ALWAYS);
	gtk_table_attach_defaults (GTK_TABLE (table), sw, 1, 2, 2, 3);
	gtk_widget_show (sw);

	wid = gtk_ctree_new (4, 0);
	conf->check_errors = wid;
	for (i = 0; i < 4; i++) {
		gtk_clist_set_column_title (GTK_CLIST (wid), i, _(titles[i]));
		gtk_clist_set_column_auto_resize (GTK_CLIST (wid), i, TRUE);
	}
	gtk_clist_column_titles_show (GTK_CLIST (wid));
	gtk_clist_column_titles_passive (GTK_CLIST (wid));
	gtk_clist_set_selection_mode (GTK_CLIST (wid), GTK_SELECTION_SINGLE);
	gtk_widget_set_usize (wid, 500, 200);
	gtk_container_add (GTK_CONTAINER (sw), wid);
	gtk_widget_show (wid);

	gtk_signal_connect (GTK_OBJECT (wid),
			    "button_press_event",
			    GTK_SIGNAL_FUNC (button_press), NULL);

	gtk_signal_connect (GTK_OBJECT (dlg), "clicked",
			    GTK_SIGNAL_FUNC (integrity_check_clicked_cb),
			    conf);
	gtk_widget_show (dlg);
	conf->check_dlg = dlg;
}

static GtkCTreeNode *
create_link_error_entry (gASQL_Main_Config * conf, SqlMemFlink * link)
{
	gchar *arow[4];
	GtkCTreeNode *node;
	SqlMemTable *tf, *tt;
	GdkColor color;
	GdkColormap *colormap;

	tf = sql_db_find_table_from_field (conf->db, link->from);
	tt = sql_db_find_table_from_field (conf->db, link->to);
	arow[0] = g_strdup_printf ("%s.%s --> %s.%s",
				   tf->name, link->from->name,
				   tt->name, link->to->name);
	arow[1] = NULL;
	arow[2] = NULL;
	arow[3] = NULL;
	node = gtk_ctree_insert_node (GTK_CTREE (conf->check_errors),
				      NULL, NULL, arow, GNOME_PAD,
				      NULL, NULL, NULL, NULL, FALSE, FALSE);

	colormap = gdk_colormap_get_system ();
	color.red = 0;
	color.green = 0xffff;
	color.blue = 0xffff;
	if (gdk_color_alloc (colormap, &color)) {
		gtk_ctree_node_set_background (GTK_CTREE (conf->check_errors),
					       node, &color);
	}

	g_free (arow[0]);
	return node;
}

/* CHECK 1:
   tests if there are multiple occurences of the same value in the
   FROM field of the link */
static GtkCTreeNode *
integrity_check_1 (gASQL_Main_Config * conf,
		   SqlMemFlink * link, GtkCTreeNode * node)
{
	gchar *query;
	gboolean error;
	SqlQueryRes *res;
	gint nrows, total, i, j;
	GtkCTreeNode *topnode = NULL;
	SqlMemTable *tf, *tt;

	tf = sql_db_find_table_from_field (conf->db, link->from);
	tt = sql_db_find_table_from_field (conf->db, link->to);
	query = g_strdup_printf ("select %s from %s order by %s",
				 link->from->name,
				 tf->name, link->from->name);
	res = sql_access_do_query (conf->srv, query);
	g_free (query);
	if (res) {
		nrows = sql_query_res_get_nbtuples (res);
		total = 0;
		i = 0;
		while (i < nrows) {
			error = FALSE;
			j = 1;
			while ((i + j < nrows)
			       && !strcmp (sql_query_res_get_item (res, i, 0),
					   sql_query_res_get_item (res, i + j,
								   0))) {
				error = TRUE;
				j++;
			}
			if (error) {
				gchar *arow[4];

				if (!topnode) {
					if (!node)	/* creating the top node */
						topnode =
							create_link_error_entry
							(conf, link);
					else
						topnode = node;
				}

				/* for that error */
				arow[0] = _("Multiple occurences");
				arow[1] = tf->name;
				arow[2] = link->from->name;
				arow[3] =
					g_strdup_printf
					("%d occurences of the value '%s'", j,
					 sql_query_res_get_item (res, i, 0));
				gtk_ctree_insert_node (GTK_CTREE
						       (conf->check_errors),
						       topnode, NULL, arow,
						       GNOME_PAD, NULL, NULL,
						       NULL, NULL, TRUE,
						       FALSE);
				g_free (arow[3]);
			}
			i += j;
			total += j - 1;
		}
		gtk_object_destroy (GTK_OBJECT (res));
	}

	return topnode;
}

/* CHECK 2:
   looking for NULL tuples in FROM fields and TO fields */
static GtkCTreeNode *
integrity_check_2 (gASQL_Main_Config * conf,
		   SqlMemFlink * link, GtkCTreeNode * node)
{
	gchar *query;
	SqlQueryRes *res;
	GtkCTreeNode *topnode = NULL;
	SqlMemTable *tf, *tt;
	gint nuls;

	tf = sql_db_find_table_from_field (conf->db, link->from);
	tt = sql_db_find_table_from_field (conf->db, link->to);

	/* test for the FROM field */
	query = g_strdup_printf ("select %s from %s where %s is null",
				 link->from->name, tf->name,
				 link->from->name);
	res = sql_access_do_query (conf->srv, query);
	g_free (query);
	if (res) {
		nuls = sql_query_res_get_nbtuples (res);
		if (nuls) {
			gchar *arow[4];

			if (!topnode) {
				if (!node)	/* creating the top node */
					topnode =
						create_link_error_entry (conf,
									 link);
				else
					topnode = node;
			}

			/* for that error */
			arow[0] = _("NULL field");
			arow[1] = tf->name;
			arow[2] = link->from->name;
			if (nuls == 1)
				arow[3] =
					g_strdup
					("There is one null value in this field");
			else
				arow[3] =
					g_strdup_printf
					("There are %d null values in this field",
					 nuls);
			gtk_ctree_insert_node (GTK_CTREE (conf->check_errors),
					       topnode, NULL, arow, GNOME_PAD,
					       NULL, NULL, NULL, NULL, TRUE,
					       FALSE);
			g_free (arow[3]);
		}
		gtk_object_destroy (GTK_OBJECT (res));
	}

	/* test for the TO field */
	query = g_strdup_printf ("select %s from %s where %s is null",
				 link->to->name, tt->name, link->to->name);
	res = sql_access_do_query (conf->srv, query);
	g_free (query);
	if (res) {
		nuls = sql_query_res_get_nbtuples (res);
		if (nuls) {
			gchar *arow[4];
			GtkCTreeNode *holdnode;

			if (!node)	/* creating the top node */
				holdnode =
					create_link_error_entry (conf, link);
			else
				holdnode = node;

			/* for that error */
			arow[0] = _("NULL field");
			arow[1] = tt->name;
			arow[2] = link->to->name;
			if (nuls == 1)
				arow[3] =
					g_strdup
					("There is one null value in this field");
			else
				arow[3] =
					g_strdup_printf
					("There are %d null values in this field",
					 nuls);
			gtk_ctree_insert_node (GTK_CTREE (conf->check_errors),
					       holdnode, NULL, arow,
					       GNOME_PAD, NULL, NULL, NULL,
					       NULL, TRUE, FALSE);
			g_free (arow[3]);
		}
		gtk_object_destroy (GTK_OBJECT (res));
	}

	return topnode;
}

/* CHECK 3:
   looking for values in the TO field which do not correspond to a value
   if the FROM field */
static GtkCTreeNode *
integrity_check_3 (gASQL_Main_Config * conf,
		   SqlMemFlink * link, GtkCTreeNode * node)
{
	gchar *query;
	SqlQueryRes *res;
	GtkCTreeNode *topnode = NULL;
	SqlMemTable *tf, *tt;
	gint nuls = 0, ncorresp = 0, ntotal = 0;

	tf = sql_db_find_table_from_field (conf->db, link->from);
	tt = sql_db_find_table_from_field (conf->db, link->to);

	/* find NULL values test for the TO field */
	query = g_strdup_printf ("select %s from %s where %s is null",
				 link->to->name, tt->name, link->to->name);
	res = sql_access_do_query (conf->srv, query);
	g_free (query);
	if (res) {
		nuls = sql_query_res_get_nbtuples (res);
		gtk_object_destroy (GTK_OBJECT (res));
	}
	/* total nb of tuples */
	query = g_strdup_printf ("select count(*) from %s", tt->name);
	res = sql_access_do_query (conf->srv, query);
	g_free (query);
	if (res) {
		ntotal = atoi (sql_query_res_get_item (res, 0, 0));
		gtk_object_destroy (GTK_OBJECT (res));
	}
	/* corresponding tuples */
	query = g_strdup_printf
		("select count(%s) from %s where %s is not null and "
		 "%s = %s.%s", link->to->name, tt->name, link->to->name,
		 link->to->name, tf->name, link->from->name);
	res = sql_access_do_query (conf->srv, query);
	g_free (query);
	if (res) {
		ncorresp = atoi (sql_query_res_get_item (res, 0, 0));
		gtk_object_destroy (GTK_OBJECT (res));
	}

	if (ncorresp + nuls != ntotal) {
		gchar *arow[4];

		if (!node)	/* creating the top node */
			topnode = create_link_error_entry (conf, link);
		else
			topnode = node;

		arow[0] = _("Non corresponding field");
		arow[1] = tt->name;
		arow[2] = link->to->name;
		if (ntotal - ncorresp - nuls == 1)
			arow[3] =
				g_strdup
				("There is one non corresponding value in "
				 "this field");
		else
			arow[3] =
				g_strdup_printf
				("There are %d non corresponding values in "
				 "this field", ntotal - ncorresp - nuls);
		gtk_ctree_insert_node (GTK_CTREE (conf->check_errors),
				       topnode, NULL, arow, GNOME_PAD, NULL,
				       NULL, NULL, NULL, TRUE, FALSE);
		g_free (arow[3]);
	}

	return topnode;
}

static void
integrity_check_do (gASQL_Main_Config * conf)
{
	GSList *list;
	SqlMemFlink *link;
	gint nb, i, now;
	gchar *str;
	SqlMemTable *tf, *tt;
	GtkCTreeNode *node = NULL, *hold;

	list = conf->db->field_links;
	nb = g_slist_length (list);
	now = 0;

	gtk_progress_configure (GTK_PROGRESS (conf->check_pbar), 0, 0, nb);

	while (conf->check_perform && list) {
		link = SQL_MEM_FLINK (list->data);
		tf = sql_db_find_table_from_field (conf->db, link->from);
		tt = sql_db_find_table_from_field (conf->db, link->to);
		str = g_strdup_printf ("%s.%s -> %s.%s", tf->name,
				       link->from->name, tt->name,
				       link->to->name);
		gtk_label_set_text (GTK_LABEL (conf->check_link_name), str);
		g_free (str);
		node = NULL;
		/* Test 1:
		   test if there are not some multiple occurences of the field which
		   is the source of the link. */
		if (!link->from->is_key)
			node = integrity_check_1 (conf, link, NULL);

		/* test 2 */
		hold = integrity_check_2 (conf, link, node);
		node = node ? node : hold;

		/* test 3 */
		if (!node)
			node = integrity_check_3 (conf, link, NULL);
		else {
			gchar *arow[4];

			arow[0] = _("Test not done");
			arow[1] = NULL;
			arow[2] = NULL;
			arow[3] =
				"test to find non corresponding field values not done";
			gtk_ctree_insert_node (GTK_CTREE (conf->check_errors),
					       node, NULL, arow, GNOME_PAD,
					       NULL, NULL, NULL, NULL, TRUE,
					       FALSE);
		}

		/* GTK main loop from time to time */
		while (gtk_main_iteration_do (FALSE));

		/* to spend some time to slow it a bit to see the progress bar! */
		i = 0;
		while (conf->check_perform && (i < 500)) {
			i++;
			if (i % 50 == 1) {
				while (gtk_main_iteration_do (FALSE));
			}
		}

		now++;
		gtk_progress_set_value (GTK_PROGRESS (conf->check_pbar), now);
		list = g_slist_next (list);
	}
	gtk_label_set_text (GTK_LABEL (conf->check_link_name), "");
	gtk_progress_set_value (GTK_PROGRESS (conf->check_pbar), 0);
}

static void
integrity_check_clicked_cb (GtkWidget * dlg, gint btn,
			    gASQL_Main_Config * conf)
{
	switch (btn) {
	case 0:		/* start */
		gnome_dialog_set_sensitive (GNOME_DIALOG (dlg), 1, TRUE);
		gnome_dialog_set_sensitive (GNOME_DIALOG (dlg), 0, FALSE);
		gtk_clist_clear (GTK_CLIST (conf->check_errors));
		conf->check_perform = TRUE;
		integrity_check_do (conf);
	case 1:		/* stop */
		gnome_dialog_set_sensitive (GNOME_DIALOG (dlg), 1, FALSE);
		gnome_dialog_set_sensitive (GNOME_DIALOG (dlg), 0, TRUE);
		conf->check_perform = FALSE;
		break;
	case 2:		/* close */
		conf->check_dlg = NULL;
		conf->check_pbar = NULL;
		conf->check_link_name = NULL;
		conf->check_errors = NULL;
		conf->check_perform = FALSE;
		gnome_dialog_close (GNOME_DIALOG (dlg));
		break;
	}
}
