/* sqlquery.h
 *
 * Copyright (C) 1999 - 2001 Vivien Malerba
 * Copyright (C) 2001 Fernando Martins
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#ifndef __SQL_QUERY__
#define __SQL_QUERY__

#include <gtk/gtkobject.h>
#include "sqldb.h"
#include "gasql_conf.h"

#ifdef __cplusplus
extern "C"
{
#endif				/* __cplusplus */

#define SQL_QUERY(obj)          GTK_CHECK_CAST (obj, sql_query_get_type(), SqlQuery)
#define SQL_QUERY_CLASS(klass)  GTK_CHECK_CLASS_CAST (klass, sql_query_get_type (), SqlQueryClass)
#define IS_SQL_QUERY(obj)       GTK_CHECK_TYPE (obj, sql_query_get_type ())


	typedef struct _Sqlquery SqlQuery;
	typedef struct _SqlqueryClass SqlQueryClass;

	/* these are static to the SqlQuery class */
	typedef struct _QueryElement QueryElement;
	typedef struct _QueryElementValue QueryElementValue;
	typedef struct _QueryCriteria QueryCriteria;
	typedef struct _QueryLink QueryLink;
	typedef struct _QueryWhereNode QueryWhereNode;	/* nodes for where conditions */
	typedef struct _QueryMissingValue QueryMissingValue;

	typedef enum
	{
		QUERY_ELT_FIELD,
		QUERY_ELT_AGGREGATE,
		QUERY_ELT_FUNCTION,
		QUERY_ELT_VALUE
	}
	QueryEltType;

	typedef enum
	{
		QUERY_VALUE_STR
	}
	QueryEltValueType;

	typedef enum
	{
		QUERY_WHERE_AND,
		QUERY_WHERE_OR,
		QUERY_WHERE_NOT
	}
	QueryWhereLogicType;

	typedef enum
	{
		QUERY_OP_EQUAL,
		QUERY_OP_DIFF,
		QUERY_OP_SUP,
		QUERY_OP_SUPEQUAL,
		QUERY_OP_INF,
		QUERY_OP_INFEQUAL,
		QUERY_OP_LIKE,
		QUERY_OP_REGEX,
		QUERY_OP_IN
	}
	QueryWhereOpType;

	typedef enum
	{
		QUERY_WHERE_ELT,	/* right operand is another QueryElement */
		QUERY_WHERE_ASK,	/* right operand is a value              */
	}
	QueryWhereRightOperandType;

	typedef enum {
		JOIN_INNER,
		JOIN_LEFT_OUTER,
		JOIN_RIGHT_OUTER,
		JOIN_FULL_OUTER,
		JOIN_CROSS,
		LAST_JOIN_TYPE
	} JoinType; /* Type of join between two tables (views) */

	struct _QueryElement
	{
		QueryEltType type;
		/* QUERY_ELT_FIELD     => main is a ptr on SqlMemField object
		   QUERY_ELT_AGGREGATE => main is a ptr on SqlDataAggregate object
		   QUERY_ELT_FUNCTION  => main is a ptr on SqlDataFunction object 
		   QUERY_ELT_VALUE     => main is a ptr on QueryElementValue struct */
		gpointer main;
		GSList *args;	/* args is a list of pointers on other QueryElement structs
				   For Aggregate */
		gchar *print_name;
		gchar *name;
	};

	/* to encapsulate a Field link and its usage in the query */
	struct _QueryLink
	{
		SqlMemFlink *link;
		/*gboolean useit;*/
		JoinType join_type;
		
	};

	/* to hold values in the objects of the query */
	struct _QueryElementValue
	{
		QueryEltValueType type_val;
		union
		{
			gchar *str;
			/* GdaField *field; ==> not used */
		}
		val;
		gchar *default_val;
		SqlDataType *data_type;
		gboolean ask_at_exec;	/* TRUE if we want to be able to modify it at execution time */
	};

	struct _QueryWhereNode
	{
		gboolean is_node;
		union
		{
			QueryWhereLogicType logic;
			struct
			{
				QueryElement *l_elt;	/* QueryElt on the left of condition */
				QueryWhereOpType op;	/* =, <, ... */
				QueryWhereRightOperandType r_type;	/* type on the right of condition */
				gpointer r_object;	/* same usage as main of QueryElement */
			}
			leaf;
		}
		val;
	};

	/* struct to hold the value of a missing value */
	struct _QueryMissingValue
	{
		QueryWhereRightOperandType type;	/* type of missing value */
		gpointer node;
		gchar *sql;	/* value */
	};

	/* struct for the object's data */
	struct _Sqlquery
	{
		GtkObject object;

		gASQL_Main_Config *conf;
		guint id;	/* comes from conf->id_serial */
		gchar *name;
		gchar *descr;
		GSList *objects;	/* pointers are to QueryElements */
		GSList *depend_list;	/* dependencies imposed by the program.
					   Pointers to QueryLink structures */

		/* GNodes tree of QueryWhereNode's. 
		   The root node holds data to the actual SqlQuery WHERE part */
		GNode *where_tree;
		GSList *group_by_list;	/* QueryElement list */
		GSList *order_by_list;	/* QueryElement list */

		gboolean text_only;
		gchar *text_sql;	/* if we have a text only query, otherwise NULL */

		GSList *envs;	/* SqlQueryEnv associated to this query */
	};

	/* struct for the object's class */
	struct _SqlqueryClass
	{
		GtkObjectClass parent_class;

		void (*elt_created) (SqlQuery * q, QueryElement * new_elt);
		void (*elt_dropped) (SqlQuery * q, QueryElement * elt);
		/* emited when elt modified and when elt is moved */
		void (*elt_modified) (SqlQuery * q, QueryElement * elt);
		void (*changed) (SqlQuery * q);
		void (*name_changed) (SqlQuery * q);
		void (*auto_link_added) (SqlQuery * q, QueryLink * ql);
		void (*auto_link_removed) (SqlQuery * q, QueryLink * ql);

		void (*wh_cond_created) (SqlQuery * q,
					 QueryWhereNode * new_node);
		/* WARNING: the next signal is emitted while the node is still in the GNodes tree,
		   and it is removed and destroyed only after the signal callbacks are finished */
		void (*wh_cond_dropped) (SqlQuery * q, QueryWhereNode * node);
		void (*wh_cond_modified) (SqlQuery * q,
					  QueryWhereNode * node);
		void (*wh_cond_moved) (SqlQuery * q, QueryWhereNode * node);
	};

	/* 
	 * generic widget's functions 
	 */
	guint sql_query_get_type (void);

	/* new query */
	GtkObject *sql_query_new (gchar * name, gASQL_Main_Config * conf);
	GtkObject *sql_query_new_noref (gchar * name, gASQL_Main_Config * conf);	/* non referenced ! */

	/* gets a copy of a query */
	GtkObject *sql_query_copy (SqlQuery * q);
	GtkObject *sql_query_copy_noref (SqlQuery * q);	/* non referenced ! */


	/*
	 * Helper functions
	 */
	/* if NULL passed as gchar*, nothing happens for that gchar* */
	void sql_query_set_name (SqlQuery * q, gchar * name, gchar * descr);

	/* sets the text SQL (and toggles the query type) */
	void sql_query_set_text_sql (SqlQuery * q, gchar * sql);

	/* DO NOT FREE MEM */
	gchar *sql_query_render_operator (QueryWhereOpType op);

	/*
	 * Loading/saving as XML
	 */
	void sql_query_build_xml_tree (SqlQuery * q,
				       xmlNodePtr toptree,
				       gASQL_Main_Config * conf);
	gboolean sql_queries_build_from_xml_tree (gASQL_Main_Config * conf,
						  xmlNodePtr node);

	/* 
	 * Query contents manipulation
	 */
	QueryElement *sql_query_new_elt_field (SqlQuery * q,
					       gchar * name,
					       gchar * print_name,
					       SqlMemField * field);
	QueryElement *sql_query_new_elt_aggregate (SqlQuery * q,
						   gchar * name,
						   gchar * print_name,
						   SqlDataAggregate * agg,
						   GSList * args);
	QueryElement *sql_query_new_elt_function (SqlQuery * q,
						  gchar * name,
						  gchar * print_name,
						  SqlDataFunction * func,
						  GSList * args);
	QueryElement *sql_query_new_elt_value (SqlQuery * q,
					       gchar * name,
					       gchar * print_name,
					       SqlDataType * data_type,
					       gchar * defval,
					       gboolean modif);

	QueryElement *sql_query_get_element (SqlQuery * q, gchar * name);

	void sql_query_del_elt (SqlQuery * q, QueryElement * elt);
	void sql_query_swap_objects (SqlQuery * q, QueryElement *o1, QueryElement *o2);

	void sql_query_modif_elt (SqlQuery * q, QueryElement * elt,
				  gchar * name, gchar * print_name);

	/*
	 * Query links usage management
	 */
	void sql_query_auto_links_refresh (SqlQuery * q);
	void sql_query_auto_links_add_link (GtkObject * obj,
					    SqlMemFlink * link, SqlQuery * q);
	void sql_query_auto_links_del_link (GtkObject * obj,
					    SqlMemFlink * link, SqlQuery * q);

	
	gchar *join_type_get_label(JoinType join_type);
	gchar *join_type_get_xml(JoinType join_type);
	gchar *join_type_get_sql(JoinType join_type);
	gchar *join_type_get_operator(JoinType join_type);
	JoinType join_type_from_xml(gchar *jt_xml);

	/*
	 * Query WHERE/HAVING management
	 */
	QueryWhereNode *sql_query_cond_new_node (SqlQuery * q,
						 QueryWhereLogicType logic,
						 QueryWhereNode * parent,
						 QueryWhereNode * sibling);
	QueryWhereNode *sql_query_cond_new_leaf (SqlQuery * q,
						 QueryElement * l_elt,
						 QueryWhereOpType op,
						 QueryWhereRightOperandType
						 r_type, gpointer r_object,
						 QueryWhereNode * parent,
						 QueryWhereNode * sibling);
	void sql_query_cond_del (SqlQuery * q, QueryWhereNode * node);
	void sql_query_cond_move (SqlQuery * q,
				  QueryWhereNode * node,
				  QueryWhereNode * to_parent,
				  QueryWhereNode * to_sibling);
	void sql_query_cond_modif_node (SqlQuery * q,
					QueryWhereNode * node,
					QueryWhereLogicType logic);
	void sql_query_cond_modif_leaf (SqlQuery * q,
					QueryWhereNode * node,
					QueryElement * l_elt,
					QueryWhereOpType op,
					QueryWhereRightOperandType r_type,
					gpointer r_object);
	GNode *sql_query_gnode_find (SqlQuery * q, QueryWhereNode * node);

	/*
	 * Interrogation helper functions 
	 */

	/* tells if a particular field appears in the print list of the Query:
	   returns -1 if not present and >=0 otherwise, the number being the column
	   number of the SqlQueryRes of the SELECT query */
	gint sql_query_is_field_printed (SqlQuery * q, SqlMemField * field);

	/* returns the QueryElement of the query given the column number it occupies
	   int the SqlQueryRes of the SELECT row */
	QueryElement *sql_query_get_query_element (SqlQuery * q, gint pos);

	/* returns the position of a QueryElement in the resulting SqlQueryRes 
	   resultat (opposite of function above) */
	gint sql_query_get_queryres_pos (SqlQuery * q, QueryElement * qe);
	gint sql_query_get_queryres_pos_f (SqlQuery * q, SqlMemField * field);

	/* tells if a table appears somewhere in the query */
	gboolean sql_query_is_table_concerned (SqlQuery * q,
					       SqlMemTable * table);

	/*
	 * Getting the SQL versions of the query
	 * the missing_value can be NULL if the missing values are not defined
	 */
	gchar *sql_query_get_select_query (SqlQuery * q,
					   GSList * missing_values);

	/* getting the SQL query to select all the fields of a table given
	   the entry number in the SqlQueryRes for the SELECT version of the query
	   Used to request a confirmation before deletion
	   WARNING: returns only the part of the query after the FROM statement,
	   so it can be used with DELETE, UPDATE as well */
	gchar *sql_query_get_select_all_query_with_values (SqlQuery * q,
							   gint row);
#ifdef __cplusplus
}
#endif				/* __cplusplus */

#endif
