/**************************************************************************\
 gatos (General ATI TV and Overlay Software)

  Project Coordinated By Insomnia (Steaphan Greene)
  (insomnia@core.binghamton.edu)

  Copyright (C) 1999 Steaphan Greene, yvind Aabling, Octavian Purdila, 
	Vladimir Dergachev and Christian Lupien.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.

\**************************************************************************/

#ifndef GATOS_GATOS_H
#define GATOS_GATOS_H 1

#ifdef __cplusplus
extern "C" {
#endif

#ifdef GATOS_GATOS_C
#define WHERE
#else
#define WHERE	extern
#endif

#ifdef __KERNEL__
#define RETURN0		return 0
#define RETURN(VALUE)	return -VALUE
#else
#ifdef GATOS_GATOS_C
#define RETURN(VALUE)	return gatos_leave(VALUE)
#else
#define RETURN0		return 0
#define RETURN(VALUE)	return (errno = VALUE)
#endif
#endif

#include "config.h"
#include "i18n.h"
#include "channels.h"

#define GATOSBUTTONS	1
#define NEWSOUNDMIXER	1  /* Use new mixer filtering, undef if you
			    * want to use the old way.
			    *
			    * NOTE: gatos.mixerdev always containing
			    *       the *real* mixer channel (soundcard.h).
			    */

typedef unsigned char	u8 ;
typedef unsigned short	u16 ;
typedef unsigned int	u32 ;

#define LIMIT(X,A,B)	((X)<(A)) ? (A) : ((X)>(B)) ? (B) : (X)

#define ATICARDSMAX	8

struct gatosaticard {
  u32 busnum ;
  u32 cardnum ;
  u32 func ;
  u32 fbaddr ;
  u32 regaddr ;
  u32 biosaddr ;
  u16 vendorid ;
  u16 deviceid ;
  u16 revision ;
  u16 m64i2cinfo ;
  double dotclock ;
  double refclock ;
  int width ;
  int bus ;
  char *ident ;
  char *name ;
  char pciaddr[16] ;
  u8 m64mminfo[5] ;
  u8 r128mminfo[12] ;
  u8 r128tvinfo[12] ;
} ;

struct gatosi2cchip {
  int	deviceid ;
  int	revision ;
  char*	ident ;
  char*	name ;
  u8	addr ;
} ;

struct gatosinfo {

  /* Vital info */

  char statename[256] ;	/* State file name */
  int debug ;		/* Verbose/debug flag */
  int videoram ;	/* Total amount of video RAM in kb */
  int buffermem ;	/* Video RAM reserved for GATOS in kb */
  int bogomips ;	/* CPU BogoMIPS rating */
  int memfd ;		/* /dev/mem filehandle */
  int level ;		/* gatos_enter/leave housekeeping */
  int pending ;		/* gatos_enter/leave housekeeping */
  double refclock ;	/* ATI chip refclock crystal frequency */

  /* Real memory and video memory control */

  u32 fbaddr ;		/* FrameBuffer Aperture, */
  u32 regaddr ;		/* Register Aperture and */
  u32 biosaddr ;	/* BIOS adresses in real memory. */
  u32 buffer0, buffer1 ;/* Video capture buffer offsets */
  int captbufsize ;	/* Max size of each capture buffer in kb */

  /* Video window attributes */

  u32 colorkey ;	/* Color key */
  int gamma, cold ;	/* Gamma and red temp correction */
  int xsize, ysize ;	/* Size of video window */
  int xpos, ypos ;	/* Position of video window */
  int xpan, ypan ;	/* X virtual desktop panning (upper left corner) */
  int video, mapped, visibility ;	/* Window visibility housekeeping */
  int donegeometry ;	/* gatos_setgeometry called */

  int xcapt, ycapt ;	/* Capture buffer dimensions */
  int xstart, ystart, xend, yend ;     /* Cropped/zoomed subwindow position */
  int hcaptidx, vcaptidx, hcaptmax, vcaptmax ;

  /* Monitor and X attributes */

  int xdim, ydim ;	/* Monitor resolution */
  double dotclock ;	/* Monitor pixelclock frequency in MHz */
  int overclock ;

  /* Video source, format and attributes */

  int tvout ;		/* TV-Out enabled (boolean) */
  int mux ;		/* Video source (Bt829 MUX) */
  int format ;		/* Video format */
  int ntsc, pal, secam ;/* Video format (boolean) */
  int brightness, contrast, saturation, hue ;	/* Bt829 settings */

  /* Channel Management */

  int channel ;
  int numchannels ;
  struct gatos_ChannelInfo channelinfo[MAX_CHANNELS] ;
  int tvtype ;

  /* Tuner */

  int K ;		/* Tuner PLL constant (frequency) */
  int tuned ;		/* Tuner frequency has been programmed */
  double freq ;		/* Tuner frequency in MHz */
  double Fpc ;		/* Video signal Picture Carrier freq in MHz */
  double minfreq, maxfreq ;	/* Tuner frequency range in MHz */
  int senscan ;		/* Use sensative channel scan */
  int fm ;		/* Select FM Tuner Audio, not TV Audio*/

  /* I2C Driver Control */

  int i2c_mode ;	/* I2C Driver Number (1..5) */
  u32 bugfix_mppread ;	/* For TB Driver (i2cmpp.c) */

  /* Misc (unsorted) */

  int pid, usr1pid, stop, audio, chan ;
  char *extrastring[256];
  int esn;

  /* Sound settings */

  int mute, volume, stereo, sap ;
  int mixerfd, mixerdev, oldvol ;

  /* Closed captionning settings */
  int CCmode;

  /* Chip info */

  u8 boardinfo, tunertype ;	/* Video board info byte and tuner type */
  int cardtype, boardrev ;	/* Card type and revision */
  int audiotype ;		/* Audio */
  int nosa ;			/* Disable stand-alone capture cards */
  /* ATI and ImpacTV chips */
  struct gatosaticard ati ;	/* ATI graphics chip */
  struct gatosi2cchip itv ;	/* ImpacTV/2/2+ chip */
  /* Video decoder, tuner and board control */
  struct gatosi2cchip board ;	/* PCF8574 8-bit I/O port (0x70 to 0x7E) */
  struct gatosi2cchip bt829 ;	/* Bt829 video decoder chip (0x88,8A) */
  struct gatosi2cchip fi12xx ;	/* FI12XX Tuner Module (0xC0,C2,C4,C6) */
  struct gatosi2cchip eeprom ;	/* EEPROM (0xA8) */
  /* TeleText */
  struct gatosi2cchip saa5281 ;	/* SAA5281 TeleText decoder (0x22) */
  /* Audio chips */
  struct gatosi2cchip tda8425 ;	/* TDA8425 stereo processor (0x82) */
  struct gatosi2cchip tda9850 ;	/* TDA9850 BTSC stereo+SAP (0xB4) */
  struct gatosi2cchip tda9851 ;	/* TDA9851 BTSC economic stereo (0xB6) */
  struct gatosi2cchip msp3410 ;	/* MSP3410 multi-sound processor (0x80) */

  /* PCI info (ATI cards only) */

  struct gatosaticard aticard[ATICARDSMAX] ;
  char *pciaddr ;		/* PCI addr of ATI card to select */
  int cardidx, aticards ;	/* Index of selected ATI card, no. of cards */

} ;

#define CC_FIFO_SIZE 16
struct CCdata {
  int num_valid;
  u8 data[CC_FIFO_SIZE];
  u8 status[CC_FIFO_SIZE];
} ;

/* CCdata.status bits */
#define CCS_EDS  0x01 /* Set when the data is EDS */
#define CCS_HIGH 0x02 /* Set when the data is the high byte (second byte read
                         of word) */
#define CCS_OVER 0x04 /* Set when there was a FIFO overload before the read */
#define CCS_PAR  0x08 /* Set when there was an odd parity error */

/* Field "r128" is for Rage128, rest are Mach64 variants */
struct atiflags {

  /* Main chip type */
  int gx:1, cx:1, ct:1, et:1, lt:1, lg:1, lt_c:1, vt:1, gt:1, r128:1 ;

  /* Capability flags */
  int doesvtb:1, doesgtc:1, doesltc:1 ;

  /* Chip groups */
  int lt_c1u1:1, lt_c2u1:1, lt_c2u2:1, lt_c2u3:1 ;	/* lt_c */
  int ltm:1, xlxc:1 ;					/* lt_c */
  int gt2c:1, gt_c:1 ;					/* gt */
  int gt2c_b3u1:1, gt2c_b3u2:1 ;			/* gt & gt2c */
  int gt_c1u1:1, gt_c1u2:1, gt_c2u1:1, gt_c2u2:1 ;	/* gt & gt_c */

  /* Specific chip asics */

  int gx_c:1, gx_d:1, gx_e:1, gx_f:1 ;			/* gx */
  int lg_a1s1:1 ;					/* lg */

  /* lt_c */
  int lb_c1u1:1, ld_c1u1:1, li_c1u1:1, lp_c1u1:1, lq_c1u1:1 ;	/* lt_c1u1 */
  int lb_c2u1:1, ld_c2u1:1, li_c2u1:1, lp_c2u1:1, lq_c2u1:1 ;	/* lt_c2u1 */
  int lb_c2u2:1, ld_c2u2:1, li_c2u2:1, lp_c2u2:1, lq_c2u2:1 ;	/* lt_c2u2 */
  int lb_c2u3:1, ld_c2u3:1, li_c2u3:1, lp_c2u3:1, lq_c2u3:1 ;	/* lt_c2u3 */
  int lm_a1t1:1, lr_a1t1:1 ;					/* ltm */
  int gm_a1t1:1, gn_a1t1:1, go_a1t1:1, gp_a1t1:1, gr_a1t1:1, gs_a1t1:1 ;/* xlxc */

  /* vt */
  int vt_a3n1:1, vt_a3s1:1, vt_a4n1:1, vt_a4s1:1 ;	/* vt */
  int vt_b1s1:1, vt_b2u1:1, vt_b2u2:1, vt_b2u3:1 ;	/* vt */

  /* gt */
  int gt_a2s1:1, gt_b1s1:1, gt_b1s2:1 ;			/* gt */
  int gt_b2u1:1, gt_b2u2:1, gt_b2u3:1 ;			/* gt */

  /* gt & gt2c */
  int gv_b3u1:1, gz_b3u1:1, gw_b3u1:1, gy_b3u1:1, vv_b3u1:1, vw_b3u1:1 ;  /* gt2c_b3u1 */
  int gv_b3u2:1, gz_b3u2:1, gw_b3u2:1, gy_b3u2:1, vv_b3u2:1, vw_b3u2:1 ;  /* gt2c_b3u2 */

  /* gt & gt_c */
  int gd_c1u1:1, gp_c1u1:1, gi_c1u1:1, gb_c1u1:1 ;		/* gt_c1u1 */
  int gd_c1u2:1, gp_c1u2:1, gi_c1u2:1, gb_c1u2:1 ;		/* gt_c1u2 */
  int gd_c2u1:1, gp_c2u1:1, gi_c2u1:1, gb_c2u1:1, gq_c2u1:1 ;	/* gt_c2u1 */
  int gd_c2u2:1, gp_c2u2:1, gi_c2u2:1, gb_c2u2:1, gq_c2u2:1 ;	/* gt_c2u2 */

} ;

WHERE struct gatosinfo gatos ;
WHERE struct atiflags ati ;

/* ATI FrameBuffer Aperture Address */
WHERE volatile u8 *ATIFB ;

/* ATI Mach64 and Rage128 Register Aperture Adresses */
WHERE volatile u32 *MEM_0, *MEM_1 ;
WHERE volatile u8 *MMR ;

WHERE int *hactive, *vactive ;

#ifdef GATOSBUTTONS
WHERE int gatosbuttons[GATOSBUTTONS] ;
#endif

#ifdef GATOS_GATOS_C
static int hactive_pal[]  = {96,192,384,576,720,768} ;
static int vactive_pal[]  = {72,144,288,432,540,576} ;
static int hactive_ntsc[] = {80,160,320,480,600,640} ;
static int vactive_ntsc[] = {60,120,240,360,450,480} ;
#endif

#ifdef GATOS_GATOS_C
WHERE char *capturesizes[] = {"ICON", "QCIF", "CIF", "XCIF", "YCIF", "FULL"} ;
#else
WHERE char *capturesizes[] ;
#endif

#ifdef USE_DELAY_S
WHERE unsigned long loops_per_sec ;
#endif

/* Debugging */
#define VERBOSE		(gatos.debug&0x0001)
#define GATOSCALL	(gatos.debug&0x0002)
#define I2CTRAFFIC	(gatos.debug&0x0004)
#define CHANNELMGMT	(gatos.debug&0x0008)
#define GATOSASYNC	(gatos.debug&0x0010)
#define GATOSQUIET	0x8000

/* Card types */
#define CARDTYPES		8
#define CARD_STAND_ALONE	1
#define CARD_INTEGRATED		2
#define CARD_EXTERNAL_POD	3
#define CARD_ALL_IN_WONDER	4
#define CARD_ALL_IN_WONDER_PRO	5
#define CARD_ALL_IN_WONDER_128	6
#define CARD_NEC		7
#define CARD_INTEL_BOARD	8

/* Audio types */
#define AUDIO_ANALOG	1
#define AUDIO_TDA8425	2
#define AUDIO_TDA9850	3
#define AUDIO_TDA9851	4
#define AUDIO_MSP3410	5
#define AUDIO_TEA5711	6

/* Mixer types */
#define NO_MIXER	0	/* No mixer available */
#define SNDCARD_MIXER	1	/* Soundcard mixer */
#define BRDCARD_MIXER	2	/* ATI card mixer */
#define BOTHCARDS_MIXER	3	/* Soundcard and ATI card mixers */
#define DISABLE_MIXER	0xFE	/* Disable sound mixer support */

/* Video formats, follows Bt829 FORMAT (in IFORM register). */
#define NTSC_MN	1	/* NTSC-M/N	US, many others */
#define NTSC_J	2	/* NTSC-J	Japan */
#define PAL	3	/* PAL-B/G, D/K, H, I	Many (Europe) */
#define PAL_M	4	/* PAL-M	Brazil (NTSC-sized PAL) */
#define PAL_N	5	/* PAL-N	Paraguay, Uruguay */
#define SECAM	6	/* SECAM	East Europe, France, Middle East */
#define PAL_NC	7	/* PAL-Ncomb	Argentina */

/* Closed Captionning mode 0 means turn circuitry off. 
   To have both on, or them */
#define GET_CC  1
#define GET_EDS 2

/* Functions for gatos_init */
#define GATOS_TVOUT		1
#define GATOS_TUNER		2
#define GATOS_OVERLAY		4
#define GATOS_WRITE_BUFFER	8
#define GATOS_EXCLUSIVE		15
#define GATOS_READ_BUFFER	16

/* Routines that may (should!) be called before gatos_init() */

int	gatos_verbosity(void) ;
void	gatos_setverbosity(int) ;
int	gatos_setpci(char*) ;

/* GATOS API initialization/termination */

#ifdef GATOS_GATOS_C
static void	gatos_enter(char*, char*, ...) ;
static int	gatos_leave(int) ;
#endif
int	gatos_init(int) ;
int	gatos_inita(void) ;
int	gatos_initb(void) ;
int	gatos_terminate(void) ;
int	gatos_configfile(void) ;
int	gatos_savestate(char*) ;
int	gatos_loadstate(char*) ;
int	gatos_getintvalue(char*, int*);
int	gatos_getstringvalue(char*, char*, int);
int	gatos_setintvalue(char*, int);
int	gatos_setstringvalue(char*, char*);

/* AA Multi-capture board test */

int	gatos_nosa(void) ;
int	gatos_setnosa(int) ;

/* AA Test button stuff */

#ifdef GATOSBUTTONS
int	gatos_settestbutton(int, int) ;
int	gatos_testbutton(int) ;
#endif

/* */

int	gatos_format(void) ;
int	gatos_setformat(int) ;
int	gatos_mux(void) ;
int	gatos_setmux(int) ;
int	gatos_tvout(void) ;
int	gatos_ismonitor(void) ;
int	gatos_settvout(int) ;

/* Overlay stuff (CHANGING!) */
u32	gatos_colorkey(void) ;
int	gatos_setcolorkey(u32) ;
void	gatos_pollscreen(int) ;
int	gatos_setcapturesize(int, int) ;
int	gatos_setgeometry(int, int, int, int) ;
int	gatos_setsubgeometry(int, int, int, int) ;
int	gatos_xcaptmax(void) ;
int	gatos_ycaptmax(void) ;
int	gatos_xcaptmin(void) ;
int	gatos_ycaptmin(void) ;
int	gatos_xcapt(void) ;
int	gatos_ycapt(void) ;
int	gatos_xdim(void) ;
int	gatos_ydim(void) ;
int	gatos_xstart(void) ;
int	gatos_ystart(void) ;
int	gatos_xend(void) ;
int	gatos_yend(void) ;
int	gatos_setoverclock(int) ;
int	gatos_overclock(void) ;

/* Closed Captioning / Teletext */
int     gatos_setCC(int);
int     gatos_CC(void);
int     gatos_getCCdata(struct CCdata *);

/* Audio/video enable/disable */

int	gatos_enable_capture(int) ;
int	gatos_enable_sound(int) ;
int	gatos_enable_video(int) ;
int	gatos_setmapped(int) ;
int	gatos_setvisibility(int) ;

/* Sound control */

int	gatos_audiotype(void) ;
int	gatos_mixer(void) ;
int	gatos_setsoundmixer(int) ;
int	gatos_setmixer(int) ;
int	gatos_nummixers(void) ;
char*	gatos_mixername(int) ;
char**	gatos_mixernames(void) ;
char**	gatos_mixerlabels(void) ;
#ifdef NEWSOUNDMIXER
int	gatos_filtmix(int) ;
#endif /* NEWSOUNDMIXER */

int	gatos_mute(void) ;
int	gatos_setmute(int) ;
int	gatos_volume(void) ;
int	gatos_setvolume(int) ;
int	gatos_stereo(void) ;
int	gatos_setstereo(int) ;
int	gatos_sap(void) ;
int	gatos_setsap(int) ;

/* Channel management interface */

int	gatos_tvtype(void) ;
int	gatos_settvtype(int) ;
int	gatos_disablechan(int) ;
int	gatos_enablechan(int) ;
int	gatos_chanenabled(int) ;
int	gatos_delchan(int) ;
int	gatos_addchan(double, char*) ;
double	gatos_chanscan(int(*)(double)) ;
int	gatos_setchan(int) ;
int	gatos_channel(void) ;
int	gatos_numchans(void) ;
char*	gatos_channame(int) ;
int	gatos_setchanname(int, char*) ;
double	gatos_chanfreq(int) ;
int	gatos_setchanfreq(int, double) ;
int	gatos_chanvolume(int) ;
int	gatos_setchanvolume(int, int) ;
int	gatos_chanbrightness(int) ;
int	gatos_setchanbrightness(int, int) ;
int	gatos_chancontrast(int) ;
int	gatos_setchancontrast(int, int) ;
int	gatos_chanhue(int) ;
int	gatos_setchanhue(int, int) ;
int	gatos_chansaturation(int) ;
int	gatos_setchansaturation(int, int) ;

/* Tuner control */

double	gatos_minfreq(void) ;
double	gatos_maxfreq(void) ;
double	gatos_tuner(void) ;
int	gatos_settuner(double) ;
double	gatos_scan(double, double, int(*)(double,char*), int(*)(double)) ;
int	gatos_senscan(void) ;
int	gatos_setsenscan(int) ;
int	gatos_fm(void) ;
int	gatos_setfm(int) ;

/* Video control */

int	gatos_brightness(void) ;
int	gatos_setbrightness(int) ;
int	gatos_contrast(void) ;
int	gatos_setcontrast(int) ;
int	gatos_saturation(void) ;
int	gatos_setsaturation(int) ;
int	gatos_hue(void) ;
int	gatos_sethue(int) ;
int	gatos_gamma(void) ;
int	gatos_setgamma(int) ;
int	gatos_cold(void) ;
int	gatos_setcold(int) ;

/* Disk capture and playback */

#ifdef GATOS_GATOS_C
static void	gatos_stopchild(int) ;
#if 0 /* NIY */
static void	gatos_sigchild(int) ;
#endif
#endif
int	gatos_stop(void) ;
int	gatos_capture(char*, int) ;
int	gatos_playback(char*, int) ;

/* Debugging */

void	gatos_debug1(void) ; /* Dump Regs */
void	gatos_debug2(void) ; /* I2C info */
void	gatos_debug3(void) ; /* Unused */
void	gatos_debug4(void) ; /* Unused */

/* */
int	gatos_rgbcapture(unsigned char*) ;

int	gatos_setcaptidx(int, int) ;

#ifdef __cplusplus
}
#endif

#undef WHERE
#endif
