# Copyright (C) 2004,2005 by SICEm S.L.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
import gtk
import gobject

import os.path

from gazpacho.path import pixmaps_dir
from gazpacho.l10n import _

class Palette(gtk.VBox):

    __gsignals__ = {
        'toggled' : (gobject.SIGNAL_RUN_LAST, None, ())
        }
    
    def __init__(self, catalogs):
        gtk.VBox.__init__(self)

        # The GladeWidgetClass corrisponding to the selected button. NULL if
        # the selector button is pressed.
        self._current = None

        self.pack_start(self._selector_new(), False)
        self.pack_start(gtk.HSeparator(), False, padding=3)

        #The vbox that contains the titles of the sections
        self._groups_vbox = gtk.VBox()
        self.pack_start(self._groups_vbox, False)
        self.pack_start(gtk.HSeparator(), False, padding=3)

        #  Where we store the different catalogs
        self._notebook = gtk.Notebook()
        self._notebook.set_show_tabs(False)
        self._notebook.set_show_border(False)
        self.pack_end(self._notebook)

        # The number of sections in this palette
        self._nb_sections = 0

        # Each section of the palette has a button. This is the
        # sections_button_group list
        self._sections_button_group = []

        for catalog in catalogs:
            for group in catalog.widget_groups:
                self.append_widget_group(group)

    # some needed property accesors
    def get_current(self): return self._current
    def set_current(self, value): self._current = value
    current = property(get_current, set_current)
    
    def _selector_new(self):
        hbox = gtk.HBox()
        # The selector is a button that is clicked to cancel the add widget
        # action. This sets our cursor back to the "select widget" mode. This
        # button is part of the widgets_button_group, so that when no widget
        # is selected, this button is pressed.
        self._selector = gtk.RadioButton(None)
        self._selector.set_mode(False)
        self._selector.set_relief(gtk.RELIEF_NONE)
        
        # Each widget in a section has a button in the palette. This is the
        # button group, since only one may be pressed.
        self._widgets_button_group = self._selector.get_group()

        image = gtk.Image()
        image.set_from_file(os.path.join(pixmaps_dir, 'selector.png'))
        self._selector.add(image)
        self._selector.connect('toggled', self._on_button_toggled)

        # A label which contains the name of the class currently selected or
        # "Selector" if no widget class is selected
        self._label = gtk.Label(_('Selector'))
        self._label.set_alignment(0.0, 0.5)

        hbox.pack_start(self._selector, False, False)
        hbox.pack_start(self._label, padding=2)
        hbox.show_all()

        return hbox
    
    def _on_button_toggled(self, button):
        if not button.get_active():
            return

        if button == self._selector:
            self._current = None
            self._label.set_text(_('Selector'))
        else:
            self._current = button.get_data('user')
            self._label.set_text(self._current.name)

        self.emit('toggled')

    def _on_catalog_button_toggled(self, button):
        page = button.get_data('page')
        self._notebook.set_current_page(page)
        return True
    
    def do_toggled(self):
        pass
    
    def append_widget_group(self, group):
        page = self._nb_sections
        self._nb_sections += 1

        # add the button
        if not self._sections_button_group:
            button = gtk.RadioButton(None, group.title)
        else:
            button = gtk.RadioButton(self._sections_button_group[0],
                                     group.title)
        self._sections_button_group = button.get_group()
        button.set_mode(False)
        button.set_data('page', page)
        button.connect('toggled', self._on_catalog_button_toggled)

        self._groups_vbox.pack_start(button, False)

        # add the selection
        self._notebook.append_page(self._widget_table_create(group),
                                   gtk.Label(''))
        self._notebook.show()

    def _widget_table_create(self, group):
        vbox = gtk.VBox()
        for widget_class in group:
            if not widget_class.in_palette:
                continue
            
            radio = gtk.RadioButton(self._widgets_button_group[0])
            radio.connect('toggled', self._on_button_toggled)
            radio.set_data('user', widget_class)
            radio.set_mode(False)
            radio.set_relief(gtk.RELIEF_NONE)
            vbox.pack_start(radio, False, False)

            hbox = gtk.HBox(spacing=2)
            hbox.pack_start(widget_class.icon, False, False)
            radio.add(hbox)

            label = gtk.Label(widget_class.palette_name)
            label.set_alignment(0.0, 0.5)
            hbox.pack_start(label, padding=1)

            self._widgets_button_group = radio.get_group()

        scrolled_window = gtk.ScrolledWindow()
        scrolled_window.set_policy(gtk.POLICY_NEVER, gtk.POLICY_AUTOMATIC)
        scrolled_window.add_with_viewport(vbox)
        scrolled_window.set_shadow_type(gtk.SHADOW_NONE)
        scrolled_window.set_size_request(-1, 400)

        return scrolled_window
            
    def unselect_widget(self):
        self._selector.set_active(True)
    
    def select_widget(self, catalog, widgetname):
        # select the catalog
        pagenum = -1
        for button in self._groups_vbox.get_children():
            if button.get_label() == catalog:
                pagenum = button.get_data('page')
                break

        if pagenum == -1:
            return
        
        vbox = self._notebook.get_nth_page(pagenum).get_child().get_child()
        # then select the widget
        for button in vbox.get_children():
            wc = button.get_data('user')
            if wc.name == widgetname:
                button.clicked()
    
gobject.type_register(Palette)
