/* GBINDADMIN, a GTK+ frontend for ISC BIND
 * Copyright C 2005, 2006 Magnus Loef <magnus-swe@telia.com> 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307, USA.
 *
*/

#include "../config.h"
#include <gtk/gtk.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "allocate.h"
#include "gettext.h"
#include "widgets.h"
#include "commands.h"
#include "make_settings_entries.h"
#include "make_settings_combos.h"
#include "populate_domain_treeview.h"
#include "populate_resource_treeview.h"
#include "populate_log_tab.h"
#include "reread_conf.h"
#include "domain_settings_type_combo_changed.h"
#include "domain_settings_window.h"

#define MAX_CONF_LINE 1000

extern char SYSLOG_PATH_BUF[1024];
extern char CHROOT_PATH_BUF[1024];

extern char global_domain_name[1024];
extern char global_domain_address[1024];
extern char global_domain_type[1024];



void show_domain_settings_window(struct w *widgets)
{
    FILE *fp;
    long file_size;
    GtkWidget *frame;
    GtkWidget *table;
    GtkWidget *domain_settings_vbox;
    GtkTooltips *tooltips;
    gchar *utf8=NULL;
    gchar *info, *fwd_zone_path, *domain_line, *named_conf;
    char *line, *tmp;
    int i, domain_found = 0;
    int soa_found = 0;

    widgets->domain_settings_window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_position(GTK_WINDOW (widgets->domain_settings_window), GTK_WIN_POS_CENTER);
    gtk_widget_set_size_request(widgets->domain_settings_window, -1, -1);

    /* Set window information */
    info = g_strdup_printf(_("GBINDADMIN %s domain settings"), VERSION);
    gtk_window_set_title(GTK_WINDOW(widgets->domain_settings_window), info);
    g_free(info);

    g_signal_connect(GTK_WINDOW(widgets->domain_settings_window), "delete_event", 
		     G_CALLBACK (gtk_widget_destroy), NULL);

    domain_settings_vbox = gtk_vbox_new(FALSE, 0);
    gtk_container_add (GTK_CONTAINER (widgets->domain_settings_window), domain_settings_vbox);


    tooltips = gtk_tooltips_new();
    frame = gtk_frame_new("Domain settings:");

    /* A table with 8 settings and 2 columns */
    table = gtk_table_new(10, 2, FALSE);

    gtk_box_pack_start(GTK_BOX(domain_settings_vbox), frame, TRUE, TRUE, 1);
    gtk_container_add(GTK_CONTAINER(frame), table);


    /* Max length and input 350 chars */


    /* Domain name */
    widgets->domain_settings_entry[0] = make_entry_with_label(GTK_TABLE(table), _(" Domain name: "),    0,1,2,3,350);
    utf8 = g_locale_to_utf8(global_domain_name, strlen(global_domain_name), NULL, NULL, NULL);
    gtk_entry_set_text(GTK_ENTRY(widgets->domain_settings_entry[0]), utf8);
    gtk_tooltips_set_tip(tooltips, widgets->domain_settings_entry[0], _("some.domain.org"), NULL);

    /* Address range */
    widgets->domain_settings_entry[1] = make_entry_with_label(GTK_TABLE(table), _(" Address range: "),    0,1,3,4,350);
    utf8 = g_locale_to_utf8(global_domain_address, strlen(global_domain_address), NULL, NULL, NULL);
    gtk_entry_set_text(GTK_ENTRY(widgets->domain_settings_entry[1]), utf8);
    gtk_tooltips_set_tip(tooltips, widgets->domain_settings_entry[1], _("192.168.0.0-254"), NULL);

    /* Domain type combo */
    widgets->domain_settings_combo[0] = make_combo_with_label(GTK_TABLE(table), _(" Domain type: "),    0,1,4,5,350);
    tmp = allocate(1024);
    snprintf(tmp, 1000, "%s", _("master"));
    utf8 = g_locale_to_utf8(tmp, strlen(tmp), NULL, NULL, NULL);
    gtk_combo_box_append_text(GTK_COMBO_BOX(widgets->domain_settings_combo[0]), utf8);
    snprintf(tmp, 1000, "%s", _("slave"));
    utf8 = g_locale_to_utf8(tmp, strlen(tmp), NULL, NULL, NULL);
    gtk_combo_box_append_text(GTK_COMBO_BOX(widgets->domain_settings_combo[0]), utf8);
    free(tmp);

    if( strstr(global_domain_type, "master") )
      gtk_combo_box_set_active(GTK_COMBO_BOX(widgets->domain_settings_combo[0]), 0);

    if( strstr(global_domain_type, "slave") )
      gtk_combo_box_set_active(GTK_COMBO_BOX(widgets->domain_settings_combo[0]), 1);

    /* Different widgets should be visible depending on whats selected in the type combo */
    g_signal_connect_swapped(G_OBJECT(widgets->domain_settings_combo[0]), "changed",
                             G_CALLBACK(domain_settings_type_combo_changed), widgets);    



    /* Administrator email */
    widgets->domain_settings_entry[2] = make_entry_with_label(GTK_TABLE(table), _(" Administrator email: "),    0,1,5,6,350);
    gtk_tooltips_set_tip(tooltips, widgets->domain_settings_entry[2], _("hostmaster@some.domain.org"), NULL);

    /* Domain serial */
    widgets->domain_settings_entry[3] = make_entry_with_label(GTK_TABLE(table), _(" Domain serial: "),    0,1,6,7,350);
    gtk_tooltips_set_tip(tooltips, widgets->domain_settings_entry[3], _("Year+Month+Day+A 2-digit serial, IE: 2006040255"), NULL);


    fwd_zone_path = g_strdup_printf("%s/etc/sites/%s/forward.zone", CHROOT_PATH_BUF, global_domain_name);


    /* Insert widget values from the selected sites forward zone */
    if((fp=fopen(fwd_zone_path, "r"))==NULL)
    {
	printf("Error: could not find the sites forward zone here: %s\n", fwd_zone_path);

	if( utf8!=NULL )
	  g_free(utf8);

	g_free(fwd_zone_path);
	return;
    }
    fseek(fp, 0, SEEK_END);
    file_size = ftell(fp);
    rewind(fp);

    line = allocate(file_size+1);
    tmp = allocate(1024);

    while(fgets(line, file_size, fp)!=NULL)
    {    
	if( strstr(line, "#") )
	  continue;

	/* Administrator email */
	if( strstr(line, "SOA") && strlen(line) < 1000 )
	{
	    soa_found = 1;    
	
	    sscanf(line, "%*s %*s %*s %*s %s", tmp); /* -1 for the end dot */
	    utf8 = g_locale_to_utf8(tmp, strlen(tmp)-1, NULL, NULL, NULL);
	    gtk_entry_set_text(GTK_ENTRY(widgets->domain_settings_entry[2]), utf8);

	    /* The serial line should come directly after the SOA line */
	    while(fgets(line, file_size, fp)!=NULL)
	      if( strlen(line) < 1000 )
	        break;
	    sscanf(line, "%s", tmp);
	    utf8 = g_locale_to_utf8(tmp, strlen(tmp), NULL, NULL, NULL);
	    gtk_entry_set_text(GTK_ENTRY(widgets->domain_settings_entry[3]), utf8);

	    /* The rest of the lines are the sites update interval */

//	    sscanf(line, "%s", tmp);

	}
	if( soa_found )
	  break;
    }
    free(line);
    free(tmp);
    fclose(fp);


    /* allow-update */
    widgets->domain_settings_entry[4] = make_entry_with_label(GTK_TABLE(table), _(" Allow update: "),    0,1,7,8,350);
    gtk_tooltips_set_tip(tooltips, widgets->domain_settings_entry[4], _("Hosts allowed to submit dynamic DNS updates (DDNS) delimited with a \";\" IE: 127.0.0.1; 192.168.0.0/24; any; or none;"), NULL);

    /* allow-transfer */
    widgets->domain_settings_entry[5] = make_entry_with_label(GTK_TABLE(table), _(" Allow transfers: "),    0,1,8,9,350);
    gtk_tooltips_set_tip(tooltips, widgets->domain_settings_entry[5], _("Hosts allowed to make zone transfers (copy the zone information) delimited with a \";\" IE: 127.0.0.1; 192.168.0.0/24; any; or none;"), NULL);

    /* allow-query */
    widgets->domain_settings_entry[6] = make_entry_with_label(GTK_TABLE(table), _(" Allow query: "),    0,1,9,10,350);
    gtk_tooltips_set_tip(tooltips, widgets->domain_settings_entry[6], _("Hosts allowed to query this DNS, delimited with a \";\" IE: 127.0.0.1; 192.168.0.0/24; any; or none;"), NULL);


    /* Notify combo */
    widgets->domain_settings_combo[2] = make_combo_with_label(GTK_TABLE(table), _(" Send notifications: "),    0,1,10,11,350);
    tmp = allocate(1024);

    snprintf(tmp, 1000, "%s", _("yes"));
    utf8 = g_locale_to_utf8(tmp, strlen(tmp), NULL, NULL, NULL);
    gtk_combo_box_append_text(GTK_COMBO_BOX(widgets->domain_settings_combo[2]), utf8);
    
    snprintf(tmp, 1000, "%s", _("no"));
    utf8 = g_locale_to_utf8(tmp, strlen(tmp), NULL, NULL, NULL);
    gtk_combo_box_append_text(GTK_COMBO_BOX(widgets->domain_settings_combo[2]), utf8);

    snprintf(tmp, 1000, "%s", _("explicit"));
    utf8 = g_locale_to_utf8(tmp, strlen(tmp), NULL, NULL, NULL);
    gtk_combo_box_append_text(GTK_COMBO_BOX(widgets->domain_settings_combo[2]), utf8);

    free(tmp);


    /* Also-notify (for masters only) */
    widgets->domain_settings_entry[7] = make_entry_with_label(GTK_TABLE(table), _(" Also notify: "),    0,1,11,12,350);
    gtk_tooltips_set_tip(tooltips, widgets->domain_settings_entry[7], _("Servers that should recieve zone update notifications delimited with a \";\" IE: 127.0.0.1; 192.168.0.100;"), NULL);

    /* Allow-notify (for slaves only) */
    widgets->domain_settings_entry[8] = make_entry_with_label(GTK_TABLE(table), _(" Allow notify: "),    0,1,12,13,350);
    gtk_tooltips_set_tip(tooltips, widgets->domain_settings_entry[8], _("Servers allowed to notify this server for zone updates delimited with a \";\" IE: 127.0.0.1; 192.168.0.100; any; or none;"), NULL);

    /* Statistics combo */
    widgets->domain_settings_combo[1] = make_combo_with_label(GTK_TABLE(table), _(" Domain statistics: "),    0,1,13,14,350);
    tmp = allocate(1024);
    snprintf(tmp, 1000, "%s", _("yes"));
    utf8 = g_locale_to_utf8(tmp, strlen(tmp), NULL, NULL, NULL);
    gtk_combo_box_append_text(GTK_COMBO_BOX(widgets->domain_settings_combo[1]), utf8);
    snprintf(tmp, 1000, "%s", _("no"));
    utf8 = g_locale_to_utf8(tmp, strlen(tmp), NULL, NULL, NULL);
    gtk_combo_box_append_text(GTK_COMBO_BOX(widgets->domain_settings_combo[1]), utf8);
    free(tmp);

    /* masters (for slave domains only) */
    widgets->domain_settings_entry[9] = make_entry_with_label(GTK_TABLE(table), _(" Master domains: "),    0,1,14,15,350);
    gtk_tooltips_set_tip(tooltips, widgets->domain_settings_entry[9], _("IP-addresses of the master domains delimited with a \";\" IE: 127.0.0.1; 192.168.0.100; none;"), NULL);


    /* Insert widget values from named.conf */
    named_conf = g_strdup_printf("%s/etc/named.conf", CHROOT_PATH_BUF);
    if((fp=fopen(named_conf, "r"))==NULL)
    {
	printf("Error: could not find named.conf here:\n%s\n", named_conf);

	if( utf8!=NULL )
	  g_free(utf8);

	g_free(named_conf);

	return;
    }
    fseek(fp, 0, SEEK_END);
    file_size = ftell(fp);
    rewind(fp);

    line = allocate(file_size+1);
    tmp = allocate(1024);


    domain_line = g_strdup_printf("zone \"%s\" IN", global_domain_name);

    while(fgets(line, file_size, fp)!=NULL)
    {    
	if( strstr(line, "#") )
	  continue;

	if( strstr(line, domain_line) )
	{
	    domain_found = 1;

	    while(fgets(line, file_size, fp)!=NULL)
	    {    
		if( strstr(line, "#") )
		  continue;

		/* Stop when the correct zone has been read */
		if( strstr(line, "zone \"") && domain_found )
	          break;

		if( strstr(line, "allow-update") && strstr(line, "{") )
		{
		    for(i=0; i<1000; i++)
		      if( line[i]=='{' )
		        break;

		    snprintf(tmp, 1000, "%s", &line[i+2]);
		    tmp[strlen(tmp)-4]='\0';
		    utf8 = g_locale_to_utf8(tmp, strlen(tmp), NULL, NULL, NULL);
		    gtk_entry_set_text(GTK_ENTRY(widgets->domain_settings_entry[4]), utf8);
		}

		if( strstr(line, "allow-transfer") && strstr(line, "{") )
		{
		    for(i=0; i<1000; i++)
		      if( line[i]=='{' )
		        break;
		    
		    snprintf(tmp, 1000, "%s", &line[i+2]);
		    tmp[strlen(tmp)-4]='\0';
		    utf8 = g_locale_to_utf8(tmp, strlen(tmp), NULL, NULL, NULL);
		    gtk_entry_set_text(GTK_ENTRY(widgets->domain_settings_entry[5]), utf8);
		}

		if( strstr(line, "allow-query") && strstr(line, "{") )
		{
		    for(i=0; i<1000; i++)
		      if( line[i]=='{' )
		        break;
		    
		    snprintf(tmp, 1000, "%s", &line[i+2]);
		    tmp[strlen(tmp)-4]='\0';
		    utf8 = g_locale_to_utf8(tmp, strlen(tmp), NULL, NULL, NULL);
		    gtk_entry_set_text(GTK_ENTRY(widgets->domain_settings_entry[6]), utf8);
		}

		if( strstr(line, "also-notify") && strstr(line, "{") )
		{
		    for(i=0; i<1000; i++)
		      if( line[i]=='{' )
		        break;
		    
		    snprintf(tmp, 1000, "%s", &line[i+2]);
		    tmp[strlen(tmp)-4]='\0';
		    if( strlen(tmp) > 5 )
		    {
			utf8 = g_locale_to_utf8(tmp, strlen(tmp), NULL, NULL, NULL);
			gtk_entry_set_text(GTK_ENTRY(widgets->domain_settings_entry[7]), utf8);
		    }
		}

		if( strstr(line, "allow-notify") && strstr(line, "{") )
		{
		    for(i=0; i<1000; i++)
		      if( line[i]=='{' )
		        break;
		    
		    snprintf(tmp, 1000, "%s", &line[i+2]);
		    tmp[strlen(tmp)-4]='\0';

		    if( strlen(tmp) > 3 )
		    {
			utf8 = g_locale_to_utf8(tmp, strlen(tmp), NULL, NULL, NULL);
			gtk_entry_set_text(GTK_ENTRY(widgets->domain_settings_entry[8]), utf8);
		    }
		}

		if( strstr(line, "masters") && strstr(line, "{") )
		{
		    for(i=0; i<1000; i++)
		      if( line[i]=='{' )
		        break;
		    
		    snprintf(tmp, 1000, "%s", &line[i+2]);
		    tmp[strlen(tmp)-4]='\0';
		    utf8 = g_locale_to_utf8(tmp, strlen(tmp), NULL, NULL, NULL);
		    gtk_entry_set_text(GTK_ENTRY(widgets->domain_settings_entry[9]), utf8);
		}

		if( strstr(line, "zone-statistics") && strstr(line, ";") )
		{
		    snprintf(tmp, 1000, "%s", &line[16]);
		    tmp[strlen(tmp)-2]='\0';

		    if( strstr(tmp, "yes") )
    		      gtk_combo_box_set_active(GTK_COMBO_BOX(widgets->domain_settings_combo[1]), 0);
		    else
		    if( strstr(tmp, "no") )
    		      gtk_combo_box_set_active(GTK_COMBO_BOX(widgets->domain_settings_combo[1]), 1);
		}

		if( strstr(line, "notify") && ! strstr(line, "also-notify") 
		&& ! strstr(line, "allow-notify") && strstr(line, ";") )
		{
		    snprintf(tmp, 1000, "%s", &line[8]);
		    tmp[strlen(tmp)-2]='\0';

		    if( strstr(tmp, "yes") )
    		      gtk_combo_box_set_active(GTK_COMBO_BOX(widgets->domain_settings_combo[2]), 0);
		    else
		    if( strstr(tmp, "no") )
    		      gtk_combo_box_set_active(GTK_COMBO_BOX(widgets->domain_settings_combo[2]), 1);
		    else
		    if( strstr(tmp, "explicit") )
    		      gtk_combo_box_set_active(GTK_COMBO_BOX(widgets->domain_settings_combo[2]), 2);
		}

	    }
	    if( domain_found )
	      break;
	}
    }
    fclose(fp);
    free(line);
    free(tmp);
    g_free(named_conf);

    if( utf8!=NULL )
      g_free(utf8);

    GtkWidget *domain_settings_hbuttonbox = gtk_hbutton_box_new();
    gtk_box_pack_start(GTK_BOX(domain_settings_vbox), domain_settings_hbuttonbox, FALSE, FALSE, 0);
    gtk_button_box_set_layout(GTK_BUTTON_BOX(domain_settings_hbuttonbox), GTK_BUTTONBOX_SPREAD);

    GtkWidget *apply_domain_settings_button = gtk_button_new_from_stock(GTK_STOCK_APPLY);
    gtk_container_add(GTK_CONTAINER(domain_settings_hbuttonbox), apply_domain_settings_button);
    g_signal_connect_swapped(G_OBJECT(apply_domain_settings_button), "clicked", 
                             G_CALLBACK(apply_domain_settings_button_clicked), widgets);
    
    GtkWidget *cancel_domain_settings_button = gtk_button_new_from_stock(GTK_STOCK_CANCEL);
    gtk_container_add(GTK_CONTAINER(domain_settings_hbuttonbox), cancel_domain_settings_button);
    g_signal_connect_swapped(G_OBJECT(cancel_domain_settings_button), "clicked", 
                             G_CALLBACK(cancel_domain_settings_button_clicked), widgets);

    gtk_widget_show_all(widgets->domain_settings_window);

    /* Domain type determines what entries are visible */
    domain_settings_type_combo_changed(widgets);
}



void cancel_domain_settings_button_clicked(struct w *widgets)
{
    gtk_widget_destroy(widgets->domain_settings_window);
}



/* FIXME... */
void apply_domain_settings_button_clicked(struct w *widgets)
{
    /* The apply button in the domain settings window */
    FILE *fp;
    long file_size, type_pos=0;
    int i, new_domain_name=0, domain_type_slave=0;
    char *line, *new_conf, *tmp;
    gint active_index = 0;
    gchar *cmd, *info, *named_conf, *utf8=NULL;
    char *forward, *reverse_file;
    int found_rew_zone = 0;
    int found_fwd_zone = 0;

    G_CONST_RETURN gchar *domain_name = gtk_entry_get_text(GTK_ENTRY(widgets->domain_settings_entry[0]));
    // Not needed atm.  G_CONST_RETURN gchar *address_range = gtk_entry_get_text(GTK_ENTRY(widgets->settings_entry[1]));
    G_CONST_RETURN gchar *admin_email = gtk_entry_get_text(GTK_ENTRY(widgets->domain_settings_entry[2]));
    G_CONST_RETURN gchar *domain_serial = gtk_entry_get_text(GTK_ENTRY(widgets->domain_settings_entry[3]));
    G_CONST_RETURN gchar *allow_update = gtk_entry_get_text(GTK_ENTRY(widgets->domain_settings_entry[4]));
    G_CONST_RETURN gchar *allow_transfer = gtk_entry_get_text(GTK_ENTRY(widgets->domain_settings_entry[5]));
    G_CONST_RETURN gchar *allow_query = gtk_entry_get_text(GTK_ENTRY(widgets->domain_settings_entry[6]));
    G_CONST_RETURN gchar *also_notify = gtk_entry_get_text(GTK_ENTRY(widgets->domain_settings_entry[7]));
    G_CONST_RETURN gchar *allow_notify = gtk_entry_get_text(GTK_ENTRY(widgets->domain_settings_entry[8]));
    G_CONST_RETURN gchar *masters = gtk_entry_get_text(GTK_ENTRY(widgets->domain_settings_entry[9]));


    // FIXME: add update interval 3H 2T etc etc (in the zone files)


    gchar *fwd_zone_path = g_strdup_printf("%s//etc/sites/%s/forward.zone", CHROOT_PATH_BUF, global_domain_name);
    gchar *rev_zone_path = g_strdup_printf("%s/etc/sites/%s/reverse.zone", CHROOT_PATH_BUF, global_domain_name);

    gchar *fwd_decl = g_strdup_printf("\"%s\"", global_domain_name);


    named_conf = g_strdup_printf("%s/etc/named.conf", CHROOT_PATH_BUF);

    /* Check that we have a complete site with all required files */
    if( ! file_exists(named_conf) )
    {
	info = g_strdup_printf(_("Error: named.conf does not exist here:\n%s\n"), named_conf);
	utf8 = g_locale_to_utf8(info, strlen(info), NULL, NULL, NULL);
	show_info(info);
	g_free(info);
	if( utf8!=NULL )
    	  g_free(utf8);
	g_free(fwd_zone_path);
	g_free(rev_zone_path);
	g_free(fwd_decl);
	g_free(named_conf);
	return;
    }

    if( ! file_exists(fwd_zone_path) )
    {
	info = g_strdup_printf(_("Error: Forward zone does not exist here:\n%s\n"), fwd_zone_path);
	utf8 = g_locale_to_utf8(info, strlen(info), NULL, NULL, NULL);
	show_info(info);
	g_free(info);
	if( utf8!=NULL )
    	  g_free(utf8);
	g_free(fwd_zone_path);
	g_free(rev_zone_path);
	g_free(fwd_decl);
	g_free(named_conf);
	return;
    }

    if( ! file_exists(rev_zone_path) )
    {
	info = g_strdup_printf(_("Error: Reverse zone does not exist here:\n%s\n"), rev_zone_path);
	utf8 = g_locale_to_utf8(info, strlen(info), NULL, NULL, NULL);
	show_info(info);
	g_free(info);
	if( utf8!=NULL )
    	  g_free(utf8);
	g_free(fwd_zone_path);
	g_free(rev_zone_path);
	g_free(fwd_decl);
	g_free(named_conf);
	return;
    }



    /* DNS type (combo) */
    gchar *domain_type = NULL;

    /* DNS statistics (combo) */
    gchar *domain_statistics = NULL;

    /* DNS Notify (combo) */
    gchar *domain_notify = NULL;

    /* DNS Type */    
    active_index = gtk_combo_box_get_active(GTK_COMBO_BOX(widgets->domain_settings_combo[0]));
    if( active_index <= 0 )
	domain_type = g_strdup_printf("master");
    else
    if( active_index == 1 )
	domain_type = g_strdup_printf("slave");
    else
      {
	 info = g_strdup_printf(_("DNS Type not set, no changes made.\n"));
	 utf8 = g_locale_to_utf8(info, strlen(info), NULL, NULL, NULL);
	 show_info(info);
	 g_free(info);
	 if( utf8!=NULL )
    	   g_free(utf8);
	 g_free(fwd_zone_path);
	 g_free(rev_zone_path);
	 g_free(fwd_decl);
	 g_free(named_conf);
	 return;
      }


    /* DNS Statistics */    
    active_index = gtk_combo_box_get_active(GTK_COMBO_BOX(widgets->domain_settings_combo[1]));
    if( active_index <= 0 )
	domain_statistics = g_strdup_printf("yes");
    else
    if( active_index == 1 )
	domain_statistics = g_strdup_printf("no");
    else
      {
	 domain_statistics = g_strdup_printf("yes");
	 info = g_strdup_printf(_("DNS statistics not set, setting it to yes\n"));
	 utf8 = g_locale_to_utf8(info, strlen(info), NULL, NULL, NULL);
	 show_info(info);
	 g_free(info);
	 if( utf8!=NULL )
    	   g_free(utf8);
	 /* Dont return. */
      }


    /* DNS Notify */    
    active_index = gtk_combo_box_get_active(GTK_COMBO_BOX(widgets->domain_settings_combo[2]));
    if( active_index <= 0 )
	domain_notify = g_strdup_printf("yes");
    else
    if( active_index == 1 )
	domain_notify = g_strdup_printf("no");
    else
    if( active_index == 2 )
	domain_notify = g_strdup_printf("explicit");
    else
      {
	 domain_notify = g_strdup_printf("no");
	 info = g_strdup_printf(_("DNS notification not set, setting it to no.\n"));
	 utf8 = g_locale_to_utf8(info, strlen(info), NULL, NULL, NULL);
	 show_info(info);
	 g_free(info);
	 if( utf8!=NULL )
    	   g_free(utf8);
	 /* Dont return. */
      }


    /* Some configuration values cant be used in slave mode */
    if( strstr(domain_type, "slave") )
      domain_type_slave = 1;
    else
      domain_type_slave = 0;

    /* If a slave domain selected it must have master domain servers */
    if( domain_type_slave && strlen(masters) < 3 )
    {
	info = g_strdup_printf(_("Error: You have selected a slave domain,\nSlave domains requires master domain servers.\n"));
	utf8 = g_locale_to_utf8(info, strlen(info), NULL, NULL, NULL);
	show_info(info);
	g_free(info);
	if( utf8!=NULL )
    	  g_free(utf8);
	g_free(fwd_zone_path);
	g_free(rev_zone_path);
	g_free(fwd_decl);
	g_free(domain_type);
	g_free(domain_statistics);
	g_free(named_conf);
	return;
    }

    
    /* Domain names differ, set new domain name as selected and sanity check */
    if( strcmp(global_domain_name, domain_name) )
    {
	new_domain_name = 1;

	gchar *new_domain_dir = g_strdup_printf("%s/etc/sites/%s", CHROOT_PATH_BUF, domain_name);
	if( file_exists(new_domain_dir) )
	{
	    info = g_strdup_printf(_("Error: This domain already exists here:\n%s\n"), new_domain_dir);
	    utf8 = g_locale_to_utf8(info, strlen(info), NULL, NULL, NULL);
	    show_info(info);
	    g_free(info);
	    if( utf8!=NULL )
    	      g_free(utf8);
	    g_free(new_domain_dir);
	    g_free(fwd_zone_path);
	    g_free(rev_zone_path);
	    g_free(fwd_decl);
	    g_free(domain_type);
	    g_free(domain_statistics);
	    g_free(named_conf);
	    return;
	}
	g_free(new_domain_dir);
    }



    /* Change this sites forward zone: SOA, serial, email and update interval */
    if((fp=fopen(fwd_zone_path, "r"))==NULL)
    {
	/* Sanity check in teh pirkinning */
	if( utf8!=NULL )
    	  g_free(utf8);
	g_free(fwd_zone_path);
	g_free(rev_zone_path);
	g_free(fwd_decl);
	g_free(domain_statistics);
	g_free(domain_type);
	g_free(named_conf);
	return;
    }
    fseek(fp, 0, SEEK_END);
    file_size = ftell(fp);
    rewind(fp);
    
    line = allocate(file_size+1);
    new_conf = allocate(file_size+16384); /* For changes */
    tmp = allocate(1024);

    i = 0;
    while(fgets(line, file_size, fp)!=NULL)
    {    
	if( strstr(line, "#") )
	{
	    strcat(new_conf, line);
	    continue;
	}
	
	/* Change domain name and email address */
	if( strstr(line, "SOA") && strstr(line, "(") )
	{
	    snprintf(tmp, 1000, "@\tIN\tSOA\t%s.\t%s. (\n", domain_name, admin_email);
	    strcat(new_conf, tmp);
	
	    /* Change the serial and update interval */
	    while(fgets(line, file_size, fp)!=NULL)
	    {    
		i++;
		
		if( strstr(line, "#") )
		{
		    strcat(new_conf, line);
		    continue;
		}

		/* Change the serial number */
		if( i == 1 )
		{
		    snprintf(tmp, 1000, "\t\t\t%s\t;\n", domain_serial);
		    strcat(new_conf, tmp);
		    continue;
		}

		/* Fixme: change update interval */
		if( i == 2 )
		{
		    strcat(new_conf, line);		
		    continue;
		}
		if( i == 3 )
		{
		    strcat(new_conf, line);		
		    continue;
		}
		if( i == 4 )
		{
		    strcat(new_conf, line);		
		    continue;
		}
		if( i == 5 )
		{
		    strcat(new_conf, line);		
		    continue;
		}

		
		strcat(new_conf, line);		
	
		if( strstr(line, ")") )
		  break;
	    }
	}
	else
	  strcat(new_conf, line);
    }
    free(line);
    free(tmp);
    fclose(fp);

    /* Write the new conf */
    if((fp=fopen(fwd_zone_path, "w+"))==NULL)
    {
	info = g_strdup_printf(_("Error: Cannot write the forward zone here:\n%s\n"), fwd_zone_path);
	utf8 = g_locale_to_utf8(info, strlen(info), NULL, NULL, NULL);
	show_info(info);
	g_free(info);
	free(new_conf);
	if( utf8!=NULL )
    	  g_free(utf8);
	g_free(fwd_zone_path);
	g_free(rev_zone_path);
	g_free(fwd_decl);
	g_free(domain_statistics);
	g_free(domain_type);
	g_free(named_conf);
	return;
    }
    fputs(new_conf, fp);
    fclose(fp);
    free(new_conf);



    if( new_domain_name )
    {
	/* sed -e s/drum.se/bbb.se/g filepath > filepath.tmp */
	cmd = g_strdup_printf("%s -e s/%s/%s/g %s > %s.tmp", 
              SED_BINARY, global_domain_name, domain_name, fwd_zone_path, fwd_zone_path);

	if( ! run_command(cmd) )
	{
	    printf("Error running command:\n%s\n", cmd);
	    if( utf8!=NULL )
    	      g_free(utf8);
	    g_free(fwd_zone_path);
	    g_free(rev_zone_path);
	    g_free(fwd_decl);
	    g_free(domain_statistics);
	    g_free(domain_type);
	    g_free(named_conf);
	    return;    
	}
	g_free(cmd);

	/* mv /path/to/zonefile.tmp path/to/zonefile */
	cmd = g_strdup_printf("mv %s.tmp %s", fwd_zone_path, fwd_zone_path);

	if( ! run_command(cmd) )
	{
	    printf("Error running command:\n%s\n", cmd);
	    if( utf8!=NULL )
    	      g_free(utf8);
	    g_free(fwd_zone_path);
	    g_free(rev_zone_path);
	    g_free(fwd_decl);
	    g_free(domain_statistics);
	    g_free(domain_type);
	    g_free(named_conf);
	    return;    
	}
	g_free(cmd);
    }



    /* Change this sites reverse zone: SOA, serial, email and update interval */
    /* Duplicate of the above incase it needs other mods */
    if((fp=fopen(rev_zone_path, "r"))==NULL)
    {
	/* Sanity checked in teh pirkinning */
	if( utf8!=NULL )
    	  g_free(utf8);
	g_free(fwd_zone_path);
	g_free(rev_zone_path);
	g_free(fwd_decl);
	g_free(domain_statistics);
	g_free(domain_type);
	g_free(named_conf);
	return;
    }
    fseek(fp, 0, SEEK_END);
    file_size = ftell(fp);
    rewind(fp);
    
    line = allocate(file_size+1);
    new_conf = allocate(file_size+16384); /* For changes */
    tmp = allocate(1024);

    i = 0;
    while(fgets(line, file_size, fp)!=NULL)
    {    
	if( strstr(line, "#") )
	{
	    strcat(new_conf, line);
	    continue;
	}
	
	/* Change domain name and email address */
	if( strstr(line, "SOA") && strstr(line, "(") )
	{
	    snprintf(tmp, 1000, "@\tIN\tSOA\t%s.\t%s. (\n", domain_name, admin_email);
	    strcat(new_conf, tmp);
	
	    /* Change the serial and update interval */
	    while(fgets(line, file_size, fp)!=NULL)
	    {    
		i++;
		
		if( strstr(line, "#") )
		{
		    strcat(new_conf, line);
		    continue;
		}

		/* Change the serial number */
		if( i == 1 )
		{
		    snprintf(tmp, 1000, "\t\t\t%s\t;\n", domain_serial);
		    strcat(new_conf, tmp);
		    continue;
		}

		/* Fixme: change update interval */
		if( i == 2 )
		{
		    strcat(new_conf, line);		
		    continue;
		}
		if( i == 3 )
		{
		    strcat(new_conf, line);		
		    continue;
		}
		if( i == 4 )
		{
		    strcat(new_conf, line);		
		    continue;
		}
		if( i == 5 )
		{
		    strcat(new_conf, line);		
		    continue;
		}

		
		strcat(new_conf, line);		
	
		if( strstr(line, ")") )
		  break;
	    }
	}
	else
	  strcat(new_conf, line);
    }
    free(line);
    free(tmp);
    fclose(fp);

    /* Write the new conf */
    if((fp=fopen(rev_zone_path, "w+"))==NULL)
    {
	info = g_strdup_printf(_("Error: Cannot write the reverse zone here:\n%s\n"), rev_zone_path);
	utf8 = g_locale_to_utf8(info, strlen(info), NULL, NULL, NULL);
	show_info(info);
	g_free(info);
	free(new_conf);
	if( utf8!=NULL )
    	  g_free(utf8);
	g_free(fwd_zone_path);
	g_free(rev_zone_path);
	g_free(fwd_decl);
	g_free(domain_statistics);
	g_free(domain_type);
	g_free(named_conf);
	return;
    }
    fputs(new_conf, fp);
    fclose(fp);
    free(new_conf);



    if( new_domain_name )
    {
	/* sed -e s/drum.se/bbb.se/g filepath > filepath.tmp */
	cmd = g_strdup_printf("%s -e s/%s/%s/g %s > %s.tmp", 
              SED_BINARY, global_domain_name, domain_name, rev_zone_path, rev_zone_path);

	if( ! run_command(cmd) )
	{
	    printf("Error running command:\n%s\n", cmd);
	    if( utf8!=NULL )
    	      g_free(utf8);
	    g_free(fwd_zone_path);
	    g_free(rev_zone_path);
	    g_free(fwd_decl);
	    g_free(domain_statistics);
	    g_free(domain_type);
	    g_free(named_conf);
	    return;    
	}
	g_free(cmd);

	/* mv /path/to/zonefile.tmp path/to/zonefile */
	cmd = g_strdup_printf("mv %s.tmp %s", rev_zone_path, rev_zone_path);

	if( ! run_command(cmd) )
	{
	    printf("Error running command:\n%s\n", cmd);
	    if( utf8!=NULL )
    	      g_free(utf8);
	    g_free(fwd_zone_path);
	    g_free(rev_zone_path);
	    g_free(fwd_decl);
	    g_free(domain_statistics);
	    g_free(domain_type);
	    g_free(named_conf);
	    return;    
	}
	g_free(cmd);
    }


    /* Rename this sites directory FIXME, only if domain name change */
    /* mv /path/to/sitedir/old-domain-dir path/to/sitedir/new-domain-dir */
    if( new_domain_name )
    {
	cmd = g_strdup_printf("mv %s/etc/sites/%s %s/etc/sites/%s", CHROOT_PATH_BUF, global_domain_name, CHROOT_PATH_BUF, domain_name);
	if( ! run_command(cmd) )
	{
	    printf("Error running command:\n%s\n", cmd);
	    if( utf8!=NULL )
    	      g_free(utf8);
	    g_free(fwd_zone_path);
	    g_free(rev_zone_path);
	    g_free(fwd_decl);
	    g_free(domain_statistics);
	    g_free(domain_type);
	    g_free(named_conf);
	    return;    
	}
	g_free(cmd);
    }


    /* Change named.conf's values for this site (only the options) */
    if((fp=fopen(named_conf, "r"))==NULL)
    {
        /* Sanity checked in teh pirkinning */
        printf("Couldnt find:\n%s\n", named_conf);
	if( utf8!=NULL )
    	  g_free(utf8);
	g_free(fwd_zone_path);
	g_free(rev_zone_path);
	g_free(fwd_decl);
	g_free(domain_statistics);
	g_free(domain_type);
	g_free(named_conf);
        return;
    }
    fseek(fp, 0, SEEK_END);
    file_size = ftell(fp);
    rewind(fp);

    line = allocate(file_size+1);
    forward = allocate(MAX_CONF_LINE+1);
    reverse_file = allocate(MAX_CONF_LINE+1);
    tmp = allocate(MAX_CONF_LINE+1);

    new_conf = allocate(file_size+16384);

    while(fgets(line, file_size, fp)!=NULL)
    {
	if( strlen(line) > MAX_CONF_LINE-3 )
	  continue;

	/* Skip comments */
	if( strstr(line, "/*") )
	{
	    while(fgets(line, file_size, fp)!=NULL)
	    {
		if( strstr(line, "*/") )
		  break;
	    }	
	}

	/* Skip comments */
	if( strstr(line, "//") )
	  continue;

	/* Forward zone begin */
	if( strstr(line, "zone \"") && strstr(line, fwd_decl) 
	&& ! found_fwd_zone )
	{
	    found_fwd_zone = 1;
	    strcat(new_conf, line);

	    /* For matching against the reverse zone file */
	    sscanf(line, "%*s %s", forward);
	    snprintf(tmp, MAX_CONF_LINE-1, "%s", &forward[1]);
	    snprintf(forward, strlen(tmp), "%s", tmp);
	    tmp[strlen(tmp)-1]='\0';
	    snprintf(reverse_file, strlen(tmp)+3, "/%s/", tmp);


	    /* Handle the options */
	    while(fgets(line, file_size, fp)!=NULL)
	    {
		/* Remove allow-update line */
		if( strstr(line, "allow-update") && strstr(line, "{") )
		  continue;
		    
		/* Remove masters line */
		if( strstr(line, "masters") && strstr(line, "{") )
		  continue;

		/* Remove allow-notify line */
		if( strstr(line, "allow-notify") && strstr(line, "{") )
		  continue;


		    
		/* The allow-transfer line must exist, also handles other options */
		if( strstr(line, "type ") && strstr(line, ";") )
		{
		    snprintf(tmp, 1000, "\ttype %s;\n", domain_type);
		    strcat(new_conf, tmp);
		}
		else
		if( strstr(line, "allow-transfer") && strstr(line, "{") )
		{
		    snprintf(tmp, 1000, "\tallow-transfer { %s };\n", allow_transfer);
		    strcat(new_conf, tmp);

		    /* If the domain type is master, we can have allow-update */
		    if( ! domain_type_slave )
		    {
			snprintf(tmp, 1000, "\tallow-update { %s };\n", allow_update);
		    	strcat(new_conf, tmp);
		    }

		    if( domain_type_slave )
		    {
			snprintf(tmp, 1000, "\tmasters { %s };\n", masters);
			strcat(new_conf, tmp);

			if( strlen(allow_notify) > 3 )
			{
			    snprintf(tmp, 1000, "\tallow-notify { %s };\n", allow_notify);
			    strcat(new_conf, tmp);
			}
		    }
		}
		else
		if( strstr(line, "allow-query") && strstr(line, "{") )
		{
		    snprintf(tmp, 1000, "\tallow-query { %s };\n", allow_query);
		    strcat(new_conf, tmp);
		}
		else
		if( strstr(line, "zone-statistics") && strstr(line, ";") )
		{
		    snprintf(tmp, 1000, "\tzone-statistics %s;\n", domain_statistics);
		    strcat(new_conf, tmp);
		}
		else
		if( strstr(line, "notify") && ! strstr(line, "also-notify")
		&& ! strstr(line, "allow-notify") && strstr(line, ";") )
		{
		    snprintf(tmp, 1000, "\tnotify %s;\n", domain_notify);
		    strcat(new_conf, tmp);
		}
		else
		if( strstr(line, "also-notify") && strstr(line, ";") )
		{
		    snprintf(tmp, 1000, "\talso-notify { %s };\n", also_notify);
		    strcat(new_conf, tmp);
		}
		else
		  strcat(new_conf, line);
		    
		if( strstr(line, "zone \"") && ! strstr(line, "forward.zone") )
		  break;
	    }
	}
	else
	  strcat(new_conf, line);



	/* Reverse zone begin (the first zone after the fwd one) */
	if( strstr(line, "zone ") && found_fwd_zone && ! found_rew_zone 
	&& strstr(line, "in-addr.arpa") )
	{
	    /* Dont handle root.hints or the local zone */
	    if( ! strcmp(forward, ".") || ! strcmp(forward, "0.0.127") )
	    {
		strcat(new_conf, line);
	        continue;
	    }
	    
	    while(fgets(line, file_size, fp)!=NULL)
	    if( strstr(line, "type ") && strstr(line, ";") )
	    {
		/* Store position of the "type" line */
		type_pos = ftell(fp);
	    }
	    else  /* The correct reverse zone has been found */
	    if( strstr(line, reverse_file) && strlen(line) < MAX_CONF_LINE-3 )
	    {
		/* We have found the correct reverse zone's "file" line */
	    	found_rew_zone = 1;

		/* Scroll back to the stored "type" position */
		fseek(fp, type_pos, SEEK_SET);
		snprintf(tmp, 1000, "\ttype %s;\n", domain_type);
		strcat(new_conf, tmp);

		/* Skip comments */
		if( strstr(line, "/*") )
		{
		    while(fgets(line, file_size, fp)!=NULL)
		    {
			strcat(new_conf, line);
			if( strstr(line, "*/") )
			  break;
		    }	
		}

		/* Skip comments */
		if( strstr(line, "//") )
		  continue;


		/* Handle the options */
		while(fgets(line, file_size, fp)!=NULL)
		{
		    /* Remove any allow-update lines */
		    if( strstr(line, "allow-update") && strstr(line, "{") )
		      continue;
		    
		    /* Remove any masters lines */
		    if( strstr(line, "masters") && strstr(line, "{") )
		      continue;

		    /* Remove any allow-update lines */
		    if( strstr(line, "allow-notify") && strstr(line, "{") )
		      continue;


		    
		    /* The allow-transfer line must exist, also handles other options */
		    if( strstr(line, "allow-transfer") && strstr(line, "{") )
		    {
		        snprintf(tmp, 1000, "\tallow-transfer { %s };\n", allow_transfer);
		        strcat(new_conf, tmp);

			/* If the domain type is master, we can have allow-update */
			if( ! domain_type_slave )
			{
		    	     snprintf(tmp, 1000, "\tallow-update { %s };\n", allow_update);
		    	     strcat(new_conf, tmp);
			}

			if( domain_type_slave )
			{
		    	    snprintf(tmp, 1000, "\tmasters { %s };\n", masters);
			    strcat(new_conf, tmp);

			    if( strlen(allow_notify) > 3 )
			    {
				snprintf(tmp, 1000, "\tallow-notify { %s };\n", allow_notify);
				strcat(new_conf, tmp);
			    }
			}
		    }
		    else
		    if( strstr(line, "allow-query") && strstr(line, "{") )
		    {
		        snprintf(tmp, 1000, "\tallow-query { %s };\n", allow_query);
		        strcat(new_conf, tmp);
		    }
		    else
		    if( strstr(line, "zone-statistics") && strstr(line, ";") )
		    {
		        snprintf(tmp, 1000, "\tzone-statistics %s;\n", domain_statistics);
		        strcat(new_conf, tmp);
		    }
		    else
		    if( strstr(line, "notify") && ! strstr(line, "also-notify")
		    && ! strstr(line, "allow-notify") && strstr(line, ";") )
		    {
			snprintf(tmp, 1000, "\tnotify %s;\n", domain_notify);
			strcat(new_conf, tmp);
		    }
		    else
		    if( strstr(line, "also-notify") && strstr(line, ";") )
		    {
			snprintf(tmp, 1000, "\talso-notify { %s };\n", also_notify);
			strcat(new_conf, tmp);
		    }
		    else
		      strcat(new_conf, line);
		    
		    if( strstr(line, "zone \"") && ! strstr(line, "zone-statistics") )
		      break;
		}
		break;
	    }
	    else
	      strcat(new_conf, line);
	}
    }
    fclose(fp);
    free(line);
    free(forward);
    free(reverse_file);
    free(tmp);

    g_free(fwd_decl);


    /* Write the new conf */
    if((fp=fopen(named_conf, "w+"))==NULL)
    {
	info = g_strdup_printf(_("Error: Cannot write named.conf here:\n%s\n"), named_conf);
	utf8 = g_locale_to_utf8(info, strlen(info), NULL, NULL, NULL);
	show_info(info);
	g_free(info);
	if( utf8!=NULL )
    	  g_free(utf8);
	g_free(fwd_zone_path);
	g_free(rev_zone_path);
	g_free(domain_statistics);
	g_free(domain_type);
	g_free(named_conf);
	return;
    }
    fputs(new_conf, fp);
    fclose(fp);

    free(new_conf);



    if( new_domain_name )
    {
	/* sed -e s/drum.se/bbb.se/g filepath > filepath.tmp */
	cmd = g_strdup_printf("%s -e s/%s/%s/g %s > %s.tmp", 
              SED_BINARY, global_domain_name, domain_name, named_conf, named_conf);

	if( ! run_command(cmd) )
	{
	    printf("Error running command:\n%s\n", cmd);
	    if( utf8!=NULL )
    	      g_free(utf8);
	    g_free(fwd_zone_path);
	    g_free(rev_zone_path);
	    g_free(domain_statistics);
	    g_free(domain_type);
	    g_free(named_conf);
	    return;    
	}
	g_free(cmd);

	/* mv /path/to/named.conf.tmp path/to/named.conf */
	cmd = g_strdup_printf("mv %s.tmp %s", named_conf, named_conf);

	if( ! run_command(cmd) )
	{
	    printf("Error running command:\n%s\n", cmd);
	    if( utf8!=NULL )
    	      g_free(utf8);
	    g_free(fwd_zone_path);
	    g_free(rev_zone_path);
	    g_free(domain_statistics);
	    g_free(domain_type);
	    g_free(named_conf);
	    return;    
	}
	g_free(cmd);
    }


    if( ! strstr(global_domain_type, domain_type) )
    {
	if( strstr(domain_type, "master") )
    	  printf("Domain type changed to master.\n");
    	if( strstr(domain_type, "slave") )
          printf("Domain type changed to slave.\n");

	if( domain_type_slave )
	  info = g_strdup_printf(_("Domain type changed to slave.\n"));
	else
	  info = g_strdup_printf(_("Domain type changed to master.\n"));

	utf8 = g_locale_to_utf8(info, strlen(info), NULL, NULL, NULL);
	show_info(info);
	g_free(info);
    }

    if( utf8!=NULL )
      g_free(utf8);
    g_free(fwd_zone_path);
    g_free(rev_zone_path);
    g_free(domain_statistics);
    g_free(domain_type);
    g_free(named_conf);

    /* Repopulate the gui */
    populate_domain_treeview(widgets);

    populate_resource_treeview(widgets);

    populate_log_tab(widgets);
    
    gtk_widget_destroy(widgets->domain_settings_window);
}
