------------------------------------------------------------------------------
--                                                                          --
--                  GNAT RUN-TIME LIBRARY (GNARL) COMPONENTS                --
--                                                                          --
--     S Y S T E M . T A S K _ P R I M I T I V E S . O P E R A T I O N S    --
--                                                                          --
--                                  B o d y                                 --
--                                                                          --
--         Copyright (C) 1992-2005, Free Software Foundation, Inc.          --
--                                                                          --
-- GNARL is free software; you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion. GNARL is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNARL; see file COPYING.  If not, write --
-- to  the  Free Software Foundation,  51  Franklin  Street,  Fifth  Floor, --
-- Boston, MA 02110-1301, USA.                                              --
--                                                                          --
-- As a special exception,  if other files  instantiate  generics from this --
-- unit, or you link  this unit with other files  to produce an executable, --
-- this  unit  does not  by itself cause  the resulting  executable  to  be --
-- covered  by the  GNU  General  Public  License.  This exception does not --
-- however invalidate  any other reasons why  the executable file  might be --
-- covered by the  GNU Public License.                                      --
--                                                                          --
-- GNARL was developed by the GNARL team at Florida State University.       --
-- Extensive contributions were provided by Ada Core Technologies, Inc.     --
--                                                                          --
------------------------------------------------------------------------------

--  This is the VxWorks version of this package

--  This package contains all the GNULL primitives that interface directly
--  with the underlying OS.

pragma Polling (Off);
--  Turn off polling, we do not want ATC polling to take place during
--  tasking operations. It causes infinite loops and other problems.

with System.Tasking.Debug;
--  used for Known_Tasks

with System.Interrupt_Management;
--  used for Keep_Unmasked
--           Abort_Task_Signal
--           Signal_ID
--           Initialize_Interrupts

with System.Soft_Links;
--  used for Defer/Undefer_Abort

--  Note that we do not use System.Tasking.Initialization directly since
--  this is a higher level package that we shouldn't depend on. For example
--  when using the restricted run time, it is replaced by
--  System.Tasking.Restricted.Stages.

with System.OS_Interface;
--  used for various type, constant, and operations

with System.Parameters;
--  used for Size_Type

with System.Tasking;
--  used for Ada_Task_Control_Block
--           Task_Id
--           ATCB components and types

with Interfaces.C;

with Unchecked_Conversion;
with Unchecked_Deallocation;

package body System.Task_Primitives.Operations is

   use System.Tasking.Debug;
   use System.Tasking;
   use System.OS_Interface;
   use System.Parameters;
   use type Interfaces.C.int;

   package SSL renames System.Soft_Links;

   subtype int is System.OS_Interface.int;

   Relative : constant := 0;

   ----------------
   -- Local Data --
   ----------------

   --  The followings are logically constants, but need to be initialized at
   --  run time.

   Single_RTS_Lock : aliased RTS_Lock;
   --  This is a lock to allow only one thread of control in the RTS at a
   --  time; it is used to execute in mutual exclusion from all other tasks.
   --  Used mainly in Single_Lock mode, but also to protect All_Tasks_List

   ATCB_Key : aliased System.Address := System.Null_Address;
   --  Key used to find the Ada Task_Id associated with a thread

   ATCB_Key_Addr : System.Address := ATCB_Key'Address;
   pragma Export (Ada, ATCB_Key_Addr, "__gnat_ATCB_key_addr");
   --  Exported to support the temporary AE653 task registration
   --  implementation. This mechanism is used to minimize impact on other
   --  targets.

   Environment_Task_Id : Task_Id;
   --  A variable to hold Task_Id for the environment task

   Unblocked_Signal_Mask : aliased sigset_t;
   --  The set of signals that should unblocked in all tasks

   --  The followings are internal configuration constants needed

   Time_Slice_Val : Integer;
   pragma Import (C, Time_Slice_Val, "__gl_time_slice_val");

   Locking_Policy : Character;
   pragma Import (C, Locking_Policy, "__gl_locking_policy");

   Dispatching_Policy : Character;
   pragma Import (C, Dispatching_Policy, "__gl_task_dispatching_policy");

   FIFO_Within_Priorities : constant Boolean := Dispatching_Policy = 'F';
   --  Indicates whether FIFO_Within_Priorities is set

   Mutex_Protocol : Priority_Type;

   Foreign_Task_Elaborated : aliased Boolean := True;
   --  Used to identified fake tasks (i.e., non-Ada Threads)

   --------------------
   -- Local Packages --
   --------------------

   package Specific is

      function Is_Valid_Task return Boolean;
      pragma Inline (Is_Valid_Task);
      --  Does executing thread have a TCB?

      procedure Set (Self_Id : Task_Id);
      pragma Inline (Set);
      --  Set the self id for the current task

      function Self return Task_Id;
      pragma Inline (Self);
      --  Return a pointer to the Ada Task Control Block of the calling task

   end Specific;

   package body Specific is separate;
   --  The body of this package is target specific

   ---------------------------------
   -- Support for foreign threads --
   ---------------------------------

   function Register_Foreign_Thread (Thread : Thread_Id) return Task_Id;
   --  Allocate and Initialize a new ATCB for the current Thread

   function Register_Foreign_Thread
     (Thread : Thread_Id) return Task_Id is separate;

   -----------------------
   -- Local Subprograms --
   -----------------------

   procedure Abort_Handler (signo : Signal);
   --  Handler for the abort (SIGABRT) signal to handle asynchronous abort

   procedure Install_Signal_Handlers;
   --  Install the default signal handlers for the current task

   function To_Address is new Unchecked_Conversion (Task_Id, System.Address);

   -------------------
   -- Abort_Handler --
   -------------------

   procedure Abort_Handler (signo : Signal) is
      pragma Unreferenced (signo);

      Self_ID : constant Task_Id := Self;
      Result  : int;
      Old_Set : aliased sigset_t;

   begin
      --  It is not safe to raise an exception when using ZCX and the GCC
      --  exception handling mechanism.

      if ZCX_By_Default and then GCC_ZCX_Support then
         return;
      end if;

      if Self_ID.Deferral_Level = 0
        and then Self_ID.Pending_ATC_Level < Self_ID.ATC_Nesting_Level
        and then not Self_ID.Aborting
      then
         Self_ID.Aborting := True;

         --  Make sure signals used for RTS internal purpose are unmasked

         Result := pthread_sigmask (SIG_UNBLOCK,
           Unblocked_Signal_Mask'Unchecked_Access, Old_Set'Unchecked_Access);
         pragma Assert (Result = 0);

         raise Standard'Abort_Signal;
      end if;
   end Abort_Handler;

   -----------------
   -- Stack_Guard --
   -----------------

   procedure Stack_Guard (T : ST.Task_Id; On : Boolean) is
      pragma Unreferenced (T);
      pragma Unreferenced (On);

   begin
      --  Nothing needed (why not???)

      null;
   end Stack_Guard;

   -------------------
   -- Get_Thread_Id --
   -------------------

   function Get_Thread_Id (T : ST.Task_Id) return OSI.Thread_Id is
   begin
      return T.Common.LL.Thread;
   end Get_Thread_Id;

   ----------
   -- Self --
   ----------

   function Self return Task_Id renames Specific.Self;

   -----------------------------
   -- Install_Signal_Handlers --
   -----------------------------

   procedure Install_Signal_Handlers is
      act     : aliased struct_sigaction;
      old_act : aliased struct_sigaction;
      Tmp_Set : aliased sigset_t;
      Result  : int;

   begin
      act.sa_flags := 0;
      act.sa_handler := Abort_Handler'Address;

      Result := sigemptyset (Tmp_Set'Access);
      pragma Assert (Result = 0);
      act.sa_mask := Tmp_Set;

      Result :=
        sigaction
          (Signal (Interrupt_Management.Abort_Task_Signal),
           act'Unchecked_Access,
           old_act'Unchecked_Access);
      pragma Assert (Result = 0);

      Interrupt_Management.Initialize_Interrupts;
   end Install_Signal_Handlers;

   ---------------------
   -- Initialize_Lock --
   ---------------------

   procedure Initialize_Lock (Prio : System.Any_Priority; L : access Lock) is
   begin
      L.Mutex := semMCreate (SEM_Q_PRIORITY + SEM_INVERSION_SAFE);
      L.Prio_Ceiling := int (Prio);
      L.Protocol := Mutex_Protocol;
      pragma Assert (L.Mutex /= 0);
   end Initialize_Lock;

   procedure Initialize_Lock (L : access RTS_Lock; Level : Lock_Level) is
      pragma Unreferenced (Level);

   begin
      L.Mutex := semMCreate (SEM_Q_PRIORITY + SEM_INVERSION_SAFE);
      L.Prio_Ceiling := int (System.Any_Priority'Last);
      L.Protocol := Mutex_Protocol;
      pragma Assert (L.Mutex /= 0);
   end Initialize_Lock;

   -------------------
   -- Finalize_Lock --
   -------------------

   procedure Finalize_Lock (L : access Lock) is
      Result : int;

   begin
      Result := semDelete (L.Mutex);
      pragma Assert (Result = 0);
   end Finalize_Lock;

   procedure Finalize_Lock (L : access RTS_Lock) is
      Result : int;

   begin
      Result := semDelete (L.Mutex);
      pragma Assert (Result = 0);
   end Finalize_Lock;

   ----------------
   -- Write_Lock --
   ----------------

   procedure Write_Lock (L : access Lock; Ceiling_Violation : out Boolean) is
      Result : int;

   begin
      if L.Protocol = Prio_Protect
        and then int (Self.Common.Current_Priority) > L.Prio_Ceiling
      then
         Ceiling_Violation := True;
         return;
      else
         Ceiling_Violation := False;
      end if;

      Result := semTake (L.Mutex, WAIT_FOREVER);
      pragma Assert (Result = 0);
   end Write_Lock;

   procedure Write_Lock
     (L           : access RTS_Lock;
      Global_Lock : Boolean := False)
   is
      Result : int;

   begin
      if not Single_Lock or else Global_Lock then
         Result := semTake (L.Mutex, WAIT_FOREVER);
         pragma Assert (Result = 0);
      end if;
   end Write_Lock;

   procedure Write_Lock (T : Task_Id) is
      Result : int;

   begin
      if not Single_Lock then
         Result := semTake (T.Common.LL.L.Mutex, WAIT_FOREVER);
         pragma Assert (Result = 0);
      end if;
   end Write_Lock;

   ---------------
   -- Read_Lock --
   ---------------

   procedure Read_Lock (L : access Lock; Ceiling_Violation : out Boolean) is
   begin
      Write_Lock (L, Ceiling_Violation);
   end Read_Lock;

   ------------
   -- Unlock --
   ------------

   procedure Unlock (L : access Lock) is
      Result  : int;

   begin
      Result := semGive (L.Mutex);
      pragma Assert (Result = 0);
   end Unlock;

   procedure Unlock (L : access RTS_Lock; Global_Lock : Boolean := False) is
      Result : int;

   begin
      if not Single_Lock or else Global_Lock then
         Result := semGive (L.Mutex);
         pragma Assert (Result = 0);
      end if;
   end Unlock;

   procedure Unlock (T : Task_Id) is
      Result : int;

   begin
      if not Single_Lock then
         Result := semGive (T.Common.LL.L.Mutex);
         pragma Assert (Result = 0);
      end if;
   end Unlock;

   -----------
   -- Sleep --
   -----------

   procedure Sleep (Self_ID : Task_Id; Reason : System.Tasking.Task_States) is
      pragma Unreferenced (Reason);

      Result : int;

   begin
      pragma Assert (Self_ID = Self);

      --  Release the mutex before sleeping

      if Single_Lock then
         Result := semGive (Single_RTS_Lock.Mutex);
      else
         Result := semGive (Self_ID.Common.LL.L.Mutex);
      end if;

      pragma Assert (Result = 0);

      --  Perform a blocking operation to take the CV semaphore. Note that a
      --  blocking operation in VxWorks will reenable task scheduling. When we
      --  are no longer blocked and control is returned, task scheduling will
      --  again be disabled.

      Result := semTake (Self_ID.Common.LL.CV, WAIT_FOREVER);
      pragma Assert (Result = 0);

      --  Take the mutex back

      if Single_Lock then
         Result := semTake (Single_RTS_Lock.Mutex, WAIT_FOREVER);
      else
         Result := semTake (Self_ID.Common.LL.L.Mutex, WAIT_FOREVER);
      end if;

      pragma Assert (Result = 0);
   end Sleep;

   -----------------
   -- Timed_Sleep --
   -----------------

   --  This is for use within the run-time system, so abort is assumed to be
   --  already deferred, and the caller should be holding its own ATCB lock.

   procedure Timed_Sleep
     (Self_ID  : Task_Id;
      Time     : Duration;
      Mode     : ST.Delay_Modes;
      Reason   : System.Tasking.Task_States;
      Timedout : out Boolean;
      Yielded  : out Boolean)
   is
      pragma Unreferenced (Reason);

      Orig     : constant Duration := Monotonic_Clock;
      Absolute : Duration;
      Ticks    : int;
      Result   : int;
      Wakeup   : Boolean := False;

   begin
      Timedout := False;
      Yielded  := True;

      if Mode = Relative then
         Absolute := Orig + Time;

         --  Systematically add one since the first tick will delay *at most*
         --  1 / Rate_Duration seconds, so we need to add one to be on the
         --  safe side.

         Ticks := To_Clock_Ticks (Time);

         if Ticks > 0 and then Ticks < int'Last then
            Ticks := Ticks + 1;
         end if;

      else
         Absolute := Time;
         Ticks    := To_Clock_Ticks (Time - Monotonic_Clock);
      end if;

      if Ticks > 0 then
         loop
            --  Release the mutex before sleeping

            if Single_Lock then
               Result := semGive (Single_RTS_Lock.Mutex);
            else
               Result := semGive (Self_ID.Common.LL.L.Mutex);
            end if;

            pragma Assert (Result = 0);

            --  Perform a blocking operation to take the CV semaphore. Note
            --  that a blocking operation in VxWorks will reenable task
            --  scheduling. When we are no longer blocked and control is
            --  returned, task scheduling will again be disabled.

            Result := semTake (Self_ID.Common.LL.CV, Ticks);

            if Result = 0 then

               --  Somebody may have called Wakeup for us

               Wakeup := True;

            else
               if errno /= S_objLib_OBJ_TIMEOUT then
                  Wakeup := True;

               else
                  --  If Ticks = int'last, it was most probably truncated so
                  --  let's make another round after recomputing Ticks from
                  --  the the absolute time.

                  if Ticks /= int'Last then
                     Timedout := True;
                  else
                     Ticks := To_Clock_Ticks (Absolute - Monotonic_Clock);

                     if Ticks < 0 then
                        Timedout := True;
                     end if;
                  end if;
               end if;
            end if;

            --  Take the mutex back

            if Single_Lock then
               Result := semTake (Single_RTS_Lock.Mutex, WAIT_FOREVER);
            else
               Result := semTake (Self_ID.Common.LL.L.Mutex, WAIT_FOREVER);
            end if;

            pragma Assert (Result = 0);

            exit when Timedout or Wakeup;
         end loop;

      else
         Timedout := True;

         --  Should never hold a lock while yielding

         if Single_Lock then
            Result := semGive (Single_RTS_Lock.Mutex);
            taskDelay (0);
            Result := semTake (Single_RTS_Lock.Mutex, WAIT_FOREVER);

         else
            Result := semGive (Self_ID.Common.LL.L.Mutex);
            taskDelay (0);
            Result := semTake (Self_ID.Common.LL.L.Mutex, WAIT_FOREVER);
         end if;
      end if;
   end Timed_Sleep;

   -----------------
   -- Timed_Delay --
   -----------------

   --  This is for use in implementing delay statements, so we assume the
   --  caller is holding no locks.

   procedure Timed_Delay
     (Self_ID  : Task_Id;
      Time     : Duration;
      Mode     : ST.Delay_Modes)
   is
      Orig     : constant Duration := Monotonic_Clock;
      Absolute : Duration;
      Ticks    : int;
      Timedout : Boolean;
      Result   : int;
      Aborted  : Boolean := False;

   begin
      SSL.Abort_Defer.all;

      if Mode = Relative then
         Absolute := Orig + Time;
         Ticks    := To_Clock_Ticks (Time);

         if Ticks > 0 and then Ticks < int'Last then

            --  First tick will delay anytime between 0 and 1 / sysClkRateGet
            --  seconds, so we need to add one to be on the safe side.

            Ticks := Ticks + 1;
         end if;

      else
         Absolute := Time;
         Ticks    := To_Clock_Ticks (Time - Orig);
      end if;

      if Ticks > 0 then

         --  Modifying State and Pending_Priority_Change, locking the TCB

         if Single_Lock then
            Result := semTake (Single_RTS_Lock.Mutex, WAIT_FOREVER);
         else
            Result := semTake (Self_ID.Common.LL.L.Mutex, WAIT_FOREVER);
         end if;

         pragma Assert (Result = 0);

         Self_ID.Common.State := Delay_Sleep;
         Timedout := False;

         loop
            if Self_ID.Pending_Priority_Change then
               Self_ID.Pending_Priority_Change := False;
               Self_ID.Common.Base_Priority    := Self_ID.New_Base_Priority;
               Set_Priority (Self_ID, Self_ID.Common.Base_Priority);
            end if;

            Aborted := Self_ID.Pending_ATC_Level < Self_ID.ATC_Nesting_Level;

            --  Release the TCB before sleeping

            if Single_Lock then
               Result := semGive (Single_RTS_Lock.Mutex);
            else
               Result := semGive (Self_ID.Common.LL.L.Mutex);
            end if;
            pragma Assert (Result = 0);

            exit when Aborted;

            Result := semTake (Self_ID.Common.LL.CV, Ticks);

            if Result /= 0 then

               --  If Ticks = int'last, it was most probably truncated
               --  so let's make another round after recomputing Ticks
               --  from the the absolute time.

               if errno = S_objLib_OBJ_TIMEOUT and then Ticks /= int'Last then
                  Timedout := True;
               else
                  Ticks := To_Clock_Ticks (Absolute - Monotonic_Clock);

                  if Ticks < 0 then
                     Timedout := True;
                  end if;
               end if;
            end if;

            --  Take back the lock after having slept, to protect further
            --  access to Self_ID

            if Single_Lock then
               Result := semTake (Single_RTS_Lock.Mutex, WAIT_FOREVER);
            else
               Result := semTake (Self_ID.Common.LL.L.Mutex, WAIT_FOREVER);
            end if;

            pragma Assert (Result = 0);

            exit when Timedout;
         end loop;

         Self_ID.Common.State := Runnable;

         if Single_Lock then
            Result := semGive (Single_RTS_Lock.Mutex);
         else
            Result := semGive (Self_ID.Common.LL.L.Mutex);
         end if;

      else
         taskDelay (0);
      end if;

      SSL.Abort_Undefer.all;
   end Timed_Delay;

   ---------------------
   -- Monotonic_Clock --
   ---------------------

   function Monotonic_Clock return Duration is
      TS     : aliased timespec;
      Result : int;
   begin
      Result := clock_gettime (CLOCK_REALTIME, TS'Unchecked_Access);
      pragma Assert (Result = 0);
      return To_Duration (TS);
   end Monotonic_Clock;

   -------------------
   -- RT_Resolution --
   -------------------

   function RT_Resolution return Duration is
   begin
      return 1.0 / Duration (sysClkRateGet);
   end RT_Resolution;

   ------------
   -- Wakeup --
   ------------

   procedure Wakeup (T : Task_Id; Reason : System.Tasking.Task_States) is
      pragma Unreferenced (Reason);
      Result : int;
   begin
      Result := semGive (T.Common.LL.CV);
      pragma Assert (Result = 0);
   end Wakeup;

   -----------
   -- Yield --
   -----------

   procedure Yield (Do_Yield : Boolean := True) is
      pragma Unreferenced (Do_Yield);
      Result : int;
      pragma Unreferenced (Result);
   begin
      Result := taskDelay (0);
   end Yield;

   ------------------
   -- Set_Priority --
   ------------------

   type Prio_Array_Type is array (System.Any_Priority) of Integer;
   pragma Atomic_Components (Prio_Array_Type);

   Prio_Array : Prio_Array_Type;
   --  Global array containing the id of the currently running task for
   --  each priority. Note that we assume that we are on a single processor
   --  with run-till-blocked scheduling.

   procedure Set_Priority
     (T                   : Task_Id;
      Prio                : System.Any_Priority;
      Loss_Of_Inheritance : Boolean := False)
   is
      Array_Item : Integer;
      Result     : int;

   begin
      Result :=
        taskPrioritySet
          (T.Common.LL.Thread, To_VxWorks_Priority (int (Prio)));
      pragma Assert (Result = 0);

      if FIFO_Within_Priorities then

         --  Annex D requirement [RM D.2.2 par. 9]:

         --    If the task drops its priority due to the loss of inherited
         --    priority, it is added at the head of the ready queue for its
         --    new active priority.

         if Loss_Of_Inheritance
           and then Prio < T.Common.Current_Priority
         then
            Array_Item := Prio_Array (T.Common.Base_Priority) + 1;
            Prio_Array (T.Common.Base_Priority) := Array_Item;

            loop
               --  Give some processes a chance to arrive

               taskDelay (0);

               --  Then wait for our turn to proceed

               exit when Array_Item = Prio_Array (T.Common.Base_Priority)
                 or else Prio_Array (T.Common.Base_Priority) = 1;
            end loop;

            Prio_Array (T.Common.Base_Priority) :=
              Prio_Array (T.Common.Base_Priority) - 1;
         end if;
      end if;

      T.Common.Current_Priority := Prio;
   end Set_Priority;

   ------------------
   -- Get_Priority --
   ------------------

   function Get_Priority (T : Task_Id) return System.Any_Priority is
   begin
      return T.Common.Current_Priority;
   end Get_Priority;

   ----------------
   -- Enter_Task --
   ----------------

   procedure Enter_Task (Self_ID : Task_Id) is
      procedure Init_Float;
      pragma Import (C, Init_Float, "__gnat_init_float");
      --  Properly initializes the FPU for PPC/MIPS systems

   begin
      Self_ID.Common.LL.Thread := taskIdSelf;
      Specific.Set (Self_ID);

      Init_Float;

      --  Install the signal handlers

      --  This is called for each task since there is no signal inheritance
      --  between VxWorks tasks.

      Install_Signal_Handlers;

      Lock_RTS;

      for J in Known_Tasks'Range loop
         if Known_Tasks (J) = null then
            Known_Tasks (J) := Self_ID;
            Self_ID.Known_Tasks_Index := J;
            exit;
         end if;
      end loop;

      Unlock_RTS;
   end Enter_Task;

   --------------
   -- New_ATCB --
   --------------

   function New_ATCB (Entry_Num : Task_Entry_Index) return Task_Id is
   begin
      return new Ada_Task_Control_Block (Entry_Num);
   end New_ATCB;

   -------------------
   -- Is_Valid_Task --
   -------------------

   function Is_Valid_Task return Boolean renames Specific.Is_Valid_Task;

   -----------------------------
   -- Register_Foreign_Thread --
   -----------------------------

   function Register_Foreign_Thread return Task_Id is
   begin
      if Is_Valid_Task then
         return Self;
      else
         return Register_Foreign_Thread (taskIdSelf);
      end if;
   end Register_Foreign_Thread;

   --------------------
   -- Initialize_TCB --
   --------------------

   procedure Initialize_TCB (Self_ID : Task_Id; Succeeded : out Boolean) is
   begin
      Self_ID.Common.LL.CV := semBCreate (SEM_Q_PRIORITY, SEM_EMPTY);
      Self_ID.Common.LL.Thread := 0;

      if Self_ID.Common.LL.CV = 0 then
         Succeeded := False;
      else
         Succeeded := True;

         if not Single_Lock then
            Initialize_Lock (Self_ID.Common.LL.L'Access, ATCB_Level);
         end if;
      end if;
   end Initialize_TCB;

   -----------------
   -- Create_Task --
   -----------------

   procedure Create_Task
     (T          : Task_Id;
      Wrapper    : System.Address;
      Stack_Size : System.Parameters.Size_Type;
      Priority   : System.Any_Priority;
      Succeeded  : out Boolean)
   is
      Adjusted_Stack_Size : size_t;
   begin
      if Stack_Size = Unspecified_Size then
         Adjusted_Stack_Size := size_t (Default_Stack_Size);

      elsif Stack_Size < Minimum_Stack_Size then
         Adjusted_Stack_Size := size_t (Minimum_Stack_Size);

      else
         Adjusted_Stack_Size := size_t (Stack_Size);
      end if;

      --  Ask for four extra bytes of stack space so that the ATCB pointer can
      --  be stored below the stack limit, plus extra space for the frame of
      --  Task_Wrapper. This is so the user gets the amount of stack requested
      --  exclusive of the needs
      --
      --  We also have to allocate n more bytes for the task name storage and
      --  enough space for the Wind Task Control Block which is around 0x778
      --  bytes. VxWorks also seems to carve out additional space, so use 2048
      --  as a nice round number. We might want to increment to the nearest
      --  page size in case we ever support VxVMI.
      --
      --  XXX - we should come back and visit this so we can set the task name
      --        to something appropriate.

      Adjusted_Stack_Size := Adjusted_Stack_Size + 2048;

      --  Since the initial signal mask of a thread is inherited from the
      --  creator, and the Environment task has all its signals masked, we do
      --  not need to manipulate caller's signal mask at this point. All tasks
      --  in RTS will have All_Tasks_Mask initially.

      if T.Common.Task_Image_Len = 0 then
         T.Common.LL.Thread := taskSpawn
           (System.Null_Address,
            To_VxWorks_Priority (int (Priority)),
            VX_FP_TASK,
            Adjusted_Stack_Size,
            Wrapper,
            To_Address (T));
      else
         declare
            Name : aliased String (1 .. T.Common.Task_Image_Len + 1);

         begin
            Name (1 .. Name'Last - 1) :=
              T.Common.Task_Image (1 .. T.Common.Task_Image_Len);
            Name (Name'Last) := ASCII.NUL;

            T.Common.LL.Thread := taskSpawn
              (Name'Address,
               To_VxWorks_Priority (int (Priority)),
               VX_FP_TASK,
               Adjusted_Stack_Size,
               Wrapper,
               To_Address (T));
         end;
      end if;

      if T.Common.LL.Thread = -1 then
         Succeeded := False;
      else
         Succeeded := True;
      end if;

      Task_Creation_Hook (T.Common.LL.Thread);
      Set_Priority (T, Priority);
   end Create_Task;

   ------------------
   -- Finalize_TCB --
   ------------------

   procedure Finalize_TCB (T : Task_Id) is
      Result  : int;
      Tmp     : Task_Id          := T;
      Is_Self : constant Boolean := (T = Self);

      procedure Free is new
        Unchecked_Deallocation (Ada_Task_Control_Block, Task_Id);

   begin
      if not Single_Lock then
         Result := semDelete (T.Common.LL.L.Mutex);
         pragma Assert (Result = 0);
      end if;

      T.Common.LL.Thread := 0;

      Result := semDelete (T.Common.LL.CV);
      pragma Assert (Result = 0);

      if T.Known_Tasks_Index /= -1 then
         Known_Tasks (T.Known_Tasks_Index) := null;
      end if;

      Free (Tmp);

      if Is_Self then
         Result := taskVarDelete (taskIdSelf, ATCB_Key'Access);
         pragma Assert (Result /= ERROR);
      end if;
   end Finalize_TCB;

   ---------------
   -- Exit_Task --
   ---------------

   procedure Exit_Task is
   begin
      Specific.Set (null);
   end Exit_Task;

   ----------------
   -- Abort_Task --
   ----------------

   procedure Abort_Task (T : Task_Id) is
      Result : int;
   begin
      Result := kill (T.Common.LL.Thread,
                      Signal (Interrupt_Management.Abort_Task_Signal));
      pragma Assert (Result = 0);
   end Abort_Task;

   ----------------
   -- Initialize --
   ----------------

   procedure Initialize (S : in out Suspension_Object) is
   begin
      --  Initialize internal state. It is always initialized to False (ARM
      --  D.10 par. 6).

      S.State := False;
      S.Waiting := False;

      --  Initialize internal mutex

      --  Use simpler binary semaphore instead of VxWorks
      --  mutual exclusion semaphore, because we don't need
      --  the fancier semantics and their overhead.

      S.L := semBCreate (SEM_Q_FIFO, SEM_FULL);

      --  Initialize internal condition variable

      S.CV := semBCreate (SEM_Q_FIFO, SEM_EMPTY);
   end Initialize;

   --------------
   -- Finalize --
   --------------

   procedure Finalize (S : in out Suspension_Object) is
      Result : STATUS;
   begin
      --  Destroy internal mutex

      Result := semDelete (S.L);
      pragma Assert (Result = OK);

      --  Destroy internal condition variable

      Result := semDelete (S.CV);
      pragma Assert (Result = OK);
   end Finalize;

   -------------------
   -- Current_State --
   -------------------

   function Current_State (S : Suspension_Object) return Boolean is
   begin
      --  We do not want to use lock on this read operation. State is marked
      --  as Atomic so that we ensure that the value retrieved is correct.

      return S.State;
   end Current_State;

   ---------------
   -- Set_False --
   ---------------

   procedure Set_False (S : in out Suspension_Object) is
      Result  : STATUS;
   begin
      Result := semTake (S.L, WAIT_FOREVER);
      pragma Assert (Result = OK);

      S.State := False;

      Result := semGive (S.L);
      pragma Assert (Result = OK);
   end Set_False;

   --------------
   -- Set_True --
   --------------

   procedure Set_True (S : in out Suspension_Object) is
      Result : STATUS;
   begin
      Result := semTake (S.L, WAIT_FOREVER);
      pragma Assert (Result = OK);

      --  If there is already a task waiting on this suspension object then
      --  we resume it, leaving the state of the suspension object to False,
      --  as it is specified in ARM D.10 par. 9. Otherwise, it just leaves
      --  the state to True.

      if S.Waiting then
         S.Waiting := False;
         S.State := False;

         Result := semGive (S.CV);
         pragma Assert (Result = OK);
      else
         S.State := True;
      end if;

      Result := semGive (S.L);
      pragma Assert (Result = OK);
   end Set_True;

   ------------------------
   -- Suspend_Until_True --
   ------------------------

   procedure Suspend_Until_True (S : in out Suspension_Object) is
      Result : STATUS;
   begin
      Result := semTake (S.L, WAIT_FOREVER);

      if S.Waiting then
         --  Program_Error must be raised upon calling Suspend_Until_True
         --  if another task is already waiting on that suspension object
         --  (ARM D.10 par. 10).

         Result := semGive (S.L);
         pragma Assert (Result = OK);

         raise Program_Error;
      else
         --  Suspend the task if the state is False. Otherwise, the task
         --  continues its execution, and the state of the suspension object
         --  is set to False (ARM D.10 par. 9).

         if S.State then
            S.State := False;

            Result := semGive (S.L);
            pragma Assert (Result = 0);
         else
            S.Waiting := True;

            --  Release the mutex before sleeping

            Result := semGive (S.L);
            pragma Assert (Result = OK);

            Result := semTake (S.CV, WAIT_FOREVER);
            pragma Assert (Result = 0);
         end if;
      end if;
   end Suspend_Until_True;

   ----------------
   -- Check_Exit --
   ----------------

   --  Dummy version

   function Check_Exit (Self_ID : ST.Task_Id) return Boolean is
      pragma Unreferenced (Self_ID);
   begin
      return True;
   end Check_Exit;

   --------------------
   -- Check_No_Locks --
   --------------------

   function Check_No_Locks (Self_ID : ST.Task_Id) return Boolean is
      pragma Unreferenced (Self_ID);
   begin
      return True;
   end Check_No_Locks;

   ----------------------
   -- Environment_Task --
   ----------------------

   function Environment_Task return Task_Id is
   begin
      return Environment_Task_Id;
   end Environment_Task;

   --------------
   -- Lock_RTS --
   --------------

   procedure Lock_RTS is
   begin
      Write_Lock (Single_RTS_Lock'Access, Global_Lock => True);
   end Lock_RTS;

   ----------------
   -- Unlock_RTS --
   ----------------

   procedure Unlock_RTS is
   begin
      Unlock (Single_RTS_Lock'Access, Global_Lock => True);
   end Unlock_RTS;

   ------------------
   -- Suspend_Task --
   ------------------

   function Suspend_Task
     (T           : ST.Task_Id;
      Thread_Self : Thread_Id) return Boolean
   is
   begin
      if T.Common.LL.Thread /= 0
        and then T.Common.LL.Thread /= Thread_Self
      then
         return taskSuspend (T.Common.LL.Thread) = 0;
      else
         return True;
      end if;
   end Suspend_Task;

   -----------------
   -- Resume_Task --
   -----------------

   function Resume_Task
     (T           : ST.Task_Id;
      Thread_Self : Thread_Id) return Boolean
   is
   begin
      if T.Common.LL.Thread /= 0
        and then T.Common.LL.Thread /= Thread_Self
      then
         return taskResume (T.Common.LL.Thread) = 0;
      else
         return True;
      end if;
   end Resume_Task;

   ----------------
   -- Initialize --
   ----------------

   procedure Initialize (Environment_Task : Task_Id) is
      Result : int;

   begin
      if Locking_Policy = 'C' then
         Mutex_Protocol := Prio_Protect;
      elsif Locking_Policy = 'I' then
         Mutex_Protocol := Prio_Inherit;
      else
         Mutex_Protocol := Prio_None;
      end if;

      if Time_Slice_Val > 0 then
         Result := kernelTimeSlice
           (To_Clock_Ticks
             (Duration (Time_Slice_Val) / Duration (1_000_000.0)));
      end if;

      Result := sigemptyset (Unblocked_Signal_Mask'Access);
      pragma Assert (Result = 0);

      for J in Interrupt_Management.Signal_ID loop
         if System.Interrupt_Management.Keep_Unmasked (J) then
            Result := sigaddset (Unblocked_Signal_Mask'Access, Signal (J));
            pragma Assert (Result = 0);
         end if;
      end loop;

      Environment_Task_Id := Environment_Task;

      --  Initialize the lock used to synchronize chain of all ATCBs

      Initialize_Lock (Single_RTS_Lock'Access, RTS_Lock_Level);

      Enter_Task (Environment_Task);
   end Initialize;

end System.Task_Primitives.Operations;
