package GCMoviesLists;

###################################################
#
#  Copyright 2005 Tian
#
#  This file is part of GCfilms.
#
#  GCfilms is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  GCfilms is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with GCfilms; if not, write to the Free Software
#  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
###################################################

use strict;

{
    package GCTextList;
    
    use Gtk2::SimpleList;
    use base "Gtk2::ScrolledWindow";

    sub new
    {
        my ($proto, $parent, $title) = @_;
        my $class = ref($proto) || $proto;
        my $self  = $class->SUPER::new;
        bless ($self, $class);

        $self->{parent} = $parent;
        
        $self->set_policy ('automatic', 'automatic');
        $self->set_shadow_type('etched-in');
        
        $self->{list} = new Gtk2::SimpleList($title=>'text');
        $self->{list}->set_name('GCMoviesTextList');
        $self->{list}->set_headers_clickable(1);
        $self->{list}->set_rules_hint(1);
        $self->{list}->get_column(0)->set_sort_indicator(1);
        
        $self->{list}->signal_connect (cursor_changed => sub {
            my ($sl, $path, $column) = @_;
            my @idx = $sl->get_selected_indices;
            $parent->display($idx[0]);
            $self->{currentIdx} = $idx[0];
        });

        $self->{list}->signal_connect ('row-activated' => sub {
           $parent->displayInWindow;
        });

        $self->add($self->{list});
        
        $self->{list}->signal_connect('button_press_event' => sub {
                my ($widget, $event) = @_;
                return 0 if $event->button ne 3;
                $self->{context}->popup(undef, undef, undef, undef, $event->button, $event->time);
                return 0;
        });
        
        $self->{list}->get_column(0)->signal_connect('clicked' => sub {
            $parent->{movies}->changeOrder;
            $self->setSortOrder;
        });

        $self->{currentIdx} = 0;

        return $self;
    }

    sub setSortOrder
    {
        my $self = shift;
        my ($order, $field) = $self->{parent}->{movies}->getSortOrder;
        $self->{list}->get_column(0)->set_sort_order($order ? 'ascending' : 'descending');
    }

    sub clearCache
    {
        my $self = shift;
    }

    sub reset
    {
        my $self = shift;
        
        @{$self->{list}->{data}} = ();
    }
    
    sub done
    {
        my $self = shift;
    }
    
    sub addMovie
    {
        my ($self, $info) = @_;
        
        push @{$self->{list}->{data}}, $self->{parent}->transformTitle($info->{title});
    }
    
    sub removeMovie
    {
        my ($self, $number) = @_;
        
        splice @{$self->{list}->{data}}, $number, 1;
        
        #Reload not needed
        return 0;
    }
    
    sub select
    {
        my ($self, $idx) = @_;
        
        $self->{list}->select($idx);
        $self->{currentIdx} = $idx;
    }
    
    sub showCurrent
    {
        my $self = shift;
        
   		my $path = $self->{list}->get_selection->get_selected_rows;
		$self->{list}->scroll_to_cell($path) if $path;
    }
    
    sub changeCurrent
    {
        my ($self, $image, $borrower) = @_;
    }
    
    sub changeCurrentTitle
    {
        my ($self, $title) = @_;
        
        $self->{list}->{data}->[$self->{currentIdx}] = $self->{parent}->transformTitle($title);
    }
    
}

{
    package GCImageList;
    
    use File::Basename;
    use GCUtils;
    use base "Gtk2::VBox";

    sub new
    {
        my ($proto, $parent, $title, $columns, $withImage) = @_;
        my $class = ref($proto) || $proto;
        my $self  = $class->SUPER::new;
        bless ($self, $class);
        
        $self->{withImage} = $parent->{options}->listBgPicture;
        
        $self->{vboxWidth} = 130;
        $self->{vboxHeight} = 170;
        $self->{vboxHeight} += 20 if $self->{withImage};;
        $self->{imgWidth} = 120;
        $self->{imgHeight} = 160;
        
        $self->{scroll} = new Gtk2::ScrolledWindow;
        $self->{scroll}->set_policy ('automatic', 'automatic');
        $self->{scroll}->set_shadow_type('etched-in');
        
        $self->{parent} = $parent;
        
        $self->{sortButton} = Gtk2::Button->new_from_stock('gtk-sort-ascending');
        
        $self->{list} = new Gtk2::Table(0,$columns,1);
        $self->{tooltips} = Gtk2::Tooltips->new();
        
        $self->{columns} = $columns;
        $self->reset;
        $self->clearCache;

        my @colors = split m/,/, $parent->{options}->listBgColor;
        ($colors[0], $colors[1], $colors[2]) = (65535, 65535, 65535) if !@colors;
        $self->{inactiveBg} = new Gtk2::Gdk::Color($colors[0], $colors[1], $colors[2]);
        @colors = split m/,/, $parent->{options}->listFgColor;
        ($colors[0], $colors[1], $colors[2]) = (0, 0, 0) if !@colors;
        $self->{activeBg} = new Gtk2::Gdk::Color($colors[0], $colors[1], $colors[2]);

        $self->{selectedPixmap} = $parent->{style}->{dir}.'/list_bg_selected.xpm';
        $self->{unselectedPixmap} = $parent->{style}->{dir}.'/list_bg_unselected.xpm';
        $self->{bgPixmap} = $parent->{style}->{dir}.'/list_bg.xpm';
        $self->{lendPicture} = $parent->{style}->{dir}.'/lend.png';
        
        $self->{defaultImage} = $parent->{defaultImage};
        
        $self->{list}->set_row_spacings(0);
        $self->{list}->set_col_spacings(0);
        $self->{list}->set_border_width(0);
        $self->{list}->set_border_width(5) if ! $self->{withImage};
        
        $self->{scroll}->add_with_viewport($self->{list});

        $self->{list}->parent->modify_bg('normal', $self->{inactiveBg});
        $self->{list}->parent->modify_bg('active', $self->{inactiveBg});
        $self->{list}->parent->modify_bg('prelight', $self->{inactiveBg});
        $self->{list}->parent->modify_bg('selected', $self->{inactiveBg});
        $self->{list}->parent->modify_bg('insensitive', $self->{inactiveBg});
        
    
        if ($self->{withImage})
        {
            GCUtils::setWidgetPixmap($self->{list}->parent, $self->{bgPixmap});
            
            $self->{backgroundPixbuf} = Gtk2::Gdk::Pixbuf->new_from_file($self->{unselectedPixmap});
            $self->{selectedPixbuf} = Gtk2::Gdk::Pixbuf->new_from_file($self->{selectedPixmap});
        }

        $self->pack_start($self->{sortButton},0,0,0);
        $self->pack_start($self->{scroll},1,1,0);
        
        $self->{list}->can_focus(1);
        
        $self->{sortButton}->signal_connect('clicked' => sub {
            $parent->{movies}->changeOrder;
            $self->setSortOrder;
        });
        
        
        return $self;
    }

    sub setSortOrder
    {
        my $self = shift;
        my ($order, $field) = $self->{parent}->{movies}->getSortOrder;
        
        $self->remove($self->{sortButton});
        $self->{sortButton}->destroy;
        $self->{sortButton} = Gtk2::Button->new_from_stock($order ? 'gtk-sort-ascending' : 'gtk-sort-descending');
        $self->pack_start($self->{sortButton},0,0,0);
        $self->reorder_child($self->{sortButton},0);
        $self->{sortButton}->show_all;
        
        $self->{sortButton}->signal_connect('clicked' => sub {
            $self->{parent}->{movies}->changeOrder;
            $self->setSortOrder;
        });
        
    }

    sub clearCache
    {
        my $self = shift;
        
        if ($self->{cache})
        {
            foreach (values %{$self->{cache}})
            {
                $_->{eventBox}->destroy;
            }
        }
        
        $self->{cache} = {};
    }

    sub reset
    {
        my $self = shift;
        
        #Restore current picture if modified
        $self->{boxes}->[$self->{current}]->child->set_from_pixbuf($self->{previousPixbuf})
            if ($self->{previousPixbuf} && $self->{boxes}->[$self->{current}]->child);
        
        my @array = ();
        $self->{boxes} = \@array;
        $self->{number} = 0;
        $self->{current} = 0;
        $self->{previous} = 0;
        $self->{previousPixbuf} = undef;
        
        my @children = $self->{list}->get_children;
        foreach (@children)
        {
            $self->{list}->remove($_);
        }
    }
    
    sub done
    {
        my $self = shift;
    }
    
    sub createEventBox
    {
        my ($self, $info) = @_;
    
        my $eventBox = new Gtk2::EventBox;
        $eventBox->can_focus(1);
                
        my $image = new Gtk2::Image;
        my $displayedImage = GCUtils::getDisplayedImage($info->{image},
                                                        $self->{defaultImage},
                                                        $self->{parent}->{options}->file);

        my $pixbuf = Gtk2::Gdk::Pixbuf->new_from_file($displayedImage);
        $pixbuf = $pixbuf->scale_simple($self->{imgWidth},$self->{imgHeight},'bilinear')
            if ($self->{withImage}) ||
               ($pixbuf->get_width > $self->{imgWidth}) ||
               ($pixbuf->get_height > $self->{imgHeight});
        if ($info->{borrower} && ($info->{borrower} ne 'none'))
        {
           my $lendPixbuf = Gtk2::Gdk::Pixbuf->new_from_file($self->{lendPicture});
           $lendPixbuf->composite($pixbuf,
                                  0, 0, 
                                  $pixbuf->get_width, $pixbuf->get_height,
                                  0, 0,
                                  1, 1, 
                                  'nearest', 255);
        }
        
        $self->{tooltips}->set_tip($eventBox, $self->{parent}->transformTitle($info->{title}), '');

        if ($self->{withImage})
        {
            $pixbuf->composite($self->{backgroundPixbuf},
                                  5, 15, 
                                  $self->{imgWidth}, $self->{imgHeight},
                                  5, 15,
                                  1, 1, 
                                  'nearest', 255);
            $pixbuf = $self->{backgroundPixbuf}->copy;
            
        }
        else
        {
            $eventBox->modify_bg('normal', $self->{inactiveBg});
        }

        $image->set_from_pixbuf($pixbuf);
        $eventBox->add($image);
        $eventBox->set_size_request($self->{vboxWidth},$self->{vboxHeight});
        
        return $eventBox;
    }

    sub getFromCache
    {
        my ($self, $info) = @_;
        
        if (! $self->{cache}->{$info->{id}})
        {
            my $item = {};
            $item->{eventBox} = $self->createEventBox($info);
            $item->{keyHandler} = 0;
            $item->{mouseHandler} = 0;
        
            $self->{cache}->{$info->{id}} = $item;
        }
        return $self->{cache}->{$info->{id}};
    }
    
    sub addMovie
    {
        my ($self, $info) = @_;
        
        my $row = int $self->{number} / $self->{columns};
        my $col = $self->{number} % $self->{columns};
        
        $self->{list}->resize($row + 1, $self->{columns});

        my $item = $self->getFromCache($info);
        my $eventBox = $item->{eventBox};
        my $i = $self->{number};
        
        $eventBox->signal_handler_disconnect($item->{mouseHandler})
            if $item->{mouseHandler};
        $item->{mouseHandler} = $eventBox->signal_connect('button_press_event' => sub {
                my ($widget, $event) = @_;
                $self->select($i);
                $self->{parent}->display($i) unless $event->type eq '2button-press';
                $self->{parent}->displayInWindow if $event->type eq '2button-press';
                $self->{context}->popup(undef, undef, undef, undef, $event->button, $event->time)
                    if $event->button eq 3;
                $widget->grab_focus;
            });

        $eventBox->signal_handler_disconnect($item->{keyHandler})
            if $item->{keyHandler};

        $item->{keyHandler} = $eventBox->signal_connect('key-press-event' => sub {
                my ($widget, $event) = @_;
                my $idx = $i;
                my $key = Gtk2::Gdk->keyval_name($event->keyval);
                if (($key eq 'Return') || ($key eq 'space'))
                {
                    $self->{parent}->displayInWindow;
                    return 1;
                }
                $idx++ if $key eq 'Right';
                $idx-- if $key eq 'Left';
                $idx += $self->{columns} if $key eq 'Down';
                $idx -= $self->{columns} if $key eq 'Up';
                return 1 if ($idx < 0) || ($idx >= scalar @{$self->{boxes}});
                my $column = $idx % $self->{columns};
                $self->select($idx);
                $self->{parent}->display($idx);
                $self->{boxes}->[$idx]->grab_focus;
                $self->showCurrent if $key eq 'Down'
                                   || $key eq 'Up'
                                   || (($key eq 'Left') && ($column == ($self->{columns} - 1)))
                                   || (($key eq 'Right') && ($column == 0));
                return 1;
                
            });

        $eventBox->show_all;       
        $self->{list}->attach($eventBox, $col, $col+1, $row, $row+1, 'fill', 'fill', 0, 0);
        push @{$self->{boxes}}, $eventBox;
        
        $self->{number}++;
    }
    
    sub removeMovie
    {
        my ($self, $number, $id) = @_;

        $self->{cache}->{$id}->{eventBox}->destroy;
        delete $self->{cache}->{$id};

        #Reload needed
        return 1;
    }
        
    sub select
    {
        my ($self, $idx, $id) = @_;
        
        my @boxes = @{$self->{boxes}};
        
        return if ! scalar(@boxes);
        
        my $previous = $self->{current};
        $self->{current} = $idx;
        
        if ($self->{withImage})
        {
            GCUtils::setWidgetPixmap($boxes[$previous], $self->{unselectedPixmap});
            GCUtils::setWidgetPixmap($boxes[$self->{current}], $self->{selectedPixmap});
        }
        else
        {
            $boxes[$previous]->modify_bg('normal', $self->{inactiveBg});  
            $boxes[$self->{current}]->modify_bg('normal', $self->{activeBg});
        
        }
        $boxes[$previous]->child->set_from_pixbuf($self->{previousPixbuf})
            if $self->{previousPixbuf} && $boxes[$previous]->child;
        my $pixbuf = $boxes[$self->{current}]->child->get_pixbuf;
        $self->{previousPixbuf} = $pixbuf->copy;
        
        
        $pixbuf->saturate_and_pixelate($pixbuf, 1.5, 0);

        $self->{selectedPixbuf}->composite($pixbuf,
                           0, 0, 
                           $self->{vboxWidth}, $self->{vboxHeight},
                           0, 0,
                           1, 1, 
                           'nearest', 255)
            if $self->{withImage};
        
            
        $boxes[$self->{current}]->child->set_from_pixbuf($pixbuf);
        
        $self->{previous} = $previous;
        
    }
    
    sub showCurrent
    {
        my $self = shift;
 
        my $adj = $self->{scroll}->get_vadjustment;
        my $totalRows = int $self->{number} / $self->{columns};
        my $row = (int $self->{current} / $self->{columns}) - 1;

        $adj->set_value((($adj->upper - $adj->lower) / $totalRows) * $row) if $totalRows > 0;
    }

    sub changeCurrent
    {
        my ($self, $image, $borrower) = @_;

        my @boxes = @{$self->{boxes}};
        
        my $displayedImage = GCUtils::getDisplayedImage($image,
                                                        $self->{defaultImage},
                                                        $self->{parent}->{options}->file);
        
        
        my $pixbuf = Gtk2::Gdk::Pixbuf->new_from_file($displayedImage);
        $pixbuf = $pixbuf->scale_simple($self->{imgWidth},$self->{imgHeight},'bilinear')
            if ($self->{withImage}) ||
               ($pixbuf->get_width > $self->{imgWidth}) ||
               ($pixbuf->get_height > $self->{imgHeight});

        if ($borrower && ($borrower ne 'none'))
        {
           my $lendPixbuf = Gtk2::Gdk::Pixbuf->new_from_file($self->{lendPicture});
           
           $lendPixbuf->composite($pixbuf,
                                  0, 0, 
                                  $pixbuf->get_width, $pixbuf->get_height,
                                  0, 0,
                                  1, 1, 
                                  'nearest', 255);
           
        }

        if ($self->{withImage})
        {
            $pixbuf->composite($self->{backgroundPixbuf},
                                  5, 15, 
                                  $self->{imgWidth}, $self->{imgHeight},
                                  5, 15,
                                  1, 1, 
                                  'nearest', 255);
            $pixbuf = $self->{backgroundPixbuf}->copy;
            
        }


        $self->{previousPixbuf} = $pixbuf->copy;
        $pixbuf->saturate_and_pixelate($pixbuf, 1.5, 0);
        $self->{selectedPixbuf}->composite($pixbuf,
                                           0, 0, 
                                           $self->{vboxWidth}, $self->{vboxHeight},
                                           0, 0,
                                           1, 1, 
                                           'nearest', 255)
            if $self->{withImage};
        
        $boxes[$self->{current}]->child->set_from_pixbuf($pixbuf);
    }

    sub changeCurrentTitle
    {
        my ($self, $title) = @_;
        
        $self->{tooltips}->set_tip($self->{boxes}->[$self->{previous}], $self->{parent}->transformTitle($title), '');
    }

    
}

{
    package GCDetailledList;
    
    use File::Basename;
    use Gtk2::SimpleList;
    use base "Gtk2::ScrolledWindow";

    sub new
    {
        my ($proto, $parent, $title) = @_;
        my $class = ref($proto) || $proto;
        my $self  = $class->SUPER::new;
        bless ($self, $class);

        $self->{parent} = $parent;
        
        $self->{imgWidth} = 60;
        $self->{imgHeight} = 80;
        $self->{lendPicture} = $parent->{style}->{dir}.'/lend.png';
        $self->{defaultImage} = $parent->{defaultImage};
        
        $self->set_policy ('automatic', 'automatic');
        $self->set_shadow_type('etched-in');
        
        $parent->{options}->details('image|title|date|director|type')
            if ! $parent->{options}->exists('details');
        my @tmpArray = split m/\|/, $parent->{options}->details;
        $self->{fieldsArray} = \@tmpArray;
        
        my @columns;
        foreach my $field(@tmpArray)
        {
            my $title = $parent->{lang}->{FieldsList}->{$field};
            $title =~ s/.*\s*:\s*//;
        
            push @columns, $title, 'text' if $field ne 'image';
            push @columns, $title, 'pixbuf' if $field eq 'image';
        }
        
        
        $self->{list} = new Gtk2::SimpleList(@columns);
        $self->{list}->set_name('GCMoviesDetailsList');
        $self->{list}->set_headers_clickable(1);
        $self->{list}->set_rules_hint(1);
        #$self->{list}->get_selection->set_mode ('multiple');
        
        $self->{list}->signal_connect (cursor_changed => sub {
            my ($sl, $path, $column) = @_;
            my @idx = $sl->get_selected_indices;
            $parent->display($idx[0]);
            $self->{currentIdx} = $idx[0];
        });

        $self->{list}->signal_connect ('row-activated' => sub {
           $parent->displayInWindow;
        });

        $self->add($self->{list});
        
        $self->{list}->signal_connect('button_press_event' => sub {
                my ($widget, $event) = @_;
                return 0 if $event->button ne 3;
                $self->{context}->popup(undef, undef, undef, undef, $event->button, $event->time);
                return 0;
        });

        $self->{titleIndex} = -1;
        $self->{imageIndex} = -1;

        my $i = -1;
        foreach my $field(@{$self->{fieldsArray}})
        {
            $i++;
            $self->{list}->get_column($i)->set_resizable(1);
            $self->{titleIndex} = $i if $field eq 'title';
            $self->{imageIndex} = $i if $field eq 'image';
            next if $field eq 'image';
            $self->{list}->get_column($i)->signal_connect('clicked' => sub {
                $parent->{movies}->changeOrder($field);
                $self->setSortOrder;
            });
        }

        $self->{currentIdx} = 0;

        return $self;
    }

    sub setSortOrder
    {
        my $self = shift;
        my ($order, $field) = $self->{parent}->{movies}->getSortOrder;
        my $i = 0;

        foreach (@{$self->{fieldsArray}})
        {
            if ($_ eq $field)
            {
                $self->{list}->get_column($i)->set_sort_indicator(1);
                $self->{list}->get_column($i)->set_sort_order($order ? 'ascending' : 'descending');
            }
            else
            {
                $self->{list}->get_column($i)->set_sort_indicator(0);
            }
        
            $i++;
        }
    }

    sub done
    {
        my $self = shift;
    }

    sub clearCache
    {
        my $self = shift;
    }

    sub reset
    {
        my $self = shift;
        
        @{$self->{list}->{data}} = ();
    }

    sub addMovie
    {
        my ($self, $info) = @_;
                
        my @data;
        
        foreach my $field(@{$self->{fieldsArray}})
        {
            my $value;
            $value = $info->{$field};
            $value =~ s/,*$//;
            $value = $self->{parent}->transformTitle($value) if $field eq 'title';
            if ($field eq 'image')
            {
                $value = GCUtils::getDisplayedImage($value,
                                                    $self->{defaultImage},
                                                    $self->{parent}->{options}->file);
                $value = Gtk2::Gdk::Pixbuf->new_from_file($value);
                $value = $value->scale_simple($self->{imgWidth},$self->{imgHeight},'bilinear')
                if ($value->get_width > $self->{imgWidth}) ||
                   ($value->get_height > $self->{imgHeight});
            }
            
            push @data, $value;
        }
        
        push @{$self->{list}->{data}}, \@data;
    }
    
    sub removeMovie
    {
        my ($self, $number) = @_;
        
        splice @{$self->{list}->{data}}, $number, 1;
        
        #Reload not needed
        return 0;
    }
        
    sub select
    {
        my ($self, $idx) = @_;
        
        $self->{list}->select($idx);
        $self->{currentIdx} = $idx;
    }
    
    sub showCurrent
    {
        my $self = shift;
        
   		my $path = $self->{list}->get_selection->get_selected_rows;
		$self->{list}->scroll_to_cell($path) if $path;
    }
    
    sub changeCurrent
    {
        my ($self, $image, $borrower) = @_;
        $image = GCUtils::getDisplayedImage($image,
                                            $self->{defaultImage},
                                            $self->{parent}->{options}->file);
        my $pixbuf = Gtk2::Gdk::Pixbuf->new_from_file($image);
        $pixbuf = $pixbuf->scale_simple($self->{imgWidth},$self->{imgHeight},'bilinear')
        if ($pixbuf->get_width > $self->{imgWidth}) ||
           ($pixbuf->get_height > $self->{imgHeight});
        
        $self->{list}->{data}->[$self->{currentIdx}]->[$self->{imageIndex}] = $pixbuf
            if $self->{imageIndex} > -1;
    }
    
    sub changeCurrentTitle
    {
        my ($self, $title) = @_;
        
        $self->{list}->{data}->[$self->{currentIdx}]->[$self->{titleIndex}] = $self->{parent}->transformTitle($title)
            if $self->{indexTitle} > -1;
    }
    
    
}



1;