package GCExport::GCExportBase;

###################################################
#
#  Copyright 2005 Tian
#
#  This file is part of GCfilms.
#
#  GCfilms is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  GCfilms is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with GCfilms; if not, write to the Free Software
#  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
###################################################

use strict;

{
    package GCExport::GCExportBaseClass;
    use File::Basename;
    use File::Copy;
    use GCUtils;
    
    if ($^O =~ /win32/i)
    {
        use File::DosGlob 'glob';
    }
    
    #Methods to be overriden in specific classes
    
    sub new
    {
        my $proto = shift;
        my $class = ref($proto) || $proto;
        my $self  = {};

        bless ($self, $class);
        return $self;
    }

    sub getName
    {
    }
    
    sub getSuffix
    {
        return '';
    }
    
    sub needsUTF8
    {
        return 0;
    }
    
    sub getOptions
    {
    }
    
    sub wantsDirectorySelection
    {
        return 0;
    }
    
    sub wantsFieldsSelection
    {
        return 0;
    }
    
    sub wantsImagesSelection
    {
        return 0;
    }
    
    sub wantsFileSelection
    {
        return 1;
    }
    
    sub getHeader
    {
    }

    sub getItem
    {
    }

    sub getFooter
    {
    }

    sub postProcess
    {
    }

    sub preProcess
    {
    }

    sub getEndInfo
    {
    }

    #End of methods to be overriden
    
    sub duplicatePicture
    {
        my ($self, $orig, $dir) = @_;
        
        $self->{saved}->{image} = $orig;
        my $newPic = $orig;
        if ($orig && $self->{options}->{withPictures})
        {
            $newPic = GCUtils::getDisplayedImage($orig,
                                                 $self->{options}->{defaultImage},
                                                 $self->{orig});

            (my $suffix = $newPic) =~ s/.*?(\.[^.]*)$/$1/;
            my $dest = $self->{options}->{parent}->getUniqueImageFileName($suffix,
                                                                          $dir);
            copy($newPic, $dest);
            $newPic = basename($dir).'/'.basename($dest);
            $newPic =~ s/\//\\/g if ($^O =~ /win32/i);
        }
        else
        {
            $newPic = basename($dir).'/'.basename($self->{options}->{defaultImage});
        }
        return $newPic;
    }
        
    sub initLists
    {
        my $self = shift;

        $self->{lists} = [];

        foreach (@{$self->{options}->{moviesContainer}->{fields}})
        {
            push @{$self->{lists}}, $_
                if ($self->{options}->{moviesContainer}->{fieldsInfo}->{$_}->{isList});
        }
    }
    
    sub cleanLists
    {
        my ($self, $info) = @_;
        
        foreach (@{$self->{lists}})
        {
            $self->{saved}->{$_} = $info->{$_};
            $info->{$_} =~ s/,*$//;
        }
    }

    sub restoreInfo
    {
        my ($self, $info) = @_;

        $info->{audio} = $info->{savedAudio};
        $info->{subt} = $info->{savedSubt};
        delete $info->{savedAudio};
        delete $info->{audioencoding};
        delete $info->{savedSubt};
        delete $info->{year};

        foreach (keys %{$self->{saved}})
        {
            $info->{$_} = $self->{saved}->{$_};
        }
    }
    
    # If you need really specific processing, you can instead override the process method
    
    sub process
    {
        my ($self, $options) = @_;

        $self->{saved} = {};

        $self->{options} = $options;
        $self->initLists;

        $options->{file}.= $self->getSuffix
            if ($self->getSuffix) 
            && ($options->{file} !~ /\.\w*$/);
        $self->{fileName} = $options->{file};
        $self->{orig} = $options->{collection};
        $self->{origDir} = dirname($self->{orig});
        $options->{collectionDir} = $self->{origDir};

        ($self->{dirName} = $self->{fileName}) =~ s/\.[^.]*?$//;
        $self->{dirName} .= '_images';
        if ( -e $self->{dirName})
        {
            my @images = glob $self->{dirName}.'/*';
            unlink foreach (@images);
            rmdir $self->{dirName};
            unlink $self->{dirName} if ( -e $self->{dirName});
        }
        mkdir $self->{dirName} if ($self->{options}->{withPictures});

        #Get a copy of default picture
        copy($self->{options}->{defaultImage},$self->{dirName});

        $self->preProcess();

        my @tmpArray = @{$options->{movies}};

        my $header = $self->getHeader($#tmpArray + 1);
        my $body = '';

        my $movie;
        my $idx = 0;
        foreach $movie(@tmpArray)
        {
            $movie->{image} = $self->duplicatePicture($movie->{image}, $self->{dirName});

            $self->cleanLists($movie);

            $movie->{savedAudio} = $movie->{audio};
            ($movie->{audioencoding} = $movie->{audio}) =~ s/;(.*?)(,|$)/ ($1)$2/g;
            $movie->{audioencoding} =~ s/ \(\)//g;
            $movie->{audioencoding} =~ s/,([^ ])/, $1/g;
            $movie->{audio} =~ s/;.*?(,|$)/$1/g;
            $movie->{audio} =~ s/,([^ ])/, $1/g;
            $movie->{savedSubt} = $movie->{subt};
            $movie->{subt} =~ s/,([^ ])/, $1/g;
            $movie->{year} = GCUtils::extractYear($movie->{date});

            $body .= $self->getItem($movie, $idx);

            $self->restoreInfo($movie);

	        $idx++;
        }
        my $footer = $self->getFooter($#tmpArray + 1);

        $self->postProcess(\$header, \$body);

        open EXPORTFILE, ">".$options->{file};
        binmode( EXPORTFILE, ':utf8') if $self->needsUTF8;
        printf EXPORTFILE "$header";
        printf EXPORTFILE "$body";
        printf EXPORTFILE "$footer";
        close EXPORTFILE;

        return $self->getEndInfo;
    }
}

1;
