/* gcompris - file_selector.c
 *
 * Time-stamp: <2004/10/28 00:45:50 bruno>
 *
 * Copyright (C) 2000 Bruno Coudoin
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/**
 * An file selector for gcompris
 *
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>

#include "gcompris.h"

#define SOUNDLISTFILE PACKAGE

#define MODE_LOAD 1
#define MODE_SAVE 2

static gint		 item_event_file_selector(GnomeCanvasItem *item, 
						  GdkEvent *event, 
						  gpointer data);
static gint		 item_event_fileset_selector(GnomeCanvasItem *item, 
						     GdkEvent *event, 
						     gpointer data);
static gint		 item_event_directory(GnomeCanvasItem *item, 
					      GdkEvent *event, 
					      char *dir);
static void		 display_files(GnomeCanvasItem *rootitem, gchar *rootdir);
static int		 display_file_selector(int mode, 
					       GcomprisBoard *gcomprisBoard, gchar *rootdir,
					       FileSelectorCallBack iscb);
static void		 entry_enter_callback( GtkWidget *widget,
					       GtkWidget *entry );
static int		 create_rootdir (gchar *rootdir);
static void		 free_stuff (GtkObject *obj, gchar* data);

static gboolean		 file_selector_displayed = FALSE;

static GnomeCanvasItem	*rootitem = NULL;
static GnomeCanvasItem	*file_root_item = NULL;

static FileSelectorCallBack fileSelectorCallBack = NULL;

static gchar *current_rootdir = NULL;
static GtkEntry *widget_entry = NULL;

/* Represent the limits of control area */
#define	CONTROL_AREA_X1	40
#define	CONTROL_AREA_Y1	20
#define	CONTROL_AREA_X2	760
#define	CONTROL_AREA_Y2	80

/* Represent the limits of the file area */
#define	DRAWING_AREA_X1	40
#define DRAWING_AREA_Y1	120
#define DRAWING_AREA_X2	760
#define DRAWING_AREA_Y2	500

#define HORIZONTAL_NUMBER_OF_IMAGE	6
#define VERTICAL_NUMBER_OF_IMAGE	5
#define IMAGE_GAP			18

#define IMAGE_WIDTH  (DRAWING_AREA_X2-DRAWING_AREA_X1)/HORIZONTAL_NUMBER_OF_IMAGE-IMAGE_GAP
#define IMAGE_HEIGHT (DRAWING_AREA_Y2-DRAWING_AREA_Y1)/VERTICAL_NUMBER_OF_IMAGE-IMAGE_GAP

/*
 * Main entry point 
 * ----------------
 *
 */

/*
 * Do all the file_selector display and register the events
 */

void gcompris_file_selector_save (GcomprisBoard *gcomprisBoard, gchar *rootdir,
				  FileSelectorCallBack iscb)
{
  display_file_selector(MODE_SAVE, gcomprisBoard, rootdir,
			iscb);
}

void gcompris_file_selector_load (GcomprisBoard *gcomprisBoard, gchar *rootdir,
				  FileSelectorCallBack iscb)
{
  display_file_selector(MODE_LOAD, gcomprisBoard, rootdir,
			iscb);
}

/*
 * Remove the displayed file_selector.
 * Do nothing if none is currently being dislayed
 */
void gcompris_file_selector_stop ()
{
  GcomprisBoard *gcomprisBoard = get_current_gcompris_board();

  if(gcomprisBoard!=NULL && file_selector_displayed)
    {
      if(gcomprisBoard->plugin->pause_board != NULL)
	{
	  gcomprisBoard->plugin->pause_board(FALSE);
	}
    }

  // Destroy the file_selector box
  /* FIXME: Crashes randomly */
  if(rootitem!=NULL)
    gtk_object_destroy(GTK_OBJECT(rootitem));

  rootitem = NULL;	  

  /* No need to destroy it since it's in rootitem but just clear it */
  file_root_item = NULL;

  gcompris_bar_hide(FALSE);
  file_selector_displayed = FALSE;
}



/*-------------------------------------------------------------------------------*/
/*-------------------------------------------------------------------------------*/
/*-------------------------------------------------------------------------------*/
/*-------------------------------------------------------------------------------*/


static int 
display_file_selector(int mode, 
		      GcomprisBoard *gcomprisBoard, gchar *rootdir,
		      FileSelectorCallBack iscb) {


  GnomeCanvasItem *item, *item2;
  GdkPixbuf	  *pixmap = NULL;
  gint		   y = 0;
  gint		   y_start = 0;
  gint		   x_start = 0;
  gchar		  *name = NULL;
  gchar           *full_rootdir;

  if(rootitem)
    return;

  gcompris_bar_hide(TRUE);

  if(gcomprisBoard!=NULL)
    {
      if(gcomprisBoard->plugin->pause_board != NULL)
	gcomprisBoard->plugin->pause_board(TRUE);
    }

  name = gcomprisBoard->name;
  fileSelectorCallBack=iscb;

  rootitem = \
    gnome_canvas_item_new (gnome_canvas_root(gcompris_get_canvas()),
			   gnome_canvas_group_get_type (),
			   "x", (double)0,
			   "y", (double)0,
			   NULL);

  pixmap = gcompris_load_skin_pixmap("file_selector_bg.png");
  y_start = (BOARDHEIGHT - gdk_pixbuf_get_height(pixmap))/2;
  x_start = (BOARDWIDTH - gdk_pixbuf_get_width(pixmap))/2;
  item = gnome_canvas_item_new (GNOME_CANVAS_GROUP(rootitem),
				gnome_canvas_pixbuf_get_type (),
				"pixbuf", pixmap, 
				"x", (double) x_start,
				"y", (double) y_start,
				NULL);
  y = BOARDHEIGHT - (BOARDHEIGHT - gdk_pixbuf_get_height(pixmap))/2 + 20;
  gdk_pixbuf_unref(pixmap);

  /* Entry area */
  widget_entry = gtk_entry_new_with_max_length (50);
  item = gnome_canvas_item_new (GNOME_CANVAS_GROUP(rootitem),
				gnome_canvas_widget_get_type (),
				"widget", GTK_WIDGET(widget_entry),
				"x", (double) CONTROL_AREA_X1 + 10,
				"y", (double) y_start + 30,
				"width", 250.0,
				"height", 20.0,
				"anchor", GTK_ANCHOR_NW,
				"size_pixels", FALSE,
				NULL);
  gtk_signal_connect(GTK_OBJECT(widget_entry), "activate",
		     GTK_SIGNAL_FUNC(entry_enter_callback),
		     widget_entry);
  /* Mandatory to have the keyboard keys going to our entry */
  gtk_grab_add(GTK_OBJECT (widget_entry));

  gtk_widget_show (widget_entry);

  y_start += 110;

  pixmap = gcompris_load_skin_pixmap("button_large.png");

  // CANCEL
  item = gnome_canvas_item_new (GNOME_CANVAS_GROUP(rootitem),
				gnome_canvas_pixbuf_get_type (),
				"pixbuf", pixmap, 
				"x", (double) (BOARDWIDTH*0.33) - gdk_pixbuf_get_width(pixmap)/2,
				"y", (double) y - gdk_pixbuf_get_height(pixmap) - 25,
				NULL);

  gtk_signal_connect(GTK_OBJECT(item), "event",
		     (GtkSignalFunc) item_event_file_selector,
		     "/cancel/");
  gtk_signal_connect(GTK_OBJECT(item), "event",
		     (GtkSignalFunc) gcompris_item_event_focus,
		     NULL);

  item2 = gnome_canvas_item_new (GNOME_CANVAS_GROUP(rootitem),
				 gnome_canvas_text_get_type (),
				 "text", _("CANCEL"),
				 "font", gcompris_skin_font_title,
				 "x", (double)  BOARDWIDTH*0.33,
				 "y", (double)  y - gdk_pixbuf_get_height(pixmap),
				 "anchor", GTK_ANCHOR_CENTER,
				 "fill_color_rgba", gcompris_skin_color_text_button,
				 NULL);
  gtk_signal_connect(GTK_OBJECT(item2), "event",
		     (GtkSignalFunc) item_event_file_selector,
		     "/cancel/");
  gtk_signal_connect(GTK_OBJECT(item2), "event",
		     (GtkSignalFunc) gcompris_item_event_focus,
		     item);

  // OK
  item = gnome_canvas_item_new (GNOME_CANVAS_GROUP(rootitem),
				gnome_canvas_pixbuf_get_type (),
				"pixbuf", pixmap, 
				"x", (double) (BOARDWIDTH*0.66) - gdk_pixbuf_get_width(pixmap)/2,
				"y", (double) y - gdk_pixbuf_get_height(pixmap) - 25,
				NULL);

  gtk_signal_connect(GTK_OBJECT(item), "event",
		     (GtkSignalFunc) item_event_file_selector,
		     "/ok/");
  gtk_signal_connect(GTK_OBJECT(item), "event",
		     (GtkSignalFunc) gcompris_item_event_focus,
		     NULL);

  item2 = gnome_canvas_item_new (GNOME_CANVAS_GROUP(rootitem),
				 gnome_canvas_text_get_type (),
				 "text", (mode==MODE_LOAD ? _("LOAD") : _("SAVE")),
				 "font", gcompris_skin_font_title,
				 "x", (double)  BOARDWIDTH*0.66,
				 "y", (double)  y - gdk_pixbuf_get_height(pixmap),
				 "anchor", GTK_ANCHOR_CENTER,
				 "fill_color_rgba", gcompris_skin_color_text_button,
				 NULL);
  gtk_signal_connect(GTK_OBJECT(item2), "event",
		     (GtkSignalFunc) item_event_file_selector,
		     "/ok/");
  gtk_signal_connect(GTK_OBJECT(item2), "event",
		     (GtkSignalFunc) gcompris_item_event_focus,
		     item);
  gdk_pixbuf_unref(pixmap);



  file_selector_displayed = TRUE;

  if(g_get_home_dir()) {
    full_rootdir = g_strconcat(g_get_home_dir(), "/.gcompris/user_data", NULL);
    create_rootdir(full_rootdir);
    g_free(full_rootdir);

    full_rootdir = g_strconcat(g_get_home_dir(), "/.gcompris/user_data/", rootdir, NULL);
  } else {
    /* On WIN98, No home dir */
    full_rootdir = g_strdup_printf("%s", "gcompris/user_data");
    create_rootdir(full_rootdir);
    g_free(full_rootdir);
  
    full_rootdir = g_strdup_printf("%s/%s", "gcompris/user_data", rootdir);
  }

  create_rootdir(full_rootdir);

  current_rootdir = full_rootdir;
  display_files(rootitem, full_rootdir);

}


/*
 * create the root dir if needed
 *
 * return 0 if OK, -1 if ERROR
 */
static int
create_rootdir (gchar *rootdir)
{

  if(g_file_test(rootdir, G_FILE_TEST_IS_DIR)) {
    return 0;
  }

  return(mkdir(rootdir, 0755));
}

static void
free_stuff (GtkObject *obj, gchar *data)
{
  g_free(data);
}

static void display_files(GnomeCanvasItem *root_item, gchar *rootdir)
{

  GdkPixbuf *pixmap_dir  = NULL;
  GdkPixbuf *pixmap_file = NULL;
  GnomeCanvasItem *item;
  double iw, ih;
  struct dirent *one_dirent;
  DIR *dir;

  /* Initial image position */
  guint ix  = DRAWING_AREA_X1;
  guint iy  = DRAWING_AREA_Y1;
  guint isy = DRAWING_AREA_Y1;

  if(!rootitem)
    return;

  /* Display the directory content */
  dir = opendir(rootdir);

  if (!dir) {
    g_warning("gcompris_file_selector : no root directory found in %s", rootdir);
    g_free(rootdir);
    return;
  }

  /* Delete the previous file root if any */
  if(file_root_item!=NULL) {
    gtk_object_destroy(GTK_OBJECT(file_root_item));
  }

  /* Create a root item to put the image list in it */
  file_root_item = \
    gnome_canvas_item_new (GNOME_CANVAS_GROUP(root_item),
			   gnome_canvas_group_get_type (),
			   "x", (double)0,
			   "y", (double)0,
			   NULL);

  item = gnome_canvas_item_new (GNOME_CANVAS_GROUP(file_root_item),
				gnome_canvas_text_get_type (),
				"text", rootdir,
				"x", (double)CONTROL_AREA_X1 + 10,
				"y", (double)CONTROL_AREA_Y1 + 50,
				"fill_color_rgba", 0x0000FFFF,
				"anchor", GTK_ANCHOR_NW,
				NULL);


  pixmap_dir  = gcompris_load_pixmap(gcompris_image_to_skin("directory.png"));
  pixmap_file = gcompris_load_pixmap(gcompris_image_to_skin("file.png"));

  iw = IMAGE_WIDTH;
  ih = IMAGE_HEIGHT;

  while((one_dirent = readdir(dir)) != NULL) {
    /* add the file to the display */
    gchar *filename;
    GdkPixbuf *pixmap_current  = pixmap_file;
      
    if((strcmp(one_dirent->d_name, "..")==0) &&
       strcmp(current_rootdir, rootdir)==0)
      continue;

    if(strcmp(one_dirent->d_name, ".")==0)
      continue;

    filename = g_strdup_printf("%s/%s",
			       rootdir, one_dirent->d_name);
    
    if(g_file_test(filename, G_FILE_TEST_IS_DIR)) {
      pixmap_current  = pixmap_dir;
    }

    item = gnome_canvas_item_new (GNOME_CANVAS_GROUP(file_root_item),
				  gnome_canvas_pixbuf_get_type (),
				  "pixbuf", pixmap_current,
				  "x", (double)ix + (IMAGE_WIDTH + IMAGE_GAP - gdk_pixbuf_get_width(pixmap_current))/2,
				  "y", (double)iy,
				  NULL);

    if(g_file_test(filename, G_FILE_TEST_IS_DIR)) {
      gtk_signal_connect(GTK_OBJECT(item), "event",
			 (GtkSignalFunc) item_event_directory,
			 filename);
    } else {
      gtk_signal_connect(GTK_OBJECT(item), "event",
			 (GtkSignalFunc) item_event_file_selector,
			 filename);
    }
    gtk_signal_connect(GTK_OBJECT(item), "event",
		       (GtkSignalFunc) gcompris_item_event_focus,
		       NULL);

    g_object_set_data (G_OBJECT (item), "path", filename);
    g_signal_connect (item, "destroy",
		      G_CALLBACK (free_stuff),
		      filename);

    item = gnome_canvas_item_new (GNOME_CANVAS_GROUP(file_root_item),
				  gnome_canvas_text_get_type (),
				  "text", g_path_get_basename(filename),
				  "x", (double)ix + (IMAGE_WIDTH + IMAGE_GAP)/2,
				  "y", (double)iy + IMAGE_HEIGHT - 5,
				  "anchor", GTK_ANCHOR_CENTER,
				  "fill_color_rgba", 0x0000FFFF,
				  NULL);

    gtk_signal_connect(GTK_OBJECT(item), "event",
		       (GtkSignalFunc) item_event_file_selector,
		       filename);

    ix+=IMAGE_WIDTH + IMAGE_GAP;
    if(ix>=DRAWING_AREA_X2-IMAGE_GAP)
      {
	ix=DRAWING_AREA_X1;
	iy+=IMAGE_HEIGHT + IMAGE_GAP;
      }

  }

  closedir(dir);

  gdk_pixbuf_unref(pixmap_dir);
  gdk_pixbuf_unref(pixmap_file);

}

/* Callback when a directory is selected */
static gint
item_event_directory(GnomeCanvasItem *item, GdkEvent *event, gchar *dir)
{

  if(!rootitem)
    return;

  switch (event->type) 
    {
    case GDK_ENTER_NOTIFY:
      break;
    case GDK_LEAVE_NOTIFY:
      break;
    case GDK_BUTTON_PRESS:
      if(strcmp(g_path_get_basename(dir), "..")==0) {
	/* Up one level. Remove .. and one directory on the right of the path */
	dir[strlen(dir)-3] = '\0';
	dir=g_path_get_dirname(dir);
      }
      display_files(rootitem, g_strdup(dir));
      gtk_entry_set_text(widget_entry, "");
      break;
    default:
      break;
    }
  return FALSE;

}

/* Callback when a file is selected */
static gint
item_event_file_selector(GnomeCanvasItem *item, GdkEvent *event, gpointer data)
{

  if(!rootitem)
    return;

  switch (event->type) 
    {
    case GDK_ENTER_NOTIFY:
      break;
    case GDK_LEAVE_NOTIFY:
      break;
    case GDK_BUTTON_PRESS:
      if(!strcmp((char *)data, "/ok/"))	{

	/* Nothing selected, please cancel instead */
	if(strcmp(gtk_entry_get_text(widget_entry),"")==0) {
	  return FALSE;
	}

	if(fileSelectorCallBack!=NULL) {
	  gchar *result;
	  result = g_strdup_printf("%s/%s", current_rootdir, gtk_entry_get_text(widget_entry));
	  fileSelectorCallBack(result);
	}
	gcompris_file_selector_stop();
      } else if(!strcmp((char *)data, "/cancel/")) {
	gcompris_file_selector_stop();
      } else {
	gtk_entry_set_text(widget_entry, g_path_get_basename((gchar *)data));
      }
      break;
    default:
      break;
    }
  return FALSE;

}

static void entry_enter_callback( GtkWidget *widget,
				  GtkWidget *entry )
{
  gchar *entry_text;

  if(!rootitem)
    return;

  entry_text = gtk_entry_get_text(GTK_ENTRY(entry));
}


/* Local Variables: */
/* mode:c */
/* eval:(load-library "time-stamp") */
/* eval:(make-local-variable 'write-file-hooks) */
/* eval:(add-hook 'write-file-hooks 'time-stamp) */
/* eval:(setq time-stamp-format '(time-stamp-yyyy/mm/dd time-stamp-hh:mm:ss user-login-name)) */
/* End: */
