package GCData;

###################################################
#
#  Copyright 2005-2006 Tian
#
#  This file is part of GCstar.
#
#  GCstar is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  GCstar is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with GCstar; if not, write to the Free Software
#  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
###################################################

use strict;

our $collectionTag = 'collection';
our $inlineCollectionTag = 'collectionInlineDescription';
our $preferencesTag = 'preferences';
our $inlinePreferencesTag = 'collectionInlinePreferences';


{
    package GCItems;
    
    use GCModel;
    use XML::Parser;
    
    sub new
    {
        my ($proto, $parent) = @_;
        my $class = ref($proto) || $proto;
        my $self  = {};

        $self->{parent} = $parent;

        $self->{imagesToBeRemoved} = [];
        $self->{imagesToBeAdded} = [];
        $self->{itemArray} = [];

        $self->{currentItem} = -1;
        $self->{currentId} = -1;
        $self->{hasBeenDeleted} = 0;
        $self->{block} = 0;
        $self->{filterSearch} = new GCFilterSearch;

        bless ($self, $class);
        return $self;
    }

    sub initModel
    {
        my ($self, $model) = @_;
        $self->{model} = $model;
        $self->setSerieFields;
        $self->restoreSort;
        $self->{parent}->notifyModelChange;
    }

    sub setPanel
    {
        my ($self, $panel) = @_;
        $self->{panel} = $panel;
    }

    sub updateCurrentFromPanel
    {
        my ($self, $panel) = @_;
        my $previousPanel = $self->{panel};
        $self->{panel} = $panel;
        $self->updateCurrent;
        $self->{panel} = $previousPanel;
    }

    sub updateCurrent
    {
        my $self = shift;

        return if $self->{currentItem} == -1;
        my $oldSortValue = $self->{itemArray}[$self->{currentItem}]->{$self->{currentSortField}};

        my $idField = $self->{model}->{commonFields}->{id};

        my $panelId = $self->{panel}->$idField;
        my $previousId = $self->{itemArray}[$self->{currentItem}]->{$idField};
        
        my $changed = 0;
        
        if ($panelId &&
           ($panelId != $previousId))
        {
		  $self->{itemArray}[$self->{currentItem}]->{$idField} = $self->{panel}->$idField;
		  $self->{maxId} = $panelId if ($panelId > $self->{maxId});
		  $self->findMaxId if $previousId == $self->{maxId};
		  $self->{currentId} = $panelId;
		  $self->{toBeDisplayed}->{$panelId} = 1;
		  $self->{toBeDisplayed}->{$previousId} = 0;
		  $changed = 1;
        }
        $self->{panel}->$idField($self->{itemArray}[$self->{currentItem}]->{$idField}) if $self->{panel}->$idField;

        my $info = $self->{itemArray}[$self->{currentItem}];
        my $previous = {$idField => $previousId};

        for my $field (@{$self->{model}->{fieldsNames}})
        {
            next if $field eq $idField;
            $previous->{$field} = $info->{$field};
            $self->{panel}->{$field}->addHistory if ($self->{model}->{fieldsInfo}->{$field}->{hasHistory});
            #$changed = $changed || $self->{panel}->{$field}->hasChanged;
            if (!$changed)
            {
                my $type = $self->{model}->{fieldsInfo}->{$field}->{type};
                if ($type =~ /list$/o)
                {
                    my $columns = GCUtils::listNameToNumber($self->{model}->{fieldsInfo}->{$field}->{type});
                    $changed = 1 if GCPreProcess::multipleList($info->{$field}, $columns)
                                 ne GCPreProcess::multipleList($self->{panel}->$field, $columns);
                }
                elsif ($type eq 'number')
                {
                    $changed = 1 if ($info->{$field} != $self->{panel}->$field);
                }
                else
                {
                    $changed = 1 if ($info->{$field} ne $self->{panel}->$field);
                }
            }
            $info->{$field} = $self->{panel}->$field;
            $self->{parent}->{menubar}->checkFilter($field) if $changed;
        }

        if (!($self->{model}->{preferences}->masterType eq 'generated'))
        {
            if ($self->isRankNumeric)
            {
                if (($previous->{$self->{serieField}})
                 && (($previous->{$self->{serieField}} != $info->{$self->{serieField}})
                  || ($self->{minRanks}->{$previous->{$self->{serieField}}} == $previous->{$self->{rankField}})
                  || ($self->{maxRanks}->{$previous->{$self->{serieField}}} == $previous->{$self->{rankField}})))
                {
                    #Min or max rank has to be changed for previous serie
                    $self->findMinMaxForSerie($previous->{$self->{serieField}});
                }
        
                if (($info->{$self->{serieField}})
                 && (($self->{minRanks}->{$info->{$self->{serieField}}} > $info->{$self->{rankField}})
                  || ($self->{maxRanks}->{$info->{$self->{serieField}}} < $info->{$self->{rankField}})))
                {
                    #Min or max rank has to be changed for new serie
                    $self->findMinMaxForSerie($info->{$self->{serieField}});
                }
                $self->{minRanks}->{$info->{$self->{serieField}}} = $info->{$self->{rankField}}
                    if  $info->{$self->{serieField}}
                    && ($info->{$self->{rankField}} < $self->{minRanks}->{$info->{$self->{serieField}}});
                $self->{maxRanks}->{$info->{$self->{serieField}}} = $info->{$self->{rankField}}
                    if  $info->{$self->{serieField}}
                    && ($info->{$self->{rankField}} > $self->{maxRanks}->{$info->{$self->{serieField}}});
            }
            else
            {
                if (($previous->{$self->{serieField}})
                 && (($previous->{$self->{serieField}} != $info->{$self->{serieField}})
                  || ($self->{minRanks}->{$previous->{$self->{serieField}}} eq $previous->{$self->{rankField}})
                  || ($self->{maxRanks}->{$previous->{$self->{serieField}}} eq $previous->{$self->{rankField}})))
                {
                    #Min or max rank has to be changed for previous serie
                    $self->findMinMaxForSerie($previous->{$self->{serieField}});
                }
        
                if (($info->{$self->{serieField}})
                 && (($self->{minRanks}->{$info->{$self->{serieField}}} gt $info->{$self->{rankField}})
                  || ($self->{maxRanks}->{$info->{$self->{serieField}}} lt $info->{$self->{rankField}})))
                {
                    #Min or max rank has to be changed for new serie
                    $self->findMinMaxForSerie($info->{$self->{serieField}});
                }
                $self->{minRanks}->{$info->{$self->{serieField}}} = $info->{$self->{rankField}}
                    if  $info->{$self->{serieField}}
                    && ($info->{$self->{rankField}} lt $self->{minRanks}->{$info->{$self->{serieField}}});
                $self->{maxRanks}->{$info->{$self->{serieField}}} = $info->{$self->{rankField}}
                    if  $info->{$self->{serieField}}
                    && ($info->{$self->{rankField}} gt $self->{maxRanks}->{$info->{$self->{serieField}}});
            }
        }

        my $reloaded = 0;                
        if ($changed)
        {
            # TODO Minimize the case where we reload (e.g. Not needed if it doesn't change the
            # item rank)
            if ($self->transformValue($oldSortValue, $self->{currentSortField}) ne
                $self->transformValue($self->{itemArray}[$self->{currentItem}]->{$self->{currentSortField}},
                                      $self->{currentSortField}))
            {
                $self->reloadList;
                $reloaded = 1;
            }
            elsif ($self->{parent}->{itemsList}->changeCurrent($previous, $info))
            {
                $self->reloadList;
                $reloaded = 1;
            }
            $self->{parent}->markAsUpdated;
        }
        return $reloaded;
    }
    
    sub getTitle
    {
        my ($self, $idx) = shift;

        my $realIdx = $idx;
        $realIdx = $self->{currentItem} if $idx == undef;
        return $self->{itemArray}[$realIdx]->{$self->{model}->{commonFields}->{title}};
    }
    
    sub displayCurrent
    {
        my $self = shift;
        $self->displayInPanel($self->{panel});
    }
    
    sub displayInPanel
    {
        my ($self, $panel, $idx) = @_;
        $idx = $self->{currentItem} if $idx == undef;
        return if $self->{currentItem} < 0;
        
        for my $field (@{$self->{model}->{fieldsNames}})
        {
            $panel->$field($self->{itemArray}[$idx]->{$field});
        }
        
        $panel->dataChanged;
    }

    sub display
    {
        my $self = shift;
        return if ! $self->{nbItems};

        my $number = shift;

        $number = 0 if $number == -1;
        $number = $self->selectToId($number);
        $self->{currentId} = $self->{itemArray}[$number]->{$self->{model}->{commonFields}->{id}};

        my $hasBeenReloaded = 0;

        if (($self->{currentItem} > -1) && !($self->{hasBeenDeleted}))
        {
            $hasBeenReloaded = $self->updateCurrent;
        }
        else
        {
            $self->{currentItem} = $number;
        }

        $self->{hasBeenDeleted} = 0;

        $self->{currentItem} = $number if ! $hasBeenReloaded;
        $self->displayCurrent;
        return $self->{currentItem};
    }

    sub setFilter
    {
        my ($self, $filter, $parameter) = @_;

        my $filtered = 1;

        $self->{filterSearch}->setFilter($filter, $parameter,
                                         $self->{model}->getFilterType($filter),
                                         $self->{model});

        $self->filter;
        $self->reloadList($self->{parent}, 1, $filtered);
        
        $self->displayCurrent;
    }

    sub setSearch
    {
        my ($self, $info) = @_;
        
        $self->{filterSearch}->clear;
        $self->{filterSearch}->setFilter($_, $info->{$_},
                                         $self->{model}->getFilterType($_),
                                         $self->{model})
            foreach (keys %$info);

        $self->filter;
        $self->reloadList($self->{parent}, 1, 1);
        
        $self->displayCurrent;
    }

    sub setSearchWithTypes
    {
        my ($self, $info, $mode) = @_;
        
        $self->{filterSearch}->clear;
        foreach (@$info)
        {
            $self->{filterSearch}->setFilter($_->{field},
                                             $_->{value},
                                             $_->{filter},
                                             $self->{model},
                                             1);
        }
        $self->{filterSearch}->setMode($mode);

        $self->filter;
        $self->reloadList($self->{parent}, 1, 1);
        $self->{filterSearch}->removeTemporaryFilters;
        $self->displayCurrent;
        $self->{filterSearch}->setMode;
    }

    sub filter
    {
        my $self = shift;

        my %tmpMap = {};        
        $self->{filterSearch}->setModel($self->{model});
        foreach (@{$self->{itemArray}})
        {
            my $id = $_->{$self->{model}->{commonFields}->{id}};
            if ($self->{filterSearch}->test($_))
            {
                $tmpMap{$id} = 1;
            }
            else
            {
                $tmpMap{$id} = 0;
            }
        }
       $self->{toBeDisplayed} = \%tmpMap;
    }

    sub changeOrder
    {
        my ($self, $field) = @_;
        
        $field = $self->{model}->{commonFields}->{title} if !$field;

        $self->{currentSortOrder} = 0 if ($field ne $self->{currentSortField});
        $self->{currentSortField} = $field;
        $self->{currentSortOrder} = 1 - $self->{currentSortOrder};
        
        $self->{model}->{preferences}->sortOrder($self->{currentSortOrder});
        $self->{model}->{preferences}->sortField($self->{currentSortField});
        
        $self->reloadList($self->{parent}, 1);
    }
    
    sub resetSortField
    {
        my $self = shift;
        $self->{currentSortField} = $self->{model}->{commonFields}->{title};
    }
    
    sub restoreSort
    {
        my $self = shift;

        my $preferences = $self->{model}->{preferences};

        $preferences->sortOrder(1) if ! $preferences->exists('sortOrder');
        $preferences->sortField($self->{model}->{commonFields}->{title}) if ! $preferences->exists('sortField');
        $self->{currentSortOrder} = $preferences->sortOrder;
        $self->{currentSortField} = $preferences->sortField;
    }
    
    sub getSortOrder
    {
        my $self = shift;
        
        return ($self->{currentSortOrder}, $self->{currentSortField});        
    }

    sub transformValue
    {
        my ($self, $value, $field, $type) = @_;
        
        $type ||= $self->{model}->{fieldsInfo}->{$field}->{type};
        $value = $self->{parent}->transformTitle($value) if $field eq $self->{model}->{commonFields}->{title};
        $value = GCPreProcess::reverseDate($value) if $type eq 'date';
        $value = GCPreProcess::multipleList($value, $type) if $type =~ /list$/o;
        return $value;
    }

    sub getValue
    {
        my ($self, $idx, $field) = @_;
        
        return $self->{itemArray}[$idx]->{$field};
    }
    
    sub setValue
    {
        my ($self, $idx, $field, $value) = @_;
        
        $self->{itemArray}[$idx]->{$field} = $value;
        if ($field eq $self->{rankField})
        {
            if ($self->isRankNumeric)
            {
                $self->{maxRanks}->{$self->{itemArray}[$idx]->{$self->{serieField}}} = $value
                    if ($self->{itemArray}[$idx]->{$self->{serieField}})
                    && ($value > ($self->{maxRanks}->{$self->{itemArray}[$idx]->{$self->{serieField}}}));
                $self->{minRanks}->{$self->{itemArray}[$idx]->{$self->{serieField}}} = $value
                    if ($self->{itemArray}[$idx]->{$self->{serieField}})
                    && (($value < ($self->{minRanks}->{$self->{itemArray}[$idx]->{$self->{serieField}}}))
                     || (!exists $self->{minRanks}->{$self->{itemArray}[$idx]->{$self->{serieField}}}));
            }
            else
            {
                $self->{maxRanks}->{$self->{itemArray}[$idx]->{$self->{serieField}}} = $value
                    if ($self->{itemArray}[$idx]->{$self->{serieField}})
                    && ($value gt ($self->{maxRanks}->{$self->{itemArray}[$idx]->{$self->{serieField}}}));
                $self->{minRanks}->{$self->{itemArray}[$idx]->{$self->{serieField}}} = $value
                    if ($self->{itemArray}[$idx]->{$self->{serieField}})
                    && (($value lt ($self->{minRanks}->{$self->{itemArray}[$idx]->{$self->{serieField}}}))
                     || (!exists $self->{minRanks}->{$self->{itemArray}[$idx]->{$self->{serieField}}}));
            }
        }
        if ($idx == $self->{currentItem})
        {
            $self->{panel}->$field($value);
            $self->{panel}->dataChanged;
        }
    }
    
    sub getItemsList
    {
        my ($self, $filter) = @_;
        
        return $self->{itemArray} if ! $filter;
        my @results = ();
        foreach (@{$self->{itemArray}})
        {
            if ($self->{toBeDisplayed}->{$_->{$self->{model}->{commonFields}->{id}}})
            {
                push @results, $_;
            }
        }
        return \@results;
    }
    
    sub sortItems
    {
        my $self = shift;

        my $initTime;

        $self->{sortCache} = {};
        sub compare
        {
            use locale;
            my $field = $self->{currentSortField};
            my $type = $self->{model}->{fieldsInfo}->{$field}->{type};
            return ($a->{$field} <=> $b->{$field})
                if ($type eq 'number');
#            return (
#                    ($self->{sortCache}->{$a->{$field}} ||= $self->transformValue($a->{$field}, $field, $type))
#                    cmp
#                    ($self->{sortCache}->{$b->{$field}} ||= $self->transformValue($b->{$field}, $field, $type))
#                   )
#                if ($type eq 'date');;
            return (
                    ($self->{sortCache}->{$a->{$field}} ||= $self->transformValue($a->{$field}, $field, $type))
                    cmp
                    ($self->{sortCache}->{$b->{$field}} ||= $self->transformValue($b->{$field}, $field, $type))
                   );
        }

        if ($self->{currentSortOrder})
        {
            @{$self->{itemArray}} = sort compare @{$self->{itemArray}};
        }
        else
        {
            @{$self->{itemArray}} = reverse sort compare @{$self->{itemArray}};
        }
    }

    sub reloadList
    {
        my ($self, $splash, $fullProgress, $filtering) = @_;

        return if $self->{block};
        if ($splash)
        {
            $splash->initProgress if $fullProgress;
            $splash->setItemsTotal($self->{nbItems});
        }
        
        $self->{parent}->{itemsList}->reset if $self->{parent}->{itemsList};
        
        $self->sortItems;

        my @displayedArray = ();
        my $lastDisplayed = -1;
        my $hasId = 0;
        my $currentDisplayed = 0;
        my $waitingForDisplayed = 0;
        my $j = 0;
        foreach (@{$self->{itemArray}})
        {
            $hasId = ($self->{currentId} eq $_->{$self->{model}->{commonFields}->{id}});

            if ($self->{toBeDisplayed}->{$_->{$self->{model}->{commonFields}->{id}}})
            {
                $self->{parent}->{itemsList}->addItem($_);
                
                if (($hasId) || $waitingForDisplayed)
                {
				    $self->{currentItem} = $j;
				    $currentDisplayed = $#displayedArray + 1;
                    if ($waitingForDisplayed)
                    {
                        $self->{currentId} =$_->{$self->{model}->{commonFields}->{id}};
                        $waitingForDisplayed = 0;   
                    }
                }
                
                push @displayedArray, $j;
                $lastDisplayed = $j;
            }
            else
            {
                if ($hasId)
                {
                    $currentDisplayed = $#displayedArray;
                    $self->{currentItem} = $lastDisplayed;
                    $self->{currentId} = $self->{itemArray}[$lastDisplayed]->{$self->{model}->{commonFields}->{id}}
                        unless $lastDisplayed < 0;
                    $waitingForDisplayed = 1 if $lastDisplayed < 0;
                }
            }

            $splash->setProgressForItemsDisplay($j) if $splash;
            $j++;
        }

        $self->{selectToIdArray} = \@displayedArray;

        if (! scalar(@displayedArray))
		{
		    $self->{panel}->hide($filtering);
		}
        else
        {
            $self->{panel}->show;
        }

        if ($splash && $fullProgress)
        {
            $splash->endProgress;
        }

        if ($self->{currentId} == -1)
        {
            #$currentDisplayed = 0;
            $self->{currentId}  = $self->selectToId(0);
        }

        if (! $self->{parent}->{initializing})
        {
            $self->{parent}->{itemsList}->done;
            $self->{parent}->{itemsList}->setSortOrder;
            $self->select($currentDisplayed, $self->{currentId}, 0);
        }

        $self->setStatus;
    }

    sub setStatus
    {
        my $self = shift;
        
        my $number = 0;
        $number = scalar @{$self->{selectToIdArray}} if $self->{selectToIdArray};
        my $status = ' ';
        $status .= $number;
        $status .= ' '.$self->{model}->getDisplayedText('Items');
        $self->{parent}->setStatus($status);
    }

    sub select
    {
		my ($self, $value, $id, $init) = @_;
		$self->{parent}->{itemsList}->select($value, $id, $init) unless  $value < -1;
    }

    sub removeCurrentItem
    {
        my $self = shift;

        my $number = $self->{currentItem};
        my $prevId = $self->{itemArray}[$number]->{$self->{model}->{commonFields}->{id}};

        foreach (@{$self->{model}->{managedImages}})
        {
            my $image = $self->{itemArray}[$number]->{$_};
            my $imagePrefix = $self->{parent}->{imagePrefix};
            if ($image =~ /(\/|\\)$imagePrefix[0-9]*\./)
            {
                $self->markToBeRemoved($image);
            }
        }

        my $oldInfo = splice @{$self->{itemArray}}, $number, 1;

        $self->{nbItems}--;

        if (!($self->{model}->{preferences}->masterType eq 'generated'))
        {
            if ($oldInfo->{$self->{serieField}} &&
               (($self->{minRanks}->{$oldInfo->{$self->{serieField}}} eq $oldInfo->{$self->{rankField}}) ||
                ($self->{maxRanks}->{$oldInfo->{$self->{serieField}}} eq $oldInfo->{$self->{rankField}})))
            {
                #Min or max rank has to be changed
                $self->findMinMaxForSerie($oldInfo->{$self->{serieField}});
            }
        }

        $self->{panel}->hide if ! $self->{nbItems};

        #$self->{currentItem}-- if $self->{currentItem};
        #$self->{currentItem}-- if $self->{currentItem} == $self->{nbItems};
        
#        my $id = $self->{itemArray}[$self->{currentItem}]->{$self->{model}->{commonFields}->{id}};
#        $self->{currentId} = $id;

        my ($needReload, $newIdx) = $self->{parent}->{itemsList}->removeItem($number, $prevId, $oldInfo);
        $self->{currentId} = $self->{itemArray}[$newIdx]->{$self->{model}->{commonFields}->{id}};
        if ($needReload)
        {
            $self->reloadList;
            $self->{parent}->{itemsList}->grab_focus;
        }
        $self->{currentItem} = $newIdx;

        $self->{hasBeenDeleted} = 1;

        $self->displayCurrent if $self->{nbItems};
    }

    sub addItem
    {
        my ($self, $info) = @_;
        $self->{panel}->show if ! $self->{nbItems};

        $self->{maxId}++;        
        my $currentId = $self->{maxId};
        $self->{itemArray}[$self->{nbItems}]->{$self->{model}->{commonFields}->{id}} = $currentId;
        
        for my $field (@{$self->{model}->{fieldsNames}})
        {
            next if $field eq $self->{model}->{commonFields}->{id};
            if ($self->{model}->{fieldsInfo}->{$field}->{hasHistory})
            {
                $self->{panel}->{$field}->addHistory($info->{$field});
            }
            $self->{itemArray}[$self->{nbItems}]->{$field} = $info->{$field};
        }

        $self->{currentId} = $self->{maxId};
        $self->{toBeDisplayed}->{$self->{currentId}} = 1;

        $self->{toBeDisplayed}->{$self->{currentId}} = 1;
        $self->{parent}->{itemsList}->addItem($self->{itemArray}[$self->{nbItems}], 1);

        $self->{currentItem} = $self->{nbItems};
        push @{$self->{selectToIdArray}}, $self->{nbItems};

        $self->select($self->{nbItems}, $self->{currentId}, 0);
        $self->setStatus;
        
        $self->{nbItems}++;        

        $self->{parent}->{itemsList}->showCurrent;
        
        return $currentId;
    }
    
    sub insertItem
    {
        my ($self, $info) = @_;

        $self->{maxId}++;
        $self->{itemArray}[$self->{nbItems}]->{$self->{model}->{commonFields}->{id}} = $self->{maxId};
        for my $field (@{$self->{model}->{fieldsNames}}) 
        {
            next if $field eq $self->{model}->{commonFields}->{id};
            $self->{itemArray}[$self->{nbItems}]->{$field} = $info->{$field};
            if ($self->{model}->{fieldsInfo}->{$field}->{hasHistory})
            {
                $self->{panel}->{$field}->addHistory($info->{$field});
            }

        }
        
        $self->{nbItems}++;

        #$self->{currentId} = $self->{maxId};
        $self->{toBeDisplayed}->{$self->{maxId}} = 1;
    }

    sub selectToId
    {
		my ($self, $i) = @_;
	
		return $self->{selectToIdArray}[$i];
    }

    sub setOptions
    {
        my ($self, $options, $splash) = @_;

        $self->{options} = $options;

        if (! $self->load($options->file, $splash, 0))
        {
            $options->file('');
            return 0;
        }
        return 1;
    }

    sub markToBeRemoved
    {
        my ($self, $image) = @_;

        push @{$self->{imagesToBeRemoved}}, $image;
    }

    sub markToBeAdded
    {
        my ($self, $image) = @_;
        push @{$self->{imagesToBeAdded}}, $image;        
    }
    
    sub removeMarkedPictures
    {
        my $self = shift;
        my $image;
        foreach $image(@{$self->{imagesToBeRemoved}})
        {
            unlink $image;
        }

        $self->{imagesToBeRemoved} = [];
    }

    sub addMarkedPictures
    {
        my $self = shift;

        $self->{imagesToBeAdded} = [];
    }
    
    sub clean
    {
        my $self = shift;
        my $image;
        foreach (@{$self->{imagesToBeAdded}})
        {
            unlink $_;
        }
    }

    sub queryReplace
    {
        my ($self, $field, $old, $new) = @_;
        foreach (@{$self->{itemArray}})
        {
            $_->{$field} =~ s/$old/$new/gi;
        }
        $self->displayCurrent;
        $self->reloadList;
    }
    
    sub isRankNumeric
    {
        my $self = shift;
        my $rankField = $self->{model}->{preferences}->orderBy;
        return ($self->{model}->{fieldsInfo}->{$rankField}->{type} eq 'number');
    }
    
    sub isMaster
    {
        my ($self, $info) = @_;
        if ($self->{model}->{preferences}->masterType eq 'generated')
        {
            #We never have a real master as it will be generated by the items list.
            # But for generated ones, a special field is added to let us the same methods
            #return 1 if $info->{isGeneratedMaster};
            return 0;
        }
        my $serieField = $self->{model}->{preferences}->groupBy;
        my $rankField = $self->{model}->{preferences}->orderBy;
        if ($self->isRankNumeric)
        {
            return 1 if $info->{$rankField} <= $self->{minRanks}->{$info->{$serieField}};
        }
        else
        {
            return 1 if $info->{$rankField} le $self->{minRanks}->{$info->{$serieField}};
        }
        return 0;
    }
    
    sub getMinRank
    {
        my ($self, $serie) = @_;
        
        return $self->{minRanks}->{$serie}
    }
    
    sub getMaxRank
    {
        my ($self, $serie) = @_;
        return $self->{maxRanks}->{$serie}
    }

    sub findMinMaxForSerie
    {
        my ($self, $serie) = @_;

        $self->{minRanks}->{$serie} = 1001;
        $self->{maxRanks}->{$serie} = -1;
        my $i;
        my $isNumeric = $self->isRankNumeric;
        #We scan all the items to find the new master
        # (May be optimized)
        foreach (@{$self->{itemArray}})
        {
            next if $_->{$self->{serieField}} ne $serie;
            my $rank = $_->{$self->{rankField}};
            if ($isNumeric)
            {
                $self->{minRanks}->{$serie} = $rank
                    if $rank < $self->{minRanks}->{$serie};
                $self->{maxRanks}->{$serie} = $rank
                    if $rank > $self->{maxRanks}->{$serie};
            }
            else
            {
                $self->{minRanks}->{$serie} = $rank
                    if $rank lt $self->{minRanks}->{$serie};
                $self->{maxRanks}->{$serie} = $rank
                    if $rank gt $self->{maxRanks}->{$serie};
            }            
        }
    }
    
    sub setSerieFields
    {
        my $self = shift;
        my $serie = $self->{model}->{preferences}->groupBy;
        my $rank = $self->{model}->{preferences}->orderBy;
        $self->{serieField} = $serie;
        $self->{rankField} = $rank;
        
        #Now we have to regenerate the min and max info
        $self->{minRanks} = {};
        $self->{maxRanks} = {};
        foreach (@{$self->{itemArray}})
        {
            my ($serie, $rank) = ($_->{$serie},
                                  $_->{$rank});
            next if ! $serie;
            
            if ($self->isRankNumeric)
            {
                $self->{minRanks}->{$serie} = 1001 if ! exists $self->{minRanks}->{$serie};
                $self->{maxRanks}->{$serie} = -1 if ! exists $self->{maxRanks}->{$serie};
                $self->{minRanks}->{$serie} = $rank if ($rank < $self->{minRanks}->{$serie});
                $self->{maxRanks}->{$serie} = $rank if ($rank > $self->{maxRanks}->{$serie});
            }
            else
            {
                $self->{minRanks}->{$serie} = 'ZZZZZ' if ! exists $self->{minRanks}->{$serie};
                $self->{maxRanks}->{$serie} = ' ' if ! exists $self->{maxRanks}->{$serie};
                $self->{minRanks}->{$serie} = $rank if ($rank lt $self->{minRanks}->{$serie});
                $self->{maxRanks}->{$serie} = $rank if ($rank gt $self->{maxRanks}->{$serie});
            }
        }
    }

    sub findMaxId
    {
        my $self = shift;

        $self->{maxId} = -1;
        foreach (@{$self->{itemArray}})
        {
            $self->{maxId} = $_->{$self->{model}->{commonFields}->{id}}
                if $_->{$self->{model}->{commonFields}->{id}} > $self->{maxId};
        }
    }
    
    sub clearList
    {
        my $self = shift;
        $self->{currentItem} = -1;
        $self->{maxId} = 0;
		$self->{itemArray} = [];
		$self->{information} = {};
		$self->{selectToIdArray} = [];
        $self->{minRanks} = {};
        $self->{maxRanks} = {};
		$self->{currentId} = -1;
		$self->{toBeDisplayed} = {};
        $self->{panel}->hide if $self->{panel};
        $self->{nbItems} = 0;
        $self->{parent}->{itemsList}->clearCache if $self->{parent}->{itemsList};
        $self->reloadList if ! $self->{parent}->{initializing};
    }

    sub setLock
    {
        my ($self, $value) = @_;
        $self->{information}->{locked} = $value;
    }

    sub getLock
    {
        my $self = shift;
        return $self->{information}->{locked};
    }

    my $globalInstance;

    sub load
    {
        my ($self, $file, $splash, $fullProgress, $noReload) = @_;

        my $initTime;
        if ($ENV{GCS_PROFILING} > 0)
        {
            eval 'use Time::HiRes';
            eval '$initTime = [Time::HiRes::gettimeofday()]';
        }

        $self->clean;
        my $i = 0;
        my $k = 0;
        if (!$file)
        {
            #$self->initModel;
            $self->{parent}->setCurrentModel;
            return 0;
        }
        if (! -e $file)
		{
            $self->{nbItems} = 0;
            $self->{panel}->hide if $self->{panel};
            if (!open DATA, ">$file")
            {
                print "Cannot create $file\n";
                $self->{parent}->setCurrentModel;
                return 0;
            }
            binmode(DATA, ':utf8' );
            my $model = ($self->{model}) ? $self->{model}->getName : 'GCfilms';
            print DATA '<?xml version="1.0" encoding="UTF-8"?>
<collection type="',$model,'" items="0">
</collection>
';
            close DATA;
		}

        if (! -r $file)
        {
            my  $dialog = Gtk2::MessageDialog->new($self->{parent},
                       [qw/modal destroy-with-parent/],
                       'error',
                       'ok',
                       $self->{parent}->{lang}->{OpenError});
            
            $dialog->set_position('center-on-parent');
            $dialog->run();
            $dialog->destroy ;
            return 0;
        }

        my $collection;

        $self->{block} = 1;
        $self->clearList;
        $self->{block} = 0;
        $self->{splash} = $splash;
        eval
        {
            $globalInstance = $self;
            $self->{inlineModel} = '';
            $self->{inlinePreferences} = '';
            #my $parser = XML::Parser->new(Style => 'Stream');
            my $parser = XML::Parser->new(Handlers => {
                Init  => \&StartDocument,
                Final => \&EndDocument,
                Start => \&StartTag,
                End   => \&EndTag,
                Char  => \&Text,
            });
            $parser->parsefile($file);

#            use XML::LibXML qw(:all);
#            my $parser = XML::LibXML->new;
#            my $doc = $parser->parse_file($file);
#            my $root = $doc->getDocumentElement();
#            my $currentCount = 0;
#            $self->{nbItems} = $root->getAttribute('items');
#            $self->{splash}->setItemsTotal($self->{nbItems})
#                if $self->{splash};
#            if ($root->getAttribute('type') ne 'inline')
#            {
#                $self->{parent}->setCurrentModel($root->getAttribute('type'));
#            }
#            
#            for my $node($root->getChildrenByTagName('item'))
#            {
#                my $item;
#                $currentCount++;
#                for my $data($node->getChildNodes)
#                {
#                    next if $data->nodeType eq XML_TEXT_NODE;
#                    $item->{$data->getName} = $data->textContent;
#                }
#                push @{$self->{itemArray}}, $item;
#                my $id = $item->{$self->{model}->{commonFields}->{id}};
#                if ($self->{filterSearch}->test($item))
#                {
#                    $self->{toBeDisplayed}->{$id} = 1;
#                }
#                else
#                {
#                    $self->{toBeDisplayed}->{$id} = 0;
#                }
#                $self->{splash}->setProgressForItemsLoad($currentCount)
#                    if $self->{splash};
#            }



#            use XML::Twig;
#
#            sub item
#            {
#                my ($twig, $item) = @_;
#                #print "Got one\n";
#            }
#
#            my $twig= new XML::Twig( 
#                                    TwigHandlers => {item => \&item}
#                                   );            
#            $twig->parsefile($file);    # build the twig
#            my $root= $twig->root;           # get the root of the twig (stats)
#            #my @players= $root->children;    # get the player list




            if ($ENV{GCS_PROFILING} > 0)
            {
                my $elapsed;
                eval '$elapsed = Time::HiRes::tv_interval($initTime)';
                print "Load time : $elapsed\n";
            }


        };
        if ($@)
        {
            print "Fatal error while reading $file: $@\n";
            return 0;
        }

        for (@{$self->{model}->{fieldsHistory}})
        {
            $self->{panel}->{$_}->setDropDown;
        }

		# Now we calculate nbItems to be sure we have the correct value
        $self->{nbItems} = scalar @{$self->{itemArray}};
	
        if (! $self->{nbItems})
        {
            $self->{panel}->hide;
        }
		
        $self->{parent}->refreshFilters;
		
        $self->reloadList($splash, $fullProgress) unless $noReload;
        
        if ($ENV{GCS_PROFILING} > 0)
        {
            my $elapsed;
            eval '$elapsed = Time::HiRes::tv_interval($initTime)';
            print "Load time : $elapsed\n";
        }

        return 1;
    }

    sub save
    {
        my ($self, $splash) = @_;

        my $initTime;
        if ($ENV{GCS_PROFILING} > 0)
        {
            eval 'use Time::HiRes';
            eval '$initTime = [Time::HiRes::gettimeofday()]';
        }

        if ($splash)
        {
            $splash->initProgress($self->{parent}->{lang}->{StatusSave});
            $splash->setItemsTotal($self->{nbItems});
        }
        $self->updateCurrent if ($self->{currentItem} > -1);
        $self->addMarkedPictures;
        $self->removeMarkedPictures;
        if (!open DATA, ">".$self->{options}->file)
        {
            my  $dialog = Gtk2::MessageDialog->new($self->{parent},
                [qw/modal destroy-with-parent/],
                'error',
                'ok',
                $self->{parent}->{lang}->{SaveError});

            $dialog->set_position('center-on-parent');
            $dialog->run();
            $dialog->destroy ;
            return 0;
        }
        
        binmode(DATA, ':utf8');

        my $xmlModel = '';
        my $xmlPreferences = '';
        my $collectionType;
        if ($self->{model}->{isInline})
        {
            $xmlModel = $self->{model}->toString($GCData::inlineCollectionTag, 1);
            $xmlPreferences = $self->{model}->{preferences}->toXmlString;
            $collectionType = 'inline';
        }
        else
        {
            $collectionType = $self->{model}->getName;
        }
        my $information = ' <information>
';
        $information .= "  <$_>".GCUtils::encodeEntities($self->{information}->{$_})."</$_>\n"
            foreach (sort keys %{$self->{information}});
        $information .= ' </information>';


        print DATA '<?xml version="1.0" encoding="UTF-8"?>
<collection type="',$collectionType,'" items="',$self->{nbItems},'">
',$xmlModel,'
',$xmlPreferences,'
',$information,'
';
        my $i = 1;
        foreach (@{$self->{itemArray}})
        {
            #Perform the transformation for each image value
            foreach my $pic(@{$self->{model}->{managedImages}})
            {
                $_->{$pic} 
                    = $self->{parent}->transformPicturePath($_->{$pic});
            }

                print DATA ' <item>
';
            foreach my $field(@{$self->{model}->{fieldsNames}})
            {
                if (ref($_->{$field}) eq 'HASH')
                {
                    print DATA '  <', $field, '>
', GCUtils::listToXml($_->{$field}), '  </', $field, '>
';
                }
                else
                {
                    (my $data = $_->{$field}) =~ s/&/&amp;/g;
                    $data =~ s/</&lt;/g;
                    $data =~ s/>/&gt;/g;
                    $data =~ s/"/&quot;/g;
                    #"
                    print DATA '  <', $field, '>', $data, '</', $field, '>
';
                }
            }
            print DATA ' </item>
';
            $splash->setProgressForItemsDisplay($i) if $splash;
            $i++;
        }
        print DATA '</collection>
';
        close DATA;
        $self->{parent}->endProgress;
        $self->{parent}->removeUpdatedMark;

        if ($ENV{GCS_PROFILING} > 0)
        {
            my $elapsed;
            eval '$elapsed = Time::HiRes::tv_interval($initTime)';
            print "Save time : $elapsed\n";
        }
        return 1;
    }
    
    # Parser routines
    
    sub HashToXMLString
    {
        my %hash = @_;
        my $result = '';
        foreach (keys %hash)
        {
            $result .= " $_=\"".$hash{$_}.'"';
        }
        return $result;
    }
    
    # Some globals to speed up things
    my $inCol;
    my $inLine;
    my $currentTag;
    my $currentCol;
    my $currentCount;
    my $currentIsList;
    my $isItem;
    my $isInfo;
    my $newItem;
    my $modCap;
    my $prefCap;
    my $anyCap;
    
    sub StartDocument
    {
        $isItem = 0;
        $inLine = 0;
        $inCol = 0;
        $currentCol = '';
        $currentCount = 0;
        $modCap = 0;
        $prefCap = 0;
        $anyCap = 0;
    }
    
    sub EndDocument
    {
        $globalInstance->{parent}->setCurrentModelFromInline($globalInstance)
            if $globalInstance->{inlineModel};
    }
    
    sub StartTag
    {
        #my ($expat, $tag, %attrs) = @_;
        if ($isItem)
        {
            if ($inLine)
            {
                #Only a col could start in a line
                $inCol = 1;
            }
            elsif ($_[1] eq 'line')
            {
                $inLine = 1;
                $currentIsList = 1;
                $newItem->{$currentTag} =  {} if (ref($newItem->{$currentTag})  ne 'HASH');
                push @{$newItem->{$currentTag}->{line}}, {col => []};
            }
            else
            {
                $currentIsList = 0;
                $currentTag = $_[1];
            }
        }
        elsif ($isInfo)
        {
            $currentTag = $_[1];
        }
        else
        {
            my ($expat, $tag, %attrs) = @_;
            if ($modCap)
            {
                $globalInstance->{inlineModel} .= "<$tag".HashToXMLString(%attrs).'>';
            }
            elsif ($prefCap)
            {
                $globalInstance->{inlinePreferences} .= "<$tag".HashToXMLString(%attrs).'>';
            }
            elsif ($tag eq 'item')
            {
                $newItem = {};
                $isItem = 1;
            }
            elsif ($tag eq 'information')
            {
                $globalInstance->{information} = {};
                $isInfo = 1;
            }
            elsif ($tag eq $GCData::inlineCollectionTag)
            {
                $modCap = 1;
                $anyCap = 1;
                $globalInstance->{inlineModel} = '<collection'.HashToXMLString(%attrs).">\n";
            }
            elsif ($tag eq $GCData::inlinePreferencesTag)
            {
                $prefCap = 1;
                $anyCap = 1;
                $globalInstance->{inlinePreferences} = '<preferences'.HashToXMLString(%attrs).">\n";
            }
            elsif ($tag eq $GCData::collectionTag)
            {
                $globalInstance->{nbItems} = $attrs{items};
                $globalInstance->{splash}->setItemsTotal($globalInstance->{nbItems})
                    if $globalInstance->{splash};
                if ($attrs{type} ne 'inline')
                {
                    $globalInstance->{parent}->setCurrentModel($attrs{type});
                }
            }
        }
    }

    sub EndTag
    {
        if ($anyCap)
        {
            if ($modCap)
            {
                if ($_[1] eq $GCData::inlineCollectionTag)
                {
                    $anyCap = $prefCap;
                    $globalInstance->{inlineModel} .= '</'.$GCData::collectionTag.'>';
                    if ($globalInstance->{inlinePreferences})
                    {
                        $globalInstance->{parent}->setCurrentModelFromInline($globalInstance);
                        $globalInstance->{inlineModel} = '';
                    }
                        
                }
                else
                {
                    $globalInstance->{inlineModel} .= '</'.$_[1].">\n";
                }
                return;
            }
            else
            {
                if ($_[1] eq $GCData::inlinePreferencesTag)
                {
                    $anyCap = $modCap;
                    $globalInstance->{inlinePreferences} .= '</'.$GCData::preferencesTag.'>';
                    if ($globalInstance->{inlineModel})
                    {
                        $globalInstance->{parent}->setCurrentModelFromInline($globalInstance);
                        $globalInstance->{inlineModel} = '';
                    }
                }
                else
                {
                    $globalInstance->{inlinePreferences} .= '</'.$_[1].">\n";
                }
                return;
            }
        }

        if ($_[1] eq 'item')
        {
            push @{$globalInstance->{itemArray}}, $newItem;
            $currentCount++;

            $globalInstance->{panel}->{$_}->addHistory($newItem->{$_}, 1)
                foreach (@{$globalInstance->{model}->{fieldsHistory}});
            foreach (@{$globalInstance->{model}->{fieldsNotNull}})
            {
                $newItem->{$_} = $globalInstance->{model}->{fieldsInfo}->{$_}->{init} if ! $newItem->{$_};
            }
            
            my $id = $newItem->{$globalInstance->{model}->{commonFields}->{id}};
            $globalInstance->{maxId} = $id
                if $id > $globalInstance->{maxId};
            if ($globalInstance->{filterSearch}->test($newItem))
            {
                $globalInstance->{toBeDisplayed}->{$id} = 1;
            }
            else
            {
                $globalInstance->{toBeDisplayed}->{$id} = 0;
            }
    
            $globalInstance->{splash}->setProgressForItemsLoad($currentCount)
                if $globalInstance->{splash};

            my ($serie, $rank) = ($newItem->{$globalInstance->{serieField}},
                                  $newItem->{$globalInstance->{rankField}});
            if ($serie)
            {
                if ($globalInstance->isRankNumeric)
                {            
                    $globalInstance->{minRanks}->{$serie} = 1001 if ! exists $globalInstance->{minRanks}->{$serie};
                    $globalInstance->{maxRanks}->{$serie} = -1 if ! exists $globalInstance->{maxRanks}->{$serie};
                    $globalInstance->{minRanks}->{$serie} = $rank if ($rank < $globalInstance->{minRanks}->{$serie});
                    $globalInstance->{maxRanks}->{$serie} = $rank if ($rank > $globalInstance->{maxRanks}->{$serie});
                }
                else
                {            
                    $globalInstance->{minRanks}->{$serie} = 'ZZZZZ' if ! exists $globalInstance->{minRanks}->{$serie};
                    $globalInstance->{maxRanks}->{$serie} = ' ' if ! exists $globalInstance->{maxRanks}->{$serie};
                    $globalInstance->{minRanks}->{$serie} = $rank if ($rank lt $globalInstance->{minRanks}->{$serie});
                    $globalInstance->{maxRanks}->{$serie} = $rank if ($rank gt $globalInstance->{maxRanks}->{$serie});
                }            
            }
            $isItem = 0;
        }
        elsif ($_[1] eq 'information')
        {
            $isInfo = 0 if !$isItem;
        }
        elsif ($inCol)
        {
            # We are closing a col as it could not have tags inside
            my $linesRef = $newItem->{$currentTag}->{line};
            push @{$linesRef->[$#{$linesRef}]->{col}}, $currentCol;
            $currentCol = '';
            $inCol = 0;            
        }
        else
        {
            # The only tag that could prevent us from closing a line is col, but it has
            # already been managed
            if ($inLine)
            {
                $inLine = 0;
            }
            else
            {
                $currentTag = '';
            }
        }
    }

    sub Text
    {
        if ($isItem)
        {
            if ((! $currentTag)
             || $inLine
             || $currentIsList
             || ((!$newItem->{$currentTag}) && ($_[1] =~ /^\s*$/oms)))
            {
                if ($inCol)
                {
                    return if $_[1] =~ /^\s*$/oms;
                    $currentCol .= $_[1];
                }
            }
            else
            {
                $newItem->{$currentTag} .= $_[1];
            }
        }
        elsif ($isInfo)
        {
            return if $_[1] =~ /^\s*$/oms;
            $globalInstance->{information}->{$currentTag} .= $_[1];
        }
        else
        {
            if ($modCap)
            {
                $globalInstance->{inlineModel} .= $_[1];
            }
            elsif ($prefCap)
            {
                $globalInstance->{inlinePreferences} .= $_[1];
            }
        }
    }
    
}

1;
