from DisplayContainer import DisplayContainer
from TargetGroup import TargetGroup
from DisplayTarget import DisplayTarget
from utils.Observable import Observable
from DisplayConfigurator import DisplayConfigurator
from main import DEFAULT_SENSOR, UNSET_COORD
from main import admin
from utils import vfs

import gtk
import os


#
# Class for display windows.
#
class Display(Observable, gtk.EventBox):

    # observer commands
    OBS_CLOSE = 0
    OBS_RESTART = 1


    def __init__(self, ident):

        # the container of this display; we use a proxy until we get
        # the real thing; the proxy remembers set values and passes them on
        # to the real container
        self.__container = DisplayContainer(self)

        # the root TargetGroup
        self.__group = None

        # the path of the .display file
        self.__path = vfs.getcwd()

        # the unique ID of this display
        self.__id = ident

        # the sensors of this window
        self.__sensors = {}

        # the last selected targets (used for detecting events)
        self.__last_targets = []

        # the last position of the mouse pointer (used for filtering out
        # unwanted events)
        self.__last_pointer_pos = (-1, -1)

        # mapping between sensors and targets; which target watches
        # which sensor?
        # (sensor, port) -> (target, property)
        self.__mapping = {}

        # temporary data for remembering the position of the last mouse click
        self.__pointer_pos = (0, 0)

        # whether the display reacts on events
        self.__is_sensitive = gtk.TRUE


        gtk.EventBox.__init__(self)
        self.show()

        # set up event handlers
        self.connect("button-press-event", self.__on_button, 0)
        self.connect("button-release-event", self.__on_button, 1)
        self.connect("scroll-event", self.__on_scroll)
        self.connect("motion-notify-event", self.__on_motion, 0)
        self.connect("leave-notify-event", self.__on_motion, 1)
        self.connect("delete-event", self.__on_close)
        self.add_events(gtk.gdk.BUTTON_PRESS_MASK |
                        gtk.gdk.BUTTON_RELEASE_MASK |
                        gtk.gdk.LEAVE_NOTIFY_MASK |
                        gtk.gdk.POINTER_MOTION_MASK)


        #self.connect("expose-event", self.__on_expose)
        
    def __on_expose(self, src, event):

        x, y, w, h = event.area
        print "EXPOSE", x, y, w, h



    #
    # Sets the container of the display.
    #
    def set_container(self, container):

        # pass the values stored by the proxy on to the real container
        self.__container.pass_values(container)
        self.__container = container


    #
    # Sets the sensitive flag of the display. Insensitive displays don't react
    # on user events.
    #
    def set_sensitive(self, value):

        self.__is_sensitive = value


    #
    # Returns the path of the .display file.
    #
    def get_path(self):

        return self.__path



    def add_children(self, childrendata):

        # create the root TargetGroup
        self.__group = TargetGroup(None, self)
        #self.__group.set_position(UNSET_COORD, UNSET_COORD)
        self.__group.add_children(childrendata)
        self.add(self.__group)
        
        self.__group.add_observer(self.__on_observe_group)
        self.__group.set_config("x", UNSET_COORD)
        self.__group.set_config("y", UNSET_COORD)




    #
    # Opens the configuration dialog for this display.
    #
    def __open_configurator(self):

        configurators = []
        sensors = self.__sensors.values()
        for s in sensors:
            configurators.append(s.get_configurator())

        dconf = DisplayConfigurator(configurators)
        dconf.set_transient_for(self.get_toplevel())


    #
    # Removes this display.
    #
    def remove_display(self):

        self.drop_observers()
        for s in self.__sensors.values():
            s.stop()

        del self.__sensors
        del self.__mapping
        del self.__last_targets

        self.remove(self.__group)
        self.__group.delete()
        self.__group.destroy()
        del self.__group
        self.__container.close()


    #
    # Reacts on closing the window.
    #
    def __on_close(self, src, event):

        #self.__remove_display()
        self.update_observer(self.OBS_CLOSE, self.__id)


    #
    # Sends the given action with arguments to the display.
    #
    def send_action(self, action, *args):

        px, py = self.get_pointer()
        self.__group.handle_action(action, px, py, [], list(args))
        self.__group.notify_handle_action(1, [])


    #
    # Reacts on button events.
    #
    def __on_button(self, src, event, is_release = 0):

        if (not self.__is_sensitive): return
        
        px, py = self.get_pointer()
        lx, ly = self.__pointer_pos
        button = event.button

        # determine action
        if (not is_release):
            self.__pointer_pos = (px, py)
            if (button == 1 and event.type == gtk.gdk._2BUTTON_PRESS):
                action = DisplayTarget.ACTION_DOUBLECLICK
            elif (button == 1):
                action = DisplayTarget.ACTION_PRESS
            elif (button == 2):
                return
            elif (button == 3):
                return
            else:
                return

        else:
            if (button == 1):
                if (abs(lx - px) < 10 and abs(ly - py) < 10):
                    action = DisplayTarget.ACTION_CLICK
                else:
                    action = DisplayTarget.ACTION_RELEASE
            elif (button == 2):
                return
            elif (button == 3):
                action = DisplayTarget.ACTION_MENU
            else:
                return

        #end if

        self.__group.handle_action(action, px, py, [], [button])
        self.__group.notify_handle_action(1, [])

        called = 0
        # make sure that there is always a popup menu
        if (action == DisplayTarget.ACTION_MENU and not called):
            self._call_sensor([(DEFAULT_SENSOR, "menu", [])], [], 3)


    #
    # Reacts on moving the mouse.
    #
    def __on_motion(self, src, event, is_leave):

        if (not self.__is_sensitive): return

        px, py = self.get_pointer()
        w, h = self.size_request()

        # some window managers send a LEAVE event for mouse clicks;
        # work around this
        if (is_leave and (px, py) == self.__last_pointer_pos
            and (0 <= px <= w) and (0 <= py <= h)):
            is_leave = 0

        # don't do redundant work
        if (not is_leave and (px, py) == self.__last_pointer_pos): return
        else: self.__last_pointer_pos = (px, py)

        if (not is_leave):
            self.__group.handle_action(DisplayTarget.ACTION_MOTION, px, py,
                                       [], [px, py])
            self.__group.notify_handle_action(1, [])

        else:
            self.__group.notify_handle_action(0, [])



    #
    # Reacts on rolling the mouse wheel.
    #
    def __on_scroll(self, src, event):

        if (not self.__is_sensitive): return

        px, py = self.get_pointer()

        if (event.direction == gtk.gdk.SCROLL_UP):
            direction = 0
        elif (event.direction == gtk.gdk.SCROLL_DOWN):
            direction = 1
        else:
            direction = -1

        self.__group.handle_action(DisplayTarget.ACTION_SCROLL, px, py, [],
                                   [direction])
        self.__group.notify_handle_action(1, [])



    #
    # Observer for sensors.
    #
    def __on_observe_sensor(self, src, cmd, data):

        # propagate the incoming sensor output
        if (cmd == src.OBS_OUTPUT):
            self.__set_settings(src, data)

        elif (cmd == src.OBS_CMD_CONFIGURE):
            self.__open_configurator()

        elif (cmd == src.OBS_CMD_REMOVE):
            self.update_observer(self.OBS_CLOSE, self.__id)

        elif (cmd == src.OBS_CMD_RESTART):
            self.update_observer(self.OBS_RESTART, self.__id)

        elif (cmd == src.OBS_CMD_DUPLICATE):
            dsp_uri = admin.get_displays().get(self.__id)
            if (dsp_uri): admin.add_display(dsp_uri)



    #
    # Observer for the root group.
    #
    def __on_observe_group(self, src, cmd, *args):

        if (cmd == src.OBS_GEOMETRY):
            x, y, w, h = self.__group.get_geometry()
            ux, uy = self.__group.get_user_geometry()[:2]

            if (ux == uy == UNSET_COORD):
                #self.__container.set_position(ux, uy)
                pass
            else:
                self.__container.set_position(x, y)

                self._call_sensor([(DEFAULT_SENSOR, "move", [])], [],
                                  ux, uy, x, y)

            if (w != 0 and h != 0):
                self.__container.set_size(w, h)
                # storing the size is useless, but it's added by request;
                # it makes life easy for desklets pagers
                self._call_sensor([(DEFAULT_SENSOR, "size", [])], [], w, h)



    #
    # Calls a function of a Sensor.
    #
    def call_sensor(self, cmd, path, *args):
        assert(cmd)

        args = list(args)
        for ident, callname, userargs in cmd:
            if (not ident): ident = DEFAULT_SENSOR
            sensor = self.__get_sensor(ident) or self.__get_sensor(DEFAULT_SENSOR)
            allargs = args + userargs

            # the sensor is an external module, so we make sure it cannot crash
            # the application

            try:
                sensor.send_action(callname, path, allargs)

            except StandardError, e:
                print "The sensor produced an error:", e
                import traceback; traceback.print_exc()
        #end for

    _call_sensor = call_sensor



    #
    # Saves the the given position of the display. The position is nw-anchored.
    #
    def save_position(self, x, y):

        if (not self.__group): return
        
        w, h = self.__group.get_geometry()[2:]
        ax, ay = self.__group.get_anchored_coords(x, y, w, h)
        dx, dy= x - ax, y - ay

        self.__group.set_config("x", str(x + dx))
        self.__group.set_config("y", str(y + dy))
    

    #
    # Sets the configuration settings.
    #
    def __set_settings(self, sensor, settings):

        for key, value in settings.get_entries():
            # extract array indexes
            if ("[" in key):
                indexpart = key[key.find("[") + 1:-1]
                indexes = indexpart.split("][")
                key = key[:key.find("[")]
            else:
                indexes = []

            # get all (target, property) tuples that are watching the given
            # sensor key and notify the targets
            entries = self.__mapping.get((sensor, key), [])
            for target, prop in entries:
                if (indexes):
                    # special handling for arrays
                    sensor_id = self.__get_sensor_id(sensor)
                    target.distribute_sensor_output(sensor_id, indexes[:],
                                                    key, value)
                else:
                    target.set_config(prop, value)
            #end for
        #end for



    #
    # Sets the configuration.
    #
    def set_config(self, key, value):

        if (key == "window-flags"):
            self.__container.set_window_flags(value)

        elif (key == "shape"):
            filename = vfs.join(self.get_path(), value)
            try:
                loader = gtk.gdk.PixbufLoader()
                fd = vfs.open(filename, "r")
                data = vfs.read_all(fd)
                fd.close()
                loader.write(data, len(data))
                loader.close()
                pixbuf = loader.get_pixbuf()
                pix, mask = pixbuf.render_pixmap_and_mask(1)
                self.__container.set_shape(mask)
            except:
                print "could not set shape ", filename
        
        else:
            self.__group.set_config(key, value)



    #
    # Adds a sensor to this display.
    #
    def add_sensor(self, ident, sensor):

        self.__sensors[ident] = sensor
        sensor.add_observer(self.__on_observe_sensor)



    #
    # Returns the sensor with the given ID. Returns None if the sensor does not
    # exist.
    #
    def __get_sensor(self, ident): return self.__sensors.get(ident)



    #
    # Returns the ID of the given sensor.
    #
    def __get_sensor_id(self, sensor):

        for k, v in self.__sensors.items():
            if (v == sensor): return k

        return ""



    #
    # Maps a sensor output to a target.
    #
    def add_mapping(self, sensorplug, target, prop):

        ident, port = sensorplug.split(":")
        sensor = self.__get_sensor(ident)
        if (not self.__mapping.has_key((sensor, port))):
            self.__mapping[(sensor, port)] = []

        if (not (target, prop) in self.__mapping[(sensor, port)]):
            self.__mapping[(sensor, port)].append((target, prop))
