/*
Copyright (C) 2003 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include <math.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <unistd.h>

#include "gdis.h"
#include "coords.h"
#include "file.h"
#include "render.h"
#include "matrix.h"
#include "opengl.h"
#include "gtkshorts.h"
#include "interface.h"
#include "dialog.h"

#define DEBUG 0
#define DEBUG2 0

/* global pak structures */
extern struct sysenv_pak sysenv;
extern struct elem_pak elements[];

GtkWidget *cf_spin;

/*************************************************/
/* store the current position of the file stream */
/*************************************************/
#define DEBUG_ADD_FRAME_OFFSET 0
gint add_frame_offset(FILE *fp, struct model_pak *data)
{
fpos_t *offset;

g_assert(fp != NULL);
g_assert(data != NULL);

offset = g_malloc(sizeof(fpos_t));
/* prepend & reverse? */
if (!fgetpos(fp, offset))
  data->frame_list = g_list_append(data->frame_list, offset); 
else
  {
  g_free(offset);
  return(1);
  }

#if DEBUG_ADD_FRAME_OFFSET
printf("stored: %p\n", offset);
#endif

return(0);
}

/**************************/
/* retrieve the nth frame */
/**************************/
#define DEBUG_READ_FRAME 0
gint read_frame(FILE *fp, gint n, struct model_pak *model)
{
gint status;
gdouble vec[3], mat[9];

g_assert(fp != NULL);
g_assert(model != NULL);

/* preserve the initial orientation & translation */
memcpy(mat, model->rotmat, 9*sizeof(gdouble));
ARR3SET(vec, model->offset);

status = read_raw_frame(fp, n, model);
if (status)
  {
  printf("Error reading frame: %d\n", status);
  return(1);
  }

/* setup */
prep_model(model);

/* restore the initial orientation and offset */
memcpy(model->rotmat, mat, 9*sizeof(gdouble));
ARR3SET(model->offset, vec);

/* apply desired constraint */
if (model->periodic)
  {
  switch (model->anim_confine)
    {
    case PBC_CONFINE_ATOMS:
      pbc_constrain_atoms(model);
      init_objs(REDO_COORDS, model);
      calc_bonds(model);
      calc_mols(model);
      break;

    case PBC_CONFINE_MOLS:
      pbc_constrain_mols(model);
      init_objs(REDO_COORDS, model);
      break;

    case UNFRAGMENT_MOLS:
      unfragment(model);
      init_objs(REDO_COORDS, model);
      break;

    default:
      init_objs(REDO_COORDS, model);
    }
  }
else
  init_objs(REDO_COORDS, model);

return(0);
}

/***************************************/
/* current frame modification callback */
/***************************************/
#define DEBUG_SELECT_FRAME 0
void select_frame(GtkWidget *w, struct model_pak *model)
{
FILE *fp;

g_assert(w != NULL);
g_assert(model != NULL);

#if DEBUG_SELECT_FRAME
printf("request frame: %d, model: %p\n", model->cur_frame, model);
#endif

fp = fopen(model->filename, "r");

read_frame(fp, model->cur_frame, model);

meas_graft_all(model);

redraw_canvas(SINGLE);

fclose(fp);
}

/*********************************************************/
/* read and draw the next frame in the current animation */
/*********************************************************/
#define DEBUG_DISPLAY_NEXT_FRAME 0
gint anim_next_frame(struct model_pak *model)
{
g_assert(model != NULL);

/* continue until we run out of frames (or a stop is flagged) */
model->cur_frame++;
if (model->cur_frame < model->num_frames && model->animating)
  {
#if DEBUG_DISPLAY_NEXT_FRAME
printf("displaying [%d]\n", model->cur_frame);
#endif
/* if a dialog exists - update via the current frame spinner */
  if (dialog_active(ANIM))
    gtksh_relation_update(model);
  else
    {
/* otherwise, update manually */
    read_frame(model->afp, model->cur_frame, model);
    meas_graft_all(model);
    redraw_canvas(SINGLE);
    }
  return(TRUE);
  }

/* done - return FALSE to terminate the timer */
model->animating = FALSE;
model->cur_frame--;
fclose(model->afp);
return(FALSE);
}

/**********************************/
/* set current frame to beginning */
/**********************************/
void anim_rewind(GtkWidget *w, gint id)
{
struct model_pak *model;

model = model_ptr(sysenv.dialog[id].model, RECALL);
g_assert(model != NULL);

model->cur_frame = 0;

gtksh_relation_update(model);
}

/****************************/
/* set current frame to end */
/****************************/
void anim_fastforward(GtkWidget *w, gint id)
{
struct model_pak *model;

model = model_ptr(sysenv.dialog[id].model, RECALL);
g_assert(model != NULL);

model->cur_frame = model->num_frames-1;

gtksh_relation_update(model);
}

/**********************/
/* one frame backward */
/**********************/
void anim_step_backward(GtkWidget *w, gint id)
{
struct model_pak *model;

model = model_ptr(sysenv.dialog[id].model, RECALL);
g_assert(model != NULL);

if (model->cur_frame > 0)
  {
  model->cur_frame--;
  gtksh_relation_update(model);
  }
}

/*********************/
/* one frame forward */
/*********************/
void anim_step_forward(GtkWidget *w, gint id)
{
struct model_pak *model;

model = model_ptr(sysenv.dialog[id].model, RECALL);
g_assert(model != NULL);

if (model->cur_frame < model->num_frames-1)
  {
  model->cur_frame++;
  gtksh_relation_update(model);
  }
}

/****************************/
/* start an animation timer */
/****************************/
void anim_start(GtkWidget *w, gint id)
{
gint freq;
struct model_pak *model;

/* don't allow more than one */
model = model_ptr(sysenv.dialog[id].model, RECALL);
if (model->animating)
  return;

/* timeout frequency */
/* NB: we can't refresh any faster than units of 25 (ie the canvas redraw frequency) */
freq = 25 * model->anim_speed;

model->afp = fopen(model->filename, "r");
if (model->afp)
  {
  g_timeout_add(freq, (GSourceFunc) &anim_next_frame, model);
  model->animating = TRUE;
  }
else
  show_text(ERROR, "Failed to open animation stream.\n");
}

/****************************/
/* stop the animation timer */
/****************************/
void anim_stop(GtkWidget *w, gint id)
{
struct model_pak *model;

model = model_ptr(sysenv.dialog[id].model, RECALL);
model->animating = FALSE;
}

/***********************************/
/* pbc confinement option handlers */
/***********************************/
void atom_pbc(struct model_pak *model)
{
model->anim_confine = PBC_CONFINE_ATOMS;
}
void mol_pbc(struct model_pak *model)
{
model->anim_confine = PBC_CONFINE_MOLS;
}
void unfrag_pbc(struct model_pak *model)
{
model->anim_confine = UNFRAGMENT_MOLS;
}
void no_pbc(struct model_pak *model)
{
model->anim_confine = PBC_CONFINE_NONE;
}

/***********************/
/* configure animation */
/***********************/
void animate_setup_widget(void)
{
gint id;
gchar *tmp;
GtkWidget *main_hbox, *frame, *vbox, *hbox, *hbox2;
GtkWidget *button, *label;
struct dialog_pak *anim_dialog;
struct model_pak *data;

/* checks */
if (!sysenv.num_models) 
  return;
data = model_ptr(sysenv.active, RECALL);
if (!data)
  return;

/* dialog setup */
if ((id = request_dialog(sysenv.active, ANIM)) < 0)
  return;
anim_dialog = &sysenv.dialog[id];
anim_dialog->model = data->number;
anim_dialog->win = gtk_dialog_new();
gtk_window_set_title(GTK_WINDOW(anim_dialog->win), "Animation");
g_signal_connect(GTK_OBJECT(anim_dialog->win), "destroy",
                 GTK_SIGNAL_FUNC(event_close_dialog), (gpointer) id);

main_hbox = gtk_hbox_new(FALSE, PANEL_SPACING);
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(anim_dialog->win)->vbox), main_hbox,
                                                         FALSE, FALSE, 0);

/* Frame 1 - general animation stuff */
frame = gtk_frame_new("Control");
gtk_box_pack_start(GTK_BOX(main_hbox),frame,TRUE,TRUE,0);
/* create a vbox in the frame */
vbox = gtk_vbox_new(TRUE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(frame), vbox);


/* num frames */
hbox = gtk_hbox_new(FALSE, 0);
gtk_container_set_border_width(GTK_CONTAINER(hbox), PANEL_SPACING);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
label = gtk_label_new("Number of frames:");
gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
tmp = g_strdup_printf("%9d", data->num_frames);
label = gtk_label_new(tmp);
gtk_box_pack_end(GTK_BOX(hbox), label, FALSE, FALSE, 0);

/* desired delay */
gtksh_direct_spin("Animation delay", &data->anim_speed,
                  1.0, 40.0, 1.0, NULL, NULL, vbox);

gtksh_direct_check("Don't recalculate connectivity", &data->anim_fix, NULL, NULL, vbox);


/* Frame 2 - cycle options */
frame = gtk_frame_new("Frame processing");
gtk_box_pack_start(GTK_BOX(main_hbox),frame,FALSE,FALSE,0);

vbox = gtk_vbox_new(FALSE, 0);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(vbox), PANEL_SPACING);

/* actions at start of each cycle */
new_radio_group(0, vbox, FF);

button = add_radio_button("Confine atoms to PBC", (gpointer) atom_pbc, data);
if (data->anim_confine == PBC_CONFINE_ATOMS)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

button = add_radio_button("Confine mols to PBC", (gpointer) mol_pbc, data);
if (data->anim_confine == PBC_CONFINE_MOLS)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

button = add_radio_button("Confine unfragmented mols to PBC", 
                                (gpointer) unfrag_pbc, data);
if (data->anim_confine == UNFRAGMENT_MOLS)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

button = add_radio_button("Cell confinement off", (gpointer) no_pbc, data);
if (data->anim_confine == PBC_CONFINE_NONE)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);


/* NEW - slider for current frame */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(anim_dialog->win)->vbox), frame,
                                                     FALSE, FALSE, 0);

vbox = gtk_vbox_new(FALSE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(frame), vbox);

gtksh_direct_hscale(0, data->num_frames-1, 1, &data->cur_frame,
                                     select_frame, data, vbox);

/* control buttons */
hbox2 = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox2, FALSE, FALSE, PANEL_SPACING);
hbox = gtk_hbox_new(TRUE, PANEL_SPACING);
gtk_box_pack_start(GTK_BOX(hbox2), hbox, TRUE, FALSE, 0);

gtksh_icon_button("GDIS_REWIND", NULL,
                  anim_rewind, GINT_TO_POINTER(id),
                  hbox);

gtksh_icon_button("GDIS_STEP_BACKWARD", NULL,
                  anim_step_backward, GINT_TO_POINTER(id),
                  hbox);

gtksh_icon_button("GDIS_PLAY", NULL,
                  anim_start, GINT_TO_POINTER(id),
                  hbox);

gtksh_icon_button("GDIS_PAUSE", NULL,
                   anim_stop, GINT_TO_POINTER(id),
                  hbox);

gtksh_icon_button("GDIS_STEP_FORWARD", NULL,
                  anim_step_forward, GINT_TO_POINTER(id),
                  hbox);

gtksh_icon_button("GDIS_FASTFORWARD", NULL,
                  anim_fastforward, GINT_TO_POINTER(id),
                  hbox);

gtksh_stock_button(GTK_STOCK_CLOSE, event_close_dialog, GINT_TO_POINTER(id),
                   GTK_DIALOG(anim_dialog->win)->action_area);


/* display the dialog */
gtk_widget_show_all(anim_dialog->win);
}

/********************************************************/
/* General routine for getting the next available frame */
/********************************************************/
gint get_frame(struct model_pak *data, gint first)
{
/* deprec. */
return(0);
}

