/*
Copyright (C) 2003 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include <stdio.h>
#include <string.h>
#include <strings.h>
#include <stdlib.h>
#include <math.h>

#include "gdis.h"
#include "coords.h"
#include "interface.h"
#include "matrix.h"
#include "spatial.h"
#include "surface.h"
#include "numeric.h"
#include "morph.h"
#include "opengl.h"
#include "select.h"

/* data structures */
extern struct sysenv_pak sysenv;
extern struct elem_pak elements[];

/**********************/
/* debugging routines */
/**********************/
void dump_links(GSList *list)
{
GSList *item;
struct core_pak *core;
struct shel_pak *shell;

for (item=list ; item ; item=g_slist_next(item))
  {
  core = (struct core_pak *) item->data;

  printf("core: %p, shell: %p ", core, core->shell);
  P3VEC(" : ", core->x);
  }
for (item=list ; item ; item=g_slist_next(item))
  {
  shell = (struct shel_pak *) item->data;

  printf("shell: %p, core: %p\n", shell, shell->core);
  P3VEC(" : ", shell->x);
  }
}
void dump_cores(GSList *list)
{
gint i=0;
GSList *item;
struct core_pak *core;
struct shel_pak *shell;

for (item=list ; item ; item=g_slist_next(item))
  {
  core = (struct core_pak *) item->data;

/*
if (i==4)
*/
  {
  printf("[%4s] %1d %1d c %10.4f %10.4f %10.4f : %6.2f\n",
          core->label, core->primary, core->orig,
          core->x[0], core->x[1], core->x[2], core->charge);

  if (core->shell)
    {
    shell = (struct shel_pak *) core->shell;

    printf("[%4s] %1d %1d  s %10.4f %10.4f %10.4f : %6.2f\n",
            shell->label, shell->primary, shell->orig,
            shell->x[0], shell->x[1], shell->x[2], shell->charge);
    }
  }
  i++;
  }
}

/****************/
/* SPECIAL OBJS */
/****************/
#define DEBUG_INIT 0
void init_objs(gint type, struct model_pak *data)
{
GSList *list;
struct core_pak *core;
struct shel_pak *shel;

g_return_if_fail(data != NULL);

switch(type)
  {
/* set all atoms back to element colour */
  case REFRESH_COLOUR:
    model_colour_scheme(data->colour_scheme, data);
    for (list=data->shels ; list ; list=g_slist_next(list))
      {
      shel = (struct shel_pak *) list->data;
      ARR3SET(shel->colour, elements[shel->atom_code].colour);
      }
    break;

/* scale all atom colours by their sof */
  case SOF_COLOUR:
    if (data->has_sof)
      for (list=data->cores ; list ; list=g_slist_next(list))
        {
        core = (struct core_pak *) list->data;
        VEC3MUL(core->colour, core->sof);
        }
    break;

  case REFRESH_BONDS:
    for (list=data->cores ; list ; list=g_slist_next(list))
      {
      core = (struct core_pak *) list->data;
      init_elem(core, type, data);
      }
    break; 

/* atomic coordinates processing */
  case INIT_COORDS:
/* find unique elements */
    g_slist_free(data->unique_atom_list);
    data->unique_atom_list = find_unique(ELEMENT, data);
/* init the charges */
    init_model_charges(data);
/* update electrostatic info */
    calc_emp(data);
/* repetition here, since there is interdependence */
/* between centroid (cent_coords) and latmat (calc_coords) calc */
    cent_coords(1, data);
    calc_coords(INITIAL, data);
  case CENT_COORDS:
    cent_coords(1, data);
  case REDO_COORDS:
    calc_coords(REFRESH, data);
    break;

  default:
    printf("Unknown object type requested!\n");
  }
}

/********/
/* AXES */
/********/
void make_axes(struct model_pak *data)
{
VEC3SET(data->axes[0].x, 1.0, 0.0, 0.0);
VEC3SET(data->axes[1].x, 0.0, 1.0, 0.0);
VEC3SET(data->axes[2].x, 0.0, 0.0, 1.0);
}

/********/
/* CELL */
/********/
void make_cell(struct model_pak *data)
{
gdouble b, c;

if (!data->periodic)
  return;

/* periodicity hacks */
b = (data->periodic < 2) ? 0.0 : 1.0;
c = (data->periodic < 3) ? 0.0 : 1.0;

/* end face 1 */
VEC3SET(data->cell[0].x, 0.0, 0.0, 0.0);
VEC3SET(data->cell[1].x, 0.0, b, 0.0);
VEC3SET(data->cell[2].x, 1.0, b, 0.0);
VEC3SET(data->cell[3].x, 1.0, 0.0, 0.0);

/* end face 2 */
VEC3SET(data->cell[4].x, 0.0, 0.0, c);
VEC3SET(data->cell[5].x, 0.0, b, c);
VEC3SET(data->cell[6].x, 1.0, b, c);
VEC3SET(data->cell[7].x, 1.0, 0.0, c);
}

/**********/
/* SHELLS */
/**********/
#define DEBUG_SHELLS 0
void update_shells(struct model_pak *data)
{
gint tot_match, match, dummy[3];
gdouble min, sep, dim[3], vec[3];
GSList *core_list, *shel_list;
struct core_pak *core;
struct shel_pak *shel;

/* pbc calc with periodic image correction */
dim[0] = data->image_limit[0] + data->image_limit[1];
dim[1] = data->image_limit[2] + data->image_limit[3];
dim[2] = data->image_limit[4] + data->image_limit[5];

tot_match = 0;
for (core_list=data->cores ; core_list ; core_list=g_slist_next(core_list))
  {
  core = (struct core_pak *) core_list->data;
  core->shell = NULL;

/* the closest separation found */
  min = MAX_SHELL_DIST*MAX_SHELL_DIST;
  match = 0;
  for (shel_list=data->shels ; shel_list ; shel_list=g_slist_next(shel_list))
    {
    shel = (struct shel_pak *) shel_list->data;

    if (shel->atom_code == core->atom_code)
      {
/* get the absolute fractional separation */
      ARR3SET(vec, core->x);
      ARR3SUB(vec, shel->x);
      VEC3ABS(vec);

/* get the minimum fractional separation */
fractional_clamp(vec, dummy, data->periodic);
      switch(data->periodic)
        {
        case 3:
          while(vec[2] > dim[2]/2.0)
            vec[2] -= 1.0;
        case 2:
          while(vec[0] > dim[0]/2.0)
            vec[0] -= 1.0;
          while(vec[1] > dim[1]/2.0)
            vec[1] -= 1.0;
          break;
        }

/* convert to cartesian & compare with cutoff */
      vecmat(data->latmat, vec);
      sep = VEC3MAGSQ(vec);
      if (sep < min)
        {
/* reference each other */
        core->shell = shel;
        shel->core = core;
        match++;
/* keep track of the minimum, so only the closest pair is recorded */
        min = sep;
        }
      }
    }
  if (match)
    tot_match++;
  }

#if DEBUG_SHELLS
printf("update_shells(): found %d core-shell connections.\n", tot_match);
#endif

/* do some shells float down here? */
if (tot_match != g_slist_length(data->shels))
  {
/* TODO - somehow only print this once for a model */
show_text(WARNING, "update_shells() warning: some shell(s) have no cores.\n");
  }
}

/*************************/
/* core matching routine */
/*************************/
struct core_pak *seek_core(struct core_pak *core, gint frac, struct model_pak *model)
{
gint dummy[3];
gdouble vec[3];
GSList *list;
struct core_pak *comp;
struct zone_pak *zone;

/* scan cores in the same zone */
zone = get_zone(core, model);
for (list=zone->cores ; list ; list=g_slist_next(list))
  {
  comp = (struct core_pak *) list->data;
  if (comp->status & DELETED)
    continue;

/* don't compare self */
  if (core != comp)
    {
    if (frac)
      {
      ARR3SET(vec, core->x);
      ARR3SUB(vec, comp->x);
/* NEW */
      fractional_clamp(vec, dummy, model->periodic);

      if (VEC3MAGSQ(vec) < FRACTION_TOLERANCE)
        return(comp);
      }
    else
      {
      ARR3SET(vec, core->rx);
      ARR3SUB(vec, comp->rx);
      if (VEC3MAGSQ(vec) < POSITION_TOLERANCE)
        return(comp);
      }
    }
  }
return(NULL);
}

/**************************/
/* shell matching routine */
/**************************/
struct shel_pak *seek_shel(struct shel_pak *shell, gint frac, struct model_pak *model)
{
gint dummy[3];
gdouble vec[3];
GSList *list;
struct shel_pak *comp;

for (list=model->shels ; list ; list=g_slist_next(list))
  {
  comp = (struct shel_pak *) list->data;
  if (comp->status & DELETED)
    continue;

/* don't compare self */
  if (shell != comp)
    {
    if (frac)
      {
      ARR3SET(vec, shell->x);
      ARR3SUB(vec, comp->x);
/* NEW */
      fractional_clamp(vec, dummy, model->periodic);

      if (VEC3MAGSQ(vec) < FRACTION_TOLERANCE*FRACTION_TOLERANCE)
        return(comp);
      }
    else
      {
      ARR3SET(vec, shell->rx);
      ARR3SUB(vec, comp->rx);
      if (VEC3MAGSQ(vec) < POSITION_TOLERANCE*POSITION_TOLERANCE)
        return(comp);
      }
    }
  }
return(NULL);
}

/*******************************/
/* coordinate matching routine */
/*******************************/
struct core_pak *seek_coord(gdouble *x, gint frac, struct model_pak *data)
{
gdouble vec[3];
GSList *list=NULL;
struct core_pak *core;

list = data->cores;
while (list)
  {
  core = (struct core_pak *) list->data;

  if (frac)
    {
    ARR3SET(vec, core->x);
    ARR3SUB(vec, x);
    if (VEC3MAGSQ(vec) < FRACTION_TOLERANCE*FRACTION_TOLERANCE)
      return(core);
    }
  else
    {
    ARR3SET(vec, core->rx);
    ARR3SUB(vec, x);
    if (VEC3MAGSQ(vec) < POSITION_TOLERANCE*POSITION_TOLERANCE)
      return(core);
    }

  list = g_slist_next(list);
  }
return(NULL);
}

/*******************************/
/* coordinate matching routine */
/*******************************/
#define DEBUG_SEEK_COORD2D 0
struct core_pak *seek_coord2d(gdouble *x, struct model_pak *data)
{
gdouble d2, tol, vec[3];
GSList *list=NULL;
struct core_pak *core, *best=NULL;

/* tolerance (angstroms squared)  */
tol = data->rmax * 20.0*data->scale/sysenv.size;

#if DEBUG_SEEK_COORD2D
printf(" size = %d\n", sysenv.size);
printf("scale = %f\n", data->scale);
printf(" rmax = %f\n", data->rmax);
printf("  tol = %f\n", tol);
#endif

for (list=data->cores ; list ; list=g_slist_next(list))
  {
  core = (struct core_pak *) list->data;

  ARR3SET(vec, core->rx);
  VEC3MUL(vec, data->scale);
  ARR3SUB(vec, x);
  d2 = vec[0]*vec[0] + vec[1]*vec[1];
  if (d2 < tol)
    {
    best = core;
    tol = d2;
    }
  }

return(best);
}

/**********/
/* COORDS */
/**********/
/* NEW - global = true -> whole model, else selection (if non-empty) */
/* TODO - if not global - need to ignore some objects (eg cell/spatials) */
#define DEBUG_CENT 0
gint cent_coords(gint global, struct model_pak *data)
{
gint i, n;
gdouble vec[3], r, r2, rmax;
GSList *list=NULL, *ilist, *clist;
struct image_pak *image;
struct spatial_pak *spatial;
struct core_pak *core;
struct vertex_pak *v;
struct vec_pak *p1;

/* initialize */
VEC3SET(data->offset, 0, 0, 0);
VEC3SET(data->centroid, 0.0, 0.0, 0.0);
r2 = 0.0;

/* core list is (non-empty) selection for non global scaling */
clist = data->cores;
if (!global && g_slist_length(data->selection))
  clist = data->selection;

/* centroid is always the origin, so just calc. scaling */
for (list=data->vertices ; list ; list=g_slist_next(list))
  {
  v = (struct vertex_pak *) list->data;

  ARR3SET(vec, v->x);
  vecmat(data->latmat, vec);
  r = VEC3MAGSQ(vec);
  if (r > r2)
    r2 = r; 
  }

/* coord centroid calc */
n=0;
for (list=clist ; list ; list=g_slist_next(list))
  {
  core = (struct core_pak *) list->data;
  if (core->status & DELETED)
    continue;

  ARR3ADD(data->centroid, core->x);
  n++;
  }
if (n)
  {
  VEC3MUL(data->centroid, 1.0 / (gdouble) n);
  }
else
  {
/* if we have no atoms in a periodic model - make pbc/2 the centroid */
  if (data->periodic)
    {
    VEC3SET(data->centroid, 0.5, 0.5, 0.5);
    }
  }

/* if centroid is substantially within pbc, make pbc/2 the centroid */
if (data->periodic)
  {
  for (i=data->periodic ; i-- ; )
    if (fabs(data->centroid[i] - 0.5) < 0.45)
      data->centroid[i] = 0.5; 
  }

/* adjust for periodic images */
n=1;
VEC3SET(vec, 0.0, 0.0, 0.0);
for (list=data->images ; list ; list=g_slist_next(list))
  {
  image = (struct image_pak *) list->data;

  ARR3ADD(vec, image->pic);
  n++;
  }
VEC3MUL(vec, 1.0 / (gdouble) n);

ARR3ADD(data->centroid, vec);

/* get distance of furtherest displayed item from centroid */
for (list=clist ; list ; list=g_slist_next(list))
  {
  core = (struct core_pak *) list->data;
  if (core->status & DELETED)
    continue;

  ilist=NULL;
  do
    {
    ARR3SET(vec, core->x);
    if (ilist)
      {
      image = (struct image_pak *) ilist->data;
/* image */
      ARR3ADD(vec, image->pic);
      ilist = g_slist_next(ilist);
      }
    else
      {
      ilist = data->images;
      }

    ARR3SUB(vec, data->centroid);
    vecmat(data->latmat,vec);

/* dist squared */
    r = VEC3MAGSQ(vec);
    if (r > r2)
      r2 = r; 
    }
  while (ilist);
  }

/* check cell vertices */
if (data->periodic)
  {
  for (i=8 ; i-- ; )
    {
    ARR3SET(vec, data->cell[i].x);
/* centroid removal */
    ARR3SUB(vec, data->centroid);
/* transform */
    vecmat(data->latmat,vec);  
/* dist sq. */
    r = VEC3MAGSQ(vec);
    if (r > r2)
      r2 = r; 
    }
  }

/* molecular surface contribution */
for (list=data->spatial ; list ; list=g_slist_next(list))
  {
  spatial = (struct spatial_pak *) list->data;

  if (spatial->type == SPATIAL_MOLSURF)
    {
    p1 = g_slist_nth_data(spatial->data, 0);

    ARR3SET(vec, p1->x);
    ARR3SUB(vec, data->centroid);
    vecmat(data->latmat,vec);  

/* dist sq. */
    r = VEC3MAGSQ(vec);
    if (r > r2)
      r2 = r; 
    }
  }

rmax = RMAX_SCALE * sqrt(r2);

/* assign the calculated value, unless 0 */
if (rmax)
  data->rmax = rmax;
else
  data->rmax = RMAX_FUDGE;
data->zoom = data->rmax/data->scale;

#if DEBUG_CENT
P3VEC("centroid = ", data->centroid);
printf("rmax = %f\n", data->rmax);
printf("zoom = %f\n", data->zoom);
printf("cent_coords() done.\n");
#endif

return(0);
}

/******************************************/
/* sort coords by z - mainly for surfaces */
/******************************************/
gint sort_cores(struct core_pak *core1, struct core_pak *core2)
{
if (core1->x[2] < core2->x[2])
  return(-1);
if (core1->x[2] > core2->x[2])
  return(1);
return(0);
}
gint sort_shels(struct shel_pak *shel1, struct shel_pak *shel2)
{
if (shel1->x[2] < shel2->x[2])
  return(-1);
if (shel1->x[2] > shel2->x[2])
  return(1);
return(0);
}
void sort_coords(struct model_pak *data)
{
data->cores = g_slist_sort(data->cores, (gpointer) sort_cores);
data->shels = g_slist_sort(data->shels, (gpointer) sort_shels);
}

/*************************/
/* get coordinate limits */
/*************************/
void cor_calc_xlimits(gdouble *min, gdouble *max, GSList *list)
{
gint i;
GSList *item;
struct core_pak *core;

/* no cores? */
if (!list)
  return;

/* init */
core = list->data;
for (i=3 ;i-- ; )
  min[i] = max[i] = core->x[i];

/* NB: even tho' we check the "fractional" part - the values */
/* will still be cartesian for isolated (or the appropriate */
/* mixed frac/cart values for 1D/2D) models */
/* loop to get limits */
for (item=list ; item ; item=g_slist_next(item))
  {
  core = item->data;

  for (i=3 ; i-- ; )
    {
    if (core->x[i] < min[i])
      min[i] = core->x[i];
    if (core->x[i] > max[i])
      max[i] = core->x[i];
    }
  }
}

/**********************/
/* CALC SCREEN COORDS */
/**********************/
#define DEBUG_UPDATE_COORDS 0
void calc_coords(gint mode, struct model_pak *data)
{
gint n, ghost, model;
gdouble scale;
gdouble rot[9], vec[3];
gdouble tmp4[16], mat4[16], vec4[4];
GSList *list=NULL, *list1, *list2;
struct core_pak *core;
struct shel_pak *shell;
struct model_pak *orig;
struct vec_pak *vector;
struct vertex_pak *v;
struct plane_pak *plane;
struct ribbon_pak *ribbon;
struct spatial_pak *spatial;
struct object_pak *odata;
struct image_pak *image;
GSList *plist, *glist=NULL, *olist;

#if DEBUG_UPDATE_COORDS
printf("calc_coords() start.\n");
#endif

g_return_if_fail(data != NULL);

/* update model & any overlayed (ghost) models */
ghost=0;
orig = data;
while(data)
  {
/* get current incremental rotation matrix */
  make_rotmat(rot, orig->da, mode);
  switch(mode)
    {
    case INITIAL:
      init_rotmat(data->rotmat);
      break;
/* set and constrain (0-360) angles */
    case YAW:
      data->angle[0] += R2D*orig->da;
      ca_deg(&data->angle[0]);
      break;
    case PITCH:
      data->angle[1] += R2D*orig->da;
      ca_deg(&data->angle[1]);
      break;
    case ROLL:
      data->angle[2] += R2D*orig->da;
      ca_deg(&data->angle[2]);
      break;
    }

#if DEBUG_UPDATE_COORDS
P3MAT("instantaneous rot matrix", rot);
#endif

/* update cummulative rotation matrix */
matmat(rot, data->rotmat);
/* compute inverse, needed for correct selection handling */
ARR3SET(&data->irotmat[0], &data->rotmat[0]);
ARR3SET(&data->irotmat[3], &data->rotmat[3]);
ARR3SET(&data->irotmat[6], &data->rotmat[6]);
invmat(data->irotmat);

#if DEBUG_UPDATE_COORDS
P3MAT("cummulative rot matrix", data->rotmat);
P3MAT("           irot matrix", data->irotmat);
#endif

/* TODO - eliminate scale? (use camera position) */
/* precalc scale */
scale = data->scale * sysenv.subscale * 0.5 / data->rmax;

/* precalc matrix products */
ARR3SET(&mat4[0], &data->latmat[0]);
ARR3SET(&mat4[4], &data->latmat[3]);
ARR3SET(&mat4[8], &data->latmat[6]);
ARR3SET(vec, data->centroid);
vecmat(data->latmat, vec);
mat4[3] = -vec[0];
mat4[7] = -vec[1];
mat4[11] = -vec[2];
VEC4SET(&mat4[12], 0.0, 0.0, 0.0, 1.0);

ARR3SET(&tmp4[0], &data->rotmat[0]);
ARR3SET(&tmp4[4], &data->rotmat[3]);
ARR3SET(&tmp4[8], &data->rotmat[6]);
tmp4[3] = 0.0;
tmp4[7] = 0.0;
tmp4[11] = 0.0;
VEC4SET(&tmp4[12], 0.0, 0.0, 0.0, 1.0);
mat4mat(tmp4, mat4);

#if DEBUG_UPDATE_COORDS
P4MAT("mat4:", mat4);
#endif

/* update image translation vectors */
for (list=data->images ; list ; list=g_slist_next(list))
  {
  image = (struct image_pak *) list->data;

  ARR3SET(vec4, image->pic);
/* NB: is a vector */
  vec4[3] = 0.0;
  vec4mat(mat4, vec4);
  ARR3SET(image->rx, vec4);
  }

/* update atoms */
for (list=data->cores ; list ; list=g_slist_next(list))
  {
  core = (struct core_pak *) list->data;
  if (core->status & DELETED)
    continue;

/* calc coords */
  ARR3SET(vec4, core->x);

ARR3ADD(vec4, core->offset);

  vec4[3] = 1.0;
  vec4mat(mat4, vec4);
  ARR3SET(core->rx, vec4);
  }

/* rotate shells (if any ) */
for (list=data->shels ; list ; list=g_slist_next(list))
  {
  shell = (struct shel_pak *) list->data;
  if (shell->status & DELETED)
    continue;

/* get coordinates */
  ARR3SET(vec4, shell->x);

ARR3ADD(vec4, shell->offset);

  vec4[3] = 1.0;
  vec4mat(mat4, vec4);
  ARR3SET(shell->rx, vec4);
  }

/********/
/* CELL */
/********/
for (n=8 ; n-- ; )
  {
  ARR3SET(vec4, data->cell[n].x);
  vec4[3] = 1.0;
  vec4mat(mat4, vec4);
  ARR3SET(data->cell[n].rx, vec4);
  }

/********/
/* AXES */
/********/
for (n=6 ; n-- ; )
  {
  ARR3SET(vec4, data->axes[n].x);
  vec4[3] = 0.0;

/* cartesian or unit cell frame */
  if (data->axes_type == CARTESIAN)
    vecmat(data->rotmat, vec4);
  else
    {
    vec4mat(mat4, vec4);
    normalize(vec4, 3);
    }

  ARR3SET(data->axes[n].rx, vec4);
/* about a 16th of rmax */
  VEC3MUL(data->axes[n].rx, 0.06*data->rmax);
  }

/******************/
/* ribbon updates */
/******************/
olist = data->ribbons;
while (olist)
  {
  odata = (struct object_pak *) olist->data;

  switch (odata->type)
    {
    case RIBBON:
      plist = (GSList *) odata->data;
      while (plist != NULL)
        {
        ribbon = (struct ribbon_pak *) plist->data;

/* end point 1 */
        ARR3SET(vec4, ribbon->x1);
        vec4[3] = 1.0;
        vec4mat(mat4, vec4); 
        ARR3SET(ribbon->r1, vec4);

/* end point 2 */
        ARR3SET(vec4, ribbon->x2);
        vec4[3] = 1.0;
        vec4mat(mat4, vec4); 
        ARR3SET(ribbon->r2, vec4);

/* normal 1 */
        ARR3SET(vec4, ribbon->u1);
        vec4[3] = 0.0;
        vec4mat(mat4, vec4);
        ARR3SET(ribbon->n1, vec4);

/* normal 2 */
        ARR3SET(vec4, ribbon->u2);
        vec4[3] = 0.0;
        vec4mat(mat4, vec4);
        ARR3SET(ribbon->n2, vec4);

/* orientation vector 1 */
        ARR3SET(vec4, ribbon->v1);
        vec4[3] = 0.0;
        vec4mat(mat4, vec4);
        ARR3SET(ribbon->o1, vec4);

/* orientation vector 2 */
        ARR3SET(vec4, ribbon->v2);
        vec4[3] = 0.0;
        vec4mat(mat4, vec4);
        ARR3SET(ribbon->o2, vec4);

        plist = g_slist_next(plist);
        }
    break;
    }
  olist = g_slist_next(olist);
  }

/* spatials */
for (olist=data->spatial ; olist ; olist=g_slist_next(olist))
  {
  spatial = (struct spatial_pak *) olist->data;

  plist = (GSList *) spatial->data;
  while (plist)
    {
    vector = (struct vec_pak *) plist->data;

/* rotate each point */
    ARR3SET(vec4, vector->x);
    vec4[3] = 1.0;
    vec4mat(mat4, vec4);
    ARR3SET(vector->rx, vec4);

/* rotate the normal */
    ARR3SET(vec4, vector->n);
    vec4[3] = 0.0;
    vec4mat(mat4, vec4);
    normalize(vec4, 3);
    ARR3SET(vector->rn, vec4);

    plist = g_slist_next(plist);
    }
  }

/**************/
/* morphology */
/**************/
if (data->id == MORPH)
  {
  for (list1=data->vertices ; list1 ; list1=g_slist_next(list1))
    {
    v = (struct vertex_pak *) list1->data;

    ARR3SET(vec4, v->x);
    vec4[3] = 1.0;
    vec4mat(mat4, vec4);
    ARR3SET(v->rx, vec4);
    }

/* facet center calc. */
  for (list1=data->planes ; list1 ; list1=g_slist_next(list1))
    {
    plane = (struct plane_pak *) list1->data;
    if (plane->present)
      {
/* set visible flag */
      if (facet_visible(data, plane))
        plane->visible = TRUE;
      else
        plane->visible = FALSE;

      VEC3SET(vec, 0.0, 0.0, 0.0);
      n = 0; 
      for (list2=plane->vertices ; list2 ; list2=g_slist_next(list2))
        {
        v = (struct vertex_pak *) list2->data;
        ARR3ADD(vec, v->rx);
        n++;
        }
/* assign */
      g_assert(n > 0);
      VEC3MUL(vec, 1.0/(gdouble) n);
      ARR3SET(plane->rx, vec);
      }
    }
  }

/*********************/
/* ghost processsing */
/*********************/
  if (!ghost)
    {
    glist = data->ghosts;
    ghost++;
    }
  else
    glist = g_slist_next(glist);
/* get the ghost model's pointer */
  if (glist)
    {
    model = GPOINTER_TO_INT(glist->data);
    data = model_ptr(model, RECALL);
    }
  else
    data = NULL;
  }

#if DEBUG_UPDATE_COORDS
printf("calc_coords() done.\n");
#endif
}

/***************************************/
/* unified pixel to coordinate routine */
/***************************************/
#define DEBUG_PCM 0
/* deprec. - use gl_get_world)coords() */
void pixel_coord_map(gint px, gint py, gdouble *r, struct model_pak *data)
{
/* checks */
g_return_if_fail(data != NULL);

/* the model should be displayed for this to make sense */
VEC3SET(r, 0.0, 0.0, 0.0);
gl_get_world_coords(px, py, r);

return;
}

/***********************************/
/* print atom coord data primitive */
/***********************************/
void print_core(struct core_pak *core)
{
gchar *txt;

g_assert(core != NULL);

txt = g_strdup_printf("%4s core at (%9.4f,%9.4f,%9.4f).\n",
                      core->label, core->x[0], core->x[1], core->x[2]);
show_text(STANDARD, txt);
g_free(txt);
}

/************************************/
/* print shell coord data primitive */
/************************************/
void print_shell(struct shel_pak *shell)
{
gchar *txt;

g_assert(shell != NULL);

txt = g_strdup_printf("%4s shel at (%9.4f,%9.4f,%9.4f).\n",
                      shell->label, shell->x[0], shell->x[1], shell->x[2]);
show_text(STANDARD, txt);
g_free(txt);
}

void print_cores(struct model_pak *data)
{
GSList *list;
struct core_pak *core;

for (list=data->cores ; list ; list=g_slist_next(list))
  {
  core = (struct core_pak *) list->data;

  printf("%4s core at (%9.4f,%9.4f,%9.4f).\n",
         core->label, core->x[0], core->x[1], core->x[2]);
  }
}

void print_shells(struct model_pak *data)
{
GSList *list;
struct shel_pak *shel;

for (list=data->shels ; list ; list=g_slist_next(list))
  {
  shel = (struct shel_pak *) list->data;

  printf("%4s shel at (%9.4f,%9.4f,%9.4f).\n",
         shel->label, shel->x[0], shel->x[1], shel->x[2]);
  }
}

/****************************************/
/* print all connected cores and shells */
/****************************************/
void print_core_shell(struct model_pak *data)
{
gint n=0;
GSList *list;
struct core_pak *core;
struct shel_pak *shel;

for (list=data->cores ; list ; list=g_slist_next(list))
  {
  core = (struct core_pak *) list->data;
  if (core->shell)
    {
    shel = (struct shel_pak *) core->shell;

if (n == 7 || n == 15)
  {
    printf("(%p) %4s core at (%9.4f,%9.4f,%9.4f).\n",
           core, core->label, core->x[0], core->x[1], core->x[2]);
    printf("(%p) %4s shel at (%9.4f,%9.4f,%9.4f).\n",
           shel, shel->label, shel->x[0], shel->x[1], shel->x[2]);
  }

    n++;
    }
  }
}

/****************************************************/
/* free all lists that reference cores or core data */
/****************************************************/
void free_core_list(struct model_pak *data)
{
GSList *list;
struct core_pak *core;

/* free cores */
for (list=data->cores ; list ; list=g_slist_next(list))
  {
  core = (struct core_pak *) list->data;

  g_slist_free(core->bonds);

  free_slist(core->vibx_list);
  free_slist(core->viby_list);
  free_slist(core->vibz_list);

  g_free(core);
  }
g_slist_free(data->cores);

/* free shells */
free_slist(data->shels);

/* free connectivity */
free_slist(data->bonds);
free_slist(data->ubonds);
free_mol_list(data);

/* others */
g_slist_free(data->selection);
/* NB: don't free data */
g_slist_free(data->unique_atom_list);

/* enforce empty lists */
data->cores = NULL;
data->shels = NULL;
data->bonds = NULL;
data->ubonds = NULL;
data->moles = NULL;
data->selection = NULL;
data->unique_atom_list = NULL;
}

