/*
Copyright (C) 2003 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <gdk/gdk.h>
#include <gtk/gtk.h>

#ifndef __WIN32
#include <sys/times.h>
#endif

#include "gdis.h"
#include "file.h"
#include "matrix.h"
#include "gtkshorts.h"
#include "interface.h"
#include "dialog.h"

/* top level data structure */
extern struct sysenv_pak sysenv;
GtkListStore *file_path_ts=NULL, *file_name_ts=NULL;
GtkWidget *file_path_tv=NULL, *file_name_tv=NULL;

GtkWidget *path_list, *file_list, *curr_path, *file_name;
gint path_width=80, file_width=160;

enum {FILE_PATH, FILE_PATH_NCOLS};
enum {FILE_NAME, FILE_NAME_NCOLS};

/************************************/
/* callback for destroying a dialog */
/************************************/
void destroy_widget(GtkWidget *w, GtkWidget *target)
{
gtk_widget_destroy(target);
}

/*********************************/
/* close type associated dialogs */
/*********************************/
#define DEBUG_DIALOGS 0
void close_dialog_type(gint type)
{
gint i;

if (!sysenv.canvas)
  return;

for (i=0 ; i<MAX_DIALOGS ; i++)
  {
  if (sysenv.dialog[i].type == type && sysenv.dialog[i].active)
    {
    sysenv.dialog[i].active = FALSE;
    gtk_widget_destroy(sysenv.dialog[i].win);
    }
  }
}

/***************************/
/* close a specific dialog */
/***************************/
void close_dialog(gint id)
{
struct model_pak *data;

/* checks */
if (id < 0 || id >= MAX_DIALOGS)
  return;

/* prevent double closure from gtk_widget_destroy() call */
if (sysenv.dialog[id].active)
  {
#if DEBUG_DIALOGS
printf("destroying dialog: %d, associated model: %d\n",id,sysenv.dialog[id].model);
#endif
  sysenv.dialog[id].active = FALSE;

/* cleanup to stop widgets being updated that have had their */
/* assoc. dialogs destroyed */
/* FIXME - a better way? */
/* eg calls to retrieve dialog data (if it exists!) for a particular */
/* model  - that way we know if we can/need to update a dialog */
  data = model_ptr(sysenv.dialog[id].model, RECALL);
  switch(sysenv.dialog[id].type)
    {
/* cleanup for the liststore/treestore stuff */
    case FILE_SELECT:
      file_cleanup();
      break;
    case GEOMETRY:
      meas_cleanup();
      break;
    case POVRAY:
      render_cleanup();
      break;
    case SYMMETRY:
      symmetry_cleanup();
      break;
    case TASKMAN:
      task_cleanup();
      break;

    case ANIM:
      data->animating = FALSE;
      break;

    case GULP:
      data->gulp.energy_entry = NULL;
      data->gulp.sdipole_entry = NULL;
      data->gulp.esurf_entry = NULL;
      data->gulp.eatt_entry = NULL;
      break;
    case CREATOR:
      switch_mode(FREE);
      break;

    case GPERIODIC:
/* TODO - search & destroy any ELEM_EDIT dialogs (+ colour selection dialogs?) */
      break;

    }
  gtk_widget_destroy(sysenv.dialog[id].win);
  sysenv.dialog[id].win = NULL;
  }
return;
}

/****************************/
/* event hook for the above */
/****************************/
void event_close_dialog(GtkWidget *w, gint id)
{
close_dialog(id);
}

/******************************************/
/* close all dialog's assoc. with a model */
/******************************************/
#define DEBUG_DELETE_MODEL_DIALOGS 0
void close_model_dialogs(gint n)
{
gint i;
struct model_pak *model;

#if DEBUG_DELETE_MODEL_DIALOGS
printf("freeing assoc. dialogs...\n");
#endif

model = model_ptr(n, RECALL);
g_assert(model != NULL);

/* destroy any open dialogs assoc with the model */
for (i=0 ; i<MAX_DIALOGS ; i++)
  if (sysenv.dialog[i].model == n || sysenv.num_models == 1)
    {
/* do we close the dialog? */ 
    switch(sysenv.dialog[i].type)
      {
/* some exceptions */ 
      case TASKMAN:
      case GEOMETRY:
        break;

      case POVRAY:
        if (sysenv.num_models > 1)
          break;

      default:
        close_dialog(i);
      }
    }

#if DEBUG_DELETE_MODEL_DIALOGS
printf("shuffling dialog numbers...\n");
#endif

/* shuffle the dialog model numbers to account */
/* for the deletion of sysenv.active */
for (i=0 ; i<MAX_DIALOGS ; i++)
  if (sysenv.dialog[i].model > n)
    sysenv.dialog[i].model--;

}

/*************************************************/
/* query a certain type of dialog's active entry */
/*************************************************/
gint dialog_active(gint type)
{
gint i;

for (i=0 ; i<MAX_DIALOGS ; i++)
  {
  if (sysenv.dialog[i].type == type)
    return(sysenv.dialog[i].active);
  }
return(FALSE);
}

/******************************/
/* seek for a specific dialog */
/******************************/
gint dialog_seek(gint type, gint model)
{
gint i;

for (i=0 ; i<MAX_DIALOGS ; i++)
  {
  if (sysenv.dialog[i].type == type && sysenv.dialog[i].model == model)
    return(i);
  }
return(-1);
}

/*********************************/
/* close type associated dialogs */
/*********************************/
#define DEBUG_DIALOGS 0
void close_active_dialog_type(gint type)
{
gint i;

if (!sysenv.canvas)
  return;

for (i=0 ; i<MAX_DIALOGS ; i++)
  {
  if (sysenv.dialog[i].type == type && sysenv.dialog[i].active)
    {
    sysenv.dialog[i].active = FALSE;
    gtk_widget_destroy(sysenv.dialog[i].win);
    }
  }
}

/******************************************/
/* ask for a model/type associated dialog */
/******************************************/
gint request_dialog(gint model, gint type)
{
gint i;

#if DEBUG_DIALOGS
printf("requested dialog type: %d, associated model: %d\n",type,model);
#endif

/* is another of the same type active? */
for (i=0 ; i<MAX_DIALOGS ; i++)
  {
  if (!sysenv.dialog[i].active)
    continue;
/* type dependent testing */
  switch(type)
    {
/* allow only only one instance of these (may have global vars) */
    case ANIM:
      if (sysenv.dialog[i].type == MD_ANALYSIS)
        {
        show_text(ERROR, "Sorry, can't have animation and analysis simultaneously.\n");
        return(-1);
        }
    case COLOUR:
    case CREATOR:
    case DIFFAX:
    case DISPLAY:
    case FILE_SELECT:
    case GENSURF:
    case GEOMETRY:
    case SPATIAL:
    case GPERIODIC:
    case ELEM_EDIT:
    case MDI:
    case POVRAY:
    case SURF:
    case TASKMAN:
      if (sysenv.dialog[i].type == type)
        {
        pick_model(model);
/* raise an existing dialog to the front */
        gdk_window_show((sysenv.dialog[i].win)->window);
        return(-1);
        }
      break;

/* allow one instance per model of these */
    case MD_ANALYSIS:
      if (sysenv.dialog[i].type == ANIM)
        {
        show_text(ERROR, "Sorry, can't have animation and analysis simultaneously.\n");
        return(-1);
        }
    case CONNECT:
    case ENERGETICS:
    case MORPH:
    case SGINFO:
    case SYMMETRY:
    case OPENGL:
    case OPENGL_OPTIONS:
      if (sysenv.dialog[i].model == model && sysenv.dialog[i].type == type)
        {
/* dialog already open - raise it to the fore & select assoc. model */
        gdk_window_show((sysenv.dialog[i].win)->window);
        pick_model(model);
        return(-1);
        }
      break;

    default:
      printf("Unknown dialog type requested.\n");
      return(-1);
    }
  }

/* find the first available dialog */ 
for (i=0 ; i<MAX_DIALOGS ; i++)
  {
  if (!sysenv.dialog[i].active)
    {
/* init and return dialog id */
    sysenv.dialog[i].active = TRUE;
    sysenv.dialog[i].model = model;
    sysenv.dialog[i].type = type;
    sysenv.dialog[i].data = NULL;
    return(i);
    }
  }
show_text(WARNING, "Maximum allowed dialogs reached.\n");
return(-1);
}

/******************************/
/* safely acquire tree models */
/******************************/
GtkTreeModel *file_path_treemodel(void)
{
if (!file_path_tv)
  return(NULL);
if (!GTK_IS_TREE_VIEW(file_path_tv))
  return(NULL);
return(gtk_tree_view_get_model(GTK_TREE_VIEW(file_path_tv)));
}

GtkTreeModel *file_name_treemodel(void)
{
if (!file_name_tv)
  return(NULL);
if (!GTK_IS_TREE_VIEW(file_name_tv))
  return(NULL);
return(gtk_tree_view_get_model(GTK_TREE_VIEW(file_name_tv)));
}

/***************************************/
/* update contents of file load widget */
/***************************************/
#define DEBUG_UPDATE_FILE_PANE 0
gint update_file_pane(void)
{
gint filter;
gchar *name, *full;
GSList *list, *dir_list;
GtkTreeIter iter;
struct stat buff;
struct file_pak *file_data;

/* NEW */
filter = sysenv.file_type;

/* getting from this directory */
gtk_label_set_text(GTK_LABEL(curr_path), sysenv.cwd);

/* clear */
/* FIXME - this is the problem - I guess you can't clear a tree store */
/* that generated a selection change event, within the handler */
gtk_list_store_clear(file_path_ts);
gtk_list_store_clear(file_name_ts);

/* get directory listing */
dir_list = get_dir_list(sysenv.cwd, 1);

/* special case for a failure (is this always a no permission error?) */
if (!dir_list)
  {
  show_text(ERROR, "Couldn't get a directory listing, check the permissions.\n");

  gtk_list_store_append(file_path_ts, &iter);
  gtk_list_store_set(file_path_ts, &iter, FILE_PATH, "..", -1);
  }

list = dir_list;
while (list)
  {
/* stat the file (MUST have the full path) */
  name = (gchar *) list->data;
  full = g_build_filename(sysenv.cwd, list->data, NULL);
  stat(full, &buff);

#if DEBUG_UPDATE_FILE_PANE
printf("[%s] : %d\n",full,buff.st_mode);
#endif

/* convert and check if directory */
  if ((buff.st_mode & S_IFMT) ==  S_IFDIR)
    {
    gtk_list_store_append(file_path_ts, &iter);
    gtk_list_store_set(file_path_ts, &iter, FILE_PATH, name, -1);
    }
  else
    {
/* is it a recognized type? */
    file_data = get_file_info((gpointer *) list->data, BY_EXTENSION);
    if (file_data)
      {
/* display name if matches current file filter */
      if (filter == file_data->group || filter == DATA) 
        {
        gtk_list_store_append(file_name_ts, &iter);
        gtk_list_store_set(file_name_ts, &iter, FILE_NAME, name, -1);
        }
      }
    }
  list = g_slist_next(list);
  g_free(full);
  }

/* done - don't free list data, it went into the store */
g_slist_free(dir_list);

return(TRUE);
}

/***********************************/
/* change to a specified directory */
/***********************************/
#define DEBUG_SET_PATH 0
gint set_path(const gchar *txt)
{
gint i, n, status=0;
gchar **buff, *text;
GString *path;

/* split by directory separator */
text = g_strdup(txt);
g_strstrip(text);
buff = g_strsplit(text, DIR_SEP, 0);
path = g_string_new(NULL);

/* find the number of tokens */
n=0;
while(*(buff+n))
  {
#if DEBUG_SET_PATH
printf("[%s] ", *(buff+n));
#endif
  n++;
  }
#if DEBUG_SET_PATH
printf(": found %d tokens.\n", n);
#endif

/* truncate token list if parent directory selected */
if (n > 1)
  if (g_ascii_strncasecmp("..",*(buff+n-1),2) == 0)
    n -= 2;

/* build new path */
/* this is a bit fiddly, as we don't want a dir_sep on the end */
/* EXCEPT if it's the root directory (blame windows for this) */
g_string_sprintf(path, "%s", *buff);
i=1;
while (i < n)
  {
  g_string_sprintfa(path, "%s%s", DIR_SEP, *(buff+i));
  i++;
  }
if (n < 2)
  g_string_sprintfa(path, "%s", DIR_SEP);

#if DEBUG_SET_PATH
printf("testing path [%s] ... \n", path->str); 
#endif

if (g_file_test(path->str, G_FILE_TEST_IS_DIR))
  {
  if (g_file_test(path->str, G_FILE_TEST_IS_EXECUTABLE))
    {
    g_free(sysenv.cwd);
    sysenv.cwd = g_strdup(path->str);
    }
  else
    {
#if DEBUG_SET_PATH
printf(" Insufficient access permission.\n");
#endif
    status++;
    }
  }
else
  {
#if DEBUG_SET_PATH
printf(" Not a directory.\n");
#endif
  status++;
  }

g_free(text);
g_string_free(path, TRUE);

return(status);
}

/*****************************/
/* file type/filter handlers */
/*****************************/
void type_change(GtkWidget *w)
{
G_CONST_RETURN gchar *tmp;
struct file_pak *file_data;

tmp = gtk_entry_get_text(GTK_ENTRY(w));

/* update if valid type */
file_data = get_file_info((gpointer *) tmp, BY_LABEL);
if (file_data)
  {
  sysenv.file_type = file_data->id;
  update_file_pane();
  }
}

void filter_change(GtkWidget *w)
{
G_CONST_RETURN gchar *tmp;
struct file_pak *file_data;

tmp = gtk_entry_get_text(GTK_ENTRY(w));

/* update if valid type */
file_data = get_file_info((gpointer *) tmp, BY_LABEL);
if (file_data)
  {
  sysenv.babel_type = file_data->id;
  update_file_pane();
  }
}

/********************************************/
/* primary event handler for file selection */
/********************************************/
void file_event_handler(GtkWidget *w, 
                        gpointer secondary_handler(gchar *, struct model_pak *))
{
gchar *name, *fullname;

/* get the current name */
/* NB: we strdup, so we don;t have to make name a gconst pointer */
name = g_strdup(gtk_entry_get_text(GTK_ENTRY(file_name)));

/* attempt to use name to change the path */
fullname = g_build_filename(sysenv.cwd, name, NULL);
if (set_path(fullname))
  secondary_handler(name, NULL);

g_free(fullname);
g_free(name);
}

/********************************/
/* cartesian/fractional toggles */
/********************************/
void toggle_save_type(struct model_pak *data)
{
data->fractional ^= 1;
}

/**********************/
/* selection handlers */
/**********************/
static void file_path_activate(GtkTreeView *treeview, GtkTreePath *treepath)
{
gchar *text, *path;
GtkTreeIter iter;
GtkTreeModel *treemodel;

treemodel = gtk_tree_view_get_model(treeview);
gtk_tree_model_get_iter(treemodel, &iter, treepath);

gtk_tree_model_get(treemodel, &iter, FILE_PATH, &text, -1);

path = g_build_path(DIR_SEP, sysenv.cwd, text, NULL);

set_path(path);
g_free(path);

update_file_pane();
}

static void cb_file_name_changed(GtkTreeSelection *selection, gpointer data)
{
gchar *text;
GtkTreeIter iter;
GtkTreeModel *treemodel;

treemodel = file_name_treemodel();
if (!treemodel)
  return;

if (gtk_tree_selection_get_selected(selection, &treemodel, &iter))
  {
  gtk_tree_model_get(treemodel, &iter, FILE_NAME, &text, -1);
  gtk_entry_set_text(GTK_ENTRY(file_name), g_strdup(text));
  }
}

/*******************/
/* cleanup on exit */
/*******************/
void file_cleanup(void)
{
/* NB: since the tree store is independant of the model's geom list */
/* it must be completely removed (and then restored) with the dialog */
gtk_list_store_clear(file_path_ts);
gtk_list_store_clear(file_name_ts);
file_path_tv = NULL;
file_path_ts = NULL;
file_name_tv = NULL;
file_name_ts = NULL;
}

/*************************************************/
/* customized load widget (with filetype filter) */
/*************************************************/
void 
file_dialog(gchar *title, 
            gchar *name, 
            gpointer secondary_handler(gchar *, struct model_pak *),
            gint filter)
{
gint id;
GSList *flist;
GList *elist, *blist;
GtkWidget *swin, *hbox, *combo, *label;
GtkCellRenderer *renderer;
GtkTreeViewColumn *column;
GtkTreeSelection *select;
struct model_pak *data=NULL;
struct dialog_pak *fl_data;
struct file_pak *fdata;

/* if save type, check we have a loaded model */
if (secondary_handler == (gpointer) file_save_as)
  {
  data = model_ptr(sysenv.active, RECALL);
  if (!data)
    return;
  }

/* get a dialog if possible */
if ((id = request_dialog(-1, FILE_SELECT)) < 0)
  return;
fl_data = &sysenv.dialog[id];

/* make and set up the dialog window */
fl_data->win = gtk_dialog_new();
gtk_window_set_title(GTK_WINDOW(fl_data->win), title);
gtk_window_set_default_size(GTK_WINDOW(fl_data->win), 400, 350);
gtk_window_set_position(GTK_WINDOW(fl_data->win), GTK_WIN_POS_CENTER);
gtk_container_set_border_width(
    GTK_CONTAINER(GTK_BOX(GTK_DIALOG(fl_data->win)->vbox)),10);
g_signal_connect(GTK_OBJECT(fl_data->win), "destroy",
                 GTK_SIGNAL_FUNC(event_close_dialog), (gpointer) id);

/* TOP ROW - cwd printed */
hbox = gtk_hbox_new(FALSE,0);
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(fl_data->win)->vbox),hbox,FALSE,FALSE,0);
label = gtk_label_new("Current path: ");
gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);

curr_path = gtk_label_new("dummy");
gtk_box_pack_start(GTK_BOX(hbox), curr_path, FALSE, FALSE, 0);

/* option to save as cartesian or fractional coords */
if (secondary_handler == (gpointer) file_save_as)
  {
  g_assert(data != NULL);
  if (data->periodic)
    {
    hbox = gtk_hbox_new(FALSE,0);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(fl_data->win)->vbox),
                                         hbox,FALSE,FALSE,0);

    new_check_button("Save as cartesian coordinates", toggle_save_type, data,
                                                    !data->fractional, hbox);
    }
  }

/* option to auto unfragment */
if (secondary_handler == (gpointer) file_load)
  {
  hbox = gtk_hbox_new(FALSE,0);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(fl_data->win)->vbox), hbox,FALSE,FALSE,0);
  gtksh_direct_check("Auto unfragment", &sysenv.unfragment, NULL, NULL, hbox);
  }

/* SECOND ROW - sub directory & file listings */
hbox = gtk_hbox_new(FALSE, PANEL_SPACING);
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(fl_data->win)->vbox), hbox, TRUE, TRUE, 0);
gtk_container_set_border_width(GTK_CONTAINER(hbox), PANEL_SPACING);

/* scrolled model pane */
swin = gtk_scrolled_window_new(NULL, NULL);
gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW (swin),
                               GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
gtk_box_pack_start(GTK_BOX(hbox), swin, TRUE, TRUE, 0);

/* path treestore/treeview */
file_path_ts = gtk_list_store_new(FILE_PATH_NCOLS, G_TYPE_STRING);
file_path_tv = gtk_tree_view_new_with_model(GTK_TREE_MODEL(file_path_ts));
gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(swin), file_path_tv);
renderer = gtk_cell_renderer_text_new();
column = gtk_tree_view_column_new_with_attributes("Tree", renderer, "text", FILE_PATH, NULL);
gtk_tree_view_append_column(GTK_TREE_VIEW(file_path_tv), column);

/* NB: use this method instead of selections to handle events */
/* as selections will core dump if the handler clears the store */
g_signal_connect(file_path_tv, "row_activated",
                 G_CALLBACK(file_path_activate), NULL);


/* scrolled model pane */
swin = gtk_scrolled_window_new(NULL, NULL);
gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW (swin),
                               GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
gtk_box_pack_start(GTK_BOX(hbox), swin, TRUE, TRUE, 0);

/* filename treestore/treeview */
file_name_ts = gtk_list_store_new(FILE_NAME_NCOLS, G_TYPE_STRING);
file_name_tv = gtk_tree_view_new_with_model(GTK_TREE_MODEL(file_name_ts));
gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(swin), file_name_tv);
renderer = gtk_cell_renderer_text_new();
column = gtk_tree_view_column_new_with_attributes("Files", renderer, "text", FILE_NAME, NULL);
gtk_tree_view_append_column(GTK_TREE_VIEW(file_name_tv), column);
/* setup the selection handler */
select = gtk_tree_view_get_selection(GTK_TREE_VIEW(file_name_tv));
gtk_tree_selection_set_mode(select, GTK_SELECTION_BROWSE);
g_signal_connect(G_OBJECT(select), "changed",
                 G_CALLBACK(cb_file_name_changed),
                 NULL);

/* THIRD ROW - filename currently selected & file extension filter */
hbox = gtk_hbox_new(FALSE, PANEL_SPACING);
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(fl_data->win)->vbox), hbox, FALSE, FALSE, 0);

/* filename */
file_name = gtk_entry_new();
gtk_box_pack_start(GTK_BOX (hbox), file_name, TRUE, TRUE, 0);

/* CURRENT */
if (name)
  gtk_entry_set_text(GTK_ENTRY(file_name), name);

gtk_entry_set_editable(GTK_ENTRY(file_name), TRUE);
/* hook a <CR> event to the load action */
g_signal_connect(GTK_OBJECT(file_name), "activate", 
                 GTK_SIGNAL_FUNC(file_event_handler), secondary_handler);

/* NEW - build the recognized extension list */
elist = NULL;
flist = sysenv.file_list;
while(flist != NULL)
  {
  fdata = (struct file_pak *) flist->data;
/* include in menu? */
  if (fdata->menu && fdata->group != BABEL)
    elist = g_list_append(elist, fdata->label);
  flist = g_slist_next(flist);
  }

/* NEW - combo boxes for file types/filters */
combo = gtk_combo_new();
gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(combo)->entry), FALSE);
gtk_combo_set_popdown_strings(GTK_COMBO(combo), elist);

/* set the currently selected type (BEFORE the changed event is connected) */
fdata = get_file_info(GINT_TO_POINTER(filter), BY_FILE_ID);
if (fdata)
  gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(combo)->entry), fdata->label);

gtk_box_pack_start(GTK_BOX (hbox), combo, FALSE, FALSE, 0);
g_signal_connect(GTK_OBJECT(GTK_COMBO(combo)->entry), "changed", 
                 GTK_SIGNAL_FUNC(type_change), NULL);
gtk_widget_set_size_request(combo, 12*sysenv.gfontsize, -1);


/* build the recognized babel filter list */
blist = NULL;
flist = sysenv.file_list;
while(flist != NULL)
  {
  fdata = (struct file_pak *) flist->data;
/* include in menu? */
  if (fdata->menu && fdata->group == BABEL)
    blist = g_list_append(blist, fdata->label);
  flist = g_slist_next(flist);
  }

/* combo boxes for file types/filters */
combo = gtk_combo_new();
gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(combo)->entry), FALSE);
gtk_combo_set_popdown_strings(GTK_COMBO(combo), blist);
/* set the currently selected filter (NB: done BEFORE the changed event is connected) */
fdata = get_file_info(GINT_TO_POINTER(sysenv.babel_type), BY_FILE_ID);
if (fdata)
  gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(combo)->entry), fdata->label);
gtk_box_pack_start(GTK_BOX (hbox), combo, FALSE, FALSE, 0);
g_signal_connect(GTK_OBJECT(GTK_COMBO(combo)->entry), "changed", 
                 GTK_SIGNAL_FUNC(filter_change), NULL);
gtk_widget_set_size_request(combo, 12*sysenv.gfontsize, -1);

/* terminating buttons */
if (secondary_handler == (gpointer) file_load)
  gtksh_stock_button(GTK_STOCK_OPEN, file_event_handler, secondary_handler, 
                     GTK_DIALOG(fl_data->win)->action_area);
else
  gtksh_stock_button(GTK_STOCK_SAVE, file_event_handler, secondary_handler, 
                     GTK_DIALOG(fl_data->win)->action_area);

gtksh_stock_button(GTK_STOCK_CANCEL, event_close_dialog, GINT_TO_POINTER(id),
                   GTK_DIALOG(fl_data->win)->action_area);

/* all done */
gtk_widget_show_all(fl_data->win);
sysenv.file_type = filter;
update_file_pane();
}

/*************************************************/
/* callback to update a widget background colour */
/*************************************************/
void set_colour_widget(GtkWidget *cs, GtkWidget *w)
{
GdkColor col;
GtkStyle *style;

gtk_color_selection_get_current_color(GTK_COLOR_SELECTION(cs), &col);

style = gtk_style_copy(gtk_widget_get_style(w));
style->bg[0].red   = col.red;
style->bg[0].green = col.green;
style->bg[0].blue  = col.blue;
gtk_widget_set_style(GTK_WIDGET(w), style);
}

/***************************************/
/* callback to update the colour value */
/***************************************/
void set_colour_value(GtkWidget *cs, gdouble *colour)
{
GdkColor col;

gtk_color_selection_get_current_color(GTK_COLOR_SELECTION(cs), &col);

colour[0] = col.red;
colour[1] = col.green;
colour[2] = col.blue;
VEC3MUL(colour, 1.0/65535.0);
}

/*******************************************************************/
/* create a colour selection dialog that automatically sets colour */
/*******************************************************************/
void colour_dialog(gchar *title, GtkWidget *w, gpointer colour,
                                   gpointer func, gpointer arg)
{
GtkWidget *csd;

/* create colour selection dialog */
csd = gtk_color_selection_dialog_new(title);

/* value colour update */
g_signal_connect(GTK_OBJECT(GTK_COLOR_SELECTION_DIALOG(csd)->colorsel),
                 "color_changed",
                  GTK_SIGNAL_FUNC(set_colour_value), (gpointer) colour);

/* widget colour update */
if (w)
  g_signal_connect(GTK_OBJECT(GTK_COLOR_SELECTION_DIALOG(csd)->colorsel),
                   "color_changed",
                    GTK_SIGNAL_FUNC(set_colour_widget), (gpointer) w);

/* call user's function */
if (func)
  g_signal_connect_after(GTK_OBJECT(GTK_COLOR_SELECTION_DIALOG(csd)->colorsel),
                        "color_changed",
                         GTK_SIGNAL_FUNC(func), (gpointer) arg);

/* default buttons */
g_signal_connect(GTK_OBJECT(GTK_COLOR_SELECTION_DIALOG(csd)->ok_button), "clicked",
                 GTK_SIGNAL_FUNC(destroy_widget), (gpointer) csd);
gtk_widget_hide(GTK_COLOR_SELECTION_DIALOG(csd)->cancel_button);

gtk_widget_show(csd);
}

/********************************************/
/* simple callback suited the colour dialog */
/********************************************/
void modify_colour_dialog(GtkWidget *w, gdouble *colour)
{
colour_dialog("Edit Colour", w, colour, NULL, NULL);
}

