/*
Copyright (C) 2003 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>

#ifndef __WIN32
#include <sys/times.h>
#endif

#include "gdis.h"
#include "coords.h"
#include "edit.h"
#include "matrix.h"
#include "opengl.h"
#include "render.h"
#include "select.h"
#include "gtkshorts.h"
#include "interface.h"

#define DEBUG 0

extern struct sysenv_pak sysenv;
extern struct elem_pak elements[];

gint current_colour[3];

/**************************/
/* remove duplicate cores */
/**************************/
#define DEBUG_REMOVE_DUPLICATES 0
void remove_duplicates(struct model_pak *data)
{
GSList *list;
struct core_pak *core, *copy;
struct shel_pak *shel, *copys;

con_init_zones(data);

#if DEBUG_REMOVE_DUPLICATES
printf("Initial cores: %d\n", g_slist_length(data->cores));
printf("Initial shels: %d\n", g_slist_length(data->shels));
#endif

/* NB: */
/* delete duplicate cores & shells separately as linkages */
/* can cause screwups */
/* commit before searching for duplicate shells, as a commit */
/* will delete some attached shells */

for (list=data->cores ; list ; list=g_slist_next(list))
  {
  core = (struct core_pak *) list->data;
  if (core->status & DELETED)
    continue;

/* loop while we keep finding (non-deleted) copies */
  copy = seek_core(core, 1, data);
  while (copy)
    {
    copy->status |= DELETED;
    copy = seek_core(core, 1, data);
    }
  }
delete_commit(data);

for (list=data->shels ; list ; list=g_slist_next(list))
  {
  shel = (struct shel_pak *) list->data;
  if (shel->status & DELETED)
    continue;

  copys = seek_shel(shel, 1, data);
  while (copys)
    {
    copys->status |= DELETED;
    copys = seek_shel(shel, 1, data);
    }
  }
delete_commit(data);

/* refresh core-shell linkages */
update_shells(data);

#if DEBUG_REMOVE_DUPLICATES
printf("Final cores: %d\n", g_slist_length(data->cores));
printf("Final shels: %d\n", g_slist_length(data->shels));
#endif
}

/****************************************/
/* clamp 3D coords to fractional limits */
/****************************************/
/* FIXME - return value is not sensible, but maybe we don't care??? */
#define DEBUG_CLAMP 0
#define EPS 0.0000000000001
gint fractional_clamp(gdouble *vec, gint *mov, gint dim)
{
gint i, moved=FALSE;

#if DEBUG_CLAMP
P3VEC("inp: ", vec);
#endif
/* NB: init ALL 3 coords as isolated molecules call this */
/* routine & it is expected that mov to be set to 0,0,0 */
for (i=3 ; i-- ; )
  mov[i] = 0;

/* clamp vec of size dim to [0,1) */
for (i=0 ; i<dim ; i++)
  {
  if (vec[i] < 0.0)
    { 
    mov[i] = (gint) (1.0 - vec[i] - EPS);
    }
  else
    mov[i] = -1 * (gint) vec[i];

  vec[i] += (gdouble) mov[i];
  }
#if DEBUG_CLAMP
P3VEC("out: ", vec);
#endif
return(moved);
}

/***************************************/
/* apply PBC to the atoms in the model */
/***************************************/
#define DEBUG_PBC_ATOMS 0
void pbc_constrain_atoms(struct model_pak *data)
{
gint xlat[3];
gdouble mov[3];
GSList *list, *blist;
struct core_pak *core;
struct shel_pak *shell;
struct bond_pak *bond;

g_return_if_fail(data != NULL);
if (!data->periodic)
  return;

#if DEBUG_PBC_ATOMS
printf("[A] num cores: %d\n", g_slist_length(data->cores));
printf("[A] num shels: %d\n", g_slist_length(data->shels));
#endif

/*
printf(" --- pca start\n");
dump_bonds(data);
if (data->periodic == 2)
  return;
*/

/* translate cores to within the cell */
for (list=data->cores ; list ; list=g_slist_next(list))
  {
  core = (struct core_pak *) list->data;

/* clamp */
  VEC3SET(xlat, 0, 0, 0);
  fractional_clamp(core->x, xlat, data->periodic);
  ARR3SET(mov, xlat);

/* move shell */
  if (core->shell)
    {
    shell = (struct shel_pak *) core->shell;
    ARR3ADD(shell->x, mov);
    }

/* no translation needed - skip to next */
  if (VEC3MAGSQ(mov) < POSITION_TOLERANCE)
    continue;

/* search for periodic bonds to update */
  for (blist=core->bonds ; blist ; blist=g_slist_next(blist))
    {
    bond = (struct bond_pak *) blist->data;

    switch (bond->periodic)
      {
      case BOND_MERGED:
      case BOND_SPLIT:
        if (core == bond->atom1)
          {
          ARR3ADD(bond->pic, mov);
          }
        else
          {
          ARR3SUB(bond->pic, mov);
          }
        break;

/* here we've found a bond that wasn't marked as periodic initially */
/* yet was broken by the pbc constraint. This can happen for cartesian */
/* models with a molecule sticking out past a cell edge */
      default:
        bond->periodic = BOND_SPLIT;
        ARR3SET(bond->pic, mov);
        if (core == bond->atom2)
          {
          VEC3MUL(bond->pic, -1.0);
          }
        break;
      }
    }
  }

/* search for ruptured periodic bonds */
for (blist=data->bonds ; blist ; blist=g_slist_next(blist))
  {
  bond = (struct bond_pak *) blist->data;

  switch (bond->periodic)
    {
    case BOND_MERGED:
      if (VEC3MAGSQ(bond->pic) > FRACTION_TOLERANCE)
        bond->periodic = BOND_SPLIT;
      break;

/* were any healed? (shouldn't happen?) */
/* this can happen if unfrag() fails & this function is called */
/*
    case BOND_PERIODIC:
      if (VEC3MAGSQ(bond->pic) < FRACTION_TOLERANCE)
        g_assert_not_reached();
*/
    }
  }

/* NEW */
/*
remove_duplicates(data);
*/

/*
printf(" --- pca end\n");
dump_bonds(data);
*/

#if DEBUG_PBC_ATOMS
printf("[B] num cores: %d\n", g_slist_length(data->cores));
printf("[B] num shels: %d\n", g_slist_length(data->shels));
#endif
}

/**************************************/
/* apply PBC to the mols in the model */
/**************************************/
#define DEBUG_CONFINE_MOLS 0
void pbc_constrain_mols(struct model_pak *data)
{
gint j, mov[3];
GSList *mlist, *clist;
struct core_pak *core;
struct mol_pak *mol;

g_return_if_fail(data != NULL);

#if DEBUG_CONFINE_MOLS
printf("pbc_constrain_mols()\n");
#endif

/* all orig coords are fractional */
for (mlist=data->moles ; mlist ; mlist=g_slist_next(mlist))
  {
  mol = (struct mol_pak *) mlist->data;

/* calc moves required to bring centroid within pbc */
  VEC3SET(mov, 0, 0, 0);
  for (j=0 ; j<data->periodic ; j++)
    {
    while (mol->centroid[j] > 1.0-FRACTION_TOLERANCE)
      {
      mol->centroid[j] -= 1.0;
      mov[j]--;
      }
    while (mol->centroid[j] < 0.0)
      {
      mol->centroid[j] += 1.0;
      mov[j]++;
      }
    }

/* apply to all atoms/shells in this molecule */
  for (clist=mol->cores ; clist ; clist=g_slist_next(clist))
    {
    core = (struct core_pak *) clist->data;

    ARR3ADD(core->x, mov);

    if (core->shell)
      {
      ARR3ADD((core->shell)->x, mov);
      }
    }
  }
}

/*******************************************/
/* draw a box - contents are the selection */
/*******************************************/
void update_box(gint x, gint y, struct model_pak *data, gint call_type)
{

switch (call_type)
  {
  case START:
/* don't clean the selection (allows multiple box selections) */
/* setup the box selection object */
    data->box_on = TRUE;
    data->select_box[0] = x;
    data->select_box[1] = y;
  case UPDATE:
    data->select_box[2] = x;
    data->select_box[3] = y;
    break;
  }
}

/**********************************************************/
/* check if a given label matches with element/atom label */
/**********************************************************/
#define DEBUG_SHELL_MATCH 0
gint shel_match(const gchar *label, struct shel_pak *shell)
{
gint code;

#if DEBUG_SHELL_MATCH
printf("[%s] : [%s]\n", label, shell->label);
#endif

code = elem_type(label);

/* if input label doesnt match the element symbol length - it means the */
/* user has put in something like H1 - compare this with the atom label */
if (code)
  {
  if (g_ascii_strcasecmp(label, elements[shell->atom_code].symbol) != 0)
    {
    if (g_ascii_strcasecmp(shell->label, label) == 0)
      return(1);
    }
  else
    return(1);
  }
else
  return(1);

#if DEBUG_SHELL_MATCH
printf("rejected.\n");
#endif

return(0);
}

/**********************************************************/
/* check if a given label matches with element/atom label */
/**********************************************************/
#define DEBUG_ATOM_MATCH 0
gint core_match(const gchar *label, struct core_pak *core)
{
gint code;

#if DEBUG_ATOM_MATCH
printf("[%s] : [%s]\n", label, core->label);
#endif

code = elem_type(label);

/* if input label doesnt match the element symbol length - it means the */
/* user has put in something like H1 - compare this with the atom label */
if (code)
  {
  if (g_ascii_strcasecmp(label, elements[core->atom_code].symbol) != 0)
    {
    if (g_ascii_strcasecmp(core->label, label) == 0)
      return(1);
    }
  else
    return(1);
  }
else
  return(1);

#if DEBUG_ATOM_MATCH
printf("rejected.\n");
#endif

return(0);
}

