/*
Copyright (C) 2003 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include <math.h>
#include <stdio.h>
#include <string.h>
#include <strings.h>
#include <ctype.h>
#include <stdlib.h>
#include <time.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>

#ifdef __sun
#include <sys/dirent.h>
#else
  #ifdef __WIN32
  #include <dirent.h>
  #else
  #include <sys/dir.h>
  #include <sys/param.h>
  #endif
#endif

#include "gdis.h"
#include "coords.h"
#include "file.h"
#include "parse.h"
#include "matrix.h"
#include "space.h"
#include "gtkshorts.h"
#include "interface.h"
#include "dialog.h"
#include "opengl.h"

#define DEBUG_MORE 0
#define MAX_KEYS 15

/* main structures */
extern struct sysenv_pak sysenv;
extern struct elem_pak elements[];

void file_init(void)
{
struct file_pak *file_data;

/* TODO - include calling routines for read/writing */
/* TODO - include the rest of the (babel) types */

sysenv.file_list = NULL;

/* supported file type */
file_data = (struct file_pak *) g_malloc(sizeof(struct file_pak));
file_data->id = DATA;                           /* unique identifier */ 
file_data->group = DATA;                        /* used to group inp/out types */
file_data->menu = TRUE;                         /* include in file menu listing? */
file_data->label = g_strdup("All known types"); /* text info for the user */
file_data->ext = NULL;                          /* extension matching */
file_data->write_file = NULL;                   /* file creation */
file_data->read_file = NULL;                    /* file reading */
sysenv.file_list = g_slist_prepend(sysenv.file_list, file_data);

/* supported file type */
file_data = (struct file_pak *) g_malloc(sizeof(struct file_pak));
file_data->id = BIOSYM;
file_data->group = BIOSYM;
file_data->menu = TRUE;
file_data->label = g_strdup("Biosym");
file_data->ext = NULL;
file_data->ext = g_slist_prepend(file_data->ext, "car");
file_data->ext = g_slist_prepend(file_data->ext, "cor");
file_data->ext = g_slist_prepend(file_data->ext, "arc");
file_data->write_file = write_arc;
file_data->read_file = read_arc;
sysenv.file_list = g_slist_prepend(sysenv.file_list, file_data);

/* supported file type */
file_data = (struct file_pak *) g_malloc(sizeof(struct file_pak));
file_data->id = CIF;
file_data->group = CIF;
file_data->menu = TRUE;
file_data->label = g_strdup("CIF");
file_data->ext = NULL;
file_data->ext = g_slist_prepend(file_data->ext, "cif");
file_data->write_file = write_cif;
file_data->read_file = read_cif;
sysenv.file_list = g_slist_prepend(sysenv.file_list, file_data);

/* supported file type */
file_data = (struct file_pak *) g_malloc(sizeof(struct file_pak));
file_data->id = FDF;
file_data->group = FDF;
file_data->menu = TRUE;
file_data->label = g_strdup("SIESTA");
file_data->ext = NULL;
file_data->ext = g_slist_prepend(file_data->ext, "fdf");
file_data->write_file = write_fdf;
file_data->read_file = read_fdf;
sysenv.file_list = g_slist_prepend(sysenv.file_list, file_data);

/* supported file type */
file_data = (struct file_pak *) g_malloc(sizeof(struct file_pak));
file_data->id = SIESTA_OUT;
file_data->group = FDF;
file_data->menu = FALSE;
file_data->label = g_strdup("SIESTA Output");
file_data->ext = NULL;
file_data->ext = g_slist_prepend(file_data->ext, "sout");
file_data->ext = g_slist_prepend(file_data->ext, "sot");
file_data->write_file = NULL;
file_data->read_file = read_sout;
sysenv.file_list = g_slist_prepend(sysenv.file_list, file_data);

/* supported file type */
file_data = (struct file_pak *) g_malloc(sizeof(struct file_pak));
file_data->id = GULP;
file_data->group = GULP;
file_data->menu = TRUE;
file_data->label = g_strdup("Gulp");
file_data->ext = NULL;
file_data->ext = g_slist_prepend(file_data->ext, "gin");
file_data->ext = g_slist_prepend(file_data->ext, "res");
file_data->write_file = write_gulp;
file_data->read_file = read_gulp;
sysenv.file_list = g_slist_prepend(sysenv.file_list, file_data);

/* supported file type */
file_data = (struct file_pak *) g_malloc(sizeof(struct file_pak));
file_data->id = GULPOUT;
file_data->group = GULP;
file_data->menu = FALSE;
file_data->label = g_strdup("Gulp output");
file_data->ext = NULL;
file_data->ext = g_slist_prepend(file_data->ext, "got");
file_data->ext = g_slist_prepend(file_data->ext, "gout");
file_data->write_file = NULL;
file_data->read_file = read_gulp_output;
sysenv.file_list = g_slist_prepend(sysenv.file_list, file_data);

/* supported file type */
file_data = (struct file_pak *) g_malloc(sizeof(struct file_pak));
file_data->id = MARVIN;
file_data->group = MARVIN;
file_data->menu = TRUE;
file_data->label = g_strdup("Marvin");
file_data->ext = NULL;
file_data->ext = g_slist_prepend(file_data->ext, "mar");
file_data->ext = g_slist_prepend(file_data->ext, "mvn");
file_data->ext = g_slist_prepend(file_data->ext, "mvn-r");
file_data->write_file = write_marvin;
file_data->read_file = read_marvin;
sysenv.file_list = g_slist_prepend(sysenv.file_list, file_data);

/* supported file type */
file_data = (struct file_pak *) g_malloc(sizeof(struct file_pak));
file_data->id = MVNOUT;
file_data->group = MARVIN;
file_data->menu = FALSE;
file_data->label = g_strdup("Marvin output");
file_data->ext = NULL;
file_data->ext = g_slist_prepend(file_data->ext, "mot");
file_data->ext = g_slist_prepend(file_data->ext, "mvout");
file_data->ext = g_slist_prepend(file_data->ext, "mvnout");
file_data->write_file = NULL;
file_data->read_file = read_mvnout;
sysenv.file_list = g_slist_prepend(sysenv.file_list, file_data);

/* supported file type */
file_data = (struct file_pak *) g_malloc(sizeof(struct file_pak));
file_data->id = MORPH;
file_data->group = MORPH;
file_data->menu = TRUE;
file_data->label = g_strdup("Morphology");
file_data->ext = NULL;
file_data->ext = g_slist_append(file_data->ext, "gmf");
file_data->write_file = write_gmf;
file_data->read_file = read_gmf;
sysenv.file_list = g_slist_prepend(sysenv.file_list, file_data);

/* supported file type */
file_data = (struct file_pak *) g_malloc(sizeof(struct file_pak));
file_data->id = XTL;
file_data->group = XTL;
file_data->menu = TRUE;
file_data->label = g_strdup("XTL");
file_data->ext = NULL;
file_data->ext = g_slist_append(file_data->ext, "xtl");
file_data->write_file = write_xtl;
file_data->read_file = read_xtl;
sysenv.file_list = g_slist_prepend(sysenv.file_list, file_data);

/* supported file type */
file_data = (struct file_pak *) g_malloc(sizeof(struct file_pak));
file_data->id = XYZ;
file_data->group = XYZ;
file_data->menu = TRUE;
file_data->label = g_strdup("XYZ");
file_data->ext = NULL;
file_data->ext = g_slist_append(file_data->ext, "xyz");
file_data->write_file = write_xyz;
file_data->read_file = read_xyz;
sysenv.file_list = g_slist_prepend(sysenv.file_list, file_data);

/* supported file type */
file_data = (struct file_pak *) g_malloc(sizeof(struct file_pak));
file_data->id = BABEL;
file_data->group = DATA;
file_data->menu = TRUE;
file_data->label = g_strdup("Filterable types");
file_data->ext = NULL;
file_data->write_file = NULL;
file_data->read_file = NULL;
sysenv.file_list = g_slist_prepend(sysenv.file_list, file_data);

/* supported file type */
file_data = (struct file_pak *) g_malloc(sizeof(struct file_pak));
file_data->id = AUTO;
file_data->group = BABEL;
file_data->menu = TRUE;
file_data->label = g_strdup("auto select filter");
file_data->ext = NULL;
file_data->write_file = NULL;
file_data->read_file = NULL;
sysenv.file_list = g_slist_prepend(sysenv.file_list, file_data);

/* supported file type */
file_data = (struct file_pak *) g_malloc(sizeof(struct file_pak));
file_data->id = PDB;
file_data->group = PDB;
file_data->menu = TRUE;
file_data->label = g_strdup("PDB");
file_data->ext = NULL;
file_data->ext = g_slist_append(file_data->ext, "pdb");
file_data->write_file = NULL;
file_data->read_file = read_pdb;
sysenv.file_list = g_slist_prepend(sysenv.file_list, file_data);

/* supported file type */
file_data = (struct file_pak *) g_malloc(sizeof(struct file_pak));
file_data->id = GAMESS;
file_data->group = GAMESS;
file_data->menu = TRUE;
file_data->label = g_strdup("GAMESS");
file_data->ext = NULL;
file_data->ext = g_slist_append(file_data->ext, "inp");
file_data->write_file = write_gms;
file_data->read_file = read_gms;
sysenv.file_list = g_slist_prepend(sysenv.file_list, file_data);

/* supported file type */
file_data = (struct file_pak *) g_malloc(sizeof(struct file_pak));
file_data->id = GAMESS_OUT;
file_data->group = GAMESS;
file_data->menu = FALSE;
file_data->label = g_strdup("GAMESS Output");
file_data->ext = NULL;
file_data->ext = g_slist_prepend(file_data->ext, "gmout");
file_data->ext = g_slist_prepend(file_data->ext, "gmot");
file_data->write_file = NULL;
file_data->read_file = read_gms_out;
sysenv.file_list = g_slist_prepend(sysenv.file_list, file_data);

/* supported file type */
file_data = (struct file_pak *) g_malloc(sizeof(struct file_pak));
file_data->id = DIFFAX_INP;
file_data->group = DIFFAX_INP;
file_data->menu = TRUE;
file_data->label = g_strdup("DIFFaX");
file_data->ext = NULL;
file_data->ext = g_slist_prepend(file_data->ext, "dfx");
file_data->write_file = write_diffax;
file_data->read_file = read_diffax;
sysenv.file_list = g_slist_prepend(sysenv.file_list, file_data);

/* supported file type */
file_data = (struct file_pak *) g_malloc(sizeof(struct file_pak));
file_data->id = XML;
file_data->group = XML;
file_data->menu = TRUE;
file_data->label = g_strdup("XML");
file_data->ext = NULL;
file_data->ext = g_slist_prepend(file_data->ext, "xml");
file_data->write_file = write_xml;
file_data->read_file = read_xml;
sysenv.file_list = g_slist_prepend(sysenv.file_list, file_data);

/* supported file type */
file_data = (struct file_pak *) g_malloc(sizeof(struct file_pak));
file_data->id = ABINIT_OUT;
file_data->group = ABINIT;
file_data->menu = FALSE;
file_data->label = g_strdup("SIESTA Output");
file_data->ext = NULL;
file_data->ext = g_slist_prepend(file_data->ext, "about");
file_data->ext = g_slist_prepend(file_data->ext, "abot");
file_data->write_file = NULL;
file_data->read_file = read_about;
sysenv.file_list = g_slist_prepend(sysenv.file_list, file_data);

sysenv.file_list = g_slist_reverse(sysenv.file_list);
}

/************************/
/* current version info */
/************************/
void gdis_blurb(FILE *fp)
{
fprintf(fp, "Created by GDIS version %4.2f.%d\n", VERSION, PATCH);
}

/*********************************************/
/* alphabetically sort the directory listing */
/*********************************************/
gint alpha_slist_sort(gpointer ptr1, gpointer ptr2)
{
return(g_ascii_strcasecmp((gchar *) ptr1, (gchar *) ptr2));
}

/*********************************************/
/* an all-platform directory listing routine */
/*********************************************/
GSList *get_dir_list(const char *path, gint sort)
{
const gchar *name;
GDir *dir;
GSList *files=NULL;

/* ensure we can go up a directory */
files = g_slist_prepend(files, "..");

dir = g_dir_open(path, 0, NULL);
do
  {
  name = g_dir_read_name(dir);
  if (name)
    files = g_slist_prepend(files, g_strdup(name));
  }
while (name);
g_dir_close(dir);

if (sort)
  files = g_slist_sort(files, (gpointer) alpha_slist_sort);

return(files);
}

/************************************************/
/* routine to determine if a file is recognized */
/************************************************/
/* returns pointer to file info if found, NULL otherwise */
#define DEBUG_GET_FILE_INFO 0
struct file_pak *get_file_info(const gpointer *ptr, gint type)
{
gint code=-1;
gchar *text=NULL, *ext;
GSList *file, *ext_list;
struct file_pak *file_data;

/* checks */
g_return_val_if_fail(ptr != NULL, NULL);

/* init for search */
switch(type)
  {
  case BY_LABEL:
    text = g_strdup((gchar *) ptr);
#if DEBUG_GET_FILE_INFO
printf("Searching for type [%s]\n", text);
#endif
    break;

  case BY_EXTENSION:
/* get file extension (if any) as separate string for matching purposes */
    ext = find_char((gchar *) ptr, '.', LAST);
    if (!ext)
      return(NULL);
    else
      text = g_strdup(++ext); /* skip the '.' */
#if DEBUG_GET_FILE_INFO
printf("Searching for extension [%s]\n", text);
#endif
    break;

  case BY_FILE_ID:
    code = GPOINTER_TO_INT(ptr);
#if DEBUG_GET_FILE_INFO
printf("Searching for code [%d]\n", code);
#endif
    break;
  }

/* search */
file = sysenv.file_list;
while(file != NULL)
  {
  file_data = (struct file_pak *) file->data;

  switch(type)
    {
/* compare to all extensions in list */
    case BY_EXTENSION: 
/* go through all extensions listed under this file type */
      ext_list = file_data->ext;
      while(ext_list != NULL)
        {
        ext = (gchar *) ext_list->data;
        if (strlen(text) == strlen(ext))
          if (g_ascii_strcasecmp(text, ext) == 0)
            {
#if DEBUG_GET_FILE_INFO
printf("Matched: %s\n", file_data->label);
#endif
            return(file_data);
            }
        ext_list = g_slist_next(ext_list);
        }
      break;

/* compare with label */
    case BY_LABEL:
      if (strlen(text) != strlen(file_data->label))
        break;
      if (g_ascii_strcasecmp(text, file_data->label) == 0)
        {
#if DEBUG_GET_FILE_INFO
printf("Matched: %s\n", file_data->label);
#endif
        return(file_data);
        }
      break;

    case BY_FILE_ID:
      if (code == file_data->id)
        {
#if DEBUG_GET_FILE_INFO
printf("Matched: %s\n", file_data->label);
#endif
        return(file_data);
        }
      break;


    default:
      printf("get_file_info() error: bad search type.\n");
    }

  file = g_slist_next(file);
  }

return(NULL);
}

/*******************************************************/
/* get an unused BASENAME (of supplied extension type) */
/*******************************************************/
/* TODO - supply a basename+ext & this inserts _? until new? */
#define DEBUG_GUN 0
gchar *gun(const gchar *ext)
{
gint i;
gchar *name;
GString *filename;
FILE *fp;

/* seek a file name that doesn't exist (avoid background overwriting) */
filename = g_string_new(NULL);
i=0;
do
  {
  g_string_sprintf(filename,"dummy_%d.%s", i, ext);
#if DEBUG_GUN
printf("testing: %s\n",filename->str);
#endif
  i++;
  }
while (g_file_test(filename->str, G_FILE_TEST_EXISTS));

/* create the file to prevent another process from taking it */
/* b4 the current caller gets around to writing anything to it */
fp = fopen(filename->str, "wt");
if (fp)
  {
  fprintf(fp, "locked.\n");
  fclose(fp);
  }
else
  {
  printf("Fatal error in gun()\n");
  return(NULL);
  }

name = g_strdup(filename->str);
g_string_free(filename, TRUE);

#if DEBUG_GUN
printf("using base: %s\n", name);
#endif

return(name);
}

/**************************************************************/
/* correct numbers in binary files with reverse byte ordering */
/**************************************************************/
void swap_bytes(void *ptr, const gint size)
{
gint i,j;
gchar tmp;

/*
printf("start: ");
for (i=0 ; i<size ; i++)
  printf("%d ", *((char *)(ptr+i)));
printf("\n");
*/
j=size-1;
for (i=0 ; i<size/2 ; i++)
  {
  tmp = *((gchar *)(ptr+j));
  *((gchar *)(ptr+j)) = *((gchar *)(ptr+i));
  *((gchar *)(ptr+i)) = tmp;
  j--;
  }
/*
printf(" stop: ");
for (i=0 ; i<size ; i++)
  printf("%d ", *((char *)(ptr+i)));
printf("\n");
*/
}

/**************************************/
/* get a non trivial line from a file */
/**************************************/
gint fgetline(FILE *fp, gchar *line)
{
gint i, linlen;

for(;;)
  {
/* get a line */
  if (fgets(line, LINELEN/2, fp) == NULL)
    return(1);
  linlen = strlen(line);
/* only treated as data if not a comment and non empty */
  if (line[0] != '#' && linlen)
    {
/* ampersand concatenation */
/* TODO - extra var in fgetline() (eg mode) that requests this */
/* TODO - handle multiple line concatenation */
    for (i=linlen ; i-- ; )
      {
      if (line[i] == '&')
        {
        if (fgets(&line[linlen], LINELEN/2, fp) == NULL)
          return(1);
        break;
        }
      }
    break;
    }
  }

/* all clear */
return(0);
}

/*****************************************/
/* read in and return a line of any size */
/*****************************************/
/* TODO - replacement routine for fgetline() */
gchar *file_read_line(FILE *fp)
{
gchar c, *line;
GString *buff;

/* checks */
g_assert(fp != NULL);
c = fgetc(fp);
if (c == EOF)
  return(NULL);

/* read single chars into an expandable buffer */
buff = g_string_new(NULL);
while (c != EOF)
  {
  g_string_append_c(buff, c);

  if (c == '\n')
    break;

  c = fgetc(fp);
  }

/* create a properly terminated line of text */
g_string_append_c(buff, '\0');
line = buff->str;

/* free the GString, but not the text */
g_string_free(buff, FALSE);

return(line);
}

/********************/
/* BABEL conversion */
/********************/
gint read_using_babel(gchar *inpfile, struct model_pak *data)
{
gint status, type;
gchar *inp, *out, *outfile;
GString *txt; 
struct file_pak *file_data;

/* checks */
g_return_val_if_fail(data != NULL, 1);
g_return_val_if_fail(inpfile != NULL, 1);

if (!sysenv.babel_path)
  {
  show_text(ERROR, "Sorry, you do not appear to have babel installed.\n");
  return(2);
  }

/* new - guess filter type if none specified */
if (sysenv.babel_type == AUTO)
  {
  file_data = get_file_info((gpointer *) inpfile, BY_EXTENSION);
  if (!file_data)
    return(3);
  type = file_data->id;
  }
else
  type = sysenv.babel_type;

switch (type)
  {
/* FIXME - what about the periodic data??? */
  case CSSR:
    inp = g_strdup("cssr");
    break;
  case FDAT:
    inp = g_strdup("fdat");
    break;
  case GAMOUT:
    inp = g_strdup("gamout");
    break;
  case MOPAC_OUT:
    inp = g_strdup("mopout");
    break;
  case PDB:
    inp = g_strdup("pdb");
    break;
  default:
    printf("babel_convert(): unknown file type.\n");
    return(4);
  }

/* set up new filename */
out = g_strdup("xyz");
txt = g_string_new(NULL);
g_string_assign(txt,inpfile);
g_string_append(txt,".xyz");
outfile = g_strdup(txt->str);
/* convert & load via standard routine */
show_text(STANDARD, "Converting with babel...\n");
g_string_sprintf(txt,"%s -i%s %s -o%s %s >& /dev/null",
                      sysenv.babel_path,
                      inp, inpfile, out, outfile);
system(txt->str);

/* pass to xyz load routine */
status = read_xyz(outfile, data);

/* free & exit */
g_string_free(txt, TRUE);
g_free(inp);
g_free(out);
g_free(outfile);
return(status);
}

/************************************************/
/* read in a raw frame (animation and analysis) */
/************************************************/
gint read_raw_frame(FILE *fp, gint n, struct model_pak *model)
{
gint i, flag=99;
gchar *filename;
fpos_t *offset;
FILE *fp2;

g_assert(fp != NULL);
g_assert(model != NULL);

/* position the file pointer */
if (model->id != GULP)
  {
  offset = g_list_nth_data(model->frame_list, n);
  if (!offset)
    return(1);
  if (fsetpos(fp, offset))
    {
    printf("Error positioning file pointer.\n");
    return(2);
    }
  }

/* perform the read */
switch(model->id)
  {
  case BIOSYM:
    flag = read_arc_frame(fp, model);
    break;

  case GULP:
/* FIXME - trj files resist every attempt at random access... why??? */
/* FIXME - ugly hack to get the nth frame */
    filename = g_strdup_printf("%s/%s", sysenv.cwd, model->gulp.trj_file);
    fp2 = fopen(filename, "r");
    if (!fp2)
      {
      printf("Failed to open: %s\n", filename);
      g_free(filename);
      break;
      }
    read_trj_header(fp2, model);
    for (i=0 ; i<n-1 ; i++)
      read_trj_frame(fp2, model, FALSE);
    read_trj_frame(fp2, model, TRUE);
    fclose(fp2);
    g_free(filename);
    flag = 0;
    break;

  case SIESTA_OUT:
    flag = read_sout_frame(fp, model);
    break;

  case GAMESS_OUT:
    flag = read_gms_out_frame(fp, model);
    break;

  case ABINIT_OUT:
    flag = read_about_frame(fp, model);
    break;
    
  case PDB:
    flag = read_pdb_frame(fp, model);
    break;
  }

return(flag);
}

/***********************************/
/* handler for normal file loading */
/***********************************/
#define DEBUG_FILE_LOAD 0
void file_load(gchar *filename, struct model_pak *mdata)
{
gint i, j;
gint flag, model, status=-1;
gchar *fullname;
struct model_pak *data;
struct file_pak *file_data;

/* get the model type */
file_data = get_file_info((gpointer *) filename, BY_EXTENSION);
if (!file_data)
  return;

#if DEBUG_FILE_LOAD
printf("Using file load routine for: [%s] files.\n", file_data->label);
#endif

if (mdata == NULL)
  {
/* get the new model number */
  model = sysenv.num_models;
  data = model_ptr(model, ASSIGN);
  if (!data)
    {
    show_text(ERROR, "Model memory allocation failed.\n");
    return;
    }
  }
else
  {
  data = mdata;
  model = mdata->number;
  }
  
/* group babel types together for this */
if (file_data->group == BABEL)
  data->id = file_data->group;
else
  data->id = file_data->id;

/* NEW - cleaner file read/write */
if (file_data->read_file)
  {
  fullname =  g_build_filename(sysenv.cwd, filename, NULL);
  status = file_data->read_file(fullname, data);
  g_free(fullname);
  }
else
  show_text(ERROR, "No read routine for this type. ");

/* check for successful file load */
if (status)
  {
  show_text(ERROR, "Load failed.\n");
#if DEBUG_FILE_LOAD
  printf("Load failed, error code: %d\n",status);
#endif

  model_ptr(model, RELEASE);
  canvas_shuffle();
  }
else
  {
/* we don't know how many new models were loaded so */
/* scan through them all & check for initialization */
  for (i=0 ; i<sysenv.num_models ; i++)
    {
    data = model_ptr(i, RECALL);

/* skip if already on the tree */
    if (data->grafted)
      continue;
/* surfaces are always conv by default */
    if (data->periodic == 2)
      data->gulp.method = CONV;
/* not on tree - must have just been loaded */
    new_tree_item(data->number, APPEND);

/* create gulp supercells */
    flag=0;
    for (j=0 ; j<3 ; j++)
      {
      if (data->gulp.super[j] > 1)
        {
        data->image_limit[2*j+1] = data->gulp.super[j];
        flag++;
        }
      }
    if (flag)
      {
      update_images(CREATE, data);
      make_supercell(NULL, data);
      }
    }
  }

/* finished - only now we destroy the file dialog */
close_dialog_type(FILE_SELECT);
}

/****************/
/* load dialog */
/***************/
void file_load_dialog(void)
{
file_dialog("Load file", NULL, (gpointer) file_load, sysenv.file_type);
}

/**********************************/
/* save file with name */
/**********************************/
gint file_save_as(gchar *filename, struct model_pak *model)
{
gint id, ret;
gchar *name;
struct file_pak *file_data;

/* setup & checks */
if (!model)
  model = model_ptr(sysenv.active, RECALL);

if (!filename || !model)
  return(1);

file_data = get_file_info((gpointer *) filename, BY_EXTENSION);
if (!file_data)
  return(2);

id = file_data->id;

/* overwrite the old filenames with the new */
strcpy(model->filename, filename);
g_free(model->basename);
model->basename = strdup_basename(filename);

/* file info indicates routine to call */
if (file_data->write_file == NULL)
  {
  printf("No write routine for this type.\n");
  return(3);
  }
else
  {
/* build the full name */
  name = g_build_filename(sysenv.cwd, model->filename, NULL);
  ret = file_data->write_file(name, model);
  g_free(name);
  }

/* update */
if (ret)
  show_text(ERROR, "Save failed.\n");
else
  {
  show_text(STANDARD, "Saved model.\n");
  close_dialog_type(FILE_SELECT);
  new_tree_item(model->number, REPLACE);
  redraw_canvas(SINGLE);
  }

return(ret);
}

/*************************************/
/* save active model using same name */
/*************************************/
void file_save(void)
{
gchar *name;
struct model_pak *model;

model = model_ptr(sysenv.active, RECALL);
if (!model)
  return;

/* strip the path, as file_save_as() prepends the cwd */
name = g_path_get_basename(model->filename);
file_save_as(name, model);
g_free(name);
}

/******************/
/* save as dialog */
/******************/
void file_save_dialog(void)
{
gchar *text=NULL;
struct model_pak *model;

model = model_ptr(sysenv.active, RECALL);
if (model)
  text = model->basename;

file_dialog("File save", text, (gpointer) file_save_as, sysenv.file_type);
}

