/*
Copyright (C) 2003 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include <math.h>
#include <stdio.h>
#include <string.h>
#include <strings.h>
#include <ctype.h>
#include <stdlib.h>
#include <time.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>

#ifdef __sun
#include <sys/dirent.h>
#else
#ifdef __WIN32
#include <dirent.h>
#else
#include <sys/dir.h>
#include <sys/param.h>
#endif
#endif

#include "gdis.h"
#include "file.h"
#include "parse.h"
#include "morph.h"
#include "matrix.h"
#include "space.h"
#include "surface.h"
#include "gtkshorts.h"
#include "interface.h"


#define DEBUG_MORE 0
#define MAX_KEYS 15

/* main structures */
extern struct sysenv_pak sysenv;
extern struct elem_pak elements[];

/************************/
/* save morphology data */
/************************/
gint write_gmf(gchar *filename, struct model_pak *data)
{
GSList *plist, *slist;
struct plane_pak *pdata;
struct shift_pak *sdata;
FILE *fp;

g_return_val_if_fail(data != NULL, 1);
/* TODO - flag visible / option to save or not? */

/* open for saving */
fp = fopen(filename,"wt");
if (!fp)
  return(2);

/* header */
fprintf(fp, "\n title: ");
gdis_blurb(fp);
fprintf(fp, "  name: %s\n", data->basename);
fprintf(fp, " space: %s\n", data->sginfo.spacename);
fprintf(fp, "  cell: %f %f %f  %f %f %f\n",
             data->pbc[0], data->pbc[1], data->pbc[2],
             180.0*data->pbc[3]/PI, 180.0*data->pbc[4]/PI, 180.0*data->pbc[5]/PI);
/* default display type */
switch(data->morph_type)
  {
  case EQUIL_UN:
    fprintf(fp, " morph: unrelaxed equilibrium\n\n");
    break;
  case EQUIL_RE:
    fprintf(fp, " morph: relaxed equilibrium\n\n");
    break;
  case GROWTH_UN:
    fprintf(fp, " morph: unrelaxed growth\n\n");
    break;
  case GROWTH_RE:
    fprintf(fp, " morph: relaxed growth\n\n");
    break;
  default:
    fprintf(fp, " morph: Dhkl based\n\n");
    break;
  }

/* print out the planes */
for (plist=data->planes ; plist ; plist=g_slist_next(plist))
  {
  pdata = (struct plane_pak *) plist->data;
  if (!pdata->primary || !pdata->present)
    continue;

  fprintf(fp,"miller:  %3d %3d %3d\n", 
              pdata->index[0], pdata->index[1], pdata->index[2]);

  for (slist=pdata->shifts ; slist ; slist=g_slist_next(slist)) 
    {
    sdata = (struct shift_pak *) slist->data;
  
    fprintf(fp, " %8.6f  %2d %2d  %10.4f %10.4f %10.4f %10.4f  %g\n",
            sdata->shift, sdata->region[0], sdata->region[1], 
            sdata->esurf[0], sdata->eatt[0],
            sdata->esurf[1], sdata->eatt[1], sdata->gnorm);
    }
  }

fclose(fp);
return(0);
}

/******************************************/
/* load miller & normal lengths from file */
/******************************************/
/* FIXME - when importing planes, this can mess up the periodic bond calc */
#define DEBUG_LOAD_PLANES 0
gint load_planes(gchar *filename, struct model_pak *data)
{
gint i, h, k, l, num_tokens;
gint *sym_hkl;
gdouble m[3];
gchar **buff;
GSList *plist;
GSList *sym_equiv, *sym_list;
struct plane_pak *pdata=NULL, *pnew=NULL;
struct shift_pak *sdata=NULL;
FILE *fp;

fp = fopen(filename, "rt");
if (!fp)
  return(1);

/* get next line */
for (;;)
  {
  buff = get_tokenized_line(fp, &num_tokens);
  if (!buff)
    break;

/* NB: only update space/cell etc. data if this call did */
/* not originate from an import planes call */
  if (data->id == MORPH)
    {
/* cell parameters */
    if (g_ascii_strncasecmp(*buff,"cell",4) == 0)
      {
      if (num_tokens >= 7)
        {
        data->pbc[0] = str_to_float(*(buff+1));
        data->pbc[1] = str_to_float(*(buff+2));
        data->pbc[2] = str_to_float(*(buff+3));
        data->pbc[3] = PI*str_to_float(*(buff+4))/180.0;
        data->pbc[4] = PI*str_to_float(*(buff+5))/180.0;
        data->pbc[5] = PI*str_to_float(*(buff+6))/180.0;
/* compute direct & reciprocal lattices */
/* NB: enables fn_make_plane() to correctly compute Dhkl */
        make_latmat(data);
        }
      else
        printf("load_planes() error: bad cell line.\n");
      }

/* space group */
    if (g_ascii_strncasecmp(*buff,"space",5) == 0)
      {
      g_free(data->sginfo.spacename);
      data->sginfo.spacename = g_strjoinv(" ", buff+1);
      genpos(data);
      }

/* default morphology type */
    if (g_ascii_strncasecmp(*buff, "morph", 5) == 0)
      {
      if (num_tokens >= 3)
        {
        if (g_ascii_strncasecmp(*(buff+1), "unrelaxed", 9) == 0)
          {
          if (g_ascii_strncasecmp(*(buff+2), "equil", 5) == 0)
            data->morph_type = EQUIL_UN;
          if (g_ascii_strncasecmp(*(buff+2), "growth", 6) == 0)
            data->morph_type = GROWTH_UN;
          }
  
        if (g_ascii_strncasecmp(*(buff+1), "relaxed", 7) == 0)
          {
          if (g_ascii_strncasecmp(*(buff+2), "equil", 5) == 0)
            data->morph_type = EQUIL_RE;
          if (g_ascii_strncasecmp(*(buff+2), "growth", 6) == 0)
            data->morph_type = GROWTH_RE;
          }
        }
      else
        printf("load_planes() error: bad type line.\n");
      }
    }

/* process miller line */
  if (g_ascii_strncasecmp(*buff,"miller",6) == 0)
    {
    if (num_tokens >= 4)
      {
      h = (gint) str_to_float(*(buff+1));
      k = (gint) str_to_float(*(buff+2));
      l = (gint) str_to_float(*(buff+3));
#if DEBUG_LOAD_PLANES
printf("creating plane: %d %d %d\n", h, k, l);
#endif
      VEC3SET(m, h, k, l);
      pdata = fn_seek_plane(m, data);
      if (!pdata)
        {
        pdata = fn_make_plane(m, data);
/* append to preserve order (eg import on existing plane set) */
        if (pdata)
          data->planes = g_slist_append(data->planes, pdata);
        }
      }
    }

/* potential data */
  if (num_tokens >= 8 && pdata)
    {
/* NB: use create_shift(), as it sets some important defaults */
    sdata = create_shift(0.0);
    if (sdata)
      {
      sdata->shift = str_to_float(*(buff+0));
      sdata->region[0] = str_to_float(*(buff+1));
      sdata->region[1] = str_to_float(*(buff+2));
      sdata->esurf[0] = str_to_float(*(buff+3));
      sdata->eatt[0] = str_to_float(*(buff+4));
      sdata->esurf[1] = str_to_float(*(buff+5));
      sdata->eatt[1] = str_to_float(*(buff+6));
      sdata->gnorm = str_to_float(*(buff+7));
#if DEBUG_LOAD_PLANES
printf("adding shift: %f\n", sdata->shift);
#endif
/* append to preserve order (eg import on existing plane set) */
      pdata->shifts = g_slist_append(pdata->shifts, sdata);
      }
    }
  g_strfreev(buff);
  }

/* compute dhkl's for the plane's list */
plist = data->planes;
while(plist != NULL)
  {
  pdata = (struct plane_pak *) plist->data;

/* create default shift if none found */
  if (!pdata->shifts)
    {
    sdata = create_shift(0.0);
    if (sdata)
      pdata->shifts = g_slist_append(pdata->shifts, sdata);
    }

/* get best energy for the plane */
  update_plane_energy(pdata, data);

  plist = g_slist_next(plist);
  i++;
  }

/* compute symmetry related faces */
i=0;
plist = data->planes;
while (plist != NULL)
  {
  pdata = (struct plane_pak *) plist->data;
/* once we reach the non-primary planes, we're done */
  if (!pdata->primary)
    break;

  sym_list = sym_equiv = get_facet_equiv(data, pdata->index);
/* skip the 1st, we've already assigned it */
  sym_list = g_slist_next(sym_list);
  while(sym_list != NULL)
    {
/* get the (normalized) hkl's */
    sym_hkl = (gint *) sym_list->data;

ARR3SET(m, sym_hkl);

    pnew = fn_make_plane(m, data);
    if (pnew)
      {
      pnew->primary = FALSE;
      pnew->dhkl = pdata->dhkl;
      pnew->esurf[0] = pdata->esurf[0];
      pnew->esurf[1] = pdata->esurf[1];
      pnew->eatt[0] = pdata->eatt[0];
      pnew->eatt[1] = pdata->eatt[1];

      data->planes = g_slist_append(data->planes, pnew);
      }
    sym_list = g_slist_next(sym_list);
    }

/* wipe the old list */
  g_slist_free(sym_equiv);
  plist = g_slist_next(plist);
  }

i=0;
plist = data->planes;
while (plist != NULL)
  {
  pdata = (struct plane_pak *) plist->data;
  plist = g_slist_next(plist);
  i++;
  }
data->num_planes = i;

fclose(fp);
return(0);
}

/***********************************************************/
/* determine if two facets in a given model are equivalent */
/***********************************************************/
/* f1 & f2 are vecs with the h k l values */
#define DEBUG_FACET_EQUIV 0
gint facet_equiv(struct model_pak *data, gint *f1, gint *f2)
{
gint i, index[3];
gdouble vec[3];

#if DEBUG_FACET_EQUIV
printf("[%d %d %d] and (%d %d %d) are ", f1[0], f1[1], f1[2], f2[0], f2[1], f2[2]);
#endif

if (data->sginfo.spacenum)
  {
  for (i=0 ; i<data->sginfo.order ; i++)
    {
    ARR3SET(vec, f1);
    vecmat(*(data->sginfo.matrix+i), vec);
    ARR3SET(index, vec);

    if (index[0] == *f2 && index[1] == *(f2+1) && index[2] == *(f2+2))
      {
#if DEBUG_FACET_EQUIV
printf("equivalent (symop %d).\n", i);
#endif
      return(TRUE);  
      }
    }
  }
else
  printf("No space group information.\n");

#if DEBUG_FACET_EQUIV
printf("not equivalent.\n");
#endif

return(FALSE);
}

/***********************************/
/* get a list of equivalent facets */
/***********************************/
#define DEBUG_GET_FACET_EQUIV 0
GSList *get_facet_equiv(struct model_pak *data, gint *f1)
{
gint i, flag, *f2, index[3];
gdouble vec[3];
GSList *flist=NULL, *list=NULL;

g_return_val_if_fail(data != NULL, NULL);
g_return_val_if_fail(f1 != NULL, NULL);

#if DEBUG_GET_FACET_EQUIV
printf("search for equiv faces to (%d %d %d)\n", f1[0], f1[1], f1[2]);
#endif

/* add the supplied face to the list (needed to eliminate repetitions) */
flist = g_slist_prepend(flist, (gpointer) f1);

if (data->sginfo.spacenum)
  {
/* skip 1st (trivial) op */
  for (i=1 ; i<data->sginfo.order ; i++)
    {
/* generate new symmetry related hkl */
    ARR3SET(vec, f1);
    vecmat(*(data->sginfo.matrix+i), vec);
    ARR3SET(index, vec);
#if DEBUG_GET_FACET_EQUIV
printf("candidate: (%d %d %d) ", index[0], index[1], index[2]);
#endif
/* add new hkl if not found in the list */
    flag = 0;
    list = flist;
    while (list != NULL)
      {
      f2 = (gint *) list->data;
      if (index[0] == f2[0] && index[1] == f2[1] && index[2] == f2[2])
        flag++;
      list = g_slist_next(list);
      }
    if (!flag)
      {
      f2 = g_malloc(3*sizeof(gint));
      ARR3SET(f2, index);
      flist = g_slist_prepend(flist, (gpointer) f2);
#if DEBUG_GET_FACET_EQUIV
printf("[YES]\n");
#endif
      }
#if DEBUG_GET_FACET_EQUIV
    else
      {
printf("[NO]\n");
      }
#endif
    }
  }
else
  printf("No space group information.\n");

flist = g_slist_reverse(flist);
return(flist);
}

/**************************************/
/* determine if systematically absent */
/**************************************/
/* CURRENT - my replacement for sginfo, so we don't need to */
/* depend on the raw sginfo data structure */
#define DEBUG_FACET_ABSENT 0
gint surf_sysabs(struct model_pak *data, gint h, gint k, gint l)
{
gint i, flag;
gint f1[3];
gdouble dp, dummy, vec[3], df[3];

#if DEBUG_FACET_ABSENT
printf("testing %d %d %d\n", h, k, l);
#endif

/* apply the symmetry matrices (skip identity) */
VEC3SET(f1, h, k, l);
for (i=1 ; i<data->sginfo.order ; i++)
  {
  VEC3SET(vec, h, k, l);
  vecmat(*(data->sginfo.matrix+i), vec);
  flag = 0;

/* test f1 . m = -f1 */
  ARR3SET(df, f1);
  ARR3ADD(df, vec);
  if (VEC3MAGSQ(df) < FRACTION_TOLERANCE)
    if (data->sginfo.centric)
      flag++;
 
/* test f1 . m = f1 */
  ARR3SET(df, f1);
  ARR3SUB(df, vec);
  if (VEC3MAGSQ(df) < FRACTION_TOLERANCE)
    flag++;

  if (flag)
    {
#if DEBUG_FACET_ABSENT
printf("symop [%d] satisfies 1st absence condition.\n", i);
P3MAT("matrix:", *(data->sginfo.matrix+i));
P3VEC("offset:", *(data->sginfo.offset+i));
#endif

/* test f1.t = non-integer */
    ARR3SET(vec, *(data->sginfo.offset+i));
    ARR3MUL(vec, f1);
/* abs. dotproduct */
    dp = fabs(vec[0] + vec[1] + vec[2]);

/* is it non-integer? */
    if (modf(dp, &dummy) > FRACTION_TOLERANCE)
      {
#if DEBUG_FACET_ABSENT
printf("symop [%d] [%f %f %f] satisfies 2nd absence condition.\n",
       i, *(*(data->sginfo.offset+i)+0), *(*(data->sginfo.offset+i)+1),
          *(*(data->sginfo.offset+i)+2));
printf("facet is extinct.\n");
#endif
/* TODO - return the order ie 1/4 -> 4 etc. */
      return(TRUE);
      }
    }
  }

#if DEBUG_FACET_ABSENT
printf(">>> Not absent\n");
#endif

return(FALSE);
}

/***********************************/
/* determine if a plane is visible */
/***********************************/
#define DEBUG_FACET 0
gint facet_visible(struct model_pak *data, struct plane_pak *plane)
{
gdouble a, norm[3], vec[3];

/* get (unrotated!) normal */
ARR3SET(norm, plane->norm);

/* rotate it */
vecmat(data->rotmat, norm);

#if DEBUG_FACET
printf("facet (%2d%2d%2d), norm: %5.2f,%5.2f,%5.2f    " 
,plane->index[0] ,plane->index[1] ,plane->index[2]
,norm[0],norm[1],norm[2]);
#endif
/* absolute viewing vector, determining if visible or not */
VEC3SET(vec, 0.0, 0.0, -1.0);

/* got correct facet normal - is it visible? */
a = via(norm,vec,3);
if (a < PI/2.0)
  {
#if DEBUG_FACET
printf("view: %5.2f,%5.2f,%5.2f (%5.1f) YES\n",vec[0],vec[1],vec[2],180.0*a/PI);
#endif
  return(1);
  }
/* else... */
#if DEBUG_FACET
printf("view: %5.2f,%5.2f,%5.2f (%5.1f) NO\n",vec[0],vec[1],vec[2],180.0*a/PI);
#endif
return(0);
}

/***********************************/
/* determine if a ridge is visible */
/***********************************/
gint ridge_visible(struct vertex_pak *v1, struct vertex_pak *v2, struct model_pak *model)
{
gint flag;
GSList *list;
struct plane_pak *plane;

for (list=model->planes ; list ; list=g_slist_next(list))
  {
  plane = (struct plane_pak *) list->data;
  flag = 0;
  if (plane->present && plane->visible)
    {
    if (g_slist_find(plane->vertices, v1))
      flag++;
    if (g_slist_find(plane->vertices, v2))
      flag++;
    }
  if (flag == 2)
    return(1);
  }
return(0);
}

/***********************/
/* Morphology creation */
/***********************/
#define DEBUG_READ_GMF 0
gint read_gmf(gchar *filename, struct model_pak *data)
{
gint model;
gdouble len;
gdouble norm[3];
GSList *plist;
struct plane_pak *plane;

/* checks */
g_return_val_if_fail(data != NULL, 1);
g_return_val_if_fail(filename != NULL, 1);

sysenv.active = model = data->number;
data->mode = FREE;
/* setup model parameters */
data->id = MORPH;
data->axes_type = OTHER;
data->hpr = TRUE;
data->morph_label = TRUE;
strcpy(data->filename, filename);
g_free(data->basename);
data->basename = strdup_basename(data->filename);

/* read from a gmf file */
if (load_planes(filename, data))
  {
  printf("File load failed, check %s.\n",data->filename);
  delete_model(model);
  return(1);
  }

/* convert miller indices to cartesian inequalities */
plist = data->planes;
while (plist != NULL)
  {
  plane = (struct plane_pak *) plist->data;
/* use reciprocal vectors to get the direct lattice plane normal */
  ARR3SET(norm, plane->m); 
  vecmat(data->rlatmat, norm);

/* normalize */
  len = VEC3MAG(norm);
  if (len > FRACTION_TOLERANCE)
    {
    VEC3MUL(norm, 1.0/len);
/* NB: cdd ineq. format needs the *negative* of the miller index */
    VEC3MUL(norm, -1.0);
    ARR3SET(plane->x, norm);
    }
  else
    {
/* make it so gdis safely ignores such a thing */
    printf("Warning: 0 length face found.\n");
    return(1);
    }
  plist = g_slist_next(plist);
  }

/* construct the hull (convert to H representation) */
data->num_vertices=0;
exec_cdd(data);

#if DEBUG_READ_GMF
printf("-------------------------\n");
printf("       Input planes: %d\n", data->num_planes);
printf("Calculated vertices: %d\n", data->num_vertices);
printf("-------------------------\n");
#endif

/* test success */
if (!g_slist_length(data->planes))
  {
  delete_model(model);
  return(1);
  }

/* vertices are cartesian - so a fudge is needed */
prep_model(data);

/* proper rmax scaling & visibility calc */
init_objs(REDO_COORDS, data);

return(0);
}

