/*
Copyright (C) 2003 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include <stdio.h>
#include <stdlib.h>

#include "gdis.h"
#include "dialog.h"

#include "pause.xpm"
#include "play.xpm"
#include "rewind.xpm"
#include "fastforward.xpm"
#include "stop.xpm"
#include "step_forward.xpm"
#include "step_backward.xpm"

extern struct sysenv_pak sysenv;

extern GtkWidget *window;

enum {AUTO_CHECK, AUTO_SPIN, AUTO_RANGE};

struct relation_pak 
{
/* true  - direct variable <-> widget relation */
/* false - relative offset (ie relation depends on active model) */
gboolean direct;

/* related object data */
struct model_pak *model;
gpointer variable;
gint offset;
GtkWidget *widget;
/* widget type (for updating) ie spin, check */
/* TODO - eliminate this using if GTK_IS_SPIN_BUTTON/CHECK_BUTTON macros */
gint type;
};

GSList *gtksh_relation_list=NULL;

#define DEBUG_RELATION 0

/**************************************/
/* auto update widget handling events */
/**************************************/
void gtksh_relation_destroy(GtkWidget *w, gpointer dummy)
{
GSList *list;
struct relation_pak *relation;

#if DEBUG_RELATION
printf("destroying widget (%p)", w);
#endif

/* find appropriate relation */
list = gtksh_relation_list;
while (list)
  {
  relation = (struct relation_pak *) list->data;
  list = g_slist_next(list);

/* free associated data */
  if (relation->widget == w)
    {
#if DEBUG_RELATION
printf(" : relation (%p)", relation);
#endif
    g_free(relation);
    gtksh_relation_list = g_slist_remove(gtksh_relation_list, relation);
    }
  }
#if DEBUG_RELATION
printf("\n");
#endif
}

/***************************************************/
/* set a value according to supplied widget status */
/***************************************************/
void gtksh_relation_set_value(GtkWidget *w, struct model_pak *model)
{
gpointer value;
GSList *list;
struct relation_pak *relation;

g_assert(w != NULL);

/* if no model supplied, use active model */
if (!model)
  model = model_ptr(sysenv.active, RECALL);

/* find appropriate relation */
list = gtksh_relation_list;
while (list)
  {
  relation = (struct relation_pak *) list->data;
  list = g_slist_next(list);

  if (relation->widget != w)
    continue;

  if (relation->direct)
    value = relation->variable;
  else
    {
    g_assert(model != NULL);
    value = (gpointer) model + relation->offset;
    }

/* update variable associated with the widget */
  switch (relation->type)
    {
    case AUTO_CHECK:
#if DEBUG_RELATION
printf("model %p : relation %p : setting variable to %d\n", 
       model, relation, gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w)));
#endif
      if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w)))
        *((gint *) value) = TRUE;
      else
        *((gint *) value) = FALSE;
      break;

    case AUTO_SPIN:
      *((gdouble *) value) = SPIN_FVAL(GTK_SPIN_BUTTON(w));
      break;

    case AUTO_RANGE:
      *((gint *) value) = gtk_range_get_value(GTK_RANGE(w));
      break;
    }
  }
}

/*****************************/
/* called for a model switch */
/*****************************/
/* ie sets all widget states according to related variable */
/* NB: pass NULL to get everything updated */
void gtksh_relation_update(struct model_pak *model)
{
GSList *list;
struct relation_pak *relation;
gpointer value;

/* loop over all existing relations */
for (list=gtksh_relation_list ; list ; list=g_slist_next(list))
  {
  relation = (struct relation_pak *) list->data;

/* update test */
  if (relation->direct)
    {
    if (model)
      if (model != relation->model)
        continue;
    value = relation->variable;
    }
  else
    {
    if (model)
      value = (gpointer) model + relation->offset;
    else
      continue;
    }

/* synchronize widget and variable */
  switch (relation->type)
    {
    case AUTO_CHECK:
#if DEBUG_RELATION
printf("model %p : relation %p : setting widget to %d\n", model, relation,  *((gint *) value));
#endif
      if (*((gint *) value))
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(relation->widget),
                                     TRUE);
      else
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(relation->widget),
                                     FALSE);
      break;

    case AUTO_SPIN:
      gtk_spin_button_set_value(GTK_SPIN_BUTTON(relation->widget),
                                *((gdouble *) value));
      break;

    case AUTO_RANGE:
      gtk_range_set_value(GTK_RANGE(relation->widget), *((gint *) value));
      break;
    }
  }
}

/**************************************************/
/* create a new variable to widget correspondance */
/**************************************************/
void
gtksh_relation_submit(GtkWidget       *widget,
                      gpointer          value,
                      gboolean         direct,
                      struct model_pak *model)
{
gint type = -1;
struct relation_pak *relation;

/* NEW - auto type determination */
if (GTK_IS_CHECK_BUTTON(widget))
  type = AUTO_CHECK;
if (GTK_IS_SPIN_BUTTON(widget))
  type = AUTO_SPIN;
if (GTK_IS_RANGE(widget))
  type = AUTO_RANGE;

g_assert(type != -1);

/* alloc and init relation */
relation = g_malloc(sizeof(struct relation_pak));
relation->direct = direct;
relation->type = type;
relation->offset = value - (gpointer) model;
relation->variable = value;
relation->model = model;
relation->widget = widget;

/* if this happens, sysenv.active is not correct  */
if (!direct)
  g_assert(relation->offset > 0);

#if DEBUG_RELATION
printf("submit: ");
printf("[%p type %d]", relation, type);
if (model)
  {
  if (direct)
    printf("[model %d][value %p]", model->number, value);
  else
    printf("[model %d][value %p][offset %d]", model->number, value, relation->offset);
  }
printf("\n");
#endif

switch (relation->type)
  {
  case AUTO_CHECK:
    if (*((gint *) value))
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(widget), TRUE);
    else
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(widget), FALSE);
    break;

  case AUTO_SPIN:
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(widget), *((gdouble *) value));
    break;

  case AUTO_RANGE:
    gtk_range_set_value(GTK_RANGE(widget), *((gint *) value));
    break;
  }

gtksh_relation_list = g_slist_prepend(gtksh_relation_list, relation);
}

/*****************************************/
/* convenience routine for check buttons */
/*****************************************/
GtkWidget *new_check_button(gchar *label, gpointer callback, gpointer arg,
                                            gint state, GtkWidget *box)
{
GtkWidget *button;

/* create, pack & show the button */
button = gtk_check_button_new_with_label(label);
gtk_box_pack_start(GTK_BOX(box), button, TRUE, TRUE, 0);
gtk_widget_show(button);

/* set the state (NB: before the callback is attached) */
if (state)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

/* attach the callback */
if (callback)
  g_signal_connect(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(callback), arg);
return(button);
}

/*********************************/
/* relative updated check button */
/*********************************/
/* NB: assumes switch_model() will call gtksh_relation_update() */
GtkWidget *gtksh_auto_check(gchar *label, gpointer cb, gpointer arg,
                            gint *state, GtkWidget *box)
{
GtkWidget *button;
struct model_pak *model;

model = model_ptr(sysenv.active, RECALL);
g_assert(model != NULL);

/* create the button */
button = gtk_check_button_new_with_label(label);
gtksh_relation_submit(button, state, FALSE, model);
if (box)
  gtk_box_pack_start(GTK_BOX(box), button, TRUE, TRUE, 0);

/* callback to set the variable to match the widget */
g_signal_connect(GTK_OBJECT(button), "clicked",
                 GTK_SIGNAL_FUNC(gtksh_relation_set_value), NULL);

/* callback to do (user defined) update tasks */
if (cb)
  g_signal_connect_after(GTK_OBJECT(button), "clicked", cb, arg);

/* callback to remove the variable <-> widget relation */
g_signal_connect(GTK_OBJECT(button), "destroy",
                 GTK_SIGNAL_FUNC(gtksh_relation_destroy), NULL);

return(button);
}

/*********************************/
/* directly updated check button */
/*********************************/
/* NB: assumes switch_model() will call gtksh_relation_update() */
GtkWidget *gtksh_direct_check(gchar *label, gint *state,
                              gpointer cb, gpointer arg,
                              GtkWidget *box)
{
GtkWidget *button;
struct model_pak *model;

model = model_ptr(sysenv.active, RECALL);

/* create the button */
button = gtk_check_button_new_with_label(label);
gtksh_relation_submit(button, state, TRUE, model);
if (box)
  gtk_box_pack_start(GTK_BOX(box), button, TRUE, TRUE, 0);


/* callback to set the variable to match the widget */
g_signal_connect(GTK_OBJECT(button), "clicked",
                 GTK_SIGNAL_FUNC(gtksh_relation_set_value), model);

/* callback to remove the variable <-> widget relation */
g_signal_connect(GTK_OBJECT(button), "destroy",
                 GTK_SIGNAL_FUNC(gtksh_relation_destroy), NULL);

/* callback to do (user defined) update tasks */
if (cb)
  g_signal_connect_after(GTK_OBJECT(button), "clicked", cb, arg);

return(button);
}

/****************************/
/* a simple labelled button */
/****************************/
GtkWidget *gtksh_button(gchar *txt, gpointer cb, gpointer arg, GtkWidget *w, gint mask)
{
gint fill1, fill2;
GtkWidget *button;

/* create the button */
button = gtk_button_new_with_label(txt);

/* setup the packing requirements */
if (w)
  {
  fill1 = fill2 = FALSE;
  if (mask & 1)
    fill1 = TRUE;
  if (mask & 2)
    fill2 = TRUE;
  if (mask & 4)
    gtk_box_pack_end(GTK_BOX(w), button, fill1, fill2, 0);
  else
    gtk_box_pack_start(GTK_BOX(w), button, fill1, fill2, 0);
  }

/* attach the callback */
if (cb)
  g_signal_connect(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(cb), arg);

return(button);
}

/************************************/
/* text label plus an X push button */
/************************************/
void gtksh_button_x(gchar *text, gpointer cb, gpointer arg, GtkWidget *box)
{
GtkWidget *hbox, *button, *label;

/* create button */
button = gtk_button_new_with_label(" X ");

/* create label */
if (box)
  {
/* packing sub-widget */
  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(box), hbox, TRUE, TRUE, 0);
  if (text)
    {
    label = gtk_label_new(text);
    gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
    }
  gtk_box_pack_end(GTK_BOX(hbox), button, FALSE, FALSE, 0);
  }

if (cb)
  g_signal_connect(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(cb), arg);
}


/****************************************/
/* text label plus labelled push button */
/****************************************/
void gtksh_button_label(gchar *t1, gchar *t2, gpointer cb, gpointer arg, GtkWidget *box)
{
GtkWidget *hbox, *button, *label;

/* packing sub-widget */
hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(box), hbox, TRUE, TRUE, 0);

/* create label */
label = gtk_label_new(t1);
gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);

/* create button */
button = gtk_button_new_with_label(t2);
g_signal_connect_swapped(GTK_OBJECT(button), "clicked", 
                         GTK_SIGNAL_FUNC(cb), arg);
gtk_box_pack_end(GTK_BOX(hbox), button, FALSE, FALSE, 0);
}

/*****************************************/
/* convenience routine for radio buttons */
/*****************************************/
gint active=0, count=0, fill1=FALSE, fill2=FALSE;
GSList *group=NULL;
GtkWidget *box=NULL;

void new_radio_group(gint i, GtkWidget *pack_box, gint mask)
{
group = NULL;
active = i;
count = 0;
box = pack_box;

/* setup the packing requirements */
fill1 = fill2 = FALSE;
if (mask & 1)
  fill1 = TRUE;
if (mask & 2)
  fill2 = TRUE;
}

GtkWidget *add_radio_button(gchar *label, gpointer call, gpointer arg)
{
GtkWidget *button;

/*
printf("Adding button: %d (%d) to (%p)\n", count, active, group);
*/

/* better error checking - even call new_radio_group() ??? */
g_return_val_if_fail(box != NULL, NULL);

/* make a new button */
button = gtk_radio_button_new_with_label(group, label);

/* get the group (for the next button) */
/* NB: it is vital that this is ALWAYS done */
group = gtk_radio_button_group(GTK_RADIO_BUTTON(button));

/* attach the callback */
g_signal_connect_swapped(GTK_OBJECT(button), "pressed",
                         GTK_SIGNAL_FUNC(call), (gpointer) arg);

/* pack the button */
gtk_box_pack_start(GTK_BOX(box), button, fill1, fill2, 0);

/* set active? */
if (active == count++)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

return(button);
}

/************************************/
/* create a colour selection dialog */
/************************************/
GtkWidget *new_csd(gchar *title, gpointer callback)
{
GtkWidget *csd;

/* create colour selection dialog */
csd = gtk_color_selection_dialog_new(title);

/* setup callbacks */
g_signal_connect_swapped(G_OBJECT(GTK_COLOR_SELECTION_DIALOG(csd)->ok_button),
                        "clicked", 
                         GTK_SIGNAL_FUNC(callback),
                        (gpointer) GTK_COLOR_SELECTION_DIALOG(csd)->colorsel);

g_signal_connect_swapped(G_OBJECT(GTK_COLOR_SELECTION_DIALOG(csd)->cancel_button),
                        "clicked",
                         GTK_SIGNAL_FUNC(gtk_widget_destroy),
                        (gpointer) csd);

/* done */
gtk_widget_show(csd);
return(csd);
}

/******************************/
/* create a label + a spinner */
/******************************/
GtkWidget *new_spinner(gchar *text, gdouble min, gdouble max, gdouble step,
                       gpointer callback, gpointer arg, GtkWidget *box)
{
GtkWidget *hbox, *label, *spin;

spin = gtk_spin_button_new_with_range(min, max, step);

/* optional */
if (box)
  {
  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(box), hbox, TRUE, TRUE, 0);
  if (text)
    {
    label = gtk_label_new(text);
    gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
    }
  gtk_box_pack_end(GTK_BOX(hbox), spin, FALSE, FALSE, 0);
  }

if (callback)
  g_signal_connect(GTK_OBJECT(spin), "value-changed",
                   GTK_SIGNAL_FUNC(callback), arg);

return(spin);
}

/************************************************************/
/* as above, but automatically attaches spinner to callback */
/************************************************************/
GtkWidget *gtksh_new_spin(gchar *text, gdouble x0, gdouble x1, gdouble dx,
                          gpointer callback, GtkWidget *box)
{
GtkWidget *hbox, *label, *spin;

spin = gtk_spin_button_new_with_range(x0, x1, dx);
gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin), x0);

/* optional */
if (box)
  {
/* create the text/spinner layout */
  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(box), hbox, TRUE, TRUE, 0);
  if (text)
    {
    label = gtk_label_new(text);
    gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
    }
  gtk_box_pack_end(GTK_BOX(hbox), spin, FALSE, FALSE, 0);
  }
if (callback)
  g_signal_connect_after(GTK_OBJECT(spin), "value-changed",
                         GTK_SIGNAL_FUNC(callback), spin);
return(spin);
}

/*********************************/
/* automatically updated spinner */
/*********************************/
/* current model relative correlation between value and spinner */
/* NB: assumes switch_model() will call gtksh_relation_update() */
GtkWidget *gtksh_auto_spin(gchar *text, gdouble *value,
                           gdouble min, gdouble max, gdouble step,
                           gpointer callback, gpointer arg, GtkWidget *box)
{
GtkWidget *spin;
struct model_pak *model;

model = model_ptr(sysenv.active, RECALL);
g_assert(model != NULL);

/* create the text/spinner combo */
spin = new_spinner(text, min, max, step, callback, arg, box);

/* set up a relationship */
gtksh_relation_submit(spin, value, FALSE, model);

/* callback to set the variable to match the widget */
g_signal_connect(GTK_OBJECT(spin), "value-changed",
                (gpointer) gtksh_relation_set_value, NULL);

/* callback to remove the variable <-> widget relation */
g_signal_connect(GTK_OBJECT(spin), "destroy",
                 GTK_SIGNAL_FUNC(gtksh_relation_destroy), NULL);

return(spin);
}

/*********************************/
/* automatically updated spinner */
/*********************************/
/* direct correlation between value and spinner */
GtkWidget *gtksh_direct_spin(gchar *text, gdouble *value,
                             gdouble min, gdouble max, gdouble step,
                             gpointer callback, gpointer arg, GtkWidget *box)
{
GtkWidget *spin;
struct model_pak *model;

model = model_ptr(sysenv.active, RECALL);

/* create the text/spinner combo */
spin = new_spinner(text, min, max, step, NULL, NULL, box);

/* set up a relationship */
gtksh_relation_submit(spin, value, TRUE, model);

/* callback to set the variable to match the widget */
g_signal_connect(GTK_OBJECT(spin), "value-changed",
                 GTK_SIGNAL_FUNC(gtksh_relation_set_value), model);

/* callback to remove the variable <-> widget relation */
g_signal_connect(GTK_OBJECT(spin), "destroy",
                 GTK_SIGNAL_FUNC(gtksh_relation_destroy), NULL);

/* connect after, so all updates done before user callback is invoked */
if (callback)
  g_signal_connect_after(GTK_OBJECT(spin), "value-changed",
                         GTK_SIGNAL_FUNC(callback), arg);

return(spin);
}

/**************************/
/* auto update slider bar */
/**************************/
GtkWidget *gtksh_direct_hscale(gdouble min, gdouble max, gdouble step,
                               gpointer value, gpointer func, gpointer arg,
                                                            GtkWidget *box)
{
GtkWidget *hscale;
struct model_pak *model;

model = model_ptr(sysenv.active, RECALL);
g_assert(model != NULL);

hscale = gtk_hscale_new_with_range(min, max, step);
gtk_range_set_update_policy(GTK_RANGE(hscale), GTK_UPDATE_CONTINUOUS);

if (box)
  gtk_box_pack_start(GTK_BOX(box), hscale, TRUE, TRUE, 0);

gtksh_relation_submit(hscale, value, TRUE, model);

/* callback to set the variable to match the widget */
g_signal_connect(GTK_OBJECT(hscale), "value_changed",
                 GTK_SIGNAL_FUNC(gtksh_relation_set_value), model);

/* user callback */
if (func)
  g_signal_connect_after(GTK_OBJECT(hscale), "value_changed",
                         GTK_SIGNAL_FUNC(func), arg);

/* callback to remove the variable <-> widget relation */
g_signal_connect(GTK_OBJECT(hscale), "destroy",
                GTK_SIGNAL_FUNC(gtksh_relation_destroy), NULL);


return(hscale);
}

/*****************************/
/* create a stock GTK button */
/*****************************/
GtkWidget * 
gtksh_stock_button(const gchar *id, gpointer cb, gpointer arg, GtkWidget *box)
{
GtkWidget *button;

button = gtk_button_new_from_stock(id);

if (box)
  gtk_box_pack_start(GTK_BOX(box), button, FALSE, FALSE, 0);

g_signal_connect(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(cb), arg);

return(button);
}

/***************************************************/
/* create a button with a stock icon & custom text */
/***************************************************/
GtkWidget * 
gtksh_icon_button(const gchar *id, const gchar *text,
                  gpointer cb, gpointer arg,
                  GtkWidget *box)
{
gint stock = TRUE;
GtkWidget *hbox, *button, *label, *image=NULL;
GdkBitmap *mask;
GdkPixmap *pixmap;
GtkStyle *style;

/* button */
button = gtk_button_new();
hbox = gtk_hbox_new(FALSE, 4);
gtk_container_add(GTK_CONTAINER(button), hbox);

/* GDIS icons */
if (g_ascii_strncasecmp(id, "GDIS_PAUSE", 10) == 0)
  {
  style = gtk_widget_get_style(window);

  pixmap = gdk_pixmap_create_from_xpm_d(window->window, &mask, &style->white,
                                        pause_xpm);
  image = gtk_image_new_from_pixmap(pixmap, mask);

  stock = FALSE;
  }
if (g_ascii_strncasecmp(id, "GDIS_PLAY", 9) == 0)
  {
  style = gtk_widget_get_style(window);

  pixmap = gdk_pixmap_create_from_xpm_d(window->window, &mask, &style->white,
                                        play_xpm);
  image = gtk_image_new_from_pixmap(pixmap, mask);

  stock = FALSE;
  }
if (g_ascii_strncasecmp(id, "GDIS_REWIND", 11) == 0)
  {
  style = gtk_widget_get_style(window);

  pixmap = gdk_pixmap_create_from_xpm_d(window->window, &mask, &style->white,
                                        rewind_xpm);
  image = gtk_image_new_from_pixmap(pixmap, mask);

  stock = FALSE;
  }
if (g_ascii_strncasecmp(id, "GDIS_FASTFORWARD", 16) == 0)
  {
  style = gtk_widget_get_style(window);

  pixmap = gdk_pixmap_create_from_xpm_d(window->window, &mask, &style->white,
                                        fastforward_xpm);
  image = gtk_image_new_from_pixmap(pixmap, mask);

  stock = FALSE;
  }
if (g_ascii_strncasecmp(id, "GDIS_STOP", 9) == 0)
  {
  style = gtk_widget_get_style(window);

  pixmap = gdk_pixmap_create_from_xpm_d(window->window, &mask, &style->white,
                                        stop_xpm);
  image = gtk_image_new_from_pixmap(pixmap, mask);

  stock = FALSE;
  }
if (g_ascii_strncasecmp(id, "GDIS_STEP_FORWARD", 17) == 0)
  {
  style = gtk_widget_get_style(window);

  pixmap = gdk_pixmap_create_from_xpm_d(window->window, &mask, &style->white,
                                        step_forward_xpm);
  image = gtk_image_new_from_pixmap(pixmap, mask);

  stock = FALSE;
  }
if (g_ascii_strncasecmp(id, "GDIS_STEP_BACKWARD", 18) == 0)
  {
  style = gtk_widget_get_style(window);

  pixmap = gdk_pixmap_create_from_xpm_d(window->window, &mask, &style->white,
                                        step_backward_xpm);
  image = gtk_image_new_from_pixmap(pixmap, mask);

  stock = FALSE;
  }


/* standard GTK */
if (stock)
  image = gtk_image_new_from_stock(id, GTK_ICON_SIZE_BUTTON);

/* label dependent packing  */
if (text)
  {
  gtk_box_pack_start(GTK_BOX(hbox), image, FALSE, FALSE, 0);
  label = gtk_label_new(text);
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
  }
else
  {
  gtk_box_pack_start(GTK_BOX(hbox), image, TRUE, FALSE, 0);
  }

/* packing & callback */
if (box)
  gtk_box_pack_start(GTK_BOX(box), button, TRUE, TRUE, 0);

g_signal_connect(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(cb), arg);

return(button);
}

