/*
Copyright (C) 2003 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

/* irix */
#define _BSD_SIGNALS 1

#include <signal.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <unistd.h>
#include <sys/types.h>
#ifndef __WIN32
#include <sys/wait.h>
#endif
#include <gtk/gtkgl.h>

#include "gdis.h"
#include "file.h"
#include "parse.h"
#include "task.h"
#include "render.h"
#include "matrix.h"
#include "opengl.h"
#include "numeric.h"
#include "interface.h"

/* main data structures */
struct sysenv_pak sysenv;
struct elem_pak elements[MAX_ELEMENTS];

/********************************/
/* sfc hash table value cleanup */
/********************************/
void free_sfc(gpointer sfc_list)
{
free_slist((GSList *) sfc_list);
}

/******************/
/* system cleanup */
/******************/
void sys_free(void)
{
/* TODO - other things to free? */
g_hash_table_destroy(sysenv.sfc_table);
}

/**********************************/
/* read the init file if possible */
/**********************************/
gint read_gdisrc(void)
{
gint i, element_flag;
gchar line[LINELEN], **buff;
gdouble version;
FILE *fp;

/* attempt to open */
fp = fopen(sysenv.init_file, "r");

/* check for an old/bad gdisrc */
if (fp)
  {
/* scan the 1st line */
  fgetline(fp, line);
/* test for right format */
  buff = get_tokens(line, 2);
  if (g_ascii_strncasecmp(*buff,"gdis",4) == 0)
    {
/* test for right version */
    version = str_to_float(*(buff+1));
    sscanf(*(buff+1),"%lf",&version);
/* 0.75 was when the new element scheme was implemented */
    if (version < 0.75)
      return(1);
    }
  else
    return(1);
  g_strfreev(buff);
  }
else
  return(1);

/* read until EOF, or (failsafe) reached elements[] array allocation */
element_flag=0;
i=0;
while(!fgetline(fp,line))
  {
  buff = get_tokens(line, 11);

/* decide what to read */
  if (g_ascii_strncasecmp("size",*buff,4) == 0)
    {
    sysenv.width = (gint) str_to_float(*(buff+1));
    sysenv.height = (gint) str_to_float(*(buff+2));
    if (sysenv.width < MIN_WIDTH)
      sysenv.width = MIN_WIDTH;
    if (sysenv.height < MIN_HEIGHT)
      sysenv.height = MIN_HEIGHT;
    } 

/* model pane width */
  if (g_ascii_strncasecmp("pane",*buff,4) == 0)
    {
    if (strlen(*(buff+1)))
      sysenv.tree_width = (gint) str_to_float(*(buff+1));
    if (strlen(*(buff+2)))
      sysenv.tray_height = (gint) str_to_float(*(buff+2));
    }

/* povray executable */
  if (g_ascii_strncasecmp("povray_exe",*buff,10) == 0)
    {
    if (strlen(*(buff+1)))
      {
      g_free(sysenv.povray_exe);
      sysenv.povray_exe = g_strdup(*(buff+1));
      }
    }

/* auto unfrag? */
  if (g_ascii_strncasecmp("unfragment",*buff,10) == 0)
    {
    if (strlen(*(buff+1)))
      sysenv.unfragment = str_to_float(*(buff+1));
    }

/* animation creation executable */
  if (g_ascii_strncasecmp("convert_exe",*buff,11) == 0)
    {
    if (strlen(*(buff+1)))
      {
      g_free(sysenv.convert_exe);
      sysenv.convert_exe = g_strdup(*(buff+1));
      }
    }
/* image viewing executable */
  if (g_ascii_strncasecmp("viewer_exe",*buff,10) == 0)
    {
    if (strlen(*(buff+1)))
      {
      g_free(sysenv.viewer_exe);
      sysenv.viewer_exe = g_strdup(*(buff+1));
      }
    }
/* gulp executable */
  if (g_ascii_strncasecmp("gulp",*buff,4) == 0)
    {
    if (strlen(*(buff+1)))
      {
      g_free(sysenv.gulp_exe);
      sysenv.gulp_exe = g_strdup(*(buff+1));
      }
    }
/* file conversion executable */
  if (g_ascii_strncasecmp("babel_exe",*buff,9) == 0)
    {
    if (strlen(*(buff+1)))
      {
      g_free(sysenv.babel_exe);
      sysenv.babel_exe = g_strdup(*(buff+1));
      }
    }
/* gtk text font (NB: due to pango bug, name can't be only whitespace) */
  if (g_ascii_strncasecmp("gtk_font",*buff,8) == 0)
    if (strlen(*(buff+1)))
      strcpy(sysenv.gfont_name, g_strstrip(&line[9]));
/* OpenGL drawing font */
  if (g_ascii_strncasecmp("gl_font",*buff,7) == 0)
    if (strlen(*(buff+1)))
      strcpy(sysenv.dfont_name, g_strstrip(&line[8]));

/* atom properties box */
  if (g_ascii_strncasecmp("apb",*buff,3) == 0)
    if (strlen(*(buff+1)))
      sysenv.apb_on = (gint) str_to_float(*(buff+1));

/* rotation adjustment box */
  if (g_ascii_strncasecmp("rab",*buff,3) == 0)
    if (strlen(*(buff+1)))
      sysenv.rab_on = (gint) str_to_float(*(buff+1));

/* halo type */
  if (g_ascii_strncasecmp("halo",*buff,4) == 0)
    if (strlen(*(buff+1)))
      sysenv.render.halos = (gint) str_to_float(*(buff+1));

/* low quality rotation */
  if (g_ascii_strncasecmp("fast",*buff,4) == 0)
    if (strlen(*(buff+1)))
      sysenv.render.fast_rotation = (gint) str_to_float(*(buff+1));

  g_strfreev(buff);
  }

/* parse for elements */
rewind(fp);
read_elem_data(fp, MODIFIED);

return(0);
}

/*********************************************/
/* write setup & elements to the gdisrc file */
/*********************************************/
gint write_gdisrc(void)
{
FILE *fp;

fp = fopen(sysenv.init_file,"w");
if (!fp)
  {
  printf("Error: unable to create %s\n", sysenv.init_file);
  return(1);
  }

/* save final dimensions */
if (sysenv.mpane)
  if (GTK_IS_WIDGET(sysenv.mpane))
    sysenv.tree_width = GTK_WIDGET(sysenv.mpane)->allocation.width;
if (sysenv.tpane)
  if (GTK_IS_WIDGET(sysenv.tpane))
    sysenv.tray_height = GTK_WIDGET(sysenv.tpane)->allocation.height;

fprintf(fp,"gdis %f\n", VERSION);
fprintf(fp,"canvas %d\n", sysenv.canvas);
fprintf(fp,"size %d %d\n", sysenv.width,sysenv.height);
fprintf(fp,"pane %d %d\n", sysenv.tree_width, sysenv.tray_height);
fprintf(fp,"gtk_font %s\n", sysenv.gfont_name);
fprintf(fp,"gl_font %s\n", sysenv.dfont_name);
fprintf(fp,"apb %d\n", sysenv.apb_on);
fprintf(fp,"rab %d\n", sysenv.rab_on);
fprintf(fp,"unfragment %d\n", sysenv.unfragment);
fprintf(fp,"halos %d\n", sysenv.render.halos);
fprintf(fp,"fast %d\n", sysenv.render.fast_rotation);
fprintf(fp,"povray_exe %s\n", sysenv.povray_exe);
fprintf(fp,"convert_exe %s\n", sysenv.convert_exe);
fprintf(fp,"viewer_exe %s\n", sysenv.viewer_exe);
fprintf(fp,"gulp_exe %s\n", sysenv.gulp_exe);
fprintf(fp,"babel_exe %s\n", sysenv.babel_exe);

/* write the non-default element data */
write_elem_data(fp);
/* NEW - write the scattering factor coefficients */
write_sfc_data(fp);

fclose(fp);
return(0);
}

/**************/
/* main setup */
/**************/
#define DEBUG_SYS_INIT 0
void sys_init(gint argc, gchar *argv[])
{
gint i;
gchar *temp;
const gchar *ctemp;
struct light_pak *light;
FILE *fp;

/* top level structure initialization */
sysenv.num_tasks = 0;
sysenv.running_tasks = 0;
sysenv.max_running_tasks = 1;
sysenv.tasks = NULL;
sysenv.num_models = 0;
sysenv.stereo = FALSE;
sysenv.canvas_list = NULL;
sysenv.canvas_rows = 1;
sysenv.canvas_cols = 1;
sysenv.width = START_WIDTH;
sysenv.height = START_HEIGHT;
sysenv.tree_width = START_WIDTH/4;
sysenv.tray_height = 60;
sysenv.mpane = NULL;
sysenv.moving = FALSE;
sysenv.redraw = TRUE;
sysenv.apb_on = TRUE;
sysenv.rab_on = TRUE;
/* default to single model display */
sysenv.displayed[0] = -1;
sysenv.active = 0;
sysenv.canvas = TRUE;
sysenv.select_mode = CORE;
sysenv.unfragment = FALSE;
sysenv.calc_pbonds = TRUE;
VEC3SET(sysenv.angle, 0.0, 0.0, 0.0);

/* default masks */
sysenv.file_type = DATA;
sysenv.babel_type = AUTO;
sysenv.num_elements = sizeof(elements) / sizeof(struct elem_pak);
sysenv.elements = NULL;

/* rendering setup */
sysenv.render.type = BALL_STICK;
sysenv.render.width = 600;
sysenv.render.height = 600;
sysenv.render.vp_dist = 5.0;

/* default light */
light = g_malloc(sizeof(struct light_pak));
light->type = DIRECTIONAL;
VEC3SET(light->x, -1.0,  1.0, -1.0);
VEC3SET(light->colour, 1.0, 1.0, 1.0);
light->ambient = 0.2;
light->diffuse = 0.9;
light->specular = 0.9;
sysenv.render.light_list = g_slist_append(sysenv.render.light_list, light);
sysenv.render.num_lights = 1;

sysenv.render.perspective = FALSE;
sysenv.render.antialias = FALSE;
sysenv.render.wire_show_hidden = FALSE;
sysenv.render.fog = FALSE;
sysenv.render.wire_model = FALSE;
sysenv.render.wire_surface = FALSE;
sysenv.render.shadowless = FALSE;
sysenv.render.animate = FALSE;
sysenv.render.animate_type = ANIM_GIF;
sysenv.render.animate_file = g_strdup("movie");
sysenv.render.atype = FALSE;
sysenv.render.axes = FALSE;
sysenv.render.morph_finish = g_strdup("F_Glass4");
sysenv.render.ref_index = 2.5;
sysenv.render.delay = 20;
sysenv.render.sphere_quality = 2;
sysenv.render.cylinder_quality = 9;
sysenv.render.ribbon_quality = 10;
sysenv.render.halos = FALSE;
sysenv.render.fast_rotation = FALSE;

sysenv.render.ambience = 0.2;
sysenv.render.diffuse = 0.9;
sysenv.render.specular = 0.9;
sysenv.render.transmit = 0.9;
sysenv.render.ball_rad = 0.3;
sysenv.render.stick_rad = 0.1;
sysenv.render.stick_thickness = GTKGL_LINE_WIDTH;
sysenv.render.line_thickness = GTKGL_LINE_WIDTH;
sysenv.render.frame_thickness = GTKGL_LINE_WIDTH;
sysenv.render.geom_line_width = GTKGL_LINE_WIDTH;
sysenv.render.cpk_scale = 1.0;
sysenv.render.fog_density = 0.35;
sysenv.render.fog_start = 0.5;
sysenv.render.ribbon_curvature = 0.5;
sysenv.render.ribbon_thickness = 1.0;
sysenv.render.phonon_scaling = 4.0;
sysenv.render.ahl_strength = 0.7;
sysenv.render.ahl_size = 20;
sysenv.render.shl_strength = 0.8;
sysenv.render.shl_size = 20;
VEC3SET(sysenv.render.fg_colour, 1.0, 1.0, 1.0);
VEC3SET(sysenv.render.bg_colour, 0.0, 0.0, 0.0);
VEC3SET(sysenv.render.morph_colour, 0.1, 0.1, 0.8);
VEC3SET(sysenv.render.rsurf_colour, 0.1, 0.5, 0.7);
VEC3SET(sysenv.render.label_colour, 1.0, 1.0, 0.0);
VEC3SET(sysenv.render.title_colour, 0.0, 1.0, 1.0);
VEC3SET(sysenv.render.ribbon_colour, 0.0, 0.0, 1.0);

/* setup directory and file pointers */
sysenv.cwd = g_get_current_dir();

#if DEBUG_SYS_INIT
printf("cwd: [%s]\n", sysenv.cwd);
#endif

/* generate element file full pathname */
/* sometimes this returns the program name, and sometimes it doesn't */
temp = g_find_program_in_path(argv[0]);
/* remove program name (if attached) */
if (g_file_test(temp, G_FILE_TEST_IS_DIR))
  sysenv.gdis_path = temp;
else
  {
  sysenv.gdis_path = g_path_get_dirname(temp);
  g_free(temp);
  }

#if DEBUG_SYS_INIT
printf("%s path: [%s]\n", argv[0], sysenv.gdis_path);
#endif

if (sysenv.gdis_path)
  sysenv.elem_file = g_build_filename(sysenv.gdis_path, ELEM_FILE, NULL);
else
  {
  printf("WARNING: gdis directory not found.\n");
  sysenv.elem_file = g_build_filename(sysenv.cwd, ELEM_FILE, NULL);
  }

/* generate gdisrc full pathname */
ctemp = g_get_home_dir();
if (ctemp)
  sysenv.init_file = g_build_filename(ctemp, INIT_FILE, NULL);
else
  {
  printf("WARNING: home directory not found.\n");
  if (sysenv.gdis_path)
    sysenv.init_file = g_build_filename(sysenv.gdis_path, INIT_FILE, NULL);
  else
    {
    if (sysenv.cwd)
      sysenv.init_file = g_build_filename(sysenv.cwd, INIT_FILE, NULL);
    else
      {
      printf("FATAL: current directory not found.\n");
      exit(-1);
      }
    }
  }

/* defaults */
sysenv.povray_exe = g_strdup("povray");
sysenv.convert_exe = g_strdup("convert");
sysenv.viewer_exe = g_strdup("display");
sysenv.gulp_exe = g_strdup("gulp");
sysenv.babel_exe = g_strdup("babel");
sysenv.grace_exe = g_strdup("xmgrace");
sysenv.gamess_exe = g_strdup("run_gms_for_gdis");
strcpy(sysenv.gfont_name, GTK_FONT);
strcpy(sysenv.dfont_name, GL_FONT);
/* atomic scattering factor coefficients */
sysenv.sfc_table = g_hash_table_new_full(&g_str_hash,
                                         &hash_strcmp,
                                         &g_free,
                                         &free_sfc);

/* IMPORTANT this must be done before gdisrc is parsed as */
/* setup the element data before any possible modifications */
/* can be read in from the gdisrc file */
printf("scanning: %-50s ", sysenv.elem_file);
fp = fopen(sysenv.elem_file, "rt");
if (fp)
  {
  read_elem_data(fp, DEFAULT);
  fclose(fp);
  printf("[ok]\n");
  }
else
  {
/* missing default element info is fatal */
  printf("[not found]\n");
  exit(1);
  }

/* if failed to read gdisrc - write a new one */
/* TODO - if overwrite old version, save/rename first? */
if (read_gdisrc())
  {
  printf("creating: %-50s ", sysenv.init_file);
  if (write_gdisrc())
    printf("[error]\n");
  else
    printf("[ok]\n");
  }
else
  printf("scanning: %-50s [ok]\n", sysenv.init_file);

/* get executable paths */
sysenv.povray_path = g_find_program_in_path(sysenv.povray_exe);
sysenv.convert_path = g_find_program_in_path(sysenv.convert_exe);
sysenv.viewer_path = g_find_program_in_path(sysenv.viewer_exe);
sysenv.gulp_path = g_find_program_in_path(sysenv.gulp_exe);
sysenv.babel_path = g_find_program_in_path(sysenv.babel_exe);
sysenv.grace_path = g_find_program_in_path(sysenv.grace_exe);
sysenv.gamess_path = g_find_program_in_path(sysenv.gamess_exe);

/* dialog init */
for (i=0 ; i<MAX_DIALOGS ; i++)
  sysenv.dialog[i].active = FALSE;
/* tree init */
sysenv.num_trees = 0;
/* copied selection */
sysenv.select_source = NULL;

/* file extension recognition stuff */
file_init();

/* numerical stuff */
init_math();
}

/********/
/* MAIN */
/********/
gint main(int argc, char *argv[])
{
gint i;

/* NBL read this 1st as it affects canvas type, but command */
/* line arguments should be able to overide */
sys_init(argc, argv);
sysenv.write_gdisrc = FALSE;

/* command line? */
if (argc > 1)
  if (g_ascii_strncasecmp(argv[1],"-c",3) == 0)
    sysenv.canvas = FALSE;

/* set up main window and event handlers (or start non-gui mode) */
if (sysenv.canvas)
  {
/* NB: threads must be initialized before calling gtk_init() */
  init_task_thread();
  gtk_init(&argc, &argv);
  gtk_gl_init(&argc, &argv);

/* functions below aren't protect by thread enter/leave */
/* since nothing else writes to the widgets they control */
/* task update timer */
  g_timeout_add(1000, (GSourceFunc) &update_task_info, NULL);
/* screen redraw timer */
  g_timeout_add(25, (GSourceFunc) &redraw_handler, NULL);

/* main interface */
  connect_events();

/* process arguments as files to load */
for (i=1 ; i<argc ; i++)
  file_load(argv[i], NULL);

/* thread-safe handling */
  gdk_threads_enter();
  gtk_main();
  gdk_threads_leave();
  }
else
  cmd_init(argc, argv);

return(0);
}

