/*
Copyright (C) 2003 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <time.h>

#include "gdis.h"
#include "coords.h"
#include "edit.h"
#include "file.h"
#include "matrix.h"
#include "task.h"
#include "gtkshorts.h"
#include "dialog.h"
#include "interface.h"
#include "analysis.h"

extern struct elem_pak elements[];
extern struct sysenv_pak sysenv;

/********************************************/
/* display results of a particular analysis */
/********************************************/
void show_results(gdouble *bins, struct analysis_pak *analysis)
{
gint i;
gchar *cmd;
FILE *fp;

/* checks */
g_assert(bins != NULL);
g_assert(analysis != NULL);

/* plot file init */
fp = fopen("rdf_spectrum.dat", "wt");
if (!fp)
  return;

fprintf(fp, "@ title \"RDF plot\"\n");
fprintf(fp, "@ xaxis label \"Distance (A)\"\n");
fprintf(fp, "@ yaxis label \"g(r)\"\n");

for (i=0 ; i<analysis->num_bins ; i++)
  {
  fprintf(fp, "%f %f\n", analysis->start + i*analysis->step, bins[i]);
  }
fclose(fp);

/* send to grace */
cmd = g_strdup_printf("%s -free \"rdf_spectrum.dat\"", sysenv.grace_path);
g_spawn_command_line_async(cmd, NULL);

g_free(cmd);
}

/**************************/
/* read in all frame data */
/**************************/
#define DEBUG_LOAD_ANALYSIS 0
gint load_analysis(struct model_pak *model, struct task_pak *task)
{
gint i, j, k, m, n;
GSList *list;
struct core_pak *core;
struct model_pak temp;
struct analysis_pak *analysis;
FILE *fp;

/* checks */
g_assert(model != NULL);
g_assert(model->analysis != NULL);

analysis = model->analysis;
m = analysis->num_frames;
n = analysis->num_atoms;

#if DEBUG_LOAD_ANALYSIS
printf("Allocating for %d frames and %d atoms.\n", m, n);
#endif

analysis->latmat = g_malloc((m+1)*sizeof(struct gd9_pak));
analysis->position = g_malloc((m+1)*n*sizeof(struct gd3_pak));

/* init temp model */
template_model(&temp);
temp.frame_list = g_list_copy(model->frame_list);
temp.id = model->id;
temp.periodic = model->periodic;
temp.fractional = model->fractional;
/* init to use external trj frames */
if (model->id == GULP)
  {
  g_free(temp.gulp.trj_file);
  temp.gulp.trj_file = g_strdup(model->gulp.trj_file);
  temp.header_size = model->header_size;
  temp.frame_size = model->frame_size;
  temp.file_size = model->file_size;
  temp.expected_cores = model->expected_cores;
  temp.expected_shells = model->expected_shells;
  temp.trj_swap = model->trj_swap;
  memcpy(temp.latmat, model->latmat, 9*sizeof(gdouble));
  memcpy(temp.ilatmat, model->ilatmat, 9*sizeof(gdouble));
  temp.construct_pbc = TRUE;
  }

fp = fopen(model->filename, "r");
if (!fp)
  {
  printf("Could not open source file.");
  return(1);
  }

/* read frames */
k=0;
for (i=0 ; i<m ; i++)
  {
  read_raw_frame(fp, i, &temp);
  prep_model(&temp);

  memcpy((analysis->latmat+i)->m, temp.latmat, 9*sizeof(gdouble));

  j = g_slist_length(temp.cores);

/* fill out coordinates */
  if (j == n)
    {
    j=0;
    for (list=temp.cores ; list ; list=g_slist_next(list))
      {
      core = (struct core_pak *) list->data;
      ARR3SET((analysis->position+i*n+j)->x, core->x);
      j++;
      }
    k++;
    }
  else
    printf("Skipping inconsistent frame %d: has %d/%d cores.\n", i, j, n);

/* progress */
  task->progress += 50.0/analysis->num_frames;
  }

analysis->num_frames = k;

#if DEBUG_LOAD_ANALYSIS
printf("Read in %d complete frames.\n", k);
#endif

/* NB: we only copied the list from source model */
/* so make sure we don't free the elements */
g_list_free(temp.frame_list);
temp.frame_list = NULL;
free_model(&temp);
return(0);
}

/***********************/
/* free all frame data */
/***********************/
void free_analysis(struct model_pak *model)
{
struct analysis_pak *analysis;

if (!model->analysis)
  return;

analysis = (struct analysis_pak *) model->analysis;

g_free(analysis->latmat);
g_free(analysis->position);
}

/**********************/
/* setup the analysis */
/**********************/
/* TODO - when analysis_pak is more complete - include directly in model_pak */
#define DEBUG_INIT_ANALYSIS 0
gint init_analysis(struct model_pak *model)
{
struct analysis_pak *analysis;

g_assert(model != NULL);

analysis = g_malloc(sizeof(struct analysis_pak));
analysis->num_atoms = g_slist_length(model->cores);
analysis->atom1 = NULL;
analysis->atom2 = NULL;
analysis->num_frames = model->num_frames;
analysis->start = 0.0;
analysis->stop = model->rmax;
analysis->step = 0.1;
analysis->position = NULL;
analysis->latmat = NULL;

model->analysis = analysis;
return(0);
}

/**********************/
/* process frame data */
/**********************/
#define DEBUG_CALC_RDF 0
void calc_rdf(struct model_pak *model, struct task_pak *task)
{
gint i, j, k, m, n;
gint z, nz;
gdouble dz, *cz, slice;
gdouble volume, c, r1, r2, xi[3], xj[3], latmat[9];
struct core_pak *corei, *corej;
struct analysis_pak *analysis;

/* checks */
g_assert(model != NULL);
g_assert(model->analysis != NULL);

/* bin setup */
analysis = model->analysis;

/* load frames? */
if (!analysis->position)
  {
  slice = 50.0;
  if (load_analysis(model, task))
    return;
  }
else
  slice = 100.0;

nz = (analysis->stop - analysis->start)/analysis->step;
if (nz <= 0)
  return;
analysis->num_bins = nz;

dz = (analysis->stop - analysis->start) / (gdouble) nz;
analysis->step = dz;

cz = g_malloc(nz * sizeof(gdouble));
for (n=0 ; n<nz ; n++)
  *(cz+n) = 0.0;

#if DEBUG_CALC_RDF
printf("%f - %f : %f\n", analysis->start, analysis->stop, analysis->step);
printf("%s - %s\n", analysis->atom1, analysis->atom2);
#endif

/* TODO - for safety, remove all dependance on the model ptr */
/* (since the user may - or at least can - modify it) */
/* in particular, scanning its core list to get element labels */

/* loop over frames */
volume = 0.0;
for (n=0 ; n<analysis->num_frames ; n++)
  {
  memcpy(latmat, (analysis->latmat+n)->m, 9*sizeof(gdouble));

volume += calc_volume(latmat);

#if DEBUG_CALC_RDF
printf(">> calc_rdf [frame = %d] latmat: ", n);
P3MAT(" ", latmat);
#endif

/* loop over unique atom pairs */
  for (i=0 ; i<analysis->num_atoms-1 ; i++)
    {
corei = g_slist_nth_data(model->cores, i);

    ARR3SET(xi, (analysis->position+n*analysis->num_atoms+i)->x);

    for (j=i+1 ; j<analysis->num_atoms ; j++)
      {
corej = g_slist_nth_data(model->cores, j);

if (!pair_match(analysis->atom1, analysis->atom2, corei, corej))
  continue;

/* calculate minimum image distance */
      ARR3SET(xj, (analysis->position+n*analysis->num_atoms+j)->x);
      ARR3SUB(xj, xi);
      if (model->periodic)
        {
/* enforce range [-0.5, 0.5] */
        for (k=0 ; k<model->periodic ; k++)
          {
          m = 2.0*xj[k];
          xj[k] -= (gdouble) m;
          }
        vecmat(latmat, xj);
        }
/* bin it */
      r1 = VEC3MAG(xj); 
r1 -= analysis->start;

      z = (0.5 + r1/dz);
/* ignore if outside range */
      if (z < nz)
        *(cz+z) += 2.0;
      }
    }

/* update progess */
  task->progress += slice/analysis->num_frames;
  }

/* normalize against ideal gas with same AVERAGE volume */
volume /= analysis->num_frames;
c = 1.333333333*PI*analysis->num_atoms/volume;
c *= analysis->num_atoms*analysis->num_frames;
r1 = r2 = analysis->start;
r2 += dz;
for (i=0 ; i<nz ; i++)
  {
  *(cz+i) /= (c * (r2*r2*r2 - r1*r1*r1));
  r1 += dz;
  r2 += dz;
  }

show_results(cz, analysis);

g_free(cz);
}

/**********************/
/* submit an rdf task */
/**********************/
void exec_rdf_task(GtkWidget *w, struct model_pak *model)
{
submit_task("RDF", &calc_rdf, model, NULL, NULL, model);
}

/**********************/
/* atom entry changed */
/**********************/
void md_atom_changed(GtkWidget *w, gchar **atom)
{
g_assert(w != NULL);

/* NB: we pass a pointer to the character pointer */
if (*atom)
  g_free(*atom);

*atom = g_strdup(gtk_entry_get_text(GTK_ENTRY(w)));
}

/**************/
/* rdf window */
/**************/
void rdf_window(GtkWidget *box, struct model_pak *model)
{
GSList *item, *list=NULL;
GList *match_list=NULL;
GtkWidget *frame, *vbox, *combo;
struct analysis_pak *analysis;

/* checks */
g_assert(model != NULL);
g_assert(model->analysis != NULL);

analysis = (struct analysis_pak *) model->analysis;

/* bin setup */
frame = gtk_frame_new("Interval");
gtk_box_pack_start(GTK_BOX(box), frame, FALSE, FALSE, 0);
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);

vbox = gtk_vbox_new(TRUE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(frame), vbox);

/* TODO - some warning about the RDF being valid only for < L/2? */
/* see Allan & Tildesley pg 199 */

gtksh_direct_spin("Start", &analysis->start, 0.0, model->rmax, 0.1,
                                                 NULL, NULL, vbox);
gtksh_direct_spin("Stop", &analysis->stop, 0.1, model->rmax, 0.1,
                                               NULL, NULL, vbox);
gtksh_direct_spin("Step", &analysis->step, 0.1, model->rmax, 0.1,
                                               NULL, NULL, vbox);

/* atom setup */
/* NB: blank matches everything */
frame = gtk_frame_new("Atoms");
gtk_box_pack_start(GTK_BOX(box), frame, FALSE, FALSE, 0);
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);

vbox = gtk_vbox_new(TRUE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(frame), vbox);

/* setup unique element label list */
list = find_unique(LABEL, model);
match_list = g_list_append(match_list, "Any");
for (item=list ; item  ; item=g_slist_next(item))
  {
  match_list = g_list_append(match_list, item->data);
  }
g_slist_free(list);

/* match 1 */
combo = gtk_combo_new();
gtk_combo_set_popdown_strings(GTK_COMBO(combo), match_list);
gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(combo)->entry), TRUE);
gtk_box_pack_start(GTK_BOX(vbox), combo, TRUE, TRUE, 0);
g_signal_connect(GTK_OBJECT(GTK_COMBO(combo)->entry), "changed",
                 GTK_SIGNAL_FUNC(md_atom_changed), (gpointer) &analysis->atom1);

/* match 2 */
combo = gtk_combo_new();
gtk_combo_set_popdown_strings(GTK_COMBO(combo), match_list);
gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(combo)->entry), TRUE);
gtk_box_pack_start(GTK_BOX(vbox), combo, TRUE, TRUE, 0);
g_signal_connect(GTK_OBJECT(GTK_COMBO(combo)->entry), "changed",
                 GTK_SIGNAL_FUNC(md_atom_changed), (gpointer) &analysis->atom2);

gtk_widget_show_all(box);
}

/***************/
/* vacf_window */
/***************/
void vacf_window(GtkWidget *box, struct model_pak *model)
{
GtkWidget *label;

/* checks */
g_assert(model != NULL);

/* init */
if (!g_file_test(model->gulp.trj_file, G_FILE_TEST_EXISTS))
  {
  label = gtk_label_new("Could not find trajectory file.");
  gtk_box_pack_start(GTK_BOX(box), label, TRUE, TRUE, 0);
  return;
  }
}

/*******************************/
/* multi-frame analysis dialog */
/*******************************/
void md_dialog(void)
{
gint id;
GtkWidget *frame, *notebook, *page, *label;
struct dialog_pak *mdd;
struct model_pak *model;

/* active model */
model = model_ptr(sysenv.active, RECALL);
g_assert(model != NULL);

/* setup analysis */
if (!model->analysis)
  {
  if (init_analysis(model))
    return;
  }

/* create new dialog */
id = request_dialog(model->number, MD_ANALYSIS);
if (id < 0)
  return;
mdd = &sysenv.dialog[id];
mdd->win = gtk_dialog_new();
gtk_window_set_title (GTK_WINDOW(mdd->win), "MD analysis");
g_signal_connect(GTK_OBJECT(mdd->win), "destroy",
                 GTK_SIGNAL_FUNC(event_close_dialog), (gpointer) id);

/* frame */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(mdd->win)->vbox), frame, FALSE, FALSE, 0);
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);

/* create notebook */
notebook = gtk_notebook_new();
gtk_notebook_set_tab_pos(GTK_NOTEBOOK(notebook), GTK_POS_TOP);
gtk_container_add(GTK_CONTAINER(frame), notebook);
gtk_notebook_set_show_border(GTK_NOTEBOOK(notebook), FALSE);

/* TODO - may not even need a notebook */
/* RDF page */
page = gtk_vbox_new(FALSE,0);
label = gtk_label_new (" RDF ");
gtk_notebook_append_page(GTK_NOTEBOOK(notebook), page, label);
rdf_window(page, model);

/* TODO - rename - time correlation */
/* VACF page */
/*
page = gtk_vbox_new(FALSE,0);
label = gtk_label_new (" VACF ");
gtk_notebook_append_page(GTK_NOTEBOOK(notebook), page, label);
vacf_window(page, model);
*/

/* terminating buttons */
gtksh_stock_button(GTK_STOCK_EXECUTE, exec_rdf_task, model,
                   GTK_DIALOG(mdd->win)->action_area);

gtksh_stock_button(GTK_STOCK_CLOSE, event_close_dialog, GINT_TO_POINTER(id),
                   GTK_DIALOG(mdd->win)->action_area);


gtk_widget_show_all(mdd->win);
}

