/*
Copyright (C) 2003 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <math.h>
#include <gdk/gdk.h>
#include <gtk/gtkgl.h>
#include <GL/gl.h>
#include <GL/glu.h>

#include "gdis.h"
#include "coords.h"
#include "matrix.h"
#include "molcon.h"
#include "morph.h"
#include "spatial.h"
#include "opengl.h"
#include "render.h"
#include "select.h"
#include "surface.h"
#include "numeric.h"
#include "interface.h"

/* externals */
extern struct sysenv_pak sysenv;
extern struct elem_pak elements[];

/* transformation/projection matrices */
GLint viewport[4];
GLdouble mvmatrix[16], projmatrix[16];
gdouble gl_acm[9];
gdouble halo_fade[16];
gint halo_segments=16;
gint flush=0;
gpointer gl_font;
gint gl_fontsize=10;
gint font_offset;

/* all rendering parameters are handled by the same dialog */
/* as serves the povray rendering */

/*******************************************/
/* get the canvas associated with a widget */
/*******************************************/
struct canvas_pak *find_canvas(GtkWidget *w)
{
GSList *list;
struct canvas_pak *canvas;

for (list=sysenv.canvas_list ; list ; list=g_slist_next(list))
  {
  canvas = (struct canvas_pak *) list->data;
  if (canvas->glarea == w)
    return(canvas);
  }
return(NULL);
}

/*******************/
/* configure event */
/*******************/
#define DEBUG_GL_CONFIG_EVENT 0
gint gl_configure_event(GtkWidget *w, GdkEventConfigure *event)
{
gint i, x=0, y=0, size;
static gint n=0;
GdkGLContext *glcontext;
GdkGLDrawable *gldrawable;
struct canvas_pak *canvas;
struct model_pak *data;

n++;
#if DEBUG_GL_CONFIG_EVENT
printf("gl_configure() : %d\n", n);
#endif

/* find canvas associated with this glarea */
canvas = find_canvas(w);
g_assert(canvas != NULL);

/* adjust viewport so spheres don't become ellipsoids if width != height */
/* NB: not having the origin at 0,0 means many of the atom/pixel */
/* finding routines (& box drawing) are wrong */
/* easiest solution at the moment is to keep origin at 0,0 dispite */
/* the fact that making the viewport sit in the centre is more attractive */
if (w->allocation.width > w->allocation.height)
  {
/* too wide */
  x = (w->allocation.width - w->allocation.height)/2;
  size = w->allocation.height;
  }
else
  {
/* too high (or equal) */
  y = (w->allocation.height - w->allocation.width)/2;
  size = w->allocation.width;
  }

/* store the canvas parameters */
canvas->x = x;
canvas->y = y;
canvas->width = w->allocation.width;
canvas->height = w->allocation.height;
canvas->size = size;

/* FIXME - can't remove this yet (model scaling?) */
/* but we need to convert to using canvas-> stuff instead */
sysenv.x = x;
sysenv.y = y;
sysenv.width = w->allocation.width;
sysenv.height = w->allocation.height;
sysenv.size = size;

#if DEBUG_GL_CONFIG_EVENT
printf("Relative canvas origin: (%d,%d)\n",sysenv.x,sysenv.y);
printf("     Canvas dimensions:  %dx%d\n",sysenv.width,sysenv.height);
printf("   Viewport dimensions:  %dx%d\n",sysenv.size,sysenv.size);
#endif

/* OpenGL window setup */
glcontext = gtk_widget_get_gl_context(w);
gldrawable = gtk_widget_get_gl_drawable(w);
if (!gdk_gl_drawable_gl_begin(gldrawable, glcontext))
  return(FALSE);

#if DEBUG_GL_CONFIG_EVENT
g_print("\n");
g_print("GL_RENDERER   = %s\n", (gchar *) glGetString(GL_RENDERER));
g_print("GL_VERSION    = %s\n", (gchar *) glGetString(GL_VERSION));
g_print("GL_VENDOR     = %s\n", (gchar *) glGetString(GL_VENDOR));
g_print("GL_EXTENSIONS = %s\n", (gchar *) glGetString(GL_EXTENSIONS));
g_print("\n");
#endif

/* defining this as width & height acts to stretch the model (if unequal)  */
glViewport(x,y,size,size);

gdk_gl_drawable_gl_end(gldrawable);

/* completely clear the double buffer */
flush = 2;

/* update coords */
for (i=0 ; i<sysenv.num_displayed ; i++)
  {
  data = model_ptr(sysenv.displayed[i], RECALL);
  if (data)
    init_objs(REDO_COORDS, data);
  }

/* new screen size to be saved as default */
sysenv.write_gdisrc = TRUE;

gdk_gl_drawable_gl_end(gldrawable);

return(TRUE);
}

/*****************/
/* expose event */
/****************/
#define DEBUG_GL_EXPOSE 0
gint gl_expose_event(GtkWidget *w, GdkEventExpose *event)
{
static gint n=0;
struct canvas_pak *canvas;

canvas = find_canvas(w);
n++;

#if DEBUG_GL_EXPOSE
printf("gl_expose_event(%p) : %d\n",w,n);
printf("canvas: %p, glarea: %p, model: %p\n", canvas, canvas->glarea, canvas->model);
#endif

opengl_draw(canvas, canvas->model);

return(TRUE);
}


/***************************************************/
/* setup the visual for subsequent canvas creation */
/***************************************************/
GdkGLConfig *glconfig;

gint gl_init_visual(void)
{
/* attempt to get a visual */
glconfig = gdk_gl_config_new_by_mode(GDK_GL_MODE_RGB |
                                     GDK_GL_MODE_DEPTH |
                                     GDK_GL_MODE_DOUBLE);
if (!glconfig)
  {
  printf("WARNING: cannot create a double-buffered visual.\n");
/* Try single-buffered visual */
  glconfig = gdk_gl_config_new_by_mode(GDK_GL_MODE_RGB | GDK_GL_MODE_DEPTH);
  if (!glconfig)
    {
    printf("ERROR: no appropriate visual could be acquired.\n");
    return(1);
    }
  }
return(0);
}

/*******************************************************/
/* create an OpenGL canvas for packing in a GTK widget */
/*******************************************************/
/* NEW - GTK 2.x only */
/* wxh = 200, 200 */
#define DEBUG_NEW_CANVAS 0
struct canvas_pak *gl_new_canvas(gint width, gint height)
{
struct canvas_pak *canvas;

g_assert(glconfig != NULL);

canvas = g_malloc(sizeof(struct canvas_pak));
sysenv.canvas_list = g_slist_prepend(sysenv.canvas_list, canvas);

#if DEBUG_NEW_CANVAS
printf("creating canvas: %p\n", canvas);
#endif

/* create an OpenGL capable drawing area */
canvas->active = FALSE;
canvas->model = NULL;
canvas->glarea = gtk_drawing_area_new();
gtk_widget_set_gl_capability(canvas->glarea,
                             glconfig, NULL, TRUE, GDK_GL_RGBA_TYPE);
gtk_widget_set_size_request(canvas->glarea, width, height);

/* init signals */
g_signal_connect(GTK_OBJECT(canvas->glarea), "expose_event",
                (GtkSignalFunc) gl_expose_event, NULL);
g_signal_connect(GTK_OBJECT(canvas->glarea), "configure_event",
                (GtkSignalFunc) gl_configure_event, NULL);
g_signal_connect(GTK_OBJECT(canvas->glarea), "motion_notify_event",
                (GtkSignalFunc) motion_notify_event, NULL);
g_signal_connect(GTK_OBJECT(canvas->glarea), "button_press_event",
                (GtkSignalFunc) button_press_event, NULL);
g_signal_connect(GTK_OBJECT(canvas->glarea), "button_release_event",
                (GtkSignalFunc) button_release_event, NULL);

gtk_widget_set_events(GTK_WIDGET(canvas->glarea), GDK_EXPOSURE_MASK
                                                | GDK_LEAVE_NOTIFY_MASK
                                                | GDK_BUTTON_PRESS_MASK
                                                | GDK_BUTTON_RELEASE_MASK
                                                | GDK_POINTER_MOTION_MASK
                                                | GDK_POINTER_MOTION_HINT_MASK);
return(canvas);
}

/****************************************/
/* setup the camera and projection mode */
/****************************************/
#define DEBUG_INIT_PROJ 0
void gl_init_projection(struct model_pak *data)
{
gdouble r, pix2ang;
gdouble mat[9];
gdouble upv[3] = {0.0, 0.0, 1.0};
gdouble cam[3] = {0.0, 0.0, 0.0};

/* viewing */
glMatrixMode(GL_MODELVIEW);
glLoadIdentity();
/* get the model axes transformation matrix */
init_rotmat(mat);
/* transform camera position & up vector */
vecmat(mat, cam);
vecmat(mat, upv);

/* scaling affects placement to avoid near/far clipping */
r = RMAX_FUDGE * data->rmax;
/* yet another magic number (0.427) - works reasonably well */
pix2ang = 0.427 * (gdouble) sysenv.size / data->rmax;

/* if not periodic - may need some extra help fitting within the window */
if (!data->periodic && data->id != MORPH)
  {
/* add a bit extra due to the sizes of rendered spheres */
  switch (sysenv.render.type)
    {
    case BALL_STICK:
      r += 2.0 * sysenv.render.ball_rad;
      break;
    case CPK:
      r += 2.0 * sysenv.render.cpk_scale;
      break;
    }
  }
sysenv.rsize = r;

#if DEBUG_INIT_PROJ
printf("      size = %d\n", sysenv.size);
printf(" real size = %f\n", sysenv.rsize);
printf("      rmax = %f\n", data->rmax);
printf("     scale = %f\n", data->scale);
printf("    vpdist = %f\n", sysenv.render.vp_dist);
printf("   pix2ang = %f\n", pix2ang);
#endif

/* mouse translation simply shifts the camera */
cam[0] -= data->offset[0] / pix2ang;
cam[1] -= data->offset[1] / pix2ang;
/* move camera back */
cam[2] = -(sysenv.render.vp_dist+data->scale)*r;

gluLookAt(cam[0],cam[1],cam[2],cam[0],cam[1],0.0,upv[0],upv[1],upv[2]);

/* projection */
glMatrixMode(GL_PROJECTION);
glLoadIdentity();

/* scaling affects projection to avoid near/far clipping */
/* NB: near/far clipping - dist from camera? */
/* if so, makes sense to have near clip plane = 0 */
if (sysenv.render.perspective)
  glFrustum(-r,r,-r,r,sysenv.render.vp_dist*r,
            (sysenv.render.vp_dist+2.0*data->scale)*r);
else
  glOrtho(-r,r,-r,r,0.0,(sysenv.render.vp_dist+2.0*data->scale)*r);

/* store matrices for proj/unproj operations */
glGetIntegerv(GL_VIEWPORT, viewport);
glGetDoublev(GL_MODELVIEW_MATRIX, mvmatrix);
glGetDoublev(GL_PROJECTION_MATRIX, projmatrix);

/* NEW - opengl -> gdis coordinate conversion */
VEC3SET(&gl_acm[0], -1.0,  0.0,  0.0);
VEC3SET(&gl_acm[3],  0.0,  0.0,  1.0);
VEC3SET(&gl_acm[6],  0.0, -1.0,  0.0);
}

/***************************/
/* setup all light sources */
/***************************/
void gl_init_lights(struct model_pak *data)
{
gint i;
gfloat light[4];
gdouble x, tmp[4];
GSList *list;
struct light_pak *ldata;

/* go through all OpenGL lights (enable/disable as required) */
list = sysenv.render.light_list;
for (i=GL_LIGHT0 ; i<=GL_LIGHT7 ; i++)
  {
/* do we have an active light */
  if (list)
    {
    glEnable(i);
/* get light data */
    ldata = (struct light_pak *) list->data;
/* position/direction */
    ARR3SET(light, ldata->x);

ARR3SET(tmp, light);
vecmat(gl_acm, tmp);
ARR3SET(light, tmp);

    switch (ldata->type)
      {
      case DIRECTIONAL:
        light[3] = 0.0;
        break;
      case POSITIONAL:
      default:
        VEC3MUL(light, -1.0);
        light[3] = 1.0;
      }
    glLightfv(i, GL_POSITION, light);
/* light properties */
    ARR3SET(light, ldata->colour);
    VEC3MUL(light, ldata->ambient);
    glLightfv(i, GL_AMBIENT, light);

    ARR3SET(light, ldata->colour);
    VEC3MUL(light, ldata->diffuse);
    glLightfv(i, GL_DIFFUSE, light);

    ARR3SET(light, ldata->colour);
    VEC3MUL(light, ldata->specular);
    glLightfv(i, GL_SPECULAR, light);
/* next */
    list = g_slist_next(list);
    }
  else
    glDisable(i);
  }

/* halo diminishing function */
for (i=0 ; i<halo_segments ; i++)
  {
  x = (gdouble) i / (gdouble) halo_segments;
  x *= x;
  halo_fade[i] = exp(-5.0 * x);
  }
}

/*********************************************/
/* window to real space conversion primitive */
/*********************************************/
void gl_get_world_coords(gint x, gint y, gdouble *w)
{
gint ry;
GLdouble r[3];

ry = sysenv.height - y - 1;
gluUnProject(x, ry, 0.0, mvmatrix, projmatrix, viewport, &r[0], &r[1], &r[2]);

ARR3SET(w, r);
}

/*******************************************/
/* set opengl RGB colour for gdis RGB data */
/*******************************************/
void set_gl_colour(gint *rgb)
{
gdouble col[3];

col[0] = (gdouble) *rgb;
col[1] = (gdouble) *(rgb+1);
col[2] = (gdouble) *(rgb+2);
VEC3MUL(col, 1.0/65535.0);
glColor4f(col[0], col[1], col[2], 1.0);
}

/*************************************************************/
/* adjust fg colour for visibility against current bg colour */
/*************************************************************/
void make_fg_visible(void)
{
gdouble fg[3], bg[3];

#define F_COLOUR_SCALE 65535.0
ARR3SET(fg, sysenv.render.fg_colour);
ARR3SET(bg, sysenv.render.bg_colour);
VEC3MUL(fg, F_COLOUR_SCALE);
VEC3MUL(bg, F_COLOUR_SCALE);
/* XOR to get a visible colour */
fg[0] = (gint) bg[0] ^ (gint) F_COLOUR_SCALE;
fg[1] = (gint) bg[1] ^ (gint) F_COLOUR_SCALE;
fg[2] = (gint) bg[2] ^ (gint) F_COLOUR_SCALE;
VEC3MUL(fg, 1.0/F_COLOUR_SCALE);
ARR3SET(sysenv.render.fg_colour, fg);

/* adjust label colour for visibility against the current background */
ARR3SET(fg, sysenv.render.label_colour);
VEC3MUL(fg, F_COLOUR_SCALE);
/* XOR to get a visible colour */
fg[0] = (gint) bg[0] ^ (gint) F_COLOUR_SCALE;
fg[1] = (gint) bg[1] ^ (gint) F_COLOUR_SCALE;
VEC3MUL(fg, 1.0/F_COLOUR_SCALE);
/* force to zero, so we get yellow (not white) for a black background */
fg[2] = 0.0;
ARR3SET(sysenv.render.label_colour, fg);

/* adjust title colour for visibility against the current background */
ARR3SET(fg, sysenv.render.title_colour);
VEC3MUL(fg, F_COLOUR_SCALE);
/* force to zero, so we get cyan (not white) for a black background */
fg[0] = 0.0;
/* XOR to get a visible colour */
fg[1] = (gint) bg[1] ^ (gint) F_COLOUR_SCALE;
fg[2] = (gint) bg[2] ^ (gint) F_COLOUR_SCALE;
VEC3MUL(fg, 1.0/F_COLOUR_SCALE);
ARR3SET(sysenv.render.title_colour, fg);

/*
printf("fg: %lf %lf %lf\n",fg[0],fg[1],fg[2]);
*/
}

/*********************************************************/
/* draw a circular arc from p1 to p2, with p0 the centre */
/*********************************************************/
void draw_arc(gdouble *p0, gdouble *p1, gdouble *p2)
{
gint i, num_points;
gdouble angle, theta, st, ct;
gdouble v[3], v1[3], v2[3], len1, len2;

/* get vector to arc start */
ARR3SET(v1, p1);
ARR3SUB(v1, p0);

/* get vector to arc end */
ARR3SET(v2, p2);
ARR3SUB(v2, p0);

/* scale back a bit */
VEC3MUL(v1, 0.5);
VEC3MUL(v2, 0.5);

/* get lengths */
len1 = VEC3MAG(v1);
len2 = VEC3MAG(v2);

/* make them both equal to the smaller */
if (len2 > len1)
  {
  VEC3MUL(v2, len1/len2);
  }
else
  {
  VEC3MUL(v1, len2/len1);
  }

angle = via(v1, v2, 3);

num_points = 32;

glBegin(GL_LINE_STRIP);
/* sweep from start to end */
for (i=0 ; i<=num_points ; i++)
  {
/* sweep out a circle */
  theta = (gdouble) i * 0.5 * PI / (gdouble) num_points;
  st = tbl_sin(theta);
  ct = tbl_cos(theta);
/* construct normal */
  ARR3SET(v, p0);
  v[0] += ct * v1[0] + st * v2[0];
  v[1] += ct * v1[1] + st * v2[1];
  v[2] += ct * v1[2] + st * v2[2];
  glVertex3dv(v);
  }
glEnd();
}

/************************************************/
/* recursive routine for approximating a sphere */
/************************************************/
void divide_triangle(struct point_pak *p, gdouble *a, gdouble *b, gdouble *c, guint depth)
{
gint i;
gdouble ab[3], bc[3], ac[3];

if (depth)
  {
/* calculate (normalized) midpoint */
  ARR3SET(ab, a);
  ARR3ADD(ab, b);
  normalize(ab, 3);
/* calculate (normalized) midpoint */
  ARR3SET(bc, b);
  ARR3ADD(bc, c);
  normalize(bc, 3);
/* calculate (normalized) midpoint */
  ARR3SET(ac, a);
  ARR3ADD(ac, c);
  normalize(ac, 3);
/* submit the next depth */
  divide_triangle(p, a, ab, ac, depth-1);
  divide_triangle(p, ab, b, bc, depth-1);
  divide_triangle(p, ab, bc, ac, depth-1);
  divide_triangle(p, c, ac, bc, depth-1);
  }
else
  {
/* required depth reached - store the triangle */
  i = p->num_points;
  *(p->x[0]+i) = c[0];
  *(p->x[1]+i) = c[1];
  *(p->x[2]+i) = c[2];
  i++;
  *(p->x[0]+i) = b[0];
  *(p->x[1]+i) = b[1];
  *(p->x[2]+i) = b[2];
  i++;
  *(p->x[0]+i) = a[0];
  *(p->x[1]+i) = a[1];
  *(p->x[2]+i) = a[2];
  p->num_points += 3;
  }
}

/*****************************/
/* create a digitized sphere */
/*****************************/
#define DEBUG_INIT_SPHERE 0
void gl_init_sphere(struct point_pak *sphere, guint depth)
{
gint n;
gdouble a[3], b[3], c[3], d[3], e[3], f[3], g[3];

/* hexagonal -> 6 face init */
n = 6 * pow(4, depth);

#if DEBUG_INIT_SPHERE
printf("num triangles = %d\n", n);
#endif

/* point allocation */
n *= 3;
sphere->x[0] = g_malloc(n * sizeof(gdouble));
sphere->x[1] = g_malloc(n * sizeof(gdouble));
sphere->x[2] = g_malloc(n * sizeof(gdouble));
sphere->num_points = 0;
sphere->type = depth;

#if DEBUG_INIT_SPHERE
printf("pre-calc points: %d\n", n);
#endif

/* sphere type 2 (6 side init) */
VEC3SET(a,  0.000,  0.000, -1.000);
VEC3SET(b,  1.000,  0.000,  0.000);
VEC3SET(c,  0.500,  0.866,  0.000);
VEC3SET(d, -0.500,  0.866,  0.000);
VEC3SET(e, -1.000,  0.000,  0.000);
VEC3SET(f, -0.500, -0.866,  0.000);
VEC3SET(g,  0.500, -0.866,  0.000);
divide_triangle(sphere, a, b, c, depth);
divide_triangle(sphere, a, c, d, depth);
divide_triangle(sphere, a, d, e, depth);
divide_triangle(sphere, a, e, f, depth);
divide_triangle(sphere, a, f, g, depth);
divide_triangle(sphere, a, g, b, depth);

#if DEBUG_INIT_SPHERE
printf("  actual points: %d\n", sphere->num_points);
#endif

g_assert(sphere->num_points == n);
}

/***************************/
/* draw a digitized sphere */
/***************************/
void gl_draw_sphere(struct point_pak *p, gdouble *x, gdouble rad)
{
gint i;
gdouble vec[3];

g_return_if_fail(p->num_points != 0);

glBegin(GL_TRIANGLES);
for (i=0 ; i<p->num_points ; i++)
  {
  vec[0] = *(p->x[0]+i);
  vec[1] = *(p->x[1]+i);
  vec[2] = *(p->x[2]+i);
  glNormal3dv(vec);
  VEC3MUL(vec, rad);
  ARR3ADD(vec, x);
  glVertex3dv(vec);
  }
glEnd();
}

/***************************/
/* free an array of points */
/***************************/
void gl_free_points(struct point_pak *p)
{
g_return_if_fail(p->num_points != 0);

if (p->num_points)
  {
  g_free(p->x[0]);
  g_free(p->x[1]);
  g_free(p->x[2]);
  p->num_points = 0;
  }
}

/***************************/
/* create digitized circle */
/***************************/
void gl_init_circle(struct point_pak *circle, gint segments)
{
gint i;
gdouble da, theta;
gdouble vec[3];

g_return_if_fail(segments != 0);

/* allocate for digitized circle */
circle->x[0] = g_malloc((segments+1) * sizeof(gdouble));
circle->x[1] = g_malloc((segments+1) * sizeof(gdouble));
circle->x[2] = g_malloc((segments+1) * sizeof(gdouble));
circle->num_points = segments + 1;

/* first point (theta=0) */
*(circle->x[0]) = 1.0;
*(circle->x[1]) = 0.0;
*(circle->x[2]) = 0.0;
/* sweep */
vec[2] = 0.0;
da = 2.0 * PI / (gdouble) segments;
for (i=1 ; i<segments ; i++) 
  {
  theta = (gdouble) i * da;
/* store point on rim (doubles as the normal) */
  vec[0] = tbl_cos(theta);
  vec[1] = tbl_sin(theta);
  normalize(vec, 2);
  *(circle->x[0]+i) = vec[0];
  *(circle->x[1]+i) = vec[1];
  *(circle->x[2]+i) = vec[2];
  }
/* duplicate first point, so the circle is closed */
*(circle->x[0]+segments) = 1.0;
*(circle->x[1]+segments) = 0.0;
*(circle->x[2]+segments) = 0.0;
}

/*******************************/
/* draw a pre-digitized circle */
/*******************************/
void gl_draw_circle(struct point_pak *circle, gdouble *x, gdouble rad)
{
gint i;
gdouble vec[3];

g_return_if_fail(circle->num_points != 0);

glBegin(GL_POLYGON);
for (i=0 ; i<circle->num_points ; i++) 
  {
/* get points */
  vec[0] = *(circle->x[0]+i);
  vec[1] = *(circle->x[1]+i);
  vec[2] = *(circle->x[2]+i);
  VEC3MUL(vec, rad);
  ARR3ADD(vec, x);
  glVertex3dv(vec);
  }
glEnd();
}

/*****************************/
/* draw a pre-digitized ring */
/*****************************/
void gl_draw_ring(struct point_pak *circle, gdouble *x, gdouble rad1, gdouble rad2)
{
gint i;
gdouble vec[3];

g_return_if_fail(circle->num_points != 0);

glBegin(GL_QUAD_STRIP);
for (i=0 ; i<circle->num_points ; i++) 
  {
/* inner point */
  vec[0] = *(circle->x[0]+i);
  vec[1] = *(circle->x[1]+i);
  vec[2] = *(circle->x[2]+i);
  VEC3MUL(vec, rad2);
  ARR3ADD(vec, x);
  glVertex3dv(vec);

/* outer point */
  vec[0] = *(circle->x[0]+i);
  vec[1] = *(circle->x[1]+i);
  vec[2] = *(circle->x[2]+i);
  VEC3MUL(vec, rad1);
  ARR3ADD(vec, x);
  glVertex3dv(vec);
  }
glEnd();
}

/*********************************************/
/* draw cylinder from a pre-digitized circle */
/*********************************************/
#define DEBUG_GL_DRAW_CYLINDER 0
void gl_draw_cylinder(struct point_pak *circle, gdouble *pa, gdouble *pb, gdouble rad)
{
gint i, flag=0;
gdouble len1, len2;
gdouble p1[3], p2[3], p[3], vec[3], mat[9], mop[9];

g_return_if_fail(circle->num_points != 0);

/* force ordering (for correct quad normals) */
if (pa[2] > pb[2])
  {
  ARR3SET(p1, pb);
  ARR3SET(p2, pa);
  }
else
  {
  ARR3SET(p1, pa);
  ARR3SET(p2, pb);
  }

/* get vector from point 1 to 2 */
ARR3SET(vec, p2);
ARR3SUB(vec, p1);

/* normalize x */
len1 = VEC3MAG(vec);
vec[0] /= len1;
/* normalize y,z (projection) */
len2 = sqrt(vec[1]*vec[1] + vec[2]*vec[2]);
if (len2 > FRACTION_TOLERANCE)
  {
  vec[1] /= len2;
  vec[2] /= len2;
  len2 /= len1;
  }
else
  flag++;

#if DEBUG_GL_DRAW_CYLINDER
printf("--------------------\n");
P3VEC("input: ", vec);
#endif

/* z axis coincidence */
VEC3SET(&mat[0], len2, 0.0, vec[0]);
VEC3SET(&mat[3], 0.0, 1.0, 0.0);
VEC3SET(&mat[6], -vec[0], 0.0, len2);
/* yz plane coincidence (if not already satisfied) */
if (!flag)
  {
  VEC3SET(&mop[0], 1.0, 0.0, 0.0);
  VEC3SET(&mop[3], 0.0, vec[2], vec[1]);
  VEC3SET(&mop[6], 0.0, -vec[1], vec[2]);
/* construct transformation matrix */
  matmat(mop, mat);
  }

/* check */
#if DEBUG_GL_DRAW_CYLINDER
P3MAT("matrix: ", mat);
VEC3SET(vec, 0.0, 0.0, 1.0);
P3VEC("before: ", vec);
vecmat(mat, vec);
P3VEC("after: ", vec);
printf("--------------------\n");
#endif

glBegin(GL_QUAD_STRIP);
for (i=0 ; i<circle->num_points ; i++) 
  {
/* get point on rim (doubles as the normal) */
  vec[0] = *(circle->x[0]+i);
  vec[1] = *(circle->x[1]+i);
  vec[2] = *(circle->x[2]+i);
/* rotate the digitized circle */
  vecmat(mat, vec);
/* submit the normal */
  glNormal3dv(vec);
/* get points at required radius */
  VEC3MUL(vec, rad);
/* point on disk 2 */
  ARR3SET(p, p2);
  ARR3ADD(p, vec);
  glVertex3dv(p);
/* point on disk 1 */
  ARR3SET(p, p1);
  ARR3ADD(p, vec);
  glVertex3dv(p);
  }
glEnd();
}

/******************************************/
/* draw a cylinder between points v1 & v2 */
/******************************************/
/* NB: used for drawing vectors */
void draw_cylinder(gdouble *va, gdouble *vb, gdouble rad, guint segments)
{
gint i;
gdouble theta, st, ct;
gdouble v1[3], v2[3], v[3], v12[3], p[3], q[3];

g_return_if_fail(segments != 0);

/* force ordering of v1 & v2 to get correct quad normals */
if (va[2] > vb[2])
  {
  ARR3SET(v1, vb);
  ARR3SET(v2, va);
  }
else
  {
  ARR3SET(v1, va);
  ARR3SET(v2, vb);
  }

/* vector from v1 to v2 */
ARR3SET(v12, v2);
ARR3SUB(v12, v1);

/* create vectors p and q, co-planar with the cylinder's cross-sectional disk */
ARR3SET(p, v12);
if (p[0] == 0.0 && p[1] == 0.0)
  p[0] += 1.0;
else
  p[2] += 1.0;
crossprod(q, p, v12);
crossprod(p, v12, q);
/* do the normalization outside the segment loop */
normalize(p, 3);
normalize(q, 3);

/* build the cylinder from rectangular segments */
/* TODO - see if vertex arrays give any speedup here */
glBegin(GL_QUAD_STRIP);
for (i=0 ; i<=segments ; i++) 
  {
/* sweep out a circle */
  theta = (gdouble) i * 2.0 * PI / (gdouble) segments;
  st = tbl_sin(theta);
  ct = tbl_cos(theta);
/* construct normal */
  v12[0] = ct * p[0] + st * q[0];
  v12[1] = ct * p[1] + st * q[1];
  v12[2] = ct * p[2] + st * q[2];
/* set the normal for the two subseqent points */
  glNormal3dv(v12);
/* get the vector from centre to rim */
  VEC3MUL(v12, rad);
/* point on disk 1 */
  ARR3SET(v, v2);
  ARR3ADD(v, v12);
  glVertex3dv(v);
/* point on disk 2 */
  ARR3SET(v, v1);
  ARR3ADD(v, v12);
  glVertex3dv(v);
  }
glEnd();
}

/*************************************/
/* draw a cone defined by two points */
/*************************************/
void draw_cone(gdouble *va, gdouble *vb, gdouble base, gint segments)
{
gint i;
gdouble theta, st, ct;
gdouble v1[3], v2[3], v[3];
gdouble p[3], q[3], v12[3];

g_return_if_fail(segments != 0);

/* force ordering of v1 & v2 to get correct quad normals */
if (va[2] > vb[2])
  {
  ARR3SET(v1, vb);
  ARR3SET(v2, va);
  }
else
  {
  ARR3SET(v1, va);
  ARR3SET(v2, vb);
  }

/* vector from v1 to v2 */
ARR3SET(v12, v2);
ARR3SUB(v12, v1);

/* create vectors p and q, co-planar with the cylinder's cross-sectional disk */
ARR3SET(p, v12);
if (p[0] == 0.0 && p[1] == 0.0)
  p[0] += 1.0;
else
  p[2] += 1.0;
crossprod(q, p, v12);
crossprod(p, v12, q);

normalize(p, 3);
normalize(q, 3);

/* build the cone from triangular segments */
glBegin(GL_TRIANGLE_FAN);

/* tip */
glVertex3dv(vb);

/* base */
for (i=0 ; i<=segments ; i++) 
  {
/* sweep out a circle */
  theta = (gdouble) i * 2.0 * PI / (gdouble) segments;
  st = tbl_sin(theta);
  ct = tbl_cos(theta);
/* construct normal */
  v12[0] = ct * p[0] + st * q[0];
  v12[1] = ct * p[1] + st * q[1];
  v12[2] = ct * p[2] + st * q[2];
  glNormal3dv(v12);
/* point 1 on disk 1 */
  ARR3SET(v, va);
  v[0] += base*v12[0];
  v[1] += base*v12[1];
  v[2] += base*v12[2];
  glVertex3dv(v);
  }
glEnd();
}

/*********************************************/
/* draw a vector cylinder between two points */
/*********************************************/
void draw_vector(gdouble *va, gdouble *vb, gdouble size)
{
gdouble vec[3];

draw_cylinder(va, vb, size, 5);

ARR3SET(vec, vb);
ARR3SUB(vec, va);
normalize(vec, 3);
VEC3MUL(vec, 4.0*size);
ARR3ADD(vec, vb);

draw_cone(vb, vec, 3*size, 9);
}

/*******************************/
/* return approx. string width */
/*******************************/
gint gl_text_width(gchar *str)
{
gint i, width=0;

for (i=0 ; i<strlen(str) ; i++)
  width += gl_fontsize;

return(width);
}

/******************************/
/* print at a window position */
/******************************/
void gl_print_win(gint x, gint y, gchar *str)
{
gdouble w[3];

/* the use of 3 coords allows us to put text above everything else */
gl_get_world_coords(x, y, w);
glRasterPos3f(w[0], w[1], w[2]); 

glListBase(font_offset);
glCallLists(strlen(str), GL_UNSIGNED_BYTE, str);
}

/*****************************/
/* print at a world position */
/*****************************/
void gl_print(gdouble x, gdouble y, gdouble z, gchar *str)
{
/* set the raster position & draw the text */
glRasterPos3f(x,y,z); 

glListBase(font_offset);
glCallLists(strlen(str), GL_UNSIGNED_BYTE, str);
}

/********************************/
/* vertex at 2D screen position */
/********************************/
void gl_vertex_win(gint x, gint y)
{
gdouble w[3];

gl_get_world_coords(x, y, w);
glVertex3dv(w);
}

/********************************/
/* OpenGL atom location routine */
/********************************/
#define DEBUG_GL_SEEK_BOND 0
GSList *gl_seek_bond(GtkWidget *w, gint x, gint y, struct model_pak *data)
{
gdouble r[3];
gdouble dx, dy, d2;
gdouble tol;
GSList *list, *match=NULL;
GdkGLContext *glcontext;
GdkGLDrawable *gldrawable;
struct bond_pak *bdata;
struct core_pak *core1, *core2;

/* is there an OpenGL canvas? */
glcontext = gtk_widget_get_gl_context(w);
gldrawable = gtk_widget_get_gl_drawable(w);
if (!gdk_gl_drawable_gl_begin(gldrawable, glcontext))
  return(NULL);

gl_get_world_coords(x, y, r);

/* this is probably close to as good as we can do, given */
/* all the different display modes (eg b&s, cpk, stick) */
tol = data->scale / data->rmax;
list = data->bonds;
while (list != NULL)
  {
  bdata = (struct bond_pak *) list->data; 
  core1 = bdata->atom1;
  core2 = bdata->atom2;

  dx = 0.5*data->scale*(core1->rx[0] + core2->rx[0]) - r[0];
  dy = 0.5*data->scale*(core1->rx[1] + core2->rx[1]) - r[1];

  d2 = dx*dx + dy*dy;
  if (d2 < tol)
    {
/* keep searching - return the best match */
    tol = d2;
    match = list;
    }
  list = g_slist_next(list);
  }

return(match);
}

/***************************************************/
/* compute atoms that lie within the selection box */
/***************************************************/
#define DEBUG_GL_SELECT_BOX 0
void gl_select_box(GtkWidget *w)
{
gint i;
gdouble tmp, wbox1[3], wbox2[3], r[3];
GSList *list;
GdkGLContext *glcontext;
GdkGLDrawable *gldrawable;
struct model_pak *data;
struct core_pak *core;

/* is there an OpenGL canvas? */
glcontext = gtk_widget_get_gl_context(w);
gldrawable = gtk_widget_get_gl_drawable(w);
if (!gdk_gl_drawable_gl_begin(gldrawable, glcontext))
  return;

data = model_ptr(sysenv.active, RECALL);
g_return_if_fail(data != NULL);

/* get real coords for selection box */
gl_get_world_coords(data->select_box[0], data->select_box[1], wbox1);
gl_get_world_coords(data->select_box[2], data->select_box[3], wbox2);

gdk_gl_drawable_gl_end(gldrawable);

/* ensure box limits have the correct order (ie low to high) */
for (i=0 ; i<2 ; i++)
  {
  if (wbox1[i] > wbox2[i])
    {
    tmp = wbox1[i];
    wbox1[i] = wbox2[i];
    wbox2[i] = tmp;
    }
  }

/* NEW - very small box -> a single click */
if (abs(data->select_box[0] - data->select_box[2]) < 5)
  {
  if (abs(data->select_box[1] - data->select_box[3]) < 5)
    {
    pixel_coord_map(data->select_box[0], data->select_box[1], r, data);
    core = seek_coord2d(r, data);
    if (core)
      {
      switch (sysenv.select_mode)
        {
        case ATOM_LABEL:
          select_all_labels(core, data);
          break;
        case CORE:
          select_add_core(core, data);
          break;
        case ELEM:
          select_all_elem(core->atom_code, data);
          break;
        case ELEM_MOL:
          select_all_elem_in_mol(core, data);
          break;
        case MOL:
          select_add_mol(core->mol, data);
          break;
        case REGION:
          select_add_region(core, data);
          break;
        }
      }
    return;
    }
  }

/* account for model scaling */
wbox1[0] /= data->scale;
wbox1[1] /= data->scale;
wbox2[0] /= data->scale;
wbox2[1] /= data->scale;

/* compare and select if... */
for (list=data->cores ; list ; list=g_slist_next(list))
  {
  core = (struct core_pak *) list->data;
  if (core->status & DELETED)
    continue;

/* within box */
  if (core->rx[0] > wbox1[0] && core->rx[0] < wbox2[0])
    if (core->rx[1] > wbox1[1] && core->rx[1] < wbox2[1])
      {
      switch(sysenv.select_mode)
        {
        case CORE:
          select_add_core(core, data);
          break;
        case MOL:
          select_add_mol(core->mol, data);
          break;
        }
      }
  }

/* done */
redraw_canvas(SINGLE);
}

/**************************/
/* atom drawing primitive */
/**************************/
void gl_draw_atom_list(GSList *list, struct model_pak *data)
{
gint omit_atom;
gdouble radius;
gdouble vec[3], colour[4];
GSList *item, *ilist;
struct point_pak sphere;
struct core_pak *core;
struct image_pak *image;

gl_init_sphere(&sphere, sysenv.render.sphere_quality);

for (item=list ; item ; item=g_slist_next(item))
  {
  core = (struct core_pak *) item->data;

/* set colour */
  ARR3SET(colour, core->colour);
  VEC3MUL(colour, 1.0/65535.0);
  colour[3] = core->colour[3];
  glColor4dv(colour);

/* set radius */
  omit_atom = FALSE;
  radius = data->scale;
  switch (sysenv.render.type)
    {
    case CPK:
      radius *= sysenv.render.cpk_scale * elements[core->atom_code].vdw;
      break;

    case LIQUORICE:
/* only one bond - omit as it's a terminating atom */
/* FIXME - this will skip isolated atoms with one periodic bond */
      if (g_slist_length(core->bonds) == 1)
        omit_atom = TRUE;
/* more than one bond - put in a small sphere to smooth bond joints */
/* no bonds (isolated) - use normal ball radius */
      if (core->bonds)
        radius *= sysenv.render.stick_rad;
      else
        radius *= sysenv.render.ball_rad;
      break;

    case STICK:
      radius *= sysenv.render.stick_rad;
      if (core->bonds)
        omit_atom = TRUE;
      break;

    case BALL_STICK:
      radius *= sysenv.render.ball_rad;
      break;
    }

/* skip? */
   if (omit_atom)
     continue;

/* original + image iteration */
  ilist = NULL;
  do
    {
    ARR3SET(vec, core->rx);
    if (ilist)
      {
/* image translation */
      image = (struct image_pak *) ilist->data;
      ARR3ADD(vec, image->rx);
      ilist = g_slist_next(ilist);
      }
    else
      ilist = data->images;

    VEC3MUL(vec, data->scale);

/* draw the atom */
/*
    glo_draw_sphere(&sphere, vec, radius);
*/
    gl_draw_sphere(&sphere, vec, radius);
    }
  while (ilist);
  }

gl_free_points(&sphere);
}

/**********************************/
/* draw the selection halo/circle */
/**********************************/
void gl_draw_halo_list(GSList *list, struct model_pak *data)
{
gint h, omit_atom;
gdouble radius, dr;
gdouble vec[3], halo[4];
GSList *item, *ilist;
struct point_pak circle;
struct core_pak *core;
struct image_pak *image;

/* variable quality halo */
h = 2 + sysenv.render.sphere_quality;
h = h*h;
gl_init_circle(&circle, h);

/* halo colour */
VEC4SET(halo, 1.0, 0.95, 0.45, 1.0);
for (item=list ; item ; item=g_slist_next(item))
  {
  core = (struct core_pak *) item->data;

/* set colour */
/*
ARR3SET(halo, core->colour);
VEC3MUL(halo, 1.0/65535.0);
*/
  glColor4dv(halo);

/* set radius */
  omit_atom = FALSE;
  radius = data->scale;
  switch (sysenv.render.type)
    {
    case CPK:
      radius *= sysenv.render.cpk_scale * elements[core->atom_code].vdw;
      break;

    case LIQUORICE:
/* only one bond - omit as it's a terminating atom */
/* FIXME - this will skip isolated atoms with one periodic bond */
      if (g_slist_length(core->bonds) == 1)
        omit_atom = TRUE;
/* more than one bond - put in a small sphere to smooth bond joints */
/* no bonds (isolated) - use normal ball radius */
      if (core->bonds)
        radius *= sysenv.render.stick_rad;
      else
        radius *= sysenv.render.ball_rad;
      break;

    case STICK:
      radius *= sysenv.render.stick_rad;
      if (core->bonds)
        omit_atom = TRUE;
      break;

    case BALL_STICK:
      radius *= sysenv.render.ball_rad;
      break;
    }

/* halo ring size increment */
/* a fn of the radius? eg 1/2 */
  dr = 0.6*radius/halo_segments;

/* original + image iteration */
  ilist = NULL;
  do
    {
    ARR3SET(vec, core->rx);
    if (ilist)
      {
/* image translation */
      image = (struct image_pak *) ilist->data;
      ARR3ADD(vec, image->rx);
      ilist = g_slist_next(ilist);
      }
    else
      ilist = data->images;

    VEC3MUL(vec, data->scale);

    if (sysenv.render.halos)
      {
/* halo fade loop */
      for (h=0 ; h<halo_segments ; h++)
        {
        halo[3] = halo_fade[h];

        glColor4dv(halo);
        gl_draw_ring(&circle, vec, radius+h*dr-0.5*dr, radius+h*dr+0.5*dr);
        }
/* reset halo transparancy */
      halo[3] = 1.0;
      }
    else
      gl_draw_ring(&circle, vec, 1.2*radius, 1.4*radius);
    }
  while (ilist);
  }

gl_free_points(&circle);
}

/**************************/
/* draw the model's atoms */
/**************************/
void gl_draw_atoms(struct model_pak *data)
{
struct core_pak *core;
GSList *clist, *list;

/*  construct list of atoms to be drawn & pass those */
list = NULL;
for (clist=data->cores ; clist ; clist=g_slist_next(clist))
  {
  core = (struct core_pak *) clist->data;

  if (core->status & (HIDDEN | DELETED))
    continue;
  if (core->ghost)
    continue;

  list = g_slist_prepend(list, core);
  }

/* perform the atom drawing */
if (sysenv.render.wire_model)
  glPolygonMode(GL_FRONT, GL_LINE);

gl_draw_atom_list(list, data);

glPolygonMode(GL_FRONT, GL_FILL);

g_slist_free(list);
}

/*******************************/
/* draw the model's ghost atom */
/*******************************/
void gl_draw_ghost_atoms(struct model_pak *data)
{
struct core_pak *core;
GSList *clist, *list;

/*  construct list of atoms to be drawn & pass those */
list = NULL;
for (clist=data->cores ; clist ; clist=g_slist_next(clist))
  {
  core = (struct core_pak *) clist->data;

  if (core->status & (HIDDEN | DELETED))
    continue;
  if (core->ghost)
    list = g_slist_prepend(list, core);
  }

/* perform the appropriate drawing */
gl_draw_atom_list(list, data);

g_slist_free(list);
}

/***********************/
/* draw model's shells */
/***********************/
void gl_draw_shells(struct model_pak *data)
{
gint omit_atom;
gdouble radius;
gdouble vec[3], colour[4];
GSList *list, *ilist;
struct point_pak sphere;
struct shel_pak *shel;
struct image_pak *image;

/* set appropriate radius */
radius = data->scale;
switch (sysenv.render.type)
  {
  case BALL_STICK:
    radius *= sysenv.render.ball_rad;
    break;
  }

gl_init_sphere(&sphere, sysenv.render.sphere_quality);

for (list=data->shels ; list ; list=g_slist_next(list))
  {
  shel = (struct shel_pak *) list->data;
  if (shel->status & (DELETED | HIDDEN))
    continue;

/* shell colour */
  ARR3SET(colour, shel->colour);
  VEC3MUL(colour, 1.0/65535.0);
/* translucency */
  colour[3] = 0.5;

/* position */
  ilist=NULL;
  do
    {
    ARR3SET(vec, shel->rx);

    if (ilist)
      {
/* image */
      image = (struct image_pak *) ilist->data;

      ARR3ADD(vec, image->rx);

      ilist = g_slist_next(ilist);
      }
    else
      {
      ilist = data->images;
      }

  VEC3MUL(vec, data->scale);

/* set appropriate radius */
  omit_atom = FALSE;
  switch (sysenv.render.type)
    {
    case CPK:
      radius = data->scale * elements[shel->atom_code].vdw;
      break;

    case LIQUORICE:
/*
      if (core->bonds)
        omit_atom = TRUE;
      if (core->bonds)
        radius = data->scale * sysenv.render.stick_rad;
      else
        radius = data->scale * sysenv.render.ball_rad;
*/
      omit_atom = TRUE;

      break;

    case STICK:
      radius = data->scale * sysenv.render.stick_rad;
/*
      if (core->bonds)
        omit_atom = TRUE;
*/
      break;
    }

  if (!omit_atom)
    {
    glColor4dv(colour);
    gl_draw_sphere(&sphere, vec, radius);
    }

    }
  while (ilist);
  }

glEnable(GL_LIGHTING);

gl_free_points(&sphere);
}

/*******************************************/
/* draw model pipes (separate bond halves) */
/*******************************************/
/* stage (drawing stage) ie colour materials/lines/etc. */
/* TODO - only do minimum necessary for each stage */
void gl_draw_pipes(gint stage, GSList *pipe_list, struct model_pak *data)
{
guint i;
gdouble v1[3], v2[3];
GSList *list, *ilist;
struct point_pak circle;
struct pipe_pak *pipe;
struct image_pak *image;

/* setup for bond drawing */
if (stage < 2)
  {
  switch(sysenv.render.type)
    {
    case BALL_STICK:
      glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_FALSE);
      glColorMaterial(GL_FRONT, GL_AMBIENT_AND_DIFFUSE);
      break;

    case LIQUORICE:
/* ensure that the underside appears */
      glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_TRUE);
/* make the back face colour light grey */
      glColorMaterial(GL_BACK, GL_AMBIENT_AND_DIFFUSE);
      glColor4f(0.7,0.7,0.7,1.0);  /* normal */
/* return to front colouring */
      glColorMaterial(GL_FRONT, GL_AMBIENT_AND_DIFFUSE);
      break;
    }

/* calc maximum distinguishable cylinder segmentation */
/* NB: enforce range (0,cylinder_quality) */
  switch(sysenv.render.type)
    {
    case BALL_STICK:
    case LIQUORICE:
      i = data->scale * sysenv.size / (10.0 * data->rmax);
      i++;
      if (i > sysenv.render.cylinder_quality)
        gl_init_circle(&circle, sysenv.render.cylinder_quality);
      else
        gl_init_circle(&circle, i);
    }
  }

/* enumerate the supplied pipes (half bonds) */
for (list=pipe_list ; list ; list=g_slist_next(list))
  {
  pipe = (struct pipe_pak *) list->data;

/* original + image iteration */
  ilist = NULL;
  do
    {
/* original */
    ARR3SET(v1, pipe->v1);
    ARR3SET(v2, pipe->v2);
    if (ilist)
      {
      image = (struct image_pak *) ilist->data;
/* image */
      ARR3ADD(v1, image->rx);
      ARR3ADD(v2, image->rx);
      ilist = g_slist_next(ilist);
      }
    else
      ilist = data->images;

    VEC3MUL(v1, data->scale);
    VEC3MUL(v2, data->scale);

    glColor4dv(pipe->colour);
    if (stage < 2)
      gl_draw_cylinder(&circle, v1, v2, pipe->radius);
    else
      {
      glBegin(GL_LINES);
      glVertex3dv(v1);
      glVertex3dv(v2);
      glEnd();
      }
    }
  while (ilist);
  }

/* cleanup after bond drawing */
if (stage < 2)
  {
  switch(sysenv.render.type)
    {
    case BALL_STICK:
    case LIQUORICE:
      gl_free_points(&circle);
    }
  }
}

/**********************************/
/* setup pipe list (bond drawing) */
/**********************************/
/* 0 - normal (solid) */
/* 1 - ghost (solid) */
/* 2 - line (stick) */
GSList *render_get_pipes(gint stage, struct model_pak *model)
{
gint part;
gdouble radius, v1[4], v2[4], mp[3], v1i[4], v2i[4], mp12i[4], mp1i2[4];
gdouble colour1[4], colour2[4];
gdouble tmp4[16], mat4[16];
GSList *list, *pipe_list=NULL;
struct pipe_pak *pipe;
struct bond_pak *bond;
struct core_pak *core1, *core2;

/* trap pointless calls */
if (sysenv.render.type == CPK)
  return(NULL);

switch (stage)
  {
  case 0:
  case 1:
    if (sysenv.render.type == STICK)
      return(NULL);
    break;

  case 2:
/*
    if (sysenv.render.type != STICK)
      return(NULL);
*/
    break;

  default:
    return(NULL);
  }

/* precalc matrix products */
ARR3SET(&mat4[0], &model->latmat[0]);
ARR3SET(&mat4[4], &model->latmat[3]);
ARR3SET(&mat4[8], &model->latmat[6]);
ARR3SET(v1, model->centroid);
vecmat(model->latmat, v1);
mat4[3] = -v1[0];
mat4[7] = -v1[1];
mat4[11] = -v1[2];
VEC4SET(&mat4[12], 0.0, 0.0, 0.0, 1.0);

ARR3SET(&tmp4[0], &model->rotmat[0]);
ARR3SET(&tmp4[4], &model->rotmat[3]);
ARR3SET(&tmp4[8], &model->rotmat[6]);
tmp4[3] = 0.0;
tmp4[7] = 0.0;
tmp4[11] = 0.0;
VEC4SET(&tmp4[12], 0.0, 0.0, 0.0, 1.0);
mat4mat(tmp4, mat4);

/* common pipe width */
radius = model->scale*sysenv.render.stick_rad;

/* enumerate bonds to construct the pipe list */
for (list=model->bonds; list ; list=g_slist_next(list))
  {
  bond = (struct bond_pak *) list->data;

  if (bond->status == DELETED)
    continue;
  if (bond->status == HIDDEN)
    continue;

/* the two atoms */
  core1 = bond->atom1;
  core2 = bond->atom2;
  ARR3SET(v1, core1->rx);
  ARR3SET(v2, core2->rx);

  switch (bond->periodic)
    {
    case BOND_SPLIT:
/* periodic image positions */
      ARR3SET(v2i, core2->x);
      v2i[3] = 1.0;
      ARR3ADD(v2i, bond->pic);
      vec4mat(mat4, v2i);

      ARR3SET(v1i, core1->x);
      v1i[3] = 1.0;
      ARR3SUB(v1i, bond->pic);
      vec4mat(mat4, v1i);

/* periodic image midpoint positions */
      ARR3SET(mp1i2, v1i);
      ARR3ADD(mp1i2, v2);
      VEC3MUL(mp1i2, 0.5);

      ARR3SET(mp12i, v1);
      ARR3ADD(mp12i, v2i);
      VEC3MUL(mp12i, 0.5);
      break;

    default:
/* merged/normal bond midpoint */
      ARR3SET(mp, v1);
      ARR3ADD(mp, v2);
      VEC3MUL(mp, 0.5);
      break;
    }

/* deleted/ zeol hidden */
  if (core1->status & (DELETED | ZEOL_HIDDEN))
    continue;
  if (core2->status & (DELETED | ZEOL_HIDDEN))
    continue;

/* colour setup */
  ARR4SET(colour1, core1->colour);
  VEC3MUL(colour1, 1.0/65535.0);

  ARR4SET(colour2, core2->colour);
  VEC3MUL(colour2, 1.0/65535.0);

/* test which parts of the bond (pipe) should be drawn */
  part = 0;
  switch (stage)
    {
/* normal */
    case 0:
      if (bond->type == BOND_HBOND)
        continue;
      if (!core1->ghost)
        if (!(core1->status & HIDDEN))
          part |= 1;
      if (!core2->ghost)
        if (!(core2->status & HIDDEN))
          part |= 2;
      break;

/* ghost */
    case 1:
      if (bond->type == BOND_HBOND)
        continue;
      if (core1->ghost)
        if (!(core1->status & HIDDEN))
          part |= 1;
      if (core2->ghost)
        if (!(core2->status & HIDDEN))
          part |= 2;
      break;

/* line bonds */
    default:
/* if not stick, only draw if it's a HBOND */
      if (sysenv.render.type != STICK)
        if (bond->type != BOND_HBOND)
          continue;
      if (!(core1->status & HIDDEN))
        part |= 1;
      if (!(core2->status & HIDDEN))
        part |= 2;
      break;
    }

/* the core1 pipe */
  if (part & 1)
    {
    pipe = g_malloc(sizeof(struct pipe_pak));

    ARR3SET(pipe->v1, v1);

    if (bond->periodic == BOND_SPLIT)
      {
      ARR3SET(pipe->v2, mp12i);
      }
    else
      {
      ARR3SET(pipe->v2, mp);
      }

    pipe->radius = radius;

    ARR4SET(pipe->colour, colour1);

    pipe_list = g_slist_prepend(pipe_list, pipe);
    }

/* the core2 pipe */
  if (part & 2)
    {
    pipe = g_malloc(sizeof(struct pipe_pak));

    ARR3SET(pipe->v1, v2);

    if (bond->periodic == BOND_SPLIT)
      {
      ARR3SET(pipe->v2, mp1i2);
      }
    else
      {
      ARR3SET(pipe->v2, mp);
      }

    pipe->radius = radius;

    ARR4SET(pipe->colour, colour2);

    pipe_list = g_slist_prepend(pipe_list, pipe);
    }
  }
return(pipe_list);
}

/**************************/
/* draw all bond types */
/**************************/
/* 0 - normal (solid) */
/* 1 - ghost (solid) */
/* 2 - line (stick) */
void gl_draw_bonds(gint stage, struct model_pak *model)
{
GSList *list;

if (sysenv.render.wire_model)
  glPolygonMode(GL_FRONT, GL_LINE);

list = render_get_pipes(stage, model);
if (list)
  {
  gl_draw_pipes(stage, list, model);
  free_slist(list);
  }

glPolygonMode(GL_FRONT, GL_FILL);
}

/***************************/
/* draw crystal morphology */
/***************************/
#define DEBUG_DRAW_MORPH 0
void gl_draw_morph(struct model_pak *data)
{
gdouble v1[3], n[3];
GSList *list1, *list2;
struct plane_pak *plane;
struct vertex_pak *v;

#if DEBUG_DRAW_MORPH
printf("gl_draw_morph()\n");
#endif

/* go through plane list */
for (list1=data->planes ; list1 ; list1=g_slist_next(list1))
  {
  plane = (struct plane_pak *) list1->data;

  if (!plane->present)
    continue;

/* NEW - dashed hidden lines */
  if (!plane->visible)
    {
    if (sysenv.render.wire_surface && sysenv.render.wire_show_hidden)
      {
      glEnable(GL_LINE_STIPPLE);
      glLineStipple(1, 0x0F0F);
      }
    else
      continue;
    }

#if DEBUG_DRAW_MORPH
printf("Drawing: %f %f %f\n", plane->m[0], plane->m[1], plane->m[2]);
#endif
/* start the face */
  glBegin(GL_POLYGON);
/* surface normal */
  ARR3SET(n, plane->norm);
  vecmat(data->rotmat, n);
  normalize(n, 3);
  glNormal3dv(n);
  for (list2=plane->vertices ; list2 ; list2=g_slist_next(list2))
    {
    v = (struct vertex_pak *) list2->data;

#if DEBUG_DRAW_MORPH
P3VEC("  : ", v->rx);
#endif

    ARR3SET(v1, v->rx);
    VEC3MUL(v1, data->scale);
    glVertex3dv(v1);
    }
  glEnd();
  glDisable(GL_LINE_STIPPLE);
  }
}

/***********************************/
/* draw the cartesian/lattice axes */
/***********************************/
void gl_draw_axes(gint mode, struct canvas_pak *canvas, struct model_pak *data)
{
gint i;
gdouble x1[3], x2[3];
gchar label[3];

/* axes type setup */
if (data->axes_type == CARTESIAN)
  strcpy(label, " x");
else
  strcpy(label, " a");

/* position - upper left corner */
gl_get_world_coords(40+canvas->x, 40+canvas->y, x1);

x1[2] = 0.0;

if (mode)
  {
/* set colour */
  glColor4f(sysenv.render.fg_colour[0], sysenv.render.fg_colour[1],
            sysenv.render.fg_colour[2], 1.0);
/* draw the axes */
  for (i=3 ; i-- ; )
    {
    ARR3SET(x2, x1);
    ARR3ADD(x2, data->axes[i].rx);
/* a 200th the size of rmax */
    draw_vector(x1, x2, 0.005*data->rmax);
    }
  }
else
  {
/* draw the labels - offset by dfontsize? */
  glColor4f(sysenv.render.title_colour[0], sysenv.render.title_colour[1],
            sysenv.render.title_colour[2], 1.0);
  for (i=0 ; i<3 ; i++)
    {
    ARR3SET(x2, x1);
    ARR3ADD(x2, data->axes[i].rx);
    gl_print(x2[0], x2[1], x2[2], label);
    label[1]++;
    }
  }
}

/*******************************************/
/* draw the cell frame for periodic models */
/*******************************************/
void gl_draw_cell(struct model_pak *data)
{
gint i, j;
gdouble v1[3], v2[3], v3[3], v4[3];

/* draw the opposite ends of the frame */
for (i=0 ; i<5 ; i+=4)
  {
  glBegin(GL_LINE_LOOP);
  ARR3SET(v1, data->cell[i+0].rx);
  ARR3SET(v2, data->cell[i+1].rx);
  ARR3SET(v3, data->cell[i+2].rx);
  ARR3SET(v4, data->cell[i+3].rx);
  VEC3MUL(v1, data->scale);
  VEC3MUL(v2, data->scale);
  VEC3MUL(v3, data->scale);
  VEC3MUL(v4, data->scale);
  glVertex3dv(v1);
  glVertex3dv(v2);
  glVertex3dv(v3);
  glVertex3dv(v4);
  glEnd();
  }
/* draw the sides of the frame */
glBegin(GL_LINES);
for (i=4 ; i-- ; )
  {
  j = i+4;
/* retrieve coordinates */
  ARR3SET(v1, data->cell[i].rx);
  ARR3SET(v2, data->cell[j].rx);
/* scale */
  VEC3MUL(v1, data->scale);
  VEC3MUL(v2, data->scale);
/* draw */
  glVertex3dv(v1);
  glVertex3dv(v2);
  }
glEnd();
}

/***************************************/
/* draw the cell frame periodic images */
/***************************************/
void gl_draw_cell_images(struct model_pak *model)
{
gint i, j;
gdouble v1[3], v2[3], v3[3], v4[3];
GSList *ilist;
struct image_pak *image;

/* image iteration (don't do original) */
ilist = model->images;
for (ilist=model->images ; ilist ; ilist=g_slist_next(ilist))
  {
/* image translation */
  image = (struct image_pak *) ilist->data;

/* draw the opposite ends of the frame */
for (i=0 ; i<5 ; i+=4)
  {
  glBegin(GL_LINE_LOOP);
  ARR3SET(v1, model->cell[i+0].rx);
  ARR3SET(v2, model->cell[i+1].rx);
  ARR3SET(v3, model->cell[i+2].rx);
  ARR3SET(v4, model->cell[i+3].rx);
ARR3ADD(v1, image->rx);
ARR3ADD(v2, image->rx);
ARR3ADD(v3, image->rx);
ARR3ADD(v4, image->rx);
  VEC3MUL(v1, model->scale);
  VEC3MUL(v2, model->scale);
  VEC3MUL(v3, model->scale);
  VEC3MUL(v4, model->scale);
  glVertex3dv(v1);
  glVertex3dv(v2);
  glVertex3dv(v3);
  glVertex3dv(v4);
  glEnd();
  }
/* draw the sides of the frame */
glBegin(GL_LINES);
for (i=4 ; i-- ; )
  {
  j = i+4;
/* retrieve coordinates */
  ARR3SET(v1, model->cell[i].rx);
  ARR3SET(v2, model->cell[j].rx);
ARR3ADD(v1, image->rx);
ARR3ADD(v2, image->rx);
/* scale */
  VEC3MUL(v1, model->scale);
  VEC3MUL(v2, model->scale);
/* draw */
  glVertex3dv(v1);
  glVertex3dv(v2);
  }
glEnd();
  }
}

/*********************/
/* draw measurements */
/*********************/
void gl_draw_geom(struct model_pak *data)
{
gint i;
gdouble n[3], v1[3], v2[3], v3[3];
GSList *list, *list2;
struct core_pak *core[4];
struct geom_pak *geom;

/* draw the lines */
for (list=data->geom ; list ; list=g_slist_next(list))
  {
  geom = (struct geom_pak *) list->data;

  i=0;
  for (list2=geom->cores ; list2 ; list2=g_slist_next(list2))
    core[i++] = (struct core_pak *) list2->data;

  switch(geom->type)
    {
    case BOND:
    case DIST:
      ARR3SET(v1, core[0]->rx);
      ARR3SET(v2, core[1]->rx);
      VEC3MUL(v1, data->scale);
      VEC3MUL(v2, data->scale);
      glBegin(GL_LINES);
      glVertex3dv(v1);
      glVertex3dv(v2);
      glEnd();
      break;

    case ANGLE:
      ARR3SET(v1, core[0]->rx);
      ARR3SET(v2, core[1]->rx);
      ARR3SET(v3, core[2]->rx);
      VEC3MUL(v1, data->scale);
      VEC3MUL(v2, data->scale);
      VEC3MUL(v3, data->scale);
/* middle one is first */
      draw_arc(v2, v1, v3);
      break;

    case DIHEDRAL:
/* middle 2 cores define the axis */
      ARR3SET(n, core[2]->rx);
      ARR3SUB(n, core[1]->rx);
      normalize(n, 3);
/* arm 1 */
      ARR3SET(v3, core[0]->rx);
      ARR3SUB(v3, core[1]->rx);
      proj_vop(v1, v3, n);
      normalize(v1, 3);
/* arm 2 */
      ARR3SET(v3, core[3]->rx);
      ARR3SUB(v3, core[2]->rx);
      proj_vop(v2, v3, n);
      normalize(v2, 3);
/* axis centre */
      ARR3SET(v3, core[1]->rx);
      ARR3ADD(v3, core[2]->rx);
      VEC3MUL(v3, 0.5);
/* arm endpoints are relative to axis centre */
      ARR3ADD(v1, v3);
      ARR3ADD(v2, v3);
/* scale */
      VEC3MUL(v1, data->scale);
      VEC3MUL(v2, data->scale);
      VEC3MUL(v3, data->scale);
/* draw arc */
      draw_arc(v3, v1, v2);
/* draw lines */
      glBegin(GL_LINE_STRIP);
      glVertex3dv(v1);
      glVertex3dv(v3);
      glVertex3dv(v2);
      glEnd();
      break;
    }
  }
}

/***************************/
/* draw morphology indices */
/***************************/
void gl_draw_miller(struct model_pak *data)
{
gchar *label;
gdouble vec[3];
GSList *plist;
struct plane_pak *plane;

/* draw facet labels */
plist = data->planes;
while (plist != NULL)
  {
  plane = (struct plane_pak *) plist->data;
  if (plane->present && plane->visible)
    {
/* TODO - scale the font with data->scale? (vanishes if too small) */
/* print the hkl label */
    label = g_strdup_printf("%d%d%d", plane->index[0],
                                      plane->index[1],
                                      plane->index[2]);
    ARR3SET(vec, plane->rx);
    VEC3MUL(vec, data->scale);
    gl_print(vec[0], vec[1], vec[2], label);
    g_free(label);

/* TODO - vector font (display list) with number + overbar number */
/*
    glBegin(GL_LINES);
    if (plane->index[0] < 0)
      {
      glVertex3d(plane->rx, plane->ry-gl_fontsize, plane->rz);
      glVertex3d(plane->rx+gl_fontsize, plane->ry-gl_fontsize, plane->rz);
      }
    if (plane->index[1] < 0)
      {
      glVertex3d(plane->rx+1*gl_fontsize, plane->ry-gl_fontsize, plane->rz);
      glVertex3d(plane->rx+2*gl_fontsize, plane->ry-gl_fontsize, plane->rz);
      }
    if (plane->index[2] < 0)
      {
      glVertex3d(plane->rx+2*gl_fontsize, plane->ry-gl_fontsize, plane->rz);
      glVertex3d(plane->rx+3*gl_fontsize, plane->ry-gl_fontsize, plane->rz);
      }
    glEnd();
*/

    }
  plist = g_slist_next(plist);
  }
}

/**************/
/* draw a box */
/**************/
void gl_draw_box(gint px1, gint py1, gint px2, gint py2)
{
glBegin(GL_LINE_LOOP);
gl_vertex_win(px1, py1);
gl_vertex_win(px1, py2);
gl_vertex_win(px2, py2);
gl_vertex_win(px2, py1);
glEnd();
}

/************************************************/
/* draw the colour scale for molecular surfaces */
/************************************************/
void gl_draw_colour_scale(gint x, gint y, struct model_pak *data)
{
gint i, n;
gdouble z1, dz;
gdouble colour[3], w1[3], w2[3];
GString *text;

/* init */
n = data->gulp.epot_divisions;
z1 = data->gulp.epot_max;
dz = (data->gulp.epot_max - data->gulp.epot_min)/ (gdouble) (n-1);

/* colour boxes */
glBegin(GL_QUADS);
for (i=0 ; i<n ; i++)
  {
  ms_epot_colour(colour, z1, data);
  glColor3f(colour[0], colour[1], colour[2]);
  z1 -= dz;

  gl_get_world_coords(x, y+i*20, w1);
  gl_get_world_coords(x, y+19+i*20, w2);
  glVertex3dv(w1);
  glVertex3dv(w2);

  gl_get_world_coords(x+19, y+19+i*20, w1);
  gl_get_world_coords(x+19, y+i*20, w2);
  glVertex3dv(w1);
  glVertex3dv(w2);
  }
glEnd();

/* init */
text = g_string_new(NULL);
z1 = data->gulp.epot_max;
glColor3f(1.0, 1.0, 1.0);

/* box labels */
for (i=0 ; i<n ; i++)
  {
  g_string_sprintf(text, "%6.2f", z1);
  gl_print_win(x+30, y+i*20+18, text->str);
  z1 -= dz;
  }
g_string_free(text, TRUE);
}

/**************************************/
/* draw text we wish to be unobscured */
/**************************************/
void gl_draw_text(struct model_pak *data)
{
gint i, j, nv=0;
gchar *text;
gdouble q, n[3], v1[3], v2[3], v3[3];
GSList *list, *list2, *item;
struct vec_pak *p1, *p2;
struct spatial_pak *spatial;
struct geom_pak *geom;
struct core_pak *core[4];
struct shel_pak *shell;

/* print mode */
text = get_mode_label(data);
gl_print_win(sysenv.x+sysenv.size-gl_text_width(text), sysenv.size - 20, text);
g_free(text);

/* print some useful info */
if (data->show_title)
  {
  if (data->title)
    {
    text = g_strdup(data->title);
    gl_print_win(sysenv.x+sysenv.size-gl_text_width(text), sysenv.y + 40, text);
    g_free(text);
    }
  }
/* print current frame */
if (data->show_frame_number)
  {
  if (data->animation)
    {
    text = g_strdup_printf("Frame [%d:%d]", data->cur_frame, data->num_frames-1);
    gl_print_win(sysenv.x+sysenv.size-gl_text_width(text), sysenv.y + 80, text);
    g_free(text);
    }
  }

/* print energy */
if (data->show_energy)
  {
/* Eatt & Esurf are now the only energies printed here */
/* NB: the total energy should be put in the title string */
  if (data->periodic == 2)
    {
    if (data->gulp.eatt[1] == 0.0)
      text = g_strdup_printf("Ea = %f %s", data->gulp.eatt[0],
                                           data->gulp.eatt_units);
    else
      text = g_strdup_printf("Ea = %f %s", data->gulp.eatt[1],
                                           data->gulp.eatt_units);
/* print Eatt */
    gl_print_win(sysenv.x, sysenv.size - 40, text);
    g_free(text);

/* get Esurf */
    if (data->gulp.esurf[1] == 0.0)
      text = g_strdup_printf("Es = %f %s",data->gulp.esurf[0],
                                          data->gulp.esurf_units);
    else
      text = g_strdup_printf("Es = %f %s",data->gulp.esurf[1],
                                          data->gulp.esurf_units);

/* print Elatt/Esurf */
    gl_print_win(sysenv.x, sysenv.size - 20, text);
    g_free(text);
    }
  }

/* hkl labels */
if (data->num_vertices && data->morph_label)
  gl_draw_miller(data);

/* unit cell lengths */
if (data->show_cell_lengths)
  {
  j=2;
  for (i=0 ; i<data->periodic ; i++)
    {
    j += pow(-1, i) * (i+1);
    text = g_strdup_printf("%5.2f", data->pbc[i]);
    ARR3SET(v1, data->cell[0].rx);
    ARR3ADD(v1, data->cell[j].rx);
    VEC3MUL(v1, 0.5*data->scale);
    gl_print(v1[0], v1[1], v1[2], text);
    g_free(text);
    }
  }

/* NEW - epot scale */
/* TODO - a toggle to show/hide this */
/* TODO - similar AFM scale */
if (data->csurf_on && data->csurf.colour_method == MS_EPOT)
  gl_draw_colour_scale(sysenv.x, 100, data);

/* the following text is likely to have partial */
/* overlapping, so XOR text fragments for clearer display */
glEnable(GL_COLOR_LOGIC_OP);
glLogicOp(GL_XOR);
glColor4f(1.0, 1.0, 1.0, 1.0);
/*
glColor4f(sysenv.render.fg_colour[0],
          sysenv.render.fg_colour[1],
	  sysenv.render.fg_colour[2], 1.0);
*/

/* print atom labels */
if (data->show_atom_labels)
  {
/* loop over atoms */
  for (list=data->cores ; list ; list=g_slist_next(list))
    {
    core[0] = (struct core_pak *) list->data;
    if (core[0]->status & (DELETED | HIDDEN))
      continue;

/* print element type */
    text = g_strdup_printf("%s", core[0]->label);
    ARR3SET(v1, core[0]->rx);
    VEC3MUL(v1, data->scale);
    gl_print(v1[0], v1[1], v1[2], text);
    g_free(text);
    }
  }

/* print charges */
if (data->show_charges)
  {
/* net charge */
  text = g_strdup_printf("Q = %6.3f", data->gulp.qsum);
  gl_print_win(sysenv.x, sysenv.size - 60, text);
  g_free(text);

/* loop over atoms */
  for (list=data->cores ; list ; list=g_slist_next(list))
    {
    core[0] = (struct core_pak *) list->data;
    if (core[0]->status & (DELETED | HIDDEN))
      continue;

/* get atom charge, add shell charge (if any) to get net result */
    q = core[0]->charge;
    if (core[0]->shell)
      {
      shell = (struct shel_pak *) core[0]->shell;
      q += shell->charge;
      }
    text = g_strdup_printf("%6.3f", q);

/* print net charge */
    ARR3SET(v1, core[0]->rx);
    VEC3MUL(v1, data->scale);
    gl_print(v1[0], v1[1], v1[2], text);

    g_free(text);
    }
  }

/* geom measurement labels */
if (data->show_geom_labels)
for (list=data->geom ; list ; list=g_slist_next(list))
  {
  geom = (struct geom_pak *) list->data;

  g_assert(g_slist_length(geom->cores) < 5);

  i=0;
  for (list2=geom->cores ; list2 ; list2=g_slist_next(list2))
    core[i++] = (struct core_pak *) list2->data;

  switch(geom->type)
    {
    case BOND:
    case DIST:
      ARR3SET(v1, core[0]->rx);
      ARR3SET(v2, core[1]->rx);
      ARR3ADD(v1, v2);
      VEC3MUL(v1, 0.5);
      VEC3MUL(v1, data->scale);
      gl_print(v1[0], v1[1], v1[2], geom->text);
      break;

    case ANGLE:
/* angle is i-j-k */
      ARR3SET(v1, core[0]->rx);
      ARR3SET(v2, core[1]->rx);
      ARR3SET(v3, core[2]->rx);
      VEC3MUL(v1, data->scale);
      VEC3MUL(v2, data->scale);
      VEC3MUL(v3, data->scale);
/* angle label */
/* FIXME - should use a similar process to the draw_arc code to */
/* determine which arm is shorter & use that to determine label position */
      ARR3ADD(v1, v2);
      ARR3ADD(v1, v3);
      VEC3MUL(v1, 0.3333);
      gl_print(v1[0], v1[1], v1[2], geom->text);
      break;

    case DIHEDRAL:
/* middle 2 cores define the axis */
      ARR3SET(n, core[2]->rx);
      ARR3SUB(n, core[1]->rx);
      normalize(n, 3);
/* arm 1 */
      ARR3SET(v3, core[0]->rx);
      ARR3SUB(v3, core[1]->rx);
      proj_vop(v1, v3, n);
      normalize(v1, 3);
/* arm 2 */
      ARR3SET(v3, core[3]->rx);
      ARR3SUB(v3, core[2]->rx);
      proj_vop(v2, v3, n);
      normalize(v2, 3);
/* middle of arm1 & arm2 */
      ARR3ADD(v1, v2);
      VEC3MUL(v1, 0.5);
/* axis centre */
      ARR3SET(v3, core[1]->rx);
      ARR3ADD(v3, core[2]->rx);
      VEC3MUL(v3, 0.5);
/* label position is relative to axis midpoint */
      ARR3ADD(v1, v3);
/* scale */
      VEC3MUL(v1, data->scale);
      gl_print(v1[0], v1[1], v1[2], geom->text);
      break;
    }
  }

/* spatial object labels */
list = data->spatial;
while (list)
  {
  spatial = (struct spatial_pak *) list->data;
  switch(spatial->type)
    {
    case SPATIAL_VECTOR:
/* FIXME - give this it's own label on/off flag */
      if (!data->show_atom_labels)
        break;
      item = (GSList *) spatial->data;
      p1 = (struct vec_pak *) item->data;
      item = g_slist_next(item);
      p2 = (struct vec_pak *) item->data;
      ARR3SET(v1, p1->rx);
      ARR3SET(v2, p2->rx);
      VEC3MUL(v1, data->scale);
      VEC3MUL(v2, data->scale);
      VEC3MUL(v1, 0.5);
      ARR3ADD(v1, v2);
      VEC3MUL(v1, 0.67);
      text = g_strdup_printf("%d", nv);
      gl_print(v1[0], v1[1], v1[2], text);
      g_free(text);
      nv++;
      break;
    }
  list = g_slist_next(list);
  }

glDisable(GL_COLOR_LOGIC_OP);
}

/********************************/
/* draw a ribbon special object */
/********************************/
void gl_draw_ribbon(struct object_pak *odata, struct model_pak *data)
{
gint i;
gdouble len, vec1[3], vec2[3];
GSList *rlist;
struct ribbon_pak *ribbon;
GLfloat ctrl[8][3];

/* go through the ribbon segment list */
rlist = (GSList *) odata->data;
while (rlist != NULL)
  {
  ribbon = (struct ribbon_pak *) rlist->data;

  glColor4f(ribbon->colour[0], ribbon->colour[1], ribbon->colour[2],
                                            sysenv.render.transmit);

/* end points */
  ARR3SET(&ctrl[0][0], ribbon->r1);
  ARR3SET(&ctrl[3][0], ribbon->r2);

/* get distance between ribbon points */
  ARR3SET(vec1, ribbon->x1);
  ARR3SUB(vec1, ribbon->x2);
  len = VEC3MAG(vec1);

/* shape control points */
  ARR3SET(&ctrl[1][0], ribbon->r1);
  ARR3SET(&ctrl[2][0], ribbon->r2);

/* segment length based curvature - controls how flat it is at the cyclic group */
  ARR3SET(vec1, ribbon->o1);
  VEC3MUL(vec1, len*sysenv.render.ribbon_curvature);
  ARR3ADD(&ctrl[1][0], vec1);
  ARR3SET(vec2, ribbon->o2);
  VEC3MUL(vec2, len*sysenv.render.ribbon_curvature);
  ARR3ADD(&ctrl[2][0], vec2);

/* compute offsets for ribbon thickness */
  crossprod(vec1, ribbon->n1, ribbon->o1);
  crossprod(vec2, ribbon->n2, ribbon->o2);
  normalize(vec1, 3);
  normalize(vec2, 3);

/* thickness vectors for the two ribbon endpoints */
  VEC3MUL(vec1, 0.5*sysenv.render.ribbon_thickness);
  VEC3MUL(vec2, 0.5*sysenv.render.ribbon_thickness);

/* ensure these are pointing the same way */
  if (via(vec1, vec2, 3) > PI/2.0)
    {
    VEC3MUL(vec2, -1.0);
    }

/* init the bottom edge control points */
  ARR3SET(&ctrl[4][0], &ctrl[0][0]);
  ARR3SET(&ctrl[5][0], &ctrl[1][0]);
  ARR3SET(&ctrl[6][0], &ctrl[2][0]);
  ARR3SET(&ctrl[7][0], &ctrl[3][0]);
/* lift points to make the top edge */
  ARR3ADD(&ctrl[0][0], vec1);
  ARR3ADD(&ctrl[1][0], vec1);
  ARR3ADD(&ctrl[2][0], vec2);
  ARR3ADD(&ctrl[3][0], vec2);
/* lower points to make the bottom edge */
  ARR3SUB(&ctrl[4][0], vec1);
  ARR3SUB(&ctrl[5][0], vec1);
  ARR3SUB(&ctrl[6][0], vec2);
  ARR3SUB(&ctrl[7][0], vec2);

  for (i=0 ; i<8 ; i++)
    {
    VEC3MUL(&ctrl[i][0], data->scale);
    }
  
/* drawing */
  glMap2f(GL_MAP2_VERTEX_3, 
          0.0, 1.0, 3, 4,
          0.0, 1.0, 3*4, 2,
          &ctrl[0][0]);
  glEnable(GL_MAP2_VERTEX_3);
  glEnable(GL_AUTO_NORMAL);
  glMapGrid2f(sysenv.render.ribbon_quality, 0.0, 1.0, 3, 0.0, 1.0);
  glEvalMesh2(GL_FILL, 0, sysenv.render.ribbon_quality, 0, 3);

  rlist = g_slist_next(rlist);
  }
}

/**************************/
/* spatial object drawing */
/**************************/
/* NB: different setup for vectors/planes - draw in seperate iterations */
void gl_draw_spatial(gint type, struct model_pak *data)
{
gdouble vec1[3], vec2[3];
GSList *list, *list1, *list2, *ilist;
struct vec_pak *p1, *p2, *p3;
struct spatial_pak *spatial;
struct image_pak *image;

list = data->spatial;
while (list)
  {
  spatial = (struct spatial_pak *) list->data;
  if (spatial->type == type)
    {
    switch(type)
      {
/* TODO - this should replace SPATIAL_VECTOR */
      case SPATIAL_VECTORS:
        list1 = (GSList *) spatial->data;
        list2 = g_slist_next(list1);
        while (list1 && list2)
          {
          p1 = (struct vec_pak *) list1->data;
          p2 = (struct vec_pak *) list2->data;
          ARR3SET(vec1, p1->rx);
          ARR3SET(vec2, p2->rx);
          VEC3MUL(vec1, data->scale);
          VEC3MUL(vec2, data->scale);
          draw_vector(vec1, vec2, data->scale*0.04);
          list1 = g_slist_next(list2);
          list2 = g_slist_next(list1);
          }
        break;

      case SPATIAL_VECTOR:
        list2 = (GSList *) spatial->data;
        p1 = (struct vec_pak *) list2->data;
        list2 = g_slist_next(list2);
        p2 = (struct vec_pak *) list2->data;
        ARR3SET(vec1, p1->rx);
        ARR3SET(vec2, p2->rx);
        VEC3MUL(vec1, data->scale);
        VEC3MUL(vec2, data->scale);
        draw_vector(vec1, vec2, data->scale*0.04);
        break;

      case SPATIAL_PLANE:
        list2 = (GSList *) spatial->data;
        glBegin(GL_POLYGON);
        while (list2)
          {
          p1 = (struct vec_pak *) list2->data;
          ARR3SET(vec1, p1->rx);
          VEC3MUL(vec1, data->scale);
          glVertex3dv(vec1);
          list2 = g_slist_next(list2);
          }
        glEnd();
        break;

/* TODO - different routine for this, GL_TRIANGLE is faster than GL_POLYGON */
      case SPATIAL_MOLSURF:
      case SPATIAL_TRIANGLE:
      case SPATIAL_POLYGON:
/* compute normal, for proper lighting */
        list2 = (GSList *) spatial->data;
        p1 = (struct vec_pak *) list2->data;
        list2 = g_slist_next(list2);
        p2 = (struct vec_pak *) list2->data;
        list2 = g_slist_next(list2);
        p3 = (struct vec_pak *) list2->data;

/* enumerate periodic images */
        ilist=NULL;
        do
          {
          if (ilist)
            {
/* image */
            image = (struct image_pak *) ilist->data;
            ARR3SET(vec2, image->rx);
            ilist = g_slist_next(ilist);
            }
          else
            {
/* original */
            VEC3SET(vec2, 0.0, 0.0, 0.0);
            ilist = data->images;
            }
/* enumerate vertices */
          list2 = (GSList *) spatial->data;
          glBegin(GL_POLYGON);
          while (list2)
            {
            p1 = (struct vec_pak *) list2->data;

            ARR3SET(vec1, vec2);
            ARR3ADD(vec1, p1->rx);
            VEC3MUL(vec1, data->scale);
            glColor4f(p1->colour[0], p1->colour[1], p1->colour[2], sysenv.render.transmit);
            glNormal3dv(p1->rn);
            glVertex3dv(vec1);
            list2 = g_slist_next(list2);
            }
          glEnd();
          }
        while (ilist);
        break;
      }
    }
  list = g_slist_next(list);
  }
}

/************************/
/* main drawing routine */
/************************/
void draw_objs(struct canvas_pak *canvas, struct model_pak *data)
{
gdouble r;
gfloat specular[4], fog[4];
gdouble fog_mark;
struct object_pak *obj_data=NULL;
GSList *list;
PangoFontDescription *pfd;

mytimer();

/* setup the lighting */
gl_init_lights(data);

/* scaling affects placement to avoid near/far clipping */
r = RMAX_FUDGE * data->rmax;

/* main drawing setup */
glFrontFace(GL_CCW);
glEnable(GL_LIGHTING);
glEnable(GL_DEPTH_TEST);
glDepthFunc(GL_LESS);
glEnable(GL_CULL_FACE);
glShadeModel(GL_SMOOTH);
glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
glLineWidth(sysenv.render.line_thickness);

/* material shininess control (atoms) */
VEC3SET(specular, sysenv.render.ahl_strength 
                , sysenv.render.ahl_strength 
                , sysenv.render.ahl_strength);
glMaterialf(GL_FRONT, GL_SHININESS, sysenv.render.ahl_size);
glMaterialfv(GL_FRONT, GL_SPECULAR, specular);

/* colour-only change for spheres */
glEnable(GL_COLOR_MATERIAL);
glColorMaterial(GL_FRONT, GL_AMBIENT_AND_DIFFUSE);

/* turn off antialiasing, in case it was previously on */
glDisable(GL_LINE_SMOOTH);
glDisable(GL_POINT_SMOOTH);
glDisable(GL_BLEND);

/* NEW - pango OpenGL font */
font_offset = glGenLists(128);
pfd = pango_font_description_from_string(sysenv.dfont_name);
if (!gdk_gl_font_use_pango_font(pfd, 0, 128, font_offset))
  show_text(ERROR, "Failed to set up Pango font for OpenGL.\n");
pango_font_description_set_size(pfd, gl_fontsize);

/* NEW - auto foreground colour */
make_fg_visible();

/*
else
  gl_fontsize = pango_font_description_get_size(pfd);
printf("Font width = %d\n", gl_fontsize);
*/

/* depth queuing via fog */
/* FIXME: the stupid translate command for isolated atoms/spheres */
/* is what screws up this fog stuff  - molecules drawn in stick */
/* or liquorice/cylinder mode are perfectly fine, but spheres */
/* and crosses are not */
/* basically, spheres will all have the fog color */
/* at the origin superimposed on them */
if (sysenv.render.fog)
  {
/*
printf("  fog start = %f\n", sysenv.render.fog_start);
printf("fog density = %f\n", sysenv.render.fog_density);
*/

  glEnable(GL_FOG);
  ARR3SET(fog, sysenv.render.bg_colour);
  glFogf(GL_FOG_MODE, GL_LINEAR);
  glFogfv(GL_FOG_COLOR, fog);
  glHint(GL_FOG_HINT, GL_DONT_CARE);
/* FIXME - why does changing this do nothing??? */
  glFogf(GL_FOG_DENSITY, 0.35);
/* convert (intuitive) dialog value to required value */

/* FIXME - this is getting close, but still could do with some tweaking  */
  fog_mark = (sysenv.render.vp_dist+data->scale-1.0+2.0*sysenv.render.fog_start)*r;
  glFogf(GL_FOG_START, fog_mark);
/* NB: fog marks should not be equal as this turns depth queueing off */
  fog_mark += 0.1 + 2.0*(1.0 - sysenv.render.fog_density)*r;
  glFogf(GL_FOG_END, fog_mark);
  }
else
  glDisable(GL_FOG);

/* main drawing */
glPolygonMode(GL_FRONT, GL_FILL);

if (data->show_cores)
  gl_draw_atoms(data);

/* CURRENT */
gl_draw_bonds(0, data);

/* material shininess control (surfaces) */
VEC3SET(specular, sysenv.render.shl_strength 
                , sysenv.render.shl_strength 
                , sysenv.render.shl_strength);
glMaterialf(GL_FRONT, GL_SHININESS, sysenv.render.shl_size);
glMaterialfv(GL_FRONT, GL_SPECULAR, specular);

/* draw vector objects */
glColorMaterial(GL_FRONT, GL_AMBIENT_AND_DIFFUSE);
glColor4f(sysenv.render.fg_colour[0], 
          sysenv.render.fg_colour[1],
          sysenv.render.fg_colour[2], 1.0);
gl_draw_spatial(SPATIAL_VECTOR, data);
gl_draw_spatial(SPATIAL_VECTORS, data);

/* double sided drawing */
glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_TRUE);
glEnable(GL_CULL_FACE);

/* set back face colour once */
glColorMaterial(GL_BACK, GL_AMBIENT_AND_DIFFUSE);
glColor4f(0.5,0.5,0.5,1.0);
glColorMaterial(GL_FRONT, GL_AMBIENT_AND_DIFFUSE);

/* alpha blending for translucent surfaces */
glEnable(GL_BLEND);
glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

/* translucent ghost atoms */
gl_draw_ghost_atoms(data);

/* CURRENT - ghost bonds */
gl_draw_bonds(1, data);

/* translucent shells */
if (data->show_shells)
  gl_draw_shells(data);

/* selection highlighting */
glDisable(GL_LIGHTING);
/* NB: make depth buffer read only for halos (fixes overlap blackening) */
glDepthMask(GL_FALSE);

gl_draw_halo_list(data->selection, data);

glDepthMask(GL_TRUE);
glEnable(GL_LIGHTING);

/* double sided drawing for spatial objects */
glDisable(GL_CULL_FACE);

/* material setup - FIXME - make indep of ribbon stuff */
glColorMaterial(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE);
glColor4f(sysenv.render.ribbon_colour[0],
          sysenv.render.ribbon_colour[1],
          sysenv.render.ribbon_colour[2], sysenv.render.transmit);

gl_draw_spatial(SPATIAL_PLANE, data);
gl_draw_spatial(SPATIAL_POLYGON, data);
gl_draw_spatial(SPATIAL_TRIANGLE, data);

if (data->show_axes)
  gl_draw_axes(TRUE, canvas, data);

/* draw spatial objects that are allowed to be wire frame models */
if (sysenv.render.wire_surface)
  {
  glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
  if (sysenv.render.antialias)
    {
    glEnable(GL_LINE_SMOOTH);
    glEnable(GL_POINT_SMOOTH);
    glEnable(GL_BLEND);
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    }
  }

/* morphology drawing */
if (data->num_vertices)
  {
  glLineWidth(sysenv.render.frame_thickness);
  glColor4f((float) sysenv.render.morph_colour[0], 
            (float) sysenv.render.morph_colour[1],
            (float) sysenv.render.morph_colour[2], sysenv.render.transmit);
  gl_draw_morph(data);
  glLineWidth(sysenv.render.line_thickness);
  }

/* 2D molsurf's should be drawn 2-sided */
if (data->periodic == 2)
  gl_draw_spatial(SPATIAL_MOLSURF, data);

/* draw the ribbons */
list = data->ribbons;
while (list)
  {
  obj_data = (struct object_pak *) list->data;
  switch(obj_data->type)
    {
    case RIBBON:
      glColorMaterial(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE);
      gl_draw_ribbon(obj_data, data);
      break;
    }
  list = g_slist_next(list);
  }

/* 3D molsurf's should be drawn 1-sided */
glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_FALSE);
if (data->periodic != 2)
  gl_draw_spatial(SPATIAL_MOLSURF, data);

/* at this point, should have completed all colour_material routines */
/* ie only simple line/text drawing stuff after this point */
glEnable(GL_CULL_FACE);
glDisable(GL_COLOR_MATERIAL);
glDisable(GL_LIGHTING);
glShadeModel(GL_FLAT);
if (sysenv.render.antialias)
  {
  glEnable(GL_LINE_SMOOTH);
  glEnable(GL_POINT_SMOOTH);
  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  }

/* always visible foreground colour */
glColor4f(sysenv.render.fg_colour[0], 
          sysenv.render.fg_colour[1],
          sysenv.render.fg_colour[2], 1.0);

/* unit cell drawing */
if (data->show_cell && data->periodic)
  {
  glLineWidth(sysenv.render.frame_thickness);
  gl_draw_cell(data);
  }

/* CURRENT - draw line bond types */
glLineWidth(sysenv.render.stick_thickness);
glColor4f(1.0, 0.85, 0.5, 1.0);
gl_draw_bonds(2, data);

/* axes labels */
glLineWidth(1.0);
if (data->show_axes)
  gl_draw_axes(FALSE, canvas, data);

/* active/measurements colour (yellow) */
glColor4f(sysenv.render.label_colour[0],
          sysenv.render.label_colour[1],
          sysenv.render.label_colour[2], 1.0);

/* set up stippling */
glEnable(GL_LINE_STIPPLE);
glLineStipple(1, 0x0F0F);

/* selection box */
if (data->box_on)
  gl_draw_box(data->select_box[0], data->select_box[1],
              data->select_box[2], data->select_box[3]);

/* periodic cell images */
if (data->show_cell_images)
  gl_draw_cell_images(data);

/* measurements */
glLineWidth(sysenv.render.geom_line_width);
gl_draw_geom(data);
glDisable(GL_LINE_STIPPLE);

/* text drawing - accept all fragments */
glDepthFunc(GL_ALWAYS);
glColor4f(sysenv.render.fg_colour[0], 
          sysenv.render.fg_colour[1],
          sysenv.render.fg_colour[2], 1.0);
gl_draw_text(data);

data->redraw_cumulative += mytimer();
data->redraw_count++;
}

/************************/
/* total canvas refresh */
/************************/
#define DEBUG_OPENGL_DRAW 0
void opengl_draw(struct canvas_pak *canvas, struct model_pak *data)
{
gdouble sq, cq;
GdkGLContext *glcontext;
GdkGLDrawable *gldrawable;

/* is there anything to draw on? */
glcontext = gtk_widget_get_gl_context(canvas->glarea);
gldrawable = gtk_widget_get_gl_drawable(canvas->glarea);
if (!gdk_gl_drawable_gl_begin(gldrawable, glcontext))
  return;

/* sysenv.active should always be visible */
force_view(sysenv.active);

/* NB: need to have this here (not in configure event) */
/* in order for the render dialog to change it */
glClearColor(sysenv.render.bg_colour[0], sysenv.render.bg_colour[1],
             sysenv.render.bg_colour[2], 1.0);

/* clear the canvas & exit if no data */
if (!data)
  {
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT | GL_STENCIL_BUFFER_BIT);
  gdk_gl_drawable_swap_buffers(gldrawable);
  atom_properties_update(NULL, NULL);
  return;
  }
else
  {
/* clear the redraw flag */
  data->redraw = FALSE;
/* clear the atom info box  */
  if (g_slist_length(data->selection) != 1)
    atom_properties_update(NULL, NULL);
  }

/* setup transformation matrices */
gl_init_projection(data);

/* buffer clean */
if (flush)
  {
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT | GL_STENCIL_BUFFER_BIT);
  gdk_gl_drawable_swap_buffers(gldrawable);
  flush--;
  }

/* low quality render when scaling/rotating */
sq = sysenv.render.sphere_quality;
cq = sysenv.render.cylinder_quality;
if (sysenv.moving && sysenv.render.fast_rotation)
  {
  sysenv.render.sphere_quality = 1;
  sysenv.render.cylinder_quality = 5;
  }

#if DEBUG_OPENGL_DRAW
printf("  flush = %d\n", flush);
#endif

glClearStencil(0x4);
glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT | GL_STENCIL_BUFFER_BIT);
draw_objs(canvas, data);

/* all done */
sysenv.render.sphere_quality = sq;
sysenv.render.cylinder_quality = cq;

gdk_gl_drawable_swap_buffers(gldrawable);
gdk_gl_drawable_gl_end(gldrawable);
}

/**************************/
/* handle redraw requests */ 
/**************************/
#define DEBUG_REDRAW_HANDLER 0
gint redraw_handler(gpointer *dummy)
{
static gint n=1;
GSList *list;
struct model_pak *model;
struct canvas_pak *canvas;

/* do we have to do anything? */
if (sysenv.redraw)
  {

/* NEW - stereo piracy */
if (sysenv.stereo)
  {
  stereo_draw();
  return(TRUE);
  }

/* TODO - the redraw_canvas() call will create a list of */
/* glarea's that need to be updated by this routine */
/* ie instead of updating them all as is done currently */
/* loop and draw if we have a pending model and a canvas */

/* NB: the active canvas should be redrawn last - since any */
/* subsequent OpenGL calls (eg get_world_coords()) will refer to this canvas */
/* ... OR does the GL context begin/end stuff fix this issue??? */
  for (list=sysenv.canvas_list ; list ; list=g_slist_next(list))
    {
    canvas = (struct canvas_pak *) list->data;

/*
printf("canvas: %p\nactive: %d\nmodel: %p\nglarea: %p\n",
        canvas, canvas->active, canvas->model, canvas->glarea);
*/

    model = canvas->model;
    if (model)
      {
      if (!model->redraw)
        return(TRUE);
      opengl_draw(canvas, model);
      model->redraw = FALSE;

/* timing analysis */
     if (model->redraw_count >= 10)
       {
       model->redraw_time = model->redraw_cumulative/model->redraw_count;

#if DEBUG_REDRAW_HANDLER
printf("cumulative = %d : current %d : n = %d x 25ms\n",
        model->redraw_cumulative, model->redraw_time, n);
#endif

       model->redraw_count = 0;
       model->redraw_cumulative = 0;

/* decrease redraw frequency */
      if (model->redraw_time > n*25000)
        {
        n = model->redraw_time/25000;
        g_timeout_add(n*25, (GSourceFunc) &redraw_handler, NULL);
        return(FALSE);
        }

/* increase redraw frequency */
      if (n > 1)
        {
        if (model->redraw_time < (n-1)*25000)
          {
          n--;
          g_timeout_add(n*25, (GSourceFunc) &redraw_handler, NULL);
          return(FALSE);
          }
        }
        }
      }
    else
      {
/* clear the canvas */
      opengl_draw(canvas, NULL);
      }
    }
/* clear the redraw flags */
  sysenv.redraw = FALSE;
  }

return(TRUE);
}

/****************************/
/* schedule redraw requests */ 
/****************************/
void redraw_canvas(gint action)
{
GSList *list;
struct model_pak *data;

switch (action)
  {
  case SINGLE:
    data = model_ptr(sysenv.active, RECALL);
    if (data)
      data->redraw = TRUE;
    break;
  case ALL:
    for (list=sysenv.mal ; list ; list=g_slist_next(list))
      {
      data = (struct model_pak *) list->data;
      data->redraw = TRUE;
      }
    break;
  }
sysenv.redraw = TRUE;
}

