/*
Copyright (C) 2003 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include <stdio.h>
#include <stdlib.h>
#include <gtk/gtkgl.h>
#include <GL/gl.h>

#include "gdis.h"
#include "coords.h"
#include "matrix.h"
#include "opengl.h"
#include "render.h"
#include "interface.h"

/* externals */
extern struct sysenv_pak sysenv;
extern struct elem_pak elements[];

gint cb_key_press(GtkWidget *, GdkEventKey *, gpointer);

#define STEREO_WIDTH 982
#define STEREO_HEIGHT 491
#define XOFF 148
#define YOFF 532

/*******************/
/* redraw handling */
/*******************/
void stereo_redraw(void)
{
gdouble eye;
struct model_pak *model;

model = model_ptr(sysenv.active, RECALL);

glClearColor(sysenv.render.bg_colour[0], sysenv.render.bg_colour[1],
             sysenv.render.bg_colour[2], 1.0);
glClearStencil(0x4);
glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT | GL_STENCIL_BUFFER_BIT);

eye = 0.05 * sysenv.rsize;

/*
printf(" r  = %f\n", sysenv.rsize);
printf("eye = %f\n", eye);
*/

/* LEFT */
glDrawBuffer(GL_BACK_LEFT);
if (model)
  {
  glViewport(XOFF, YOFF, STEREO_WIDTH, STEREO_HEIGHT);

  gl_init_projection(model);

  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  glFrustum(-sysenv.rsize+eye, sysenv.rsize+eye,
            -sysenv.rsize,sysenv.rsize,sysenv.render.vp_dist*sysenv.rsize,
            (sysenv.render.vp_dist+2.0*model->scale)*sysenv.rsize);

  glTranslatef(+eye, 0.0, 0.0);


/* NB: disable axes as drawing requires canvas info */
/* TODO - save state & restore after */
  model->show_axes = FALSE;
  draw_objs(NULL, model);
  }
glFlush();

/* RIGHT */
glDrawBuffer(GL_BACK_RIGHT);
if (model)
  {
  glViewport(XOFF, 0, STEREO_WIDTH, STEREO_HEIGHT);

  gl_init_projection(model);

  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  glFrustum(-sysenv.rsize-eye, sysenv.rsize-eye,
            -sysenv.rsize,sysenv.rsize, sysenv.render.vp_dist*sysenv.rsize,
            (sysenv.render.vp_dist+2.0*model->scale)*sysenv.rsize);

  glTranslatef(-eye, 0.0, 0.0);


/* NB: disable axes as drawing requires canvas info */
/* TODO - save state & restore after */
  model->show_axes = FALSE;
  draw_objs(NULL, model);
  }
glFlush();
}

/********************************/
/* stereo window expose handler */
/********************************/
gint stereo_expose_event(GtkWidget *w, GdkEventExpose *event)
{
GdkGLContext *glcontext;
GdkGLDrawable *gldrawable;

glcontext = gtk_widget_get_gl_context(w);
gldrawable = gtk_widget_get_gl_drawable(w);
if (!gdk_gl_drawable_gl_begin(gldrawable, glcontext))
  return(FALSE);

stereo_redraw();

gdk_gl_drawable_swap_buffers(gldrawable);
gdk_gl_drawable_gl_end(gldrawable);

return(TRUE);
}

/**************************/
/* refresh stereo drawing */
/**************************/
GtkWidget *stereo_window=NULL, *stereo_glarea=NULL;

void stereo_draw(void)
{
g_assert(stereo_window != NULL);
g_assert(stereo_glarea != NULL);

stereo_expose_event(stereo_glarea, NULL);
}

/*******************************/
/* init fullscreen stereo mode */
/*******************************/
void stereo_open_window(void)
{
gint x, y, width, height, depth;
GdkWindow *win;
GdkGLConfig *glconfig;

#ifdef __sgi
if (system("/usr/gfx/setmon -n STR_RECT") != 0)
  {
  printf("setmon attempt failed!\n");
  return;
  }
/* NB: stereo visual not needed with STR_RECT mode */
glconfig = gdk_gl_config_new_by_mode(GDK_GL_MODE_RGB |
                                     GDK_GL_MODE_DEPTH |
                                     GDK_GL_MODE_DOUBLE);
#else
/* non SGI - request a proper stereo canvas */
glconfig = gdk_gl_config_new_by_mode(GDK_GL_MODE_RGB |
                                     GDK_GL_MODE_DEPTH |
                                     GDK_GL_MODE_DOUBLE |
                                     GDK_GL_STEREO);
if (!glconfig)
  {
  printf("WARNING: Failed to acquire a stereo visual.\n");
  glconfig = gdk_gl_config_new_by_mode(GDK_GL_MODE_RGB |
                                       GDK_GL_MODE_DEPTH |
                                       GDK_GL_MODE_DOUBLE);
  }
#endif

if (!glconfig)
  {
  printf("ERROR: Failed to acquire a visual for stereo display.\n");
  return;
  }

/* get root window dimensions so we can create a fullscreen window */
win = gdk_get_default_root_window();
gdk_window_get_geometry(win, &x, &y, &width, &height, &depth);
stereo_window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
gtk_window_set_decorated(GTK_WINDOW(stereo_window), FALSE);
gtk_window_set_default_size(GTK_WINDOW(stereo_window), width, height);

g_signal_connect(GTK_OBJECT(stereo_window), "key_press_event",
                (GtkSignalFunc) cb_key_press, NULL);

stereo_glarea = gtk_drawing_area_new();
gtk_widget_set_gl_capability(stereo_glarea, glconfig, NULL, TRUE, GDK_GL_RGBA_TYPE);
gtk_container_add(GTK_CONTAINER(stereo_window), stereo_glarea);

/* the only missing handler is configure - which is not needed as a resize is not permitted */
g_signal_connect(GTK_OBJECT(stereo_glarea), "expose_event",
                (GtkSignalFunc) stereo_expose_event, NULL);
g_signal_connect(GTK_OBJECT(stereo_glarea), "motion_notify_event",
                (GtkSignalFunc) motion_notify_event, NULL);
g_signal_connect(GTK_OBJECT(stereo_glarea), "button_press_event",
                (GtkSignalFunc) button_press_event, NULL);
g_signal_connect(GTK_OBJECT(stereo_glarea), "button_release_event",
                (GtkSignalFunc) button_release_event, NULL);

gtk_widget_set_events(GTK_WIDGET(stereo_glarea), GDK_EXPOSURE_MASK
                                               | GDK_LEAVE_NOTIFY_MASK
                                               | GDK_BUTTON_PRESS_MASK
                                               | GDK_BUTTON_RELEASE_MASK
                                               | GDK_POINTER_MOTION_MASK
                                               | GDK_POINTER_MOTION_HINT_MASK);


gtk_widget_show_all(stereo_window);

/* CURRENT - invisible cursor */
{
GdkCursor *cursor;
GdkPixmap *source, *mask;
GdkGLWindow *glwindow;
GdkColor fg = { 0, 0, 0, 0 };
GdkColor bg = { 0, 0, 0, 0 };
static unsigned char cursor1_bits[] = {0x00};
static unsigned char mask1_bits[] = {0x00};

/* create empty cursor */
source = gdk_bitmap_create_from_data(NULL, cursor1_bits, 1, 1);
mask = gdk_bitmap_create_from_data(NULL, mask1_bits, 1, 1);
cursor = gdk_cursor_new_from_pixmap(source, mask, &fg, &bg, 0, 0);

/* set cursor */
glwindow = gtk_widget_get_gl_window(stereo_glarea);
gdk_window_set_cursor(gdk_gl_window_get_window(glwindow), cursor);
}

}

/************************************/
/* shut down fullscreen stereo mode */
/************************************/
void stereo_close_window(void)
{
if (stereo_window)
  gtk_widget_destroy(stereo_window);
stereo_window = NULL;
stereo_glarea = NULL;

#ifdef __sgi
system("/usr/gfx/setmon -n 72hz");
#endif
}
