# Copyright (C) 2011 - Curtis Hovey <sinzui.is at verizon.net>
# This software is licensed under the GNU General Public License version 2
# (see the file COPYING).
"""GDP Gedit Developer Plugin base."""

__metaclass__ = type

__all__ = [
    'GDPPluginMixin',
    'GDPWindowActivatable',
    ]


from gi.repository import GObject
from gi.repository import Gtk
try:
    from gi.repository import Gedit
    # Hush lint.
    Gedit
except ImportError:
    from testing import Gedit


# Signals shared in GDP.

GObject.signal_new(
    'syntax-error-python', Gedit.Document, GObject.SIGNAL_RUN_LAST,
    GObject.TYPE_NONE, ())


GObject.signal_new(
    'bzr-branch-open', Gedit.Window, GObject.SIGNAL_RUN_LAST,
    GObject.TYPE_NONE, (GObject.TYPE_STRING, ))


# Common GDP classes.


class GDPWindowActivatable(GObject.Object, Gedit.WindowActivatable):
    """A test-safe wrapper that can also be used in real code.."""
    __gtype_name__ = "GDPNone"
    window = GObject.property(type=GObject.Object)

    def __init__(self):
        """Initialize the plugin for the Gedit application."""
        GObject.Object.__init__(self)
        self.gdp_activatable = None


class GDPPluginMixin:
    """Decorate a `GeditWindow` with GDP state"""

    ACTION_GROUP_NAME = None
    MENU_XML = None

    def actions(self, controller):
        """Return a list of action tuples.

        (name, stock_id, label, accelerator, tooltip, callback)
        """
        pass

    def activate(self):
        self.signal_ids = {}
        self.ui_id = None
        self.controller = self.CONTROLLER_CLASS(self.window)
        if self.ACTION_GROUP_NAME is None:
            return
        self.action_group = Gtk.ActionGroup(name=self.ACTION_GROUP_NAME)
        self.action_group.set_translation_domain('gedit')
        self.action_group.add_actions(self.actions(self.controller))
        manager = self.window.get_ui_manager()
        manager.insert_action_group(self.action_group, -1)
        self.ui_id = manager.add_ui_from_string(self.MENU_XML)

    def deactivate(self):
        """Deactivate the plugin for the window."""
        if self.ui_id is None:
            return
        manager = self.window.get_ui_manager()
        manager.remove_ui(self.ui_id)
        manager.remove_action_group(self.action_group)
        manager.ensure_update()
        self.controller.deactivate()
        self.controller = None

    def connect_signal(self, obj, signal, method):
        """Connect the signal from the provided object to a method."""
        if obj is None:
            return
        self.signal_ids[signal] = obj.connect(signal, method)

    def disconnect_signal(self, obj, signal):
        """Disconnect the signal from the provided object."""
        if obj is None:
            return
        if signal in self.signal_ids:
            obj.disconnect(self.signal_ids[signal])
            del self.signal_ids[signal]

    @property
    def active_document(self):
        """The active document in the window."""
        self.window.get_active_document()
