/*
** 1998-05-29 -	A rename command might be useful. Since writing this also caused me
**		to implement the generic command interface in cmd_generic, it was
**		very useful indeed!
** 1998-09-18 -	Did some pretty massive changes/additions to provide overwrite protection.
** 1999-01-30 -	Bug fix: always called ovw_overwrite_end() in closing, regardless of
**		whether the _begin() function was ever called. This caused nesting errors.
** 1999-02-23 -	Um... That "bug fix" was buggy. I think I fixed it this time, though.
** 1999-03-05 -	Altered to comply with new selection handling (and its changes on the
**		generic command invocation parameters).
*/

#include "gentoo.h"
#include "cmd_delete.h"
#include "convutil.h"
#include "errors.h"
#include "dirpane.h"
#include "overwrite.h"

#include "cmd_generic.h"
#include "cmd_rename.h"

#define	CMD_ID	"rename"

/* ----------------------------------------------------------------------------------------- */

typedef struct {
	GtkWidget	*vbox;
	GtkWidget	*label;
	GtkWidget	*entry;
	MainInfo	*min;
	gboolean	ovw_open;
} RenInfo;

/* ----------------------------------------------------------------------------------------- */

static void ren_body(MainInfo *min, DirPane *src, DirRow *row, gpointer user)
{
	gchar		temp[PATH_MAX + 256];
	const gchar	*name;
	RenInfo		*ren = user;

	name = DP_ROW_NAMED(row);
	g_snprintf(temp, sizeof temp, _("Enter New Name For \"%s\""), name);
	gtk_label_set_text(GTK_LABEL(ren->label), temp);
	gtk_entry_set_text(GTK_ENTRY(ren->entry), name);
	gtk_entry_select_region(GTK_ENTRY(ren->entry), 0, strlen(name));
	gtk_widget_grab_focus(ren->entry);

	if(!ren->ovw_open)
	{
		ovw_overwrite_begin(ren->min, _("\"%s\" Already Exists - Proceed With Rename?"), 0U);
		ren->ovw_open = TRUE;
	}
}

static gint ren_action(MainInfo *min, DirPane *src, DirPane *dst, DirRow *row, gpointer user)
{
	RenInfo		*ren = user;
	const gchar	*old_name, *new_name;
	Conv		conv;

	old_name = DP_ROW_NAME(row);
	new_name = conv_begin(&conv, gtk_entry_get_text(GTK_ENTRY(ren->entry)));

	/* Disallow attempts to use rename to move files. */
	if(strchr(new_name, G_DIR_SEPARATOR))
	{
		err_set(min, EINVAL, CMD_ID, old_name);		/* Not quite perfect error message, but simple. */
		conv_end(&conv);
		return 0;
	}
	if(strcmp(old_name, new_name) == 0)			/* Ignore renames to self. */
	{
		dp_unselect(src, DP_ROW_INDEX(src, row));
		conv_end(&conv);
		return 1;
	}
	/* Check for overwrite, and attempt to clear the way. */
	switch(ovw_overwrite_file(ren->min, new_name, NULL))
	{
	case OVW_SKIP:
		return 1;
	case OVW_CANCEL:
		return 0;
	case OVW_PROCEED:
		break;
	case OVW_PROCEED_FILE:
		if(!del_delete_file(min, new_name))
		{
			conv_end(&conv);
			return 0;
		}
		break;
	case OVW_PROCEED_DIR:
		if(!del_delete_dir(min, new_name, FALSE))
		{
			conv_end(&conv);
			return 0;
		}
		break;
	}

	err_clear(min);
	if(rename(old_name, new_name) != 0)
		err_set(min, errno, CMD_ID, old_name);
	else
		dp_unselect(src, DP_ROW_INDEX(src, row));
	conv_end(&conv);
	return errno == 0;
}

static void ren_free(gpointer user)
{
	if(((RenInfo *) user)->ovw_open)
		ovw_overwrite_end(((RenInfo *) user)->min);
}

/* ----------------------------------------------------------------------------------------- */

gint cmd_rename(MainInfo *min, DirPane *src, DirPane *dst, const CmdArg *ca)
{
	static RenInfo	ri;

	ri.ovw_open = FALSE;
	ri.vbox	    = gtk_vbox_new(FALSE, 0);
	ri.label    = gtk_label_new("");
	ri.entry    = gtk_entry_new_with_max_length(MAXNAMLEN - 1);
	gtk_box_pack_start(GTK_BOX(ri.vbox), ri.label, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(ri.vbox), ri.entry, FALSE, FALSE, 0);

	return cmd_generic(min, _("Rename"), CGF_NOALL | CGF_SRC | CGF_NODST, ren_body, ren_action, ren_free, &ri);
}
