/*
** 1998-08-31 -	This little module provides helper functions when dealing with representations
**		of numbers, specifically numbers describing the size of things. For example,
**		this module knows that 1457664 bytes can be expressed conveniently as 1.39 MB.
**		I haven't tried teaching it to make that into the "official" 1.44 MB yet, though...
** 1999-05-09 -	Added explicit support for 64-bit sizes. Might not be very portable. :(
** 2000-02-18 -	Redid handling of 64-bit constants in sze_put_size(). More portability now?
** 2000-07-02 -	Added support for I18N. Fairly hairy.
*/

#include "gentoo.h"

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "strutil.h"
#include "sizeutil.h"

/* ----------------------------------------------------------------------------------------- */

/* An array of named multipliers, used for different units. Keep sorted on name, since
** we use a binary-search when querying it (hmm, file that under "software engineering:
** overkills, blatant and obvious", I guess).
*/
static struct UnitInfo {
	const gchar	*name;
	gdouble		multiplier;
} unit_info[] = { { N_("BYTES"), 1.0},
		  { N_("GB"), (gdouble) (1 << 30) },
		  { N_("KB"), (gdouble) (1 << 10) },
		  { N_("MB"), (gdouble) (1 << 20) }
		};

/* An array of pre-built formatting strings. We use these since the exact g_snprintf()
** format specifier needed for 64-bit numbers is a compile-time constant, so we can't
** just write it in the source. It's G_GUINT64_FORMAT. We therefore use an extra step
** during the sze_initialize() call, to build the actual run-time formatting strings.
** It's all very meta. The size here is exaggerated, for translation.
*/
static struct
{
	gchar	none[24];
	gchar	bytes[24];
	gchar	kb[24];
	gchar	mb[24];
	gchar	gb[24];
} size_format;

static gdouble get_multiplier(const gchar *unit);

/* ----------------------------------------------------------------------------------------- */

static gint comp_ui(gconstpointer a, gconstpointer b)
{
	return strcoll(((const struct UnitInfo *) a)->name, ((const struct UnitInfo *) b)->name);
}

/* 2000-07-02 -	Initialize unit_info, by translating it and then sorting it. */
void sze_initialize(void)
{
	guint	i;

	for(i = 0; i < sizeof unit_info / sizeof unit_info[0]; i++)
		unit_info[i].name = _(unit_info[i].name);
	qsort(unit_info, sizeof unit_info / sizeof unit_info[0], sizeof unit_info[0], comp_ui);

	g_snprintf(size_format.none, sizeof size_format.none, _("%%%s"), G_GUINT64_FORMAT);
	g_snprintf(size_format.bytes, sizeof size_format.bytes, _("%%%s bytes"), G_GUINT64_FORMAT);
	g_snprintf(size_format.kb, sizeof size_format.kb, _("%%%s KB"), G_GUINT64_FORMAT);
	g_snprintf(size_format.mb, sizeof size_format.mb, _("%%%s MB"), G_GUINT64_FORMAT);
	g_snprintf(size_format.gb, sizeof size_format.gb, _("%%%s GB"), G_GUINT64_FORMAT);
}

/* ----------------------------------------------------------------------------------------- */

/* 1998-09-01 -	Initialize a binary search for the multiplier associated with the given
**		unit name.
*/
static gdouble get_multiplier(const gchar *unit)
{
	struct UnitInfo	*ui;

	ui = bsearch(&unit, unit_info, sizeof unit_info / sizeof *unit_info, sizeof *unit_info, comp_ui);
	return ui ? ui->multiplier : -1;
}

/* ----------------------------------------------------------------------------------------- */

/* 1998-08-31 -	Attempt to parse a size from the string beginning at <buf>. Parsing rules:
**		* Initial whitespace is ignored.
**		* The first thing that can be parsed as a positive number will be. Floating
**		  point notation is (somewhat) understood.
**		* After the number, a unit can appear, optionally prefixed by whitespace.
**		  Legal units are: bytes, KB (1024 bytes), MB (1024^2 bytes) and GB (1024^3).
**		  If no unit appears, bytes (multiplier=1) is assumed. Unknown units are ignored.
*/
gsize sze_get_size(const gchar *buf)
{
	gchar	unit[16];
	gdouble	ipart = 0.0f, fpart = 0.0f, weight = 0.1f, num, mult = 1.0f;

	while(isspace((guchar) *buf))
		buf++;

	if(isdigit((guchar) *buf))
	{
		while(isdigit((guchar) *buf))
		{
			ipart *= 10.0f;
			ipart += *buf - '0';
			buf++;
		}
		if(*buf == '.')
		{
			for(buf++; isdigit((guchar) *buf); buf++, weight /= 10.0f)
				fpart += (*buf - '0') * weight;
		}
		num = ipart + fpart;
		while(isspace((guchar) *buf))
			buf++;
		if(sscanf(buf, "%8s", unit) == 1)
		{
			g_strup(unit);
			if((mult = get_multiplier(unit)) < 0)
				mult = 1.0f;
		}
		return (gsize) mult * num;
	}
	return (gsize) -1;
}

/* ----------------------------------------------------------------------------------------- */

/* 1998-09-01 -	Output the given <size> as a string in <buf>, using no more than <buf_max>
**		bytes to do so. The <unit> controls which unit will be appended to the number.
**		It should be one of:
**		SZE_NONE	No unit is appended. Function degenerates into snprintf().
**		SZE_BYTES	The unit will be "bytes".
**		SZE_KB		Unit is "KB"; divides the size down by 1024. 4711 => "4.60 KB".
**		SZE_MB		Megabytes, divisor is 1024 squared.
**		SZE_GB		Gigabytes, using a divisor of 1024 cubed.
**		SZE_AUTO	Automatic unit selection; uses heuristics to determine which
**				unit works "best". Avoids "0.87 GB", digs "890.08 MB".
**		This function returns the number of characters written to <buf>, not including
**		the terminating '\0'-character.
*/
guint sze_put_size(gchar *buf, gsize buf_max, off_t size, SzUnit unit)
{
	const gchar	*fmt = NULL;
	gdouble		temp = 0.0;

	switch(unit)
	{
		case SZE_NONE:
			return g_snprintf(buf, buf_max, size_format.none, (guint64) size);
		case SZE_BYTES:
			return g_snprintf(buf, buf_max, size_format.bytes, (guint64) size);
		case SZE_KB:
			if((size & ((1 << 10) - 1)) == 0)
				return g_snprintf(buf, buf_max, size_format.kb, (guint64) size >> 10);
			temp = (gdouble) size / (1 << 10);
			fmt  = _("%.2f KB");
			break;
		case SZE_MB:
			if((size & ((1 << 20) - 1)) == 0)
				return g_snprintf(buf, buf_max, size_format.mb, (guint64) size >> 20);
			temp = (gdouble) size / (1 << 20);
			fmt  = _("%.2f MB");
			break;
		case SZE_GB:
			if((size & ((1 << 30) - 1)) == 0)
				return g_snprintf(buf, buf_max, size_format.gb, (guint64) size >> 30);
			temp = (gdouble) size / (1 << 30);
			fmt  = _("%.2f GB");
			break;
		case SZE_AUTO:
			if(size < (1 << 10))
				return sze_put_size(buf, buf_max, size, SZE_BYTES);
			else if(size < (1 << 20))
				return sze_put_size(buf, buf_max, size, SZE_KB);
			else if(size < (1 << 30))
				return sze_put_size(buf, buf_max, size, SZE_MB);
			else
				return sze_put_size(buf, buf_max, size, SZE_GB);
	}
	if(fmt != NULL)
		return g_snprintf(buf, buf_max, fmt, temp);
	return 0U;
}

/* 1999-05-09 -	A version of the above that deals explicitly with 64-bit sizes. Handy
**		for the total selection, file system free bytes, and stuff.
*/
guint sze_put_size64(gchar *buf, gsize buf_max, guint64 size, SzUnit unit)
{
	const gchar	*fmt = NULL;
	gdouble		temp = 0.0;

	switch(unit)
	{
		case SZE_NONE:
			return g_snprintf(buf, buf_max, size_format.none, size);
		case SZE_BYTES:
			return g_snprintf(buf, buf_max, size_format.bytes, size);
		case SZE_KB:
			if((size & ((G_GINT64_CONSTANT(1) << 10) - 1)) == 0)
				return g_snprintf(buf, buf_max, size_format.kb, size >> 10);
			temp = (gdouble) size / (G_GINT64_CONSTANT(1) << 10);
			fmt  = _("%.2f KB");
			break;
		case SZE_MB:
			if((size & ((G_GINT64_CONSTANT(1) << 20) - 1)) == 0)
				return g_snprintf(buf, buf_max, size_format.mb, size >> 20);
			temp = (gdouble) size / (G_GINT64_CONSTANT(1) << 20);
			fmt  = _("%.2f MB");
			break;
		case SZE_GB:
			if((size & ((G_GINT64_CONSTANT(1) << 30) - 1)) == 0)
				return g_snprintf(buf, buf_max, size_format.gb, size >> 30);
			temp = (gdouble) size / (G_GINT64_CONSTANT(1) << 30);
			fmt  = _("%.2f GB");
			break;
		case SZE_AUTO:
			if(size < (G_GINT64_CONSTANT(1) << 10))
				return sze_put_size64(buf, buf_max, size, SZE_BYTES);
			else if(size < (G_GINT64_CONSTANT(1) << 20))
				return sze_put_size64(buf, buf_max, size, SZE_KB);
			else if(size < (G_GINT64_CONSTANT(1) << 30))
				return sze_put_size64(buf, buf_max, size, SZE_MB);
			else
				return sze_put_size64(buf, buf_max, size, SZE_GB);
	}
	if(fmt != NULL)
		return g_snprintf(buf, buf_max, fmt, temp);
	return 0U;
}
