/* Gerris - The GNU Flow Solver
 * Copyright (C) 2001 National Institute of Water and Atmospheric Research
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.  
 */

#include <stdlib.h>
#include <math.h>

#include "adaptive.h"

/**
 * gfs_cell_coarse_init:
 * @cell: a #FttCell.
 * @domain: a #GfsDomain containing @cell.
 *
 * Initialises the variables of @cell using the values of its children
 * cells.
 */
void gfs_cell_coarse_init (FttCell * cell, GfsDomain * domain)
{
  GfsVariable * v;

  g_return_if_fail (cell != NULL);
  g_return_if_fail (!FTT_CELL_IS_LEAF (cell));
  g_return_if_fail (domain != NULL);

  v = domain->variables;
  while (v) {
    if (v->fine_coarse)
      (* v->fine_coarse) (cell, v);
    else
      gfs_get_from_below_intensive (cell, v);
    v = v->next;
  }
}

/**
 * gfs_cell_fine_init:
 * @cell: a #FttCell.
 * @domain: a #GfsDomain containing @cell.
 *
 * Initializes the variables of @cell using interpolation from its
 * parent cell.
 *
 * First-order interpolation (straight injection) is used for boundary
 * cells and second-order interpolation for the other cells.  
 */
void gfs_cell_fine_init (FttCell * cell, GfsDomain * domain)
{
  FttCell * parent;
  GfsVariable * v;

  g_return_if_fail (cell != NULL);
  g_return_if_fail (!FTT_CELL_IS_ROOT (cell));
  g_return_if_fail (domain != NULL);

  parent = ftt_cell_parent (cell);
  /* refinement of mixed cell is not implemented (yet) */
  g_assert (GFS_IS_FLUID (parent));

  gfs_cell_init (cell, domain);
  v = domain->variables;
  while (v) {
    GFS_VARIABLE (cell, v->i) = GFS_VARIABLE (parent, v->i);
    v = v->next;
  }

  if (!GFS_CELL_IS_BOUNDARY (parent)) {
    FttVector p;
    FttComponent c;
    
    ftt_cell_relative_pos (cell, &p);
    v = domain->variables;
    while (v) {
      for (c = 0; c < FTT_DIMENSION; c++)
	GFS_VARIABLE (cell, v->i) += 
	  ((gdouble *)&p)[c]*
#if 1
	  gfs_center_van_leer_gradient (parent, c, v->i);
#else
          gfs_center_gradient (parent, c, v->i);
#endif
      v = v->next;
    }
  }
}

static gboolean coarsen_cell (FttCell * cell, guint * minlevel)
{
  if (GFS_CELL_IS_BOUNDARY (cell))
    return TRUE;
  if (ftt_cell_level (cell) < *minlevel || GFS_IS_MIXED (cell))
    return FALSE;
  return (GFS_STATE (cell)->dp == 0.);
}

static void coarsen_box (GfsBox * box, guint * minlevel)
{
  ftt_cell_coarsen (box->root,
		    (FttCellCoarsenFunc) coarsen_cell, minlevel,
		    gfs_cell_cleanup);
}

static void refine_cell (FttCell * cell, gpointer * data)
{
  guint * depth = data[4];
  GfsDomain * domain = data[5];

  GFS_STATE (cell)->div = 0.;
  if (GFS_STATE (cell)->dp > 0.) {
    guint level = ftt_cell_level (cell);

    ftt_cell_refine_corners (cell, (FttCellInitFunc) gfs_cell_fine_init,
			     domain);
    ftt_cell_refine_single (cell, (FttCellInitFunc) gfs_cell_fine_init,
			    domain);
    if (level + 1 > *depth)
      *depth = level + 1;
  }
}

static void refine_cell_corner (FttCell * cell, GfsDomain * domain)
{
  if (ftt_refine_corner (cell))
    ftt_cell_refine_single (cell, (FttCellInitFunc) gfs_cell_fine_init, 
			    domain);
}

static void refine_cell_mark (FttCell * cell, gpointer * data)
{
  FttCellRefineFunc refine = data[0];
  gpointer refine_data = data[1];
  guint * minlevel = data[2];
  guint * maxlevel = data[3];
  guint level = ftt_cell_level (cell);

  if (!GFS_IS_FLUID (cell) ||
      /* refinement of solid cells not implemented (yet) */
      level >= *maxlevel)
    GFS_STATE (cell)->dp = 0.;
  else if (level < *minlevel)
    GFS_STATE (cell)->dp = 1.;
  else
    GFS_STATE (cell)->dp = (* refine) (cell, refine_data);
}

/**
 * gfs_domain_adapt:
 * @domain: a #GfsDomain.
 * @minlevel: the minimum level reachable when coarsening.
 * @maxlevel: the maximum level reachable when refining.
 * @refine: a #FttCellRefineFunc used to decide when to refine or coarsen.
 * @refine_data: user data to pass to @refine.
 *
 * Adapts the local mesh size according to the criteria defined by
 * @refine. Leaf cells for which @refine returns %TRUE are refined and
 * non-leaf cells for which @refine returns %FALSE are coarsened. The
 * variables on the refined or coarsened cells are initialized by
 * interpolation from the initial mesh.
 */
void gfs_domain_adapt (GfsDomain * domain,
		       guint minlevel, 
		       guint maxlevel,
		       FttCellRefineFunc refine,
		       gpointer refine_data)
{
  gpointer data[6];
  guint depth;
  gint l;
  GfsVariable * v;

  g_return_if_fail (domain != NULL);
  g_return_if_fail (refine != NULL);

  depth = gfs_domain_depth (domain);
  gfs_domain_cell_traverse (domain, FTT_POST_ORDER, FTT_TRAVERSE_NON_LEAFS, -1,
			   (FttCellTraverseFunc) gfs_cell_coarse_init,
			    domain);
  data[0] = refine;
  data[1] = refine_data;
  data[2] = &minlevel;
  data[3] = &maxlevel;
  data[4] = &depth;
  data[5] = domain;

  gfs_domain_cell_traverse (domain, 
			    FTT_PRE_ORDER, FTT_TRAVERSE_ALL, -1,
			    (FttCellTraverseFunc) refine_cell_mark, data);
  gfs_domain_cell_traverse (domain, 
			    FTT_PRE_ORDER, FTT_TRAVERSE_LEAFS, -1,
			    (FttCellTraverseFunc) refine_cell, data);
  gts_container_foreach (GTS_CONTAINER (domain), (GtsFunc) coarsen_box, 
			 &minlevel);

  for (l = depth - 2; l >= 0; l--)
    gfs_domain_cell_traverse (domain, 
			     FTT_PRE_ORDER, FTT_TRAVERSE_LEVEL, l,
			     (FttCellTraverseFunc) refine_cell_corner, 
			      domain);
  gfs_domain_match (domain);
  gfs_set_merged (domain);
  v = domain->variables;
  while (v) {
    gfs_domain_center_bc (domain, v);
    v = v->next;
  }
}

/* GfsAdapt: Object */

static void gfs_adapt_read (GtsObject ** o, GtsFile * fp)
{
  GfsAdapt * a = GFS_ADAPT (*o);
  GtsFileVariable var[] = {
    {GTS_UINT, "minlevel", TRUE},
    {GTS_UINT, "maxlevel", TRUE},
    {GTS_NONE}
  };

  if (GTS_OBJECT_CLASS (gfs_adapt_class ())->parent_class->read)
    (* GTS_OBJECT_CLASS (gfs_adapt_class ())->parent_class->read) 
      (o, fp);
  if (fp->type == GTS_ERROR)
    return;
  
  var[0].data = &a->minlevel;
  var[1].data = &a->maxlevel;
  gts_file_assign_variables (fp, var);
}

static void gfs_adapt_write (GtsObject * o, FILE * fp)
{
  GfsAdapt * a = GFS_ADAPT (o);

  if (GTS_OBJECT_CLASS (gfs_adapt_class ())->parent_class->write)
    (* GTS_OBJECT_CLASS (gfs_adapt_class ())->parent_class->write) 
      (o, fp);
  fputs (" { ", fp);
  if (a->minlevel > 0)
    fprintf (fp, "minlevel = %u ", a->minlevel);
  if (a->maxlevel < G_MAXINT)
    fprintf (fp, "maxlevel = %u ", a->maxlevel);
  fputc ('}', fp);
}

static gboolean gfs_adapt_event (GfsEvent * event, GfsSimulation * sim)
{
  if (GFS_ADAPT (event)->refine == NULL) {
    gts_object_destroy (GTS_OBJECT (event));
    return FALSE;
  }
  if ((* GFS_EVENT_CLASS (GTS_OBJECT_CLASS (gfs_adapt_class ())->parent_class)->event) (event, sim)) {
    GFS_ADAPT (event)->active = TRUE;
    return TRUE;
  }
  GFS_ADAPT (event)->active = FALSE;
  return FALSE;
}

static void gfs_adapt_class_init (GfsEventClass * klass)
{
  GFS_EVENT_CLASS (klass)->event = gfs_adapt_event;
  GTS_OBJECT_CLASS (klass)->read = gfs_adapt_read;
  GTS_OBJECT_CLASS (klass)->write = gfs_adapt_write;
}

static void gfs_adapt_init (GfsAdapt * object)
{
  object->active = FALSE;
  object->not = FALSE;
  object->minlevel = 0;
  object->maxlevel = G_MAXINT;
  object->refine = NULL;
}

GfsEventClass * gfs_adapt_class (void)
{
  static GfsEventClass * klass = NULL;

  if (klass == NULL) {
    GtsObjectClassInfo gfs_adapt_info = {
      "GfsAdapt",
      sizeof (GfsAdapt),
      sizeof (GfsEventClass),
      (GtsObjectClassInitFunc) gfs_adapt_class_init,
      (GtsObjectInitFunc) gfs_adapt_init,
      (GtsArgSetFunc) NULL,
      (GtsArgGetFunc) NULL
    };
    klass = gts_object_class_new (GTS_OBJECT_CLASS (gfs_event_class ()),
				  &gfs_adapt_info);
  }

  return klass;
}

/* GfsAdaptVorticity: Object */

static void gfs_adapt_vorticity_read (GtsObject ** o, GtsFile * fp)
{
  if (GTS_OBJECT_CLASS (gfs_adapt_vorticity_class ())->parent_class->read)
    (* GTS_OBJECT_CLASS (gfs_adapt_vorticity_class ())->parent_class->read) 
      (o, fp);
  if (fp->type == GTS_ERROR)
    return;

  if (fp->type != GTS_INT && fp->type != GTS_FLOAT) {
    gts_file_error (fp, "expecting a number (maxangle)");
    return;
  }
  GFS_ADAPT_VORTICITY (*o)->maxangle = atof (fp->token->str);
  gts_file_next_token (fp);
}

static void gfs_adapt_vorticity_write (GtsObject * o, FILE * fp)
{
  if (GTS_OBJECT_CLASS (gfs_adapt_vorticity_class ())->parent_class->write)
    (* GTS_OBJECT_CLASS (gfs_adapt_vorticity_class ())->parent_class->write) 
      (o, fp);
  fprintf (fp, " %g", GFS_ADAPT_VORTICITY (o)->maxangle);
}

static gboolean gfs_adapt_vorticity_event (GfsEvent * event, 
					   GfsSimulation * sim)
{
  if ((* GFS_EVENT_CLASS (GTS_OBJECT_CLASS (gfs_adapt_vorticity_class ())->parent_class)->event) (event, sim)) {
    GfsAdaptVorticity * a = GFS_ADAPT_VORTICITY (event);

    a->maxa = a->maxangle*gfs_domain_norm_velocity (GFS_DOMAIN (sim), 
						   FTT_TRAVERSE_LEAFS, 
						   -1).infty;
    return TRUE;
  }
  return FALSE;
}

static void gfs_adapt_vorticity_class_init (GfsEventClass * klass)
{
  GFS_EVENT_CLASS (klass)->event = gfs_adapt_vorticity_event;
  GTS_OBJECT_CLASS (klass)->read = gfs_adapt_vorticity_read;
  GTS_OBJECT_CLASS (klass)->write = gfs_adapt_vorticity_write;
}

static gboolean refine_vorticity (FttCell * cell, GfsAdaptVorticity * a)
{
  g_return_val_if_fail (cell != NULL, FALSE);

  gfs_variable_set_parent (gfs_div, gfs_object_simulation (a));
  gfs_vorticity (cell, gfs_div);
  if (fabs (GFS_STATE (cell)->div)*ftt_cell_size (cell) > a->maxa)
    return TRUE;
  return FALSE;
}

static void gfs_adapt_vorticity_init (GfsAdaptVorticity * object)
{
  GFS_ADAPT (object)->refine = (FttCellRefineFunc) refine_vorticity;
  object->maxangle = 1e-2;
}

GfsEventClass * gfs_adapt_vorticity_class (void)
{
  static GfsEventClass * klass = NULL;

  if (klass == NULL) {
    GtsObjectClassInfo gfs_adapt_vorticity_info = {
      "GfsAdaptVorticity",
      sizeof (GfsAdaptVorticity),
      sizeof (GfsEventClass),
      (GtsObjectClassInitFunc) gfs_adapt_vorticity_class_init,
      (GtsObjectInitFunc) gfs_adapt_vorticity_init,
      (GtsArgSetFunc) NULL,
      (GtsArgGetFunc) NULL
    };
    klass = gts_object_class_new (GTS_OBJECT_CLASS (gfs_adapt_class ()),
				  &gfs_adapt_vorticity_info);
  }

  return klass;
}

/* GfsAdaptGradient: Object */

static void gfs_adapt_gradient_read (GtsObject ** o, GtsFile * fp)
{
  GfsVariable * v;
  GfsDomain * domain;

  if (GTS_OBJECT_CLASS (gfs_adapt_gradient_class ())->parent_class->read)
    (* GTS_OBJECT_CLASS (gfs_adapt_gradient_class ())->parent_class->read) 
      (o, fp);
  if (fp->type == GTS_ERROR)
    return;

  if (fp->type != GTS_STRING) {
    gts_file_error (fp, "expecting a string (variable name)");
    return;
  }
  domain = gfs_object_simulation (*o);
  v = gfs_variable_from_name (domain->variables, fp->token->str);
  if (v == NULL) {
    gts_file_error (fp, "unknown variable `%s'", fp->token->str);
    return;
  }
  GFS_ADAPT_GRADIENT (*o)->v = v;
  gts_file_next_token (fp);

  if (fp->type != GTS_INT && fp->type != GTS_FLOAT) {
    gts_file_error (fp, "expecting a number (max)");
    return;
  }
  GFS_ADAPT_GRADIENT (*o)->max = atof (fp->token->str);
  gts_file_next_token (fp);
}

static void gfs_adapt_gradient_write (GtsObject * o, FILE * fp)
{
  if (GTS_OBJECT_CLASS (gfs_adapt_gradient_class ())->parent_class->write)
    (* GTS_OBJECT_CLASS (gfs_adapt_gradient_class ())->parent_class->write) 
      (o, fp);
  fprintf (fp, " %s %g", 
	   GFS_ADAPT_GRADIENT (o)->v->name,
	   GFS_ADAPT_GRADIENT (o)->max);
}

static void gfs_adapt_gradient_class_init (GfsEventClass * klass)
{
  GTS_OBJECT_CLASS (klass)->read = gfs_adapt_gradient_read;
  GTS_OBJECT_CLASS (klass)->write = gfs_adapt_gradient_write;
}

static gboolean refine_gradient (FttCell * cell, GfsAdaptGradient * a)
{
  FttComponent c;
  gdouble sum2 = 0;
  gdouble * lambda;

  g_return_val_if_fail (cell != NULL, FALSE);

  lambda = (gdouble *) &GFS_DOMAIN (gfs_object_simulation (a))->lambda;
  for (c = 0; c < FTT_DIMENSION; c++) {
    gdouble g = lambda[c]*gfs_center_gradient (cell, c, a->v->i);

    sum2 += g*g;
  }
  if (sum2 > a->max*a->max)
    return TRUE;
  return FALSE;
}

static void gfs_adapt_gradient_init (GfsAdaptGradient * object)
{
  GFS_ADAPT (object)->refine = (FttCellRefineFunc) refine_gradient;
}

GfsEventClass * gfs_adapt_gradient_class (void)
{
  static GfsEventClass * klass = NULL;

  if (klass == NULL) {
    GtsObjectClassInfo gfs_adapt_gradient_info = {
      "GfsAdaptGradient",
      sizeof (GfsAdaptGradient),
      sizeof (GfsEventClass),
      (GtsObjectClassInitFunc) gfs_adapt_gradient_class_init,
      (GtsObjectInitFunc) gfs_adapt_gradient_init,
      (GtsArgSetFunc) NULL,
      (GtsArgGetFunc) NULL
    };
    klass = gts_object_class_new (GTS_OBJECT_CLASS (gfs_adapt_class ()),
				  &gfs_adapt_gradient_info);
  }

  return klass;
}

/* GfsAdaptNotBox: Object */

static gboolean gfs_adapt_not_box_event (GfsEvent * event, 
					 GfsSimulation * sim)
{
  if ((* GFS_EVENT_CLASS (GTS_OBJECT_CLASS (gfs_adapt_not_box_class ())->parent_class)->event) (event, sim)) {
    GfsAdaptNotBox * a = GFS_ADAPT_NOT_BOX (event);

    if (a->box && a->box->root) {
      FttVector p;
      gdouble size;

      ftt_cell_pos (a->box->root, &p);
      size = ftt_cell_size (a->box->root)/2.;
      a->p1.x = p.x - size; a->p2.x = p.x + size; 
      a->p1.y = p.y - size; a->p2.y = p.y + size;
      a->p1.z = p.z - size; a->p2.z = p.z + size;
      return TRUE;
    }
    else 
      gts_object_destroy (GTS_OBJECT (event));
  }
  return FALSE;
}

static void gfs_adapt_not_box_class_init (GfsEventClass * klass)
{
  GFS_EVENT_CLASS (klass)->event = gfs_adapt_not_box_event;
}

static gboolean refine_not_box (FttCell * cell, GfsAdaptNotBox * a)
{
  FttVector p;

  ftt_cell_pos (cell, &p);
  if (p.x < a->p1.x || p.x > a->p2.x ||
      p.y < a->p1.y || p.y > a->p2.y ||
      p.z < a->p1.z || p.z > a->p2.z)
    return TRUE;
  return FALSE;
}

static void gfs_adapt_not_box_init (GfsAdaptNotBox * object)
{
  GFS_ADAPT (object)->not = TRUE;
  GFS_ADAPT (object)->refine = (FttCellRefineFunc) refine_not_box;
}

GfsEventClass * gfs_adapt_not_box_class (void)
{
  static GfsEventClass * klass = NULL;

  if (klass == NULL) {
    GtsObjectClassInfo gfs_adapt_not_box_info = {
      "GfsAdaptNotBox",
      sizeof (GfsAdaptNotBox),
      sizeof (GfsEventClass),
      (GtsObjectClassInitFunc) gfs_adapt_not_box_class_init,
      (GtsObjectInitFunc) gfs_adapt_not_box_init,
      (GtsArgSetFunc) NULL,
      (GtsArgGetFunc) NULL
    };
    klass = gts_object_class_new (GTS_OBJECT_CLASS (gfs_adapt_class ()),
				  &gfs_adapt_not_box_info);
  }

  return klass;
}

GfsAdaptNotBox * gfs_adapt_not_box_new (GfsEventClass * klass,
					GfsBox * box)
{
  GfsAdaptNotBox * a;

  g_return_val_if_fail (klass != NULL, NULL);
  g_return_val_if_fail (box != NULL, NULL);

  a = GFS_ADAPT_NOT_BOX (gts_object_new (GTS_OBJECT_CLASS 
					 (gfs_adapt_not_box_class ())));
  a->box = box;
  gfs_event_set (GFS_EVENT (a), -1., -1., -1., -1, -1, 1);
  return a;
}

static gboolean refine_simulation (FttCell * cell, GfsSimulation * sim)
{
  GSList * i = sim->adapts->items;

  while (i) {
    GfsAdapt * a = i->data;

    if (a->active && a->not && !(*a->refine) (cell, a))
      return FALSE;
    i = i->next;
  }

  i = sim->adapts->items;
  while (i) {
    GfsAdapt * a = i->data;

    if (a->active && !a->not && (*a->refine) (cell, a))
      return TRUE;
    i = i->next;
  }
  return FALSE;
}

/**
 * gfs_simulation_adapt:
 * @simulation: a #GfsSimulation.
 *
 * Checks if any mesh adaptation is necessary and adapts the mesh
 * using an OR combination of all the regular criteria defined in
 * @simulation->adapts.
 * 
 * If any one or several criteria are defined as "not" refinements,
 * the mesh will be refined only if all of this criteria AND any other
 * regular criterium is verified.  
 */
void gfs_simulation_adapt (GfsSimulation * simulation)
{
  GSList * i;
  gboolean active = FALSE;
  guint minlevel = 0, maxlevel = G_MAXINT;
  gdouble start, end;
  GfsDomain * domain;

  g_return_if_fail (simulation != NULL);

  domain = GFS_DOMAIN (simulation);
  start = g_timer_elapsed (domain->timer, NULL);

  gfs_simulation_event (simulation, simulation->adapts->items);
  i = simulation->adapts->items;
  while (i) {
    GfsAdapt * a = i->data;

    if (a->active && !a->not) {
      active = TRUE;
      if (a->minlevel > minlevel) minlevel = a->minlevel;
      if (a->maxlevel < maxlevel) maxlevel = a->maxlevel;
    }
    i = i->next;
  }
  if (active) {
    gfs_domain_adapt (GFS_DOMAIN (simulation), minlevel, maxlevel,
		     (FttCellRefineFunc) refine_simulation, simulation);
    end = g_timer_elapsed (domain->timer, NULL);
    gts_range_add_value (&domain->adapt, end - start);
    gts_range_update (&domain->adapt);    
  }
}
