#include <math.h>
#include <stdlib.h>
#include <string.h>
#include "config.h"
#ifdef HAVE_GETOPT_H
#  include <getopt.h>
#endif /* HAVE_GETOPT_H */
#ifdef HAVE_UNISTD_H
#  include <unistd.h>
#endif /* HAVE_UNISTD_H */

#include "graphic.h"
#include "solid.h"
#include "init.h"
#include "simulation.h"

static gboolean is_mixed (FttCell * cell, guint level)
{
  if (GFS_IS_MIXED (cell))
    return TRUE;
  if (!FTT_CELL_IS_ROOT (cell) && ftt_cell_level (cell) > level)
    return is_mixed (ftt_cell_parent (cell), level);
  return FALSE;
}

static void inject (FttCell * cell)
{
  if (!FTT_CELL_IS_LEAF (cell)) {
    FttCellChildren child;
    guint i;

    ftt_cell_children (cell, &child);
    for (i = 0; i < FTT_CELLS; i++)
      if (child.c[i]) {
	GFS_STATE (child.c[i])->dp = GFS_STATE (cell)->dp;
	inject (child.c[i]);
      }
  }
}

static gboolean difference_tree (FttCell * cell,
				 GfsDomain * ref,
				 GfsVariable * v,
				 gdouble period)
{
  guint level = ftt_cell_level (cell);
  FttVector pos;
  FttCell * locate;
  gboolean added = FALSE;
  
  ftt_cell_pos (cell, &pos);
  pos.x += period;
  locate = gfs_domain_locate (ref, pos, level);
  if (locate == NULL) {
    pos.x -= 2.*period;
    locate = gfs_domain_locate (ref, pos, level);
  }
  if (locate == NULL) {
    fprintf (stderr, "gfscompare: the files are not comparable\n");
    exit (1);
  }
  if (ftt_cell_level (locate) != level)
    return FALSE;
  if (!FTT_CELL_IS_LEAF (cell)) {
    FttCellChildren child;
    guint i;

    ftt_cell_children (cell, &child);
    for (i = 0; i < FTT_CELLS; i++)
      if (child.c[i] && difference_tree (child.c[i], ref, v, period))
	added = TRUE;
  }
  if (!added) {
    GFS_STATE (cell)->dp = GFS_VARIABLE (locate, v->i) - 
                           GFS_VARIABLE (cell, v->i);
    inject (cell);
  }
  return TRUE;
}

static void difference_box (GfsBox * box, gpointer * data)
{
  gdouble * period = data[2];

  difference_tree (box->root, data[0], data[1], *period);
}

static void difference (FttCell * cell, gpointer * data)
{
  gint full = *((gint *) data[0]);
  GfsNorm * norm = data[1];
  gboolean * histogram = data[2];

  if (full == -2 || 
      (full == -1 && !GFS_IS_MIXED (cell)) ||
      (full >= 0 && !is_mixed (cell, full))) {
    gfs_norm_add (norm, GFS_STATE (cell)->dp, 
		  ftt_cell_volume (cell)*(GFS_IS_MIXED (cell) ? 
					  GFS_STATE (cell)->solid->a : 1.));
    if (*histogram)
      printf ("%g %g\n", GFS_STATE (cell)->dp, 
	      GFS_IS_MIXED (cell) ? GFS_STATE (cell)->solid->a : 1.);
  }
}

static void compute_gradient (FttCell * cell, gpointer * data) 
{
  GfsVariable * v = data[0];
  FttComponent * c = data[1];

  GFS_STATE (cell)->g[0] = 
    gfs_center_gradient (cell, *c, v->i)/ftt_cell_size (cell);
}

static void compute_log (FttCell * cell) 
{
  GFS_STATE (cell)->dp = log10 (fabs (GFS_STATE (cell)->dp) + 1e-10);
}

int main (int argc, char * argv[])
{
  GtsFile * fp;
  FILE * f;
  int c = 0;
  GfsVariable * var;
  GfsSimulation * s1, * s2;
  
  gboolean verbose = FALSE;
  gint full = -2;
  gboolean no_check = FALSE;
  gboolean output = FALSE;
  gboolean squares = FALSE;
  gboolean take_log = FALSE;
  gchar * fname1, * fname2;
  gdouble period = 0.;

  FttComponent gradient = FTT_DIMENSION;

  GfsNorm norm;
  gpointer data[3];

  gboolean refined_error = FALSE;
  gboolean histogram = FALSE;

  gfs_init (&argc, &argv);

  /* parse options using getopt */
  while (c != EOF) {
#ifdef HAVE_GETOPT_LONG
    static struct option long_options[] = {
      {"period", required_argument, NULL, 'p'},
      {"histogram", no_argument, NULL, 'H'},
      {"refined", no_argument, NULL, 'r'},
      {"log", no_argument, NULL, 'l'},
      {"full", required_argument, NULL, 'f'},
      {"gradient", required_argument, NULL, 'g'},
      {"output", no_argument, NULL, 'o'},
      {"squares", no_argument, NULL, 'S'},
      {"nocheck", no_argument, NULL, 'n'},
      {"help", no_argument, NULL, 'h'},
      {"verbose", no_argument, NULL, 'v'},
    };
    int option_index = 0;
    switch ((c = getopt_long (argc, argv, "hvnog:f:lSrHp:",
			      long_options, &option_index))) {
#else /* not HAVE_GETOPT_LONG */
    switch ((c = getopt (argc, argv, "hvnog:f:lSrHp:"))) {
#endif /* not HAVE_GETOPT_LONG */
    case 'p': /* period */
      period = atof (optarg);
      break;
    case 'H': /* histogram */
      histogram = TRUE;
      break;
    case 'r': /* refined */
      refined_error = TRUE;
      break;
    case 'l': /* log */
      take_log = TRUE;
      break;
    case 'f': /* full */
      full = atoi (optarg);
      break;
    case 'g': /* gradient */
      gradient = atoi (optarg);
      if (gradient >= FTT_DIMENSION) {
	fprintf (stderr, 
		 "gfscompare: invalid argument for option `gradient'.\n"
		 "Try `gfscompare --help' for more information.\n");
	return 1; /* failure */
      }
      break;
    case 'S': /* squares */
      squares = TRUE;
      break;
    case 'o': /* output */
      output = TRUE;
      break;
    case 'n': /* nocheck */
      no_check = TRUE;
      break;
    case 'v': /* verbose */
      verbose = TRUE;
      break;
    case 'h': /* help */
      fprintf (stderr,
     "Usage: gfscompare [OPTION] FILE1 FILE2 VAR\n"
     "Computes the difference between the solutions in FILE1 and FILE2\n"
     "for variable VAR.\n"
     "\n"
     "  -p P  --period=P    shifts FILE1 by P along the x axis\n"
     "  -H    --histogram   output (error,volume) pairs for each cell used\n"
     "                      to compute the error norms\n"
     "  -o    --output      output a GTS representation of the error field\n"
     "  -S    --squares     output an OOGL representation of the error field\n"
     "  -l    --log         output the log10 of the absolute value of the error field\n"
     "  -f L  --full=L      compare only leaf cells descendants of a cell full at level L\n"
     "                      or all full leaf cells if L = -1\n"
     "  -r    --refined     display error norm on the finest grid\n"
     "  -n    --nocheck     do not check solid fractions\n"
     "  -g C  --gradient=C  use the C component of the gradient of VAR\n"
     "  -v    --verbose     display difference statistics and other info\n"
     "  -h    --help        display this help and exit\n"
     "\n"
     "Reports bugs to %s\n",
	       FTT_MAINTAINER);
      return 0; /* success */
      break;
    case '?': /* wrong options */
      fprintf (stderr, "Try `gfscompare --help' for more information.\n");
      return 1; /* failure */
    }
  }

  if (optind >= argc) { /* missing FILE1 */  
    fprintf (stderr, 
	     "gfscompare: missing FILE1\n"
	     "Try `gfscompare --help' for more information.\n");
    return 1; /* failure */
  }
  fname1 = argv[optind++];

  if (optind >= argc) { /* missing FILE2 */  
    fprintf (stderr, 
	     "gfscompare: missing FILE2\n"
	     "Try `gfscompare --help' for more information.\n");
    return 1; /* failure */
  }
  fname2 = argv[optind++];

  if (optind >= argc) { /* missing VAR */  
    fprintf (stderr, 
	     "gfscompare: missing VAR\n"
	     "Try `gfscompare --help' for more information.\n");
    return 1; /* failure */
  }
  var = gfs_variable_from_name (gfs_p, argv[optind]);
  if (var == NULL) {
    fprintf (stderr, 
	     "gfscompare: unknown variable `%s'\n"
	     "Try `gfscompare --help' for more information.\n",
	     argv[optind]);
    return 1; /* failure */
  }
  optind++;

  f = fopen (fname1, "rt");
  if (f == NULL) {
    fprintf (stderr, "gfscompare: cannot open file `%s'\n", fname1);
    return 1;
  }
  fp = gts_file_new (f);
  s1 = gfs_simulation_new (gfs_simulation_class ());
  if (gfs_simulation_read (s1, fp)) {
    fprintf (stderr, 
	     "gfscompare: file `%s' is not a valid simulation file\n"
	     "%s:%d:%d: %s\n",
	     fname1, fname1, fp->line, fp->pos, fp->error);
    return 1;
  }
  gts_file_destroy (fp);
  fclose (f);

  f = fopen (fname2, "rt");
  if (f == NULL) {
    fprintf (stderr, "gfscompare: cannot open file `%s'\n", fname2);
    return 1;
  }
  fp = gts_file_new (f);
  s2 = gfs_simulation_new (gfs_simulation_class ());
  if (gfs_simulation_read (s2, fp)) {
    fprintf (stderr, 
	     "gfscompare: file `%s' is not a valid simulation file\n"
	     "%s:%d:%d: %s\n",
	     fname2, fname2, fp->line, fp->pos, fp->error);
    return 1;
  }
  gts_file_destroy (fp);
  fclose (f);

  if (verbose) {
    norm = gfs_domain_norm_variable (GFS_DOMAIN (s1),
				    var, FTT_TRAVERSE_LEAFS, -1);
    fprintf (stderr, "%s: first: %g second: %g infty: %g w: %g\n",
	     fname1, norm.first, norm.second, norm.infty, norm.w);
    norm = gfs_domain_norm_variable (GFS_DOMAIN (s2),
				    var, FTT_TRAVERSE_LEAFS, -1);
    fprintf (stderr, "%s: first: %g second: %g infty: %g w: %g\n",
	     fname2, norm.first, norm.second, norm.infty, norm.w);
  }

  if (gradient < FTT_DIMENSION) {
    gpointer data[2];

    data[0] = var;
    data[1] = &gradient;

    gfs_domain_cell_traverse (GFS_DOMAIN (s1), 
			      FTT_PRE_ORDER, FTT_TRAVERSE_LEAFS, -1,
			      (FttCellTraverseFunc) compute_gradient, data);
    gfs_domain_cell_traverse (GFS_DOMAIN (s2), 
			      FTT_PRE_ORDER, FTT_TRAVERSE_LEAFS, -1,
			      (FttCellTraverseFunc) compute_gradient, data);
    var = gfs_gx;
  }

  gfs_domain_cell_traverse (GFS_DOMAIN (s1), 
			    FTT_POST_ORDER, FTT_TRAVERSE_NON_LEAFS, -1,
			    (FttCellTraverseFunc) gfs_get_from_below_intensive,
			    var);
  gfs_domain_cell_traverse (GFS_DOMAIN (s2), 
			    FTT_POST_ORDER, FTT_TRAVERSE_NON_LEAFS, -1,
			    (FttCellTraverseFunc) gfs_get_from_below_intensive,
			    var);
  data[0] = s2;
  data[1] = var;
  data[2] = &period;
  gts_container_foreach (GTS_CONTAINER (s1), (GtsFunc) difference_box, data);
  
  gfs_norm_init (&norm);
  data[0] = &full;
  data[1] = &norm;
  data[2] = &histogram;
  gfs_domain_cell_traverse (GFS_DOMAIN (s1), 
			    FTT_PRE_ORDER, FTT_TRAVERSE_LEAFS, -1,
			    (FttCellTraverseFunc) difference, data);
  gfs_norm_update (&norm);
  if (verbose) {
    fprintf (stderr, 
	  "total err first: %10.3e second: %10.3e infty: %10.3e w: %g\n",
	     norm.first, norm.second, norm.infty, norm.w);
    if (refined_error) {
      norm = gfs_domain_norm_variable (GFS_DOMAIN (s1),
				       gfs_dp, FTT_TRAVERSE_LEVEL,
				       gfs_domain_depth (GFS_DOMAIN (s1)));
      fprintf (stderr, 
	  "refined err first: %10.3e second: %10.3e infty: %10.3e w: %g\n",
	       norm.first, norm.second, norm.infty, norm.w);
    }
  }

  if (output || squares) {
    if (take_log)
      gfs_domain_cell_traverse (GFS_DOMAIN (s1), 
			       FTT_PRE_ORDER, FTT_TRAVERSE_LEAFS, -1,
			       (FttCellTraverseFunc) compute_log, NULL);
    if (squares) {
      GtsRange stats = gfs_domain_stats_variable (GFS_DOMAIN (s1), gfs_dp,
						  FTT_TRAVERSE_LEAFS, -1);

      gfs_write_squares (GFS_DOMAIN (s1), gfs_dp, stats.min, stats.max,
			 FTT_TRAVERSE_LEAFS, -1, 
			 NULL, stdout);
    }
    else
      gfs_write_gts (GFS_DOMAIN (s1), 
		     gfs_dp, FTT_TRAVERSE_LEAFS, -1, NULL, stdout);
  }

  return 0;
}
