/*  GFC-Core: GTK+ Foundation Classes (Core Library)
 *  Copyright (C) 2003-2004 The GFC Development Team.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/// @file gfc/glib/main.hh
/// @brief A C++ interface for the GLib main event loop.
///
/// Provides G::MainContext, G::MainLoop and G::Source.

#ifndef GFC_G_MAIN_HH
#define GFC_G_MAIN_HH

#ifndef GFC_OBJECT_HH
#include <gfc/object.hh>
#endif

#ifndef GFC_G_IOCHANNEL_HH
#include <gfc/glib/iochannel.hh>
#endif

#ifndef SIGCXX_SIGCXX_H
#include <sigc++/sigc++.h>
#endif

namespace GFC {

namespace G {

class TimeVal;

enum
{
	PRIORITY_HIGH = G_PRIORITY_HIGH,
	///< Use this for high priority event sources; It is not used within GLib or GTK+ (value -100).

	PRIORITY_DEFAULT = G_PRIORITY_DEFAULT,
	///< Use this for default priority event sources; In GLib this priority is used	
	///< when adding timeout functions with g_timeout_add(); In GDK this priority
	///< is used for events from the X server (value 0).

	PRIORITY_HIGH_IDLE = G_PRIORITY_HIGH_IDLE,
	///< Use this for high priority idle functions (value 100); GTK+ uses G_PRIORITY_HIGH_IDLE + 10
	///< for window resizes and G_PRIORITY_HIGH_IDLE + 20 for window redraws, to ensure that any
	///< pending resizes are processed before any pending redraws, so that widgets are not redrawn
	///< twice unnecessarily; <B>Note</B> that you will interfere with GTK+ if you use a priority above
	///< G_PRIORITY_HIGH_IDLE + 10 (that is, GTK_PRIORITY_RESIZE).

	PRIORITY_DEFAULT_IDLE = G_PRIORITY_DEFAULT_IDLE,
	///< Use this for default priority idle functions; In GLib this priority is used when
	///< adding idle functions with g_idle_add() (value 200).

	PRIORITY_LOW = G_PRIORITY_LOW
	///<  Use this for very low priority background tasks; It is not used within GLib or GTK+ (value 300).
};

/// @name Time Methods
/// @{

void get_current_time(TimeVal& timeval);
///< Equivalent to the UNIX gettimeofday() function, but portable.
///< @param timeval A TimeVal object in which to store the current time.

/// @}

/// @class MainContext main.hh gfc/glib/main.hh
/// A C++ interface for the GMainContext.
///
/// MainContext is an object representing a set of sources to be handled in a main loop.

class MainContext : public GFC::Object
{
	MainContext(const MainContext&);
	MainContext& operator=(const MainContext&);

	GMainContext *context_;

public:
/// @name Constructors
/// @{

	MainContext();
	///< Construct a new main context object.

	explicit MainContext(GMainContext *context, bool owns_reference = true);
	///< Construct a main context from an existing GMainContext.
	///< @param context A pointer to a GMainContext.
	///< @param owns_reference Set <EM>true</EM> if the initial reference count is owned by this object.
	///<
	///< <BR>The MainContext object takes over the ownership of the GMainContext and
	///< unreferences it when the destructor is called.

	virtual ~MainContext();
	///< Destructor.

/// @}

	static Pointer<MainContext> get_default();
	///< Gets the default main context.
	///< @return The default main context.
	///<
	///< <BR>This is the main context used for main loop functions when a main loop is not explicitly specified.

/// @name Accessors
/// @{

	GMainContext* g_main_context() const;
	///< Get a pointer to the GMainContext object.

	operator GMainContext* () const;
	///< Conversion operator; Safely converts a G::MainContext object into a GMainContext pointer.

	bool pending() const;
	///< Checks if any sources have pending events for the given context.
	///< @return <EM>true</EM> if events are pending.

/// @}
/// @name Methods
/// @{

	virtual void ref();
	///< Increases the reference count of the main context by one.

	virtual void unref();
	///< Decreases the reference count of the main context by one.
	///< When the reference count becomes zero delete is called. Remember, as with all GFC Objects
	///< you must call unref() on a dynamically allocated IOChannel, not delete. If you use a smart
	///< you don't need to call unref(), the smart pointer will do that for you. You must also call
	///< unref() on an IOChannel allocated on the stack because IOChannels don't get passed on to
	///< owner objects that manage their reference count (like widgets in a container).

	bool iteration(bool may_block);
	///< Runs a single iteration of the main loop.
	///< @param may_block Whether the call may block.
	///< @return <EM>true</EM> if events were dispatched.
	///<
	///< <BR>This involves checking to see if any event sources are ready to be processed, then if
	///< no events sources are ready and <EM>may_block</EM> is true, waiting for a source to become
	///< ready, then dispatching the highest priority events sources that are ready. Note that even
	///< when <EM>may_block</EM> is true, it is still possible for iteration() to return false,
	///< since the the wait may be interrupted for other reasons than an event source becoming ready.

/// @}
};

/// @class MainLoop main.hh gfc/glib/main.hh
/// A C++ interface for the GMainLoop.
///
/// It manages all the available sources of events
/// for GLib and GTK+ applications. These events can come from any number of different types
/// of sources such as file descriptors (plain files, pipes or sockets) and timeouts. New
/// types of event sources can also be added using G::Source::attach().
///
/// To allow multiple independent sets of sources to be handled in different threads, each
/// source is associated with a MainContext. A MainContext can only be running in a single
/// thread, but sources can be added to it and removed from it from other threads.
///
/// Each event source is assigned a priority. The default priority, G::PRIORITY_DEFAULT,
/// is 0. Values less than 0 denote higher priorities. Values greater than 0 denote lower
/// priorities. Events from high priority sources are always processed before events from
/// lower priority sources.
///
/// Idle functions can also be added, and assigned a priority. These will be run whenever
/// no events with a higher priority are ready to be processed.
///
/// MainLoop represents a main event loop. After constructing a MainLoop and adding the
/// initial event sources, run() is called. This continuously checks for new events from
/// each of the event sources and dispatches them. Finally, the processing of an event from
/// one of the sources leads to a call to quit() to exit the main loop, and run() returns.
///
/// It is possible to create new instances of MainLoop recursively. This is often used in
/// GTK+ applications when showing modal dialog boxes. Note that event sources are associated
/// with a particular MainContext, and will be checked and dispatched for all main loops
/// associated with that MainContext.

class MainLoop : public GFC::Object
{
	MainLoop(const MainLoop&);
	MainLoop& operator=(const MainLoop&);

	GMainLoop *loop_;

public:
/// @name Constructors
/// @{

	MainLoop(bool is_running = false);
	///< Construct a new main loop object using the default MainContext.
	///< @param is_running Set <EM>true</EM> to indicate that the loop is running.
	///<
	///< <BR>The <EM>is_running</EM> argument is not very important since calling
	///< run() will set this to true anyway.

	MainLoop(MainContext& context, bool is_running = false);
	///< Construct a new main loop object using the specified MainContext.
	///< @param context A MainContext.
	///< @param is_running Set <EM>true</EM> to indicate that the loop is running.
	///<
	///< <BR>The <EM>is_running</EM> argument is not very important since calling
	///< run() will set this to true anyway.

	virtual ~MainLoop();
	///< Destructor.

/// @}
/// @name Accessors
/// @{

	GMainLoop* g_main_loop() const;
	///< Get a pointer to the GMainLoop object.

	operator GMainLoop* () const;
	///< Conversion operator; Safely converts a G::MainLoop object into a GMainLoop pointer.

	bool is_running() const;
	///< Checks to see if the main loop is currently being run via run().
	///< @return <EM>true</EM> if the main loop is currently being run.

	Pointer<MainContext> get_context() const;
	///< Obtains the MainContext of loop.
	///< @return The MainContext of loop

/// @}
	
	static int depth();
	///< Get the main loop recursion level in the current thread.
	///< @return The main loop recursion level in the current thread.
	///<
	///< <BR>When called from the toplevel, this method returns 0. When called from
	///< within a callback from G::MainContext::iteration() (or G::MainLoop::run(), etc.)
	///< it returns 1. When called from within a callback to a recursive call to 
	///< G::MainContext::iteration(), it returns 2. And so forth.

/// @name Methods
/// @{

	virtual void ref();
	///< Increases the reference count on a main loop object by one.

	virtual void unref();
	///< Decreases the reference count on a main loop object by one.
	///< When the reference count becomes zero for a heap object delete is called.
	///< Remember, as with all GFC Objects you must call unref() on a dynamically
	///< allocated MainLoop, not delete. If you use a smart you don't need to call
	///< unref(), the smart pointer will do that for you. You don't need to call
	///< unref() on a loop allocated on the stack unless you first called ref().

	void run();
	///< Runs a main loop until quit() is called on the loop.
	///< If this is called for the thread of the loop's MainContext, it will process
	///< events from the loop, otherwise it will simply wait.

	void quit();
	///< Stops a main loop from running, and any calls to run() for the loop will return.

/// @}
};

/// @class Source main.hh gfc/glib/main.hh
/// A GSource C++ interface.

class Source : public Trackable
{
	Source(const Source&);
	Source& operator=(const Source&);

	GSource *source_;

public:
/// @name Constructors
/// @{

	Source(GSource *source, bool owns_reference = true);
	///< Construct a source from an existing GSource.
	///< @param source A pointer to a GSource.
	///< @param owns_reference Set <EM>true</EM> if the initial reference count is owned by this object.
	///<
	///< <BR>The Source object takes over the ownership of the GSource and unreferences
	///< it when the destructor is called.

	virtual ~Source();
	///< Destructor.

/// @}
/// @name Accessors
/// @{

	GSource* g_source() const;
	///< Get a pointer to the GSource object.

	operator GSource* () const;
	///< Conversion operator; Safely converts a G::Source object into a GSource pointer.

	int get_priority() const;
	///< Gets the priority of the source.
	///< @return The priority of the source.

	bool get_can_recurse() const;
	///< Checks whether a source is allowed to be called recursively (see set_can_recurse()).
	///< @return Whether recursion is allowed.

	unsigned int get_id() const;
	///< Gets the numeric ID for a particular source.
	///< @return The ID for the source.

	Pointer<MainContext> get_context() const;
	///< Gets the main context with which the source is associated.
	///< @return The associated MainContext, or null if a context has not yet been added to the source.
	///<
	///< <BR>Calling this function on a destroyed source is an error.

	void get_current_time(TimeVal& timeval);
	///< Gets the "current time" to be used when checking this source.
	///< @param timeval A TimeVal object in which to store the current time.
	///<
	///< <BR>The advantage of calling this function over calling G::get_current_time()
	///< directly is that when checking multiple sources, GLib can cache a single value
	///< instead of having to repeatedly get the system time.

	bool is_attached(G::MainContext& context) const;
	///< Determines whether <EM>context</EM> is attached to this source.
	///< @return <EM>true</EM> if context is attached to this source.

/// @}
/// @name Methods
/// @{

	virtual void ref();
	///< Increase the source reference count by one.

	virtual void unref();
	///< Decrease the source reference count by one.
	///< When the reference count becomes zero for a heap object delete is called.
	///< Remember, as with all GFC Objects you must call unref() on a dynamically
	///< allocated IOChannel, not delete. If you use a smart you don't need to call
	///< unref(), the smart pointer will do that for you. You don't need to call unref()
	///< on a source allocated on the stack unless you first called ref().

	unsigned int attach(MainContext *context = 0);
	///< Adds the source to <EM>context</EM> so that it will be executed within that context.
	///< @param context A MainContext, or null to use the default context.
	///< @return The ID for the source within the MainContext.

	void destroy();
	///< Removes a source from its MainContext, if any, and marks it as destroyed.
	///< After calling this method the source cannot be subsequently added to another context.

	void set_priority(int priority);
	///< Sets the priority of the source.
	///< @param priority The new priority.
	///<
	///< <BR>While the main loop is being run, the source will be dispatched if it is ready
	///< to be dispatched and no sources at a higher (numerically smaller) priority are
	///< ready to be dispatched.

	void set_can_recurse(bool can_recurse);
	///< Sets whether a source can be called recursively.
	///< @param can_recurse Whether recursion is allowed for this source.
	///<
	///< <BR>If <EM>can_recurse</EM> is true, while the source is being dispatched
	///< the source will be processed normally. Otherwise, all processing of the
	///< source is blocked until the dispatch function returns.

/// @}
};

/// @class ChildWatchSource main.hh gfc/glib/main.hh
/// A child watch GSource C++ interface.


class ChildWatchSource : public Source
{
	ChildWatchSource(const ChildWatchSource&);
	ChildWatchSource& operator=(const ChildWatchSource&);

public:
	typedef sigc::slot<void, GPid, int> WatchSlot;				       
	/// Signature for the callback slot to call when a child process being watched exits,
	/// at a default priority G::PRIORITY_DEFAULT.
	///<
	///< <B>Example:</B> Method signature for WatchSlot.
	///< @code
	///< void method(GPid pid, int status);
	///< @endcode

/// @name Constructors
/// @{

	ChildWatchSource(GPid pid);
	///< Constructs a new child watch source. 
	///< @param pid The process id of a child process to watch. 
	///<
	///< <BR>The source will not initially be associated with any main context and
	///< must be added to one with G::Source::attach() before it will be executed. 
	///< Note that on platforms where GPid must be explicitely closed pid must not
	///< be closed while the source is still active (see G::Spawn::close_pid()).
	///< Typically, you will want to call G::Spawn::close_pid() in the callback
	///< slot method for the source.
	///<
	///< To set the callback slot call set_callback(). To override the default
	///< priority <EM>PRIORITY_DEFAULT</EM> call set_priority().
	
	ChildWatchSource(GPid pid, const WatchSlot& slot);
	///< Constructs a new child watch source. 
	///< @param pid The process id of a child process to watch. 
	///< @param slot The child watch callback slot, of type sigc::slot<void, GPid, int>.
	///<
	///< <BR>The source will not initially be associated with any main context and
	///< must be added to one with G::Source::attach() before it will be executed. 
	///< Note that on platforms where GPid must be explicitely closed pid must not
	///< be closed while the source is still active (see G::Spawn::close_pid()).
	///< Typically, you will want to call G::Spawn::close_pid() in the callback
	///< slot method for the source.

/// @}
/// @name Methods
/// @{

	void set_callback(const WatchSlot& slot);
	///< Sets the callback slot for the child watch source.
	///< @param slot The callback slot, of type sigc::slot<void, GPid, int>.

/// @}
};

/// @class TimeoutSource main.hh gfc/glib/main.hh
/// A timeout GSource C++ interface.
///
/// TimeoutSource is a source that's called at regular intervals, with the given priority.
/// The TimeoutSlot is called repeatedly until it returns false, at which point the timeout
/// is automatically destroyed and will not be called again. The first call to the slot
/// will be at the end of the first interval.

class TimeoutSource : public Source
{
	TimeoutSource(const TimeoutSource&);
	TimeoutSource& operator=(const TimeoutSource&);

public:
	typedef sigc::slot<bool> TimeoutSlot;
	///< Signature of the callback slot to passed to TimeoutSource.
	///< <B>Example:</B> Method signature for TimeoutSlot.
	///< @code
	///< bool method();
	///< // return: The slot should return false if the source should be removed.
	///< @endcode

/// @name Constructors
/// @{

	TimeoutSource(unsigned int interval);
	///< Constructs a new timeout source that can be associated with any main context.
	///< @param interval The time between calls to the function, in milliseconds.
	///<
	///< <BR>G::Source::attach() must be called to attach the source to a context.
	///< To set the callback slot call set_callback(). To override the default
	///< priority <EM>PRIORITY_HIGH</EM> call set_priority().

	TimeoutSource(const TimeoutSlot& slot, unsigned int interval);
	///< Constructs a new timeout source that can be associated with any main context.
	///< @param slot The callback slot, of type sigc::slot<bool>.
	///< @param interval The time between calls to the function, in milliseconds.
	///<
	///< <BR>G::Source::attach() must be called to attach the source to a context.
	///< To override the default priority <EM>PRIORITY_HIGH</EM> call set_priority().

/// @}
/// @name Methods
/// @{

	void set_callback(const TimeoutSlot& slot);
	///< Sets the callback slot for the timeout source.
	///< @param slot The callback slot, of type sigc::slot<bool>.

/// @}
};

/// @class IdleSource main.hh gfc/glib/main.hh
/// A idle GSource C++ interface.
///
/// IdleSource is a source that's called whenever there are no higher priority events pending.
/// If the IdleSlot returns false it is automatically removed from the list of event sources
/// and will not be called again.

class IdleSource : public Source
{
	IdleSource(const IdleSource&);
	IdleSource& operator=(const IdleSource&);

public:
	typedef sigc::slot<bool> IdleSlot;
	///< Signature of the callback slot to passed to IdleSource.
	///< <B>Example:</B> Method signature for IdleSlot.
	///< @code
	///< bool method();
	///< // return: The slot should return false if the source should be removed.
	///< @endcode

/// @name Constructors
/// @{

	IdleSource();
	///< Constructs a new idle source that can be associated with any main context.
	///< G::Source::attach() must be called to attach the source to a context.
	///< To set the callback slot call set_callback(). To override the default
	///< priority <EM>PRIORITY_DEFAULT_IDLE</EM> call set_priority().

	IdleSource(const IdleSlot& slot);
	///< Constructs a new idle source that can be associated with any main context.
	///< @param slot The callback slot, of type sigc::slot<bool>.
	///<
	///< <BR>G::Source::attach() must be called to attach the source to a context.
	///< To override the default priority <EM>PRIORITY_DEFAULT_IDLE</EM> call
	///< set_priority().

/// @}
/// @name Methods
/// @{

	void set_callback(const IdleSlot& slot);
	///< Sets the callback slot for the idle source.
	///< @param slot The callback slot, of type sigc::slot<bool>.

/// @}
};

/// @class IOSource main.hh gfc/glib/main.hh
/// A input/output GSource C++ interface.
///
/// IOSource is a source that's dispatched when <EM>condition</EM> is met for the io channel.
/// For example, if condition is G::IO_IN, the source will be dispatched when there's data
/// available for reading. If the IOSlot returns false it is automatically removed from the
/// list of event sources and will not be called again.

class IOSource : public Source
{
	IOSource(const IOSource&);
	IOSource& operator=(const IOSource&);

public:
	typedef sigc::slot<bool, IOChannel&, IOConditionField> IOSlot;
	///< Signature of the callback slot to be called when the requested condition on a IOChannel is satisfied.
	///< <B>Example:</B> Method signature for IOSlot.
	///< @code
	///< bool method(IOChannel& source, IOConditionField condition);
	///< // source: The IOChannel event source.
	///< // condition: The condition which has been satisfied.
	///< // return: The slot should return false if the event source should be removed.
	///< @endcode

/// @name Constructors
/// @{

	IOSource(G::IOChannel& channel, G::IOConditionField condition);
	///< Constructs a new io source that can be associated with any main context.
	///< @param channel A G::IOChannel.
	///< @param condition The condition to watch for.
	///<
	///< <BR>G::Source::attach() must be called to attach the source to a context.
	///< To set the callback slot call set_callback(). To override the default
	///< priority PRIORITY_DEFAULT you can call set_priority().

	IOSource(G::IOChannel& channel, G::IOConditionField condition, const IOSlot& slot);
	///< Constructs a new io source that can be associated with any main context.
	///< @param channel A G::IOChannel.
	///< @param condition The condition to watch for.
	///< @param slot The callback slot, of type sigc::slot<bool, IOChannel&, IOConditionField>.
	///<
	///< <BR>G::Source::attach() must be called to attach the source to a context. To
	///< override the default priority PRIORITY_DEFAULT you can call set_priority().

/// @}
/// @name Methods
/// @{

	void set_callback(const IOSlot& slot);
	///< Sets the callback slot for the io source.
	///< @param slot The callback slot, of type sigc::slot<bool, IOChannel&, IOConditionField>.

/// @}
};

/// @class ChildWatchSignal main.hh gfc/main.hh
/// @brief A child watch signal class.

class ChildWatchSignal : public sigc::trackable
{
public:
	typedef sigc::slot<void, GPid, int> SlotType;
	///< Function signature for handlers connected to this signal.

	sigc::connection connect(const SlotType& slot, GPid pid, int priority = PRIORITY_DEFAULT);
	///< Sets a callback slot to be called when the child indicated by <EM>pid</EM> exits,
	///< at a default priority, G::PRIORITY_DEFAULT.
	///< @param slot The sigc::slot to call periodically.
	///< @param pid The process id of a child process to watch. 
	///< @param priority The priority of the idle source. 
	///< @return A sigc::connection object that can be used to break or alter the connection.
	///<	
	///< <BR>Note that on platforms where GPid must be explicitely closed pid must not be
	///< closed while the source is still active (see G::Spawn::close_pid()). Typically,
	///< you will want to call (see G::Spawn::close_pid()) in the callback slot for the
	///< source.
	///<
	///< Typically the priority will be in the range between G::PRIORITY_DEFAULT_IDLE and 
	///< G::PRIORITY_HIGH_IDLE. 
};

/// A namespace instance of the ChildWatchSignal for connecting slots to be invoked periodically.

extern ChildWatchSignal child_watch_signal;

/// @class TimeoutSignal main.hh gfc/main.hh
/// @brief A Timeout signal class.
///
/// Timeout signal is a convenience class that lets you connect a callback slot to the default
/// main context. If your using a different context, for example, inside a thread, you should
/// use TimeoutSource instead. If the slot returns false the timeout source will be removed.

class TimeoutSignal : public sigc::trackable
{
public:
	typedef sigc::slot<bool> SlotType;
	///< Function signature for handlers connected to this signal.

	sigc::connection connect(const SlotType& slot, unsigned int interval, int priority = PRIORITY_DEFAULT_IDLE);
	///< Sets a callback slot to be called at regular intervals, with the given priority.
	///< @param slot The sigc::slot to call periodically.
	///< @param interval The time between calls to the function, in milliseconds.
	///< @param priority The priority of the idle source.
	///< @return A sigc::connection object that can be used to break or alter the connection.
	///<
	///< The slot is called repeatedly until it returns <EM>false</EM>, at which point
	///< the timeout is automatically destroyed and the function will not be called again.
	///< Typically the priority will be in the range between G::PRIORITY_DEFAULT_IDLE and
	///< G::PRIORITY_HIGH_IDLE.
	///<
	///< Note that timeout functions may be delayed, due to the processing of other event
	///< sources. Thus they should not be relied on for precise timing. After each call to
	///< the timeout function, the time of the next timeout is recalculated based on the
	///< current time and the given interval (it does not try to 'catch up' time lost in
	///< delays).
};

/// A namespace instance of the TimeoutSignal for connecting slots to be invoked periodically.

extern TimeoutSignal timeout_signal;

/// @class IdleSignal main.hh gfc/main.hh
/// @brief Idle signal class.
///
/// IdleSignal is a convenience class that adds a callback slot to the default main context,
/// to be called whenever there are no higher priority events pending. If the slot returns
/// false it is automatically removed from the list of event sources and will not be called
/// again.

class IdleSignal : public sigc::trackable
{
public:
	typedef sigc::slot<bool> SlotType;
	///< Function signature for handlers connected to this signal.

	sigc::connection connect(const SlotType& slot, int priority = PRIORITY_DEFAULT_IDLE);
	///< Connect a slot to be called when the event loop is idle.
	///< @param slot The slot to call.
	///< @param priority The priority which should not be above G::PRIORITY_HIGH_IDLE.
	///< @return A connection object that can be used to break or alter the connection.
	///<
	///< <BR>You can give a priority different from G::PRIORITY_DEFAULT_IDLE to the idle function.
	///< Note that you will interfere with GTK+ if you use a priority above G::PRIORITY_RESIZE.
	///< The user function returns <EM>false</EM> to remove itself or <EM>true</EM> to have it
	///< called again.
};

/// A namespace instance of IdleSignal for connecting slots to be invoked when the event loop is idle.

extern IdleSignal idle_signal;

/// @class IOSignal main.hh gfc/main.hh
/// @brief IO signal class.
///
/// IOSource is a convenience class that adds a callback slot to the default main context, to be called
/// when a given condition is met. If the slot returns false it is automatically removed from the list
/// of event sources and will not be called again.

class IOSignal : public sigc::trackable
{
public:
	typedef sigc::slot<bool, IOChannel&, IOConditionField> SlotType;
	///< Function signature for handlers connected to this signal.

	sigc::connection connect(G::IOChannel& channel, G::IOConditionField condition, const SlotType& slot, int priority = PRIORITY_DEFAULT);
	///< Connect a slot to be called when the event loop is idle.
	///< @param channel A G::IOChannel.
	///< @param condition The condition to watch for.
	///< @param slot The callback slot to call when the condition is satisfied.
	///< @param priority The priority of the io source.
	///< @return A connection object that can be used to break or alter the connection.
};

/// A namespace instance of IdleSignal for connecting slots to be invoked when the event loop is idle.

extern IOSignal io_signal;

} // namespace G

} // namespace GFC

#include <gfc/glib/inline/main.inl>

#endif // GFC_G_MAIN_HH

