/* $Id: gg.c,v 1.37 2003/01/27 11:58:08 thrull Exp $ */

/*
 * (C) Copyright 2001-2002 Igor Popik. Released under terms of GPL license.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <unistd.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <arpa/inet.h>
#include <netinet/in.h>

#include "gg-types.h"
#include "gg.h"
#include "ggcore.h"
#include "http.h"
#include "support.h"
#include "interface.h"
#include "net.h"
#include "userstatus.h"
#include "callbacks.h"

#if HAVE_ESD || HAVE_ARTS
#include "sound.h"
#endif
#ifdef USE_DOCKAPP
#include "dock.h"
#endif

extern GGConfig config;
extern GGStatus status;

#define MAX_BUF 2048

extern GList *kontakty;
extern GList *sessions;
extern GList *ignore;
extern GList *msg_spool;

GList *watches = NULL;

extern GtkWidget *window;
extern GtkWidget *wyniki_szukaj;

extern GList *wyniki_kontakty;

static void gg_animate_off();

#define MAX_CHUNK_SIZE 1024

void gg_remove_watch(int type)
{
	GList *tmplist;
	GGWatch *r;

	tmplist = watches;
	while (tmplist) {
		r = ((GGWatch *) (tmplist->data));
		if (r->type == GG_HTTP_QUERY) {
			GGHTTPQuery *q = r->data;
			if (q->type == type) {
				watches = g_list_remove(watches, r);
				tcp_watch_destroy(r);
				return;
			}
		}
		tmplist = tmplist->next;
	}
}
static
void gg_watch_timeout(GGWatch * r)
{
	if (r->type == GG_HTTP_QUERY) {
		GGHTTPQuery *q = r->data;
		switch (q->type) {
		case GG_RESOLVE_SERVER:
			if (status.retry) {
			    status.retry -= 1;
			    gg_reconnect();
			} else {
			    show_error_dialog(_("Nastapi bd podczas czenia do serwera GG!"));
			    gg_animate_off();
			    gg_change_status(GG_STATUS_OFFLINE);
			}
			break;
		case GG_CHANGE_PASSWORD:
			show_error_dialog(_("Nastapi bd podczas zmiany hasa!"));
			break;
		case GG_REMIND_PASSWORD:
			show_error_dialog(_("Nastapi bd podczas przypominania hasa na e-mail!"));
			break;
		case GG_SEARCH:
			show_error_dialog(_("Nastpi bd! Wyszukiwanie nie powiodo si!"));
			break;
		case GG_REGISTER:
			show_error_dialog(_("Nastpi bd podczas rejestracji nowego uytkownika!"));
			break;
		case GG_EXPORT_USERLIST:
			show_error_dialog(_("Nastpi bd podczas eksportu listy uytkownikw!"));
			break;
		case GG_IMPORT_USERLIST:
			show_error_dialog(_("Nastpi bd podczas importu listy uytkownikw!"));
			break;
		case GG_DELETE_USERLIST:
			show_error_dialog(_("Nastpi bd podczas kasowania listy uzytkownikw!"));
			break;
		case GG_CHANGE_PUBDIR_INFO:
			show_error_dialog(_("Bd podczas wysyania danych do katalogu publicznego!!!"));
			break;
		default:
			g_warning(_("Nieznany typ zaptyania HTTP %d"),
				  q->type);
		}
	} else {
		g_warning(_("Nieznany typ GGWatch->type %d"), r->type);
	}
	watches = g_list_remove(watches, r);
	tcp_watch_destroy(r);
}

static
gint gg_watches_timeout(gpointer user_data)
{
	GList *tmplist;
	GGWatch *r = NULL;
	gint cnt = 0;

	tmplist = watches;
	g_print("TIMEOUT %d\n", (int) tmplist);
	while (tmplist) {
		r = ((GGWatch *) (tmplist->data));

		if (r != NULL) {
			g_print("timeout = %d\n", r->timeout);

			if (r->timeout) {
				r->timeout--;
			} else {
				gg_watch_timeout(r);
				return TRUE;
			}
		}
		cnt++;
		tmplist = tmplist->next;
	}
	g_print("OUT\n");
	if (cnt)
		return TRUE;
	else {
		// Jeli nie ma adnych eventw na skadzie wyczamy timeat
		status.watches_timeout = 0;
		return FALSE;
	}
}
static
gint gg_reconnect_timeout(gpointer user_data)
{
	gg_connect_to_server();
	return FALSE;
}

void gg_reconnect()
{
	gtk_timeout_add(GG_RECONNECT_TIMEOUT,
			(GtkFunction) gg_reconnect_timeout, NULL);

}

static gint gg_client_ping(gpointer data)
{
	GGCmd *cmd;
	g_print("gg_client_ping(): PING!\n");
	if (status.connected == GG_LOGIN_OK) {
		cmd = g_new0(GGCmd, 1);
		cmd->type = GG_CLIENT_ONLINE;
		gg_send_cmd(cmd, status.sock);
		g_free(cmd);
	}
	return TRUE;
}

static
gint gg_animate_func(gpointer userdata)
{
	if (status.animate_pic) {
		set_status_img_status(GG_STATUS_ONLINE);
		status.animate_pic = 0;
	} else {
		set_status_img_status(GG_STATUS_OFFLINE);
		status.animate_pic = 1;
	}
	return TRUE;
}

#if (USE_DOCKAPP || USE_DOCKLET || USE_APPLET)
static
gint gg_new_msg_animate_func(gpointer userdata)
{
	if (!status.new_msg_pic) {
		set_dock_icon(GG_EMPTY);
		status.new_msg_pic = 1;
	} else {
		set_dock_icon(GG_STATUS_NEW_MSG);
		status.new_msg_pic = 0;
	}
	status.new_msg_anim_count--;
	
	if (!status.new_msg_anim_count) {
	    set_dock_icon(GG_STATUS_NEW_MSG);
	    status.new_msg_pic = 0;
	    return FALSE;
	} else {
	    return TRUE;
	}
	
	return TRUE;
}
#endif

void gg_animate_off()
{
	if (status.animate_timeout != 0)
		gtk_timeout_remove(status.animate_timeout);
	set_status_img_status(GG_STATUS_OFFLINE);
}

void gg_new_msg_animate_off()
{
	if (status.new_msg_timeout != 0)
		gtk_timeout_remove(status.new_msg_timeout);
	set_dock_icon(GG_STATUS_NEW_MSG);
}


/* jesli przekroczymny timeout poczenia do wyswietli sie komunikat */
static
gint gg_timeout_func(gpointer userdata)
{
	if (status.retry) {
		status.retry -= 1;
		gg_reconnect();
		return TRUE;
	} else {
		gg_error
		    ("Przekroczono czas oczekiwania na odpowied serwera!");
		gg_change_status(GG_STATUS_OFFLINE);
		return FALSE;
	}
}
static
void gg_second_stage_connect()
{
	struct in_addr a;

	if (!status.hostname) {
		status.hostname = g_strdup(GG_DEFAULT_HOST);
		status.port = GG_DEFAULT_PORT;
	}
	
	a.s_addr = inet_addr(status.hostname);

	status.sock = connect_to_server(&a, status.port, 1);

	if (status.sock == -1) {
		status.retry -= 1;
		if (status.retry) {
			gg_connect();
		} else {
			gg_animate_off();
			gg_error("Nie mog si podczy do serwera");
		}
		return;
	}
	status.chan = g_io_channel_unix_new(status.sock);
	status.tcp_line = g_io_add_watch(status.chan, G_IO_IN | G_IO_ERR | G_IO_HUP, (GIOFunc) socket_callback, NULL);	
}

/* Rozpoczcie procesu czenia z serwerem */
void gg_connect_to_server()
{
	gg_animate_off();

	status.animate_timeout =
	    gtk_timeout_add(500, (GtkFunction) gg_animate_func, NULL);

	if (status.online_timeout != 0)
		gtk_timeout_remove(status.online_timeout);

	gg_connect();
}

static void gg_send_user_add(guint uin)
{
	GGCmd *cmd;
	gchar b = 0x03;

	if (status.connected == GG_LOGIN_OK) {

		cmd = g_new0(GGCmd, 1);
		cmd->type = GG_NOTIFY_ADD;
		cmd->datalen = GG_NOTIFY_DATALEN;
		cmd->data = g_malloc(GG_NOTIFY_DATALEN);

		uin = fix32(uin);

		memcpy(cmd->data, &uin, 4);
		memcpy(cmd->data + 4, &b, 1);

		gg_send_cmd(cmd, status.sock);
		g_free(cmd->data);
		g_free(cmd);

	}
}

static void gg_send_user_list()
{
	GGCmd *cmd;
	GList *tmplist;
	guint count, pos = 0;

	if (status.connected == GG_LOGIN_OK) {
		gchar *tmpdata;

		tmplist = kontakty;
		count = g_list_length(tmplist);

		if (count == 0)
			return;

		cmd = g_new0(GGCmd, 1);
		cmd->type = GG_NOTIFY;
		tmpdata = g_malloc(GG_NOTIFY_DATALEN * count);

		while (tmplist) {
			guint uin;
			gchar b = 0x03;
			GGContact *kontakt;

			kontakt = ((GGContact *) (tmplist->data));

			uin = fix32(kontakt->uin);
			if (uin) {
				memcpy(tmpdata + pos, &uin, 4);
				memcpy(tmpdata + pos + 4, &b, 1);
				pos += GG_NOTIFY_DATALEN;
			} else {
				count--;
			}
			tmplist = tmplist->next;
		}
		cmd->datalen = GG_NOTIFY_DATALEN * count;
		cmd->data = g_malloc(GG_NOTIFY_DATALEN * count);

		memcpy(cmd->data, tmpdata, (GG_NOTIFY_DATALEN * count));

		gg_send_cmd(cmd, status.sock);
		g_free(cmd->data);
		g_free(cmd);
		g_free(tmpdata);
	}
}

void gg_login_ok()
{
	gchar title[MAX_BUF];

	status.connected = GG_LOGIN_OK;
	
	g_snprintf(title, MAX_BUF - 1, "GG - %d", config.uin);
	gtk_window_set_title(GTK_WINDOW(window), _(title));

#if HAVE_ESD || HAVE_ARTS
	if(config.login_sound)
		gg_play_sound(GG_SOUND_LOGIN_OK);
#endif
	if (status.animate_timeout != 0)
		gtk_timeout_remove(status.animate_timeout);
	if (status.online_timeout != 0)
		gtk_timeout_remove(status.online_timeout);
	status.online_timeout =
	    gtk_timeout_add(100000, (GtkFunction) gg_client_ping, NULL);

	set_status_img_status(status.state);
	set_optionmenu_menu(status.state);
	gg_send_user_list();
}

gchar *gg_get_nick_by_uin(guint uin)
{
	gchar *nick = NULL;
	GList *tmplist;
	GGContact *kontakt;

	tmplist = kontakty;

	while (tmplist) {
		kontakt = ((GGContact *) (tmplist->data));
		if (kontakt->uin == uin) {
			nick = kontakt->nick;
			break;
		}
		tmplist = tmplist->next;
	}
	return nick ? nick : "";
}

GGContact *gg_get_contact_by_uin(guint uin)
{
	GGContact *kontakt = NULL;
	GList *tmplist;

	tmplist = kontakty;

	while (tmplist) {
		kontakt = ((GGContact *) (tmplist->data));

		if (kontakt->uin == uin)
			return kontakt;

		tmplist = tmplist->next;
	}
	return NULL;
}

GGContact *gg_get_contact_by_mobile(gchar *mobile)
{
	GGContact *kontakt = NULL;
	GList *tmplist;

	tmplist = kontakty;

	while (tmplist) {
		kontakt = ((GGContact *) (tmplist->data));

		if (kontakt->mobile && !g_strcasecmp(kontakt->mobile, mobile)) {
			g_warning("ZNALEZNIONO MOBILE KONTAKT %s %s", kontakt->nick, kontakt->mobile);
			return kontakt;
		}

		tmplist = tmplist->next;
	}
	
	return NULL;
}

gchar *gg_get_status_as_string(guint uin)
{
	GGContact *kontakt = gg_get_contact_by_uin(uin);
	
	if (kontakt != NULL) {
	    switch (kontakt->status) {
		case GG_STATUS_OFFLINE:
		case GG_STATUS_OFFLINE_DESCR:
		    return "niedostpny";
    		case GG_STATUS_ONLINE:
    		case GG_STATUS_ONLINE_DESCR:
		    return "dostpny";
		case GG_STATUS_AWAY:
		case GG_STATUS_AWAY_DESCR:
		    return "zajty";
		default:
		    return "nieznany";
	    }
	} else {
	    return "nieznany";
	}
}

static
void gg_log_msg(gint sess_uin, gint uin, gchar *nick, gchar *text, gint more_than_one_day, gint time)
{

	gchar *path, *lognick;
	gchar *timestamp1, *timestamp2;
	FILE *fp;

	path = g_strdup_printf("%s/history", config.homedir);
	mkdir(path, 0755);
	g_free(path);

	path =
	    g_strdup_printf("%s/history/%d", config.homedir, sess_uin);
	fp = fopen(path, "a");
	g_free(path);

	if (!fp) {
		show_error_dialog
		    ("Nie mog utworzy pliku z histori rozmw!!! Historia rozmw zostaa wyczona.\n");
		config.log = 0;
		return;
	}

	if (nick != NULL) 
		lognick = g_strdup(nick);
	else 
		lognick = g_strdup_printf("%d", uin);
	
	timestamp1 = get_full_timestamp(0);
	timestamp2 = (more_than_one_day) ? get_full_timestamp(time) : get_timestamp(time);

        fprintf(fp, "%s :: %s (%s)\n%s\n", lognick, timestamp1, timestamp2, text);

	g_free(lognick);
	g_free(timestamp1);
	g_free(timestamp2);
	fclose(fp);
}

void gg_recv_msg(GGCmd * cmd)
{
	GtkWidget *lista, *textarea, *target = NULL;
	gchar *text, *title = NULL, *nick, *header;
	GGContact *kontakt;
	GGSession *sesja;
	gboolean freecontact = FALSE;
	GdkFont *font;
	struct gg_recv_msg *m = cmd->data;
	guint more_than_one_day = 0;
	gint sec;
	gchar *timestamp1, *timestamp2;
	
	lista = lookup_widget(window, "lista");

	g_return_if_fail(lista != NULL);
	
	m->sender = fix32(m->sender);
	m->time = fix32(m->time);
	m->class = fix32(m->class);
	m->seq = fix32(m->seq);

	if (gg_is_ignored(m->sender)) {
		g_free(cmd->data);
		g_free(cmd);
		return;
	}

	text = g_malloc0(cmd->datalen - sizeof(struct gg_recv_msg) + 1);
	memcpy(text, cmd->data + sizeof(struct gg_recv_msg), cmd->datalen - sizeof(struct gg_recv_msg));

	kontakt = gg_get_contact_by_uin(m->sender);

	if (kontakt == NULL) {
		kontakt = g_new0(GGContact, 1);
		kontakt->uin = m->sender;
		freecontact = TRUE;
	}

	sesja = gg_find_session(m->sender);

	if (sesja == NULL) {
		sesja = gg_create_session(m->sender);
		sessions = g_list_append(sessions, sesja);
	}
	g_print("gg_recv_msg(): msg class=%d\n", m->class);
	if (m->class & GG_CLASS_CHAT) {

		if (kontakt->nick && *kontakt->nick) {
			title =
			    g_strdup_printf("Rozmowa z %s (%d) - %s",
					    kontakt->nick, m->sender,
						gg_get_status_as_string(kontakt->uin));
		} else {
			title = g_strdup_printf("Rozmowa z %d", m->sender);
		}
		if (!GTK_IS_WIDGET(sesja->chat_window)
		    || sesja->chat_window == NULL
		    || !GTK_IS_WINDOW(sesja->chat_window))
			sesja->chat_window = create_chat(sesja);
		target = sesja->chat_window;
		textarea = lookup_widget(target, "chat_text");
	} else {
		if (kontakt->nick && *kontakt->nick) {
			title =
			    g_strdup_printf("Wiadomo od %s (%d)",
					    kontakt->nick, kontakt->uin);
		} else {
			title =
			    g_strdup_printf("Wiadomo od  %d", m->sender);
		}
		if (!GTK_IS_WIDGET(sesja->msg_window)
		    || sesja->msg_window == NULL
		    || !GTK_IS_WINDOW(sesja->msg_window))
			sesja->msg_window = create_msg(sesja);
		target = sesja->msg_window;
		textarea = lookup_widget(target, "text");
	} 

	gtk_window_set_title(GTK_WINDOW(target), title);

	g_free(title);
	
	nick = gg_get_nick_by_uin(m->sender);
	
	sec = time(NULL) - m->time;
	if (sec > 86400) {
	    more_than_one_day = 1;
	}
	timestamp1 = get_timestamp(0);
	timestamp2 = (more_than_one_day) ? get_full_timestamp(m->time) : get_timestamp(m->time);

	if (nick && *nick) {
		header =
		    g_strdup_printf("%s :: %s (%s)\n", nick, timestamp1, timestamp2);
	} else {
		header =
		    g_strdup_printf("%d :: %s (%s)\n", m->sender, timestamp1, timestamp2);
	}
	
	g_free(timestamp1);
	g_free(timestamp2);
	
	if (config.log) {
		gg_log_msg(m->sender, m->sender, nick, text, more_than_one_day, m->time);
	}
	font = gdk_font_load(config.bold_font);
	gtk_text_insert(GTK_TEXT(textarea), font, NULL, NULL, header,
			strlen(header));
	cp_to_iso(text);
	remove_cr(text);

	gtk_text_insert(GTK_TEXT(textarea), NULL, NULL, NULL, text,
			strlen(text));
	gtk_text_insert(GTK_TEXT(textarea), NULL, NULL, NULL, "\n", 1);

#if USE_APPLET || USE_DOCKLET || USE_DOCKAPP
	if (config.popup_msg_windows) {
	    gtk_widget_show(target);
	} else if (!GTK_WIDGET_VISIBLE(target)) {
	    msg_spool = g_list_remove(msg_spool, target); // usun z jakiegos miejsca na liscie
	    msg_spool = g_list_prepend(msg_spool, target); // wstaw na pierwsza pozycje
	    set_dock_icon(GG_NEW_MSG_IMG);
	    gg_new_msg_animate_off();
	    status.new_msg_anim_count = 4;

	    status.new_msg_timeout =
		 gtk_timeout_add(500, (GtkFunction) gg_new_msg_animate_func, NULL);
#ifdef USE_DOCKAPP
	    wmgg_new_msg(TRUE);
#endif
	}
#else
	gtk_widget_show(target);
#endif

#if HAVE_ESD || HAVE_ARTS
	if(config.msg_sound)
		gg_play_sound(GG_SOUND_MSG);
#endif

	if(status.state == GG_STATUS_AWAY && config.away_msg_send != 0) {
		gg_send_message(m->sender, m->class, config.away_msg,
				strlen(config.away_msg));
	}

	if (freecontact)
		g_free(kontakt);

	g_free(header);
	g_free(text);
}

void gg_add_kontakt(GGContact * k)
{
	GGContact *kontakt;
	
	kontakt = g_new0(GGContact, 1);
	kontakt->uin = k->uin;
	kontakt->nick = g_strdup(k->nick);
	kontakt->status = GG_STATUS_OFFLINE;
	kontakt->first_name = g_strdup(k->first_name);
	kontakt->last_name = g_strdup(k->last_name);
	kontakt->mobile = g_strdup(k->mobile);
	kontakt->comment = g_strdup(k->comment);
	
	kontakty = g_list_append(kontakty, kontakt);

	add_kontakt_to_list(kontakt);

	gg_send_user_add(kontakt->uin);
}

static void gg_change_user_status(guint uin, guint status, gulong ip, gchar *descr)
{
	GList *tmplist;
	GGContact *kontakt;
	guint prev_status;

	tmplist = kontakty;

	while (tmplist) {
		kontakt = ((GGContact *) (tmplist->data));
		if (kontakt->uin == uin) {
			prev_status = kontakt->status;
			kontakt->status = status;
			
			if (kontakt->description) {
			    g_free(kontakt->description);
			    kontakt->description = NULL;
			}
			
			if (descr) {
			    cp_to_iso(descr);
			    kontakt->description = descr;
			}
			if (ip)  {
			    kontakt->remote_ip = ip;
			} else if (status == GG_STATUS_OFFLINE) {
				kontakt->remote_ip = 0;
			}
			
			change_kontakt_status(kontakt, prev_status);

#if HAVE_ESD || HAVE_ARTS
			if (status == GG_STATUS_ONLINE) {
				if(config.user_sound)
					gg_play_sound(GG_SOUND_NOTIFY);
			}
#endif

			g_print ("gg_change_user_status(): Znaleziono kontakt %d\n", uin);
			break;
		}
		tmplist = tmplist->next;
	}
}

void gg_notify_replay(GGCmd * cmd)
{

	gint pos = 0, count = cmd->datalen;
	gchar *descr = NULL;
	
	while (pos < count) {
		struct gg_notify_reply *r = cmd->data + pos;
		
		pos += sizeof(struct gg_notify_reply);
		if (r->status == GG_STATUS_ONLINE_DESCR || 
		    r->status == GG_STATUS_AWAY_DESCR || 
		    r->status == GG_STATUS_OFFLINE_DESCR) {
		    descr = strdup(cmd->data + pos);
		    pos += strlen(descr);
		    g_print("Uwaga!!! Status jest z opisem!\n");
		}
		
		gg_change_user_status(fix32(r->uin), fix32(r->status), r->remote_ip, descr);
		g_print("gg_user_status_replay(): Dostaem odpowied: uin=%ld status=%ld\n",
		     fix32(r->uin), fix32(r->status));
	}
}

void gg_change_contact_status(GGCmd * cmd)
{
	struct gg_status *s = cmd->data;	
	gchar *descr = NULL;
	
	if (s->status == GG_STATUS_ONLINE_DESCR || 
	    s->status == GG_STATUS_AWAY_DESCR || 
	    s->status == GG_STATUS_OFFLINE_DESCR) {
	    descr = g_strdup(cmd->data + sizeof(struct gg_status));
	}
	g_print("Zmiana statusu usera\n");
	gg_change_user_status(fix32(s->uin), fix32(s->status), 0, descr);
}

gboolean gg_send_message(guint uin, guint class, gchar * text,
			 gint text_len)
{
	GGCmd *cmd;
	struct gg_send_msg m;
	gchar *msg = insert_cr(text);
	
	cmd = g_new0(GGCmd, 1);
	cmd->type = GG_SEND_MSG;
	cmd->datalen = sizeof(struct gg_send_msg) + strlen(msg) + 1;
	cmd->data = g_malloc0(cmd->datalen);

	srand(uin);
	m.recipient = fix32(uin);
	m.seq = fix32(776611 & rand());
	m.class = fix32(class);
	
	memcpy(cmd->data, &m, sizeof(struct gg_send_msg));

	iso_to_cp(msg);
	memcpy(cmd->data + 12, msg, strlen(msg));
	gg_send_cmd(cmd, status.sock);

	if (config.log) {
		gg_log_msg(uin, config.uin, getenv("USER"), text, 0, 0);
	}

	g_free(cmd->data);
	g_free(cmd);
	g_free(msg);
	
	return TRUE;
}

void gg_msg_send_ack(GGCmd * cmd)
{
#ifdef DEBUG
	struct gg_send_msg_ack *s = cmd->data;
#endif
	/* Moe kiedy bdziemy sprawdza czy doszo i co bdzie
	z tego faktu wynika ;-) */
#ifdef DEBUG
	s->status = fix32(s->status);
	g_print("gg_msg_send_ack(): Msg ACK=%d\n", s->status);
#endif
}

GGSession *gg_find_session(guint uin)
{
	GList *tmpsesje;
	GGSession *sesja = NULL;

	tmpsesje = sessions;

	while (tmpsesje) {

		sesja = ((GGSession *) (tmpsesje->data));

		if (sesja->uin == uin)
			return sesja;

		tmpsesje = tmpsesje->next;
	}
	return NULL;
}

GGSession *gg_create_session(guint uin)
{
	GGSession *sesja;

	sesja = g_new0(GGSession, 1);

	sesja->uin = uin;
	sesja->msg_window = NULL;
	sesja->chat_window = NULL;

	return sesja;
}

GGSession *gg_find_session_by_sms_pid(gint sms_pid)
{
	GList *tmpsesje;
	GGSession *sesja = NULL;

	tmpsesje = sessions;

	while (tmpsesje) {

		sesja = ((GGSession *) (tmpsesje->data));

		if (sesja->sms_pid == sms_pid)
			return sesja;

		tmpsesje = tmpsesje->next;
	}
	return NULL;
}

void gg_close_connection()
{
	GList *tmplist;
	GGContact *kontakt;

	gg_animate_off();
	gtk_timeout_remove(status.online_timeout);

	tmplist = kontakty;
	while (tmplist) {
		kontakt = ((GGContact *) tmplist->data);
		gg_change_user_status(kontakt->uin, (kontakt->uin ? GG_STATUS_OFFLINE :
				       GG_STATUS_MOBILE_ONLY), 0, NULL);
		kontakt->remote_ip = 0;
		tmplist = tmplist->next;
	}

	gg_change_status(GG_STATUS_OFFLINE);
	tcp_watch_cleanup(&status.tcp_line);
	close(status.sock);
	status.connected = 0;
	status.sock = -1;
	set_status_img_status(GG_STATUS_OFFLINE);
	gg_remove_watch(GG_RESOLVE_SERVER);
	if (status.chan)
	    g_io_channel_unref(status.chan);
}

void gg_change_find_kontakt_status(GtkWidget * lista, GGContact * kontakt)
{
	gint row;
	gchar *nick;
	GdkPixmap *pixmap;
	GdkBitmap *mask;
	GdkColormap *colormap;
	gchar *picname;
	gchar *fullpath;
		
	g_return_if_fail(kontakt != NULL);
	g_return_if_fail(lista != NULL);

	nick = g_strdup_printf("%d", kontakt->uin);

	row = gtk_clist_find_row_from_data(GTK_CLIST(lista), kontakt);

	g_return_if_fail(row != -1);

	colormap = gtk_widget_get_colormap(lista);

	switch (kontakt->status) {
	case GG_STATUS_OFFLINE:
		picname = OFFLINE_IMG;
		break;
	case GG_STATUS_ONLINE:
		picname = ONLINE_IMG;
		break;
	case GG_STATUS_AWAY:
		picname = AWAY_IMG;
		break;
	default:
		picname = OFFLINE_IMG;
		break;
	}
	
	fullpath = GPP(picname);
	pixmap = gdk_pixmap_colormap_create_from_xpm(NULL, colormap, &mask,
						     NULL,
						     fullpath);
	g_free(fullpath);
	if (pixmap != NULL) {
		gtk_clist_set_pixtext(GTK_CLIST(lista), row, 0, nick, 3,
				      pixmap, mask);
	}
	g_free(nick);
}

void gg_add_wyniki_kontakt_to_list(GGContact * kontakt)
{
	gint row;
	gchar *line[4];
	GtkWidget *lista;

	line[0] = g_strdup_printf("%d", kontakt->uin);
	line[1] =
	    g_strjoin(" ",
		      (kontakt->first_name) ? kontakt->first_name : "",
		      kontakt->last_name, NULL);

	lista = lookup_widget(wyniki_szukaj, "wyniki_lista");
	g_return_if_fail(lista != NULL);

	row = gtk_clist_append(GTK_CLIST(lista), line);
	gtk_clist_set_row_data(GTK_CLIST(lista), row, kontakt);
	gg_change_find_kontakt_status(lista, kontakt);

}

gboolean gg_is_ignored(guint uin)
{
	GList *tmplist;
	guint luin;

	tmplist = ignore;

	while (tmplist) {
		luin = *(guint *) tmplist->data;
		if (luin == uin) {
			return TRUE;
		}
		tmplist = tmplist->next;
	}
	return FALSE;
}

void gg_connection_error()
{
	gg_close_connection();
	if (status.retry) {
		status.retry -= 1;
		gg_reconnect();
	} else {
		show_error_dialog
		    ("Bd podczas czenia z serwerem");
	}
}

void gg_send_query(GGWatch * r)
{
	GGHTTPQuery *q = r->data;
	gint c;

	if (r == NULL || q == NULL)
		return;
	g_print("gg_send_query(): %s\n", q->http_query);
	c = write(r->fd, q->http_query, strlen(q->http_query));

	tcp_watch_cleanup(&r->watch);

	if (!c || c == -1) {
		r->timeout = 0;
		g_warning(_("Bd przy wysyania zapytania HTTP!"));
		g_free(q->http_query);
	} else {
	    r->chan = g_io_channel_unix_new(r->fd);
	    r->watch = g_io_add_watch(r->chan, G_IO_IN | G_IO_ERR | G_IO_HUP, (GIOFunc) http_result, r);
	    g_io_channel_unref(r->chan);
	    g_print("gg_send_query() TYPE %d %d\n", q->type,(int) q);
	}

}

void gg_resolve_callback(GGWatch * r)
{
	struct in_addr a;
	a.s_addr = 0;

	if (r == NULL || r->data == NULL)
		return;

	if (read(r->fd, &a.s_addr, sizeof(a.s_addr)) < sizeof(a.s_addr)
	    || a.s_addr == 0) {
		show_error_dialog(_("Nie mog znale nazwy serwera!"));
		close(r->fd);
		watches = g_list_remove(watches, r);
		tcp_watch_destroy(r);
	} else {
		g_print("gg_resolve_callback(): Resolved: %s\n", inet_ntoa(a));

		tcp_watch_cleanup(&r->watch);
		close(r->fd);
		
		switch (r->type) {

		case GG_HTTP_QUERY:
			r->fd =
			    connect_to_server(&a,
					      (config.use_proxy ? config.
					       proxy_port : HTTP_PORT), 1);
			if (r->fd == -1) {
				r->timeout = 0;
				perror("gg_resolve_callback()");
				return;
			} 
			
			r->chan = g_io_channel_unix_new(r->fd);
			r->watch = g_io_add_watch(r->chan, G_IO_OUT | G_IO_ERR | G_IO_HUP, (GIOFunc) http_query_write, r);
			g_io_channel_unref(r->chan);
		}
	}
//	waitpid(r->pid, NULL, 0);
}

gint gg_async_http_query(GGHTTPQuery * q, gchar *host)
{
	int fd;
	int pid;
	GGWatch *r;
	
	if (!status.watches_timeout) {
		status.watches_timeout =
		    gtk_timeout_add(GG_RECONNECT_TIMEOUT,
				    (GtkFunction) gg_watches_timeout,
				    NULL);
	}

	switch (q->type) {
	case GG_CHANGE_PASSWORD:
	case GG_REMIND_PASSWORD:
	case GG_RESOLVE_SERVER:
	case GG_SEARCH:
	case GG_REGISTER:
	case GG_EXPORT_USERLIST:
	case GG_IMPORT_USERLIST:
	case GG_DELETE_USERLIST:
	case GG_CHANGE_PUBDIR_INFO:
		ekg_resolve(&fd, &pid,
			    (config.use_proxy ? config.
			     proxy_host : host));
		break;
	default:
		g_free(q->http_query);
		g_free(q);
		g_warning("Nieznany type zapytania HTTP");
		return FALSE;
	}


	r = g_new0(GGWatch, 1);
	r->timeout = GG_HTTP_TIMEOUT;
	r->type = GG_HTTP_QUERY;
	r->data = q;
	r->fd = fd;
	r->pid = pid;
	r->chan = g_io_channel_unix_new(fd);
	r->watch = g_io_add_watch(r->chan, G_IO_IN | G_IO_ERR | G_IO_HUP, (GIOFunc) resolve_callback, r);
	g_io_channel_unref(r->chan);
	watches = g_list_append(watches, r);
	if (pid == -1) {
		r->timeout = 0;
	}
	
	return TRUE;
}

void gg_change_password_res(GGWatch * r)
{
	gint datalen;
	gchar line[MAX_BUF];
	gint err = 0;

	memset(line, 0, MAX_BUF);


	datalen = http_query_read_head(r->fd);
	if (read(r->fd, line, datalen) > 0) {
		gchar success[MAX_BUF];
		snprintf(success, MAX_BUF, GG_REG_SUCCESS "%d", config.uin);
		if (strncmp(line, success, strlen(success)) != 0) {
			err = 1;
		}
	} else {
		err = 1;
	}

	close(r->fd);
	watches = g_list_remove(watches, r);
	tcp_watch_destroy(r);

	if (err) {
		show_error_dialog("Bd podczas zmiany hasa!!!");
	} else {
		show_message_dialog("Haso zostao zmienione!");
		write_config();
	}
}

void gg_change_password(gchar * password, gchar * email)
{
	gchar *http_query;
	gchar *query;
	GGHTTPQuery *q;
	gchar *_email;
	gchar *_password;
	gchar *_fmpassword;

	gg_remove_watch(GG_CHANGE_PASSWORD);
	
	_email = gg_urlencode(email);
	_password = gg_urlencode(password);
	_fmpassword = gg_urlencode(config.password);

	query =
	    g_strdup_printf
	    ("fmnumber=%d&fmpwd=%s&pwd=%s&email=%s&code=%ld", config.uin, _fmpassword, 
	    _password, _email, gg_http_hash(email, password));

	g_free(_email);
	g_free(_password);
	g_free(_fmpassword);

	http_query =
	    g_strdup_printf("POST %s/appsvc/fmregister.asp HTTP/1.0\r\n"
			    "Host: " GG_REGISTER_HOST "\r\n"
			    "Content-Type: application/x-www-form-urlencoded\r\n"
			    "User-Agent: " USER_AGENT "\r\n"
			    "Content-Length: %d\r\n" "Pragma: no-cache\r\n"
			    "\r\n" "%s\r\n",
			    (config.
			     use_proxy ? "http://" GG_REGISTER_HOST :
			     ""), strlen(query), query);

	q = g_new0(GGHTTPQuery, 1);
	q->type = GG_CHANGE_PASSWORD;
	q->http_query = http_query;

	if (gg_async_http_query(q, GG_REGISTER_HOST)) {
    	    if (config.password != NULL) {
		    g_free(config.password);
	    }
	    config.password = g_strdup(password);

    	    if (config.email != NULL) {
		    g_free(config.email);
	    }
	    config.email = g_strdup(email);
	}
	g_free(query);
}

void gg_register_res(GGWatch * r)
{
	gint datalen;
	gchar line[MAX_BUF];
	gint err = 0;

	memset(line, 0, MAX_BUF);

	datalen = http_query_read_head(r->fd);
	g_print("datalen %d\n", datalen);
	if (read(r->fd, line, datalen) > 0) {
		if (strncmp(line, GG_REG_SUCCESS, strlen(GG_REG_SUCCESS)) != 0) {
			err = 1;
		} else {
			config.uin = atoi(line + strlen(GG_REG_SUCCESS));
		}
	} else {
		err = 1;
	}

	close(r->fd);
	watches = g_list_remove(watches, r);
	tcp_watch_destroy(r);

	if (err) {
		show_error_dialog(_("Bd podczas rejestracji uytkownika"));
	} else {
		show_message_dialog(_("Nowy uytkownik zosta zarejestrowany"));
		write_config();
	}
}

void gg_register(gchar *password, gchar *email)
{
	gchar *http_query;
	gchar *query;
	GGHTTPQuery *q;
	gchar *_email;
	gchar *_password;

	gg_remove_watch(GG_REGISTER);
	
	_email = gg_urlencode(email);
	_password = gg_urlencode(password);
	
	query =
	    g_strdup_printf
	    ("pwd=%s&email=%s&code=%ld",  _password, _email, gg_http_hash(email, password));
	g_free(_email);
	g_free(_password);
		
	http_query =
	    g_strdup_printf("POST %s/appsvc/fmregister.asp HTTP/1.0\r\n"
			    "Host: " GG_REGISTER_HOST "\r\n"
			    "Content-Type: application/x-www-form-urlencoded\r\n"
			    "User-Agent: " USER_AGENT "\r\n"
			    "Content-Length: %d\r\n" 
			    "Pragma: no-cache\r\n"
			    "\r\n" "%s\r\n",
			    (config.
			     use_proxy ? "http://" GG_REGISTER_HOST :
			     ""), strlen(query), query);

	q = g_new0(GGHTTPQuery, 1);
	q->type = GG_REGISTER;
	q->http_query = http_query;

	if (gg_async_http_query(q, GG_REGISTER_HOST)) {
    	    if (config.password != NULL) {
		    g_free(config.password);
	    }
	    config.password = g_strdup(password);

    	    if (config.email != NULL) {
		    g_free(config.email);
	    }
	    config.email = g_strdup(email);
	}

	g_free(query);
}

void gg_restore_password_res(GGWatch * r)
{
	gint datalen;
	gchar line[MAX_BUF];
	gint err = 0;

	memset(line, 0, MAX_BUF);

	datalen = http_query_read_head(r->fd);

	if (read(r->fd, line, datalen) > 0) {
		if (strncmp
		    (line, GG_PWDSEND_SUCCESS,
		     strlen(GG_PWDSEND_SUCCESS)) != 0) {
			err = 1;
		}
	} else {
		err = 1;
	}

	close(r->fd);
	watches = g_list_remove(watches, r);
	tcp_watch_destroy(r);

	if (err) {
		show_error_dialog("Bd wysyania hasa na e-mail!!!");
	} else {
		show_message_dialog
		    ("Haso zostao wysane na Twj adres e-mail");
	}
}

void gg_restore_password()
{
	gchar *http_query;
	gchar *query;
	gchar uin_char[12];
	GGHTTPQuery *q;

	gg_remove_watch(GG_REMIND_PASSWORD);

	snprintf(uin_char, 12, "%d", config.uin);

	query =
	    g_strdup_printf("userid=%d&code=%ld", config.uin,
			    gg_http_hash(uin_char, ""));

	http_query =
	    g_strdup_printf("POST %s/appsvc/fmsendpwd.asp HTTP/1.0\r\n"
			    "Host: " GG_RETR_HOST "\r\n"
			    "Content-Type: application/x-www-form-urlencoded\r\n"
			    "User-Agent: " USER_AGENT "\r\n"
			    "Content-Length: %d\r\n"
			    "Pragma: no-cache\r\n"
			    "\r\n"
			    "%s\r\n",
			    (config.
			     use_proxy ? "http://" GG_RETR_HOST : ""),
			    strlen(query), query);


	q = g_new0(GGHTTPQuery, 1);
	q->type = GG_REMIND_PASSWORD;
	q->http_query = http_query;

	gg_async_http_query(q, GG_RETR_HOST);

	g_free(query);
}


/* pobiera adres i port serwera GG*/

static
void gg_get_server_addr(gchar * line)
{
	gchar *tmpline = NULL;
	tmpline = line;

	if (line != NULL) {
		gchar **info = g_strsplit(line, " ", 6);
		gchar **addr;
		
		if (!info || !info[3] || !*info[3]) {
		    g_warning("Server sent bad header!");
		    return;
		}
		
		g_print("LINE:\n%s", line);

		addr = g_strsplit(info[3], ":", 2);

		if (!addr || !addr[0] || !*addr[0] || !addr[1] || !*addr[1]) {
		    g_warning("Cannot find server address!");
		    return;
		}
		
		if (status.hostname != NULL)
			g_free(status.hostname);

		status.hostname = g_strdup(addr[0]);

		status.port = strtol(addr[1], NULL, 0);
		
		if (config.use_proxy) {
		    status.port = GG_FAKE_PORT;
		}
		tmpline = strchr(line, '\n');
		tmpline++;
		if (*tmpline && strlen(tmpline) != config.last_sysmsg) {
		    cp_to_iso(tmpline);
		    show_message_dialog(tmpline);
		    config.last_sysmsg = strlen(tmpline);
		    write_config();
		}
		g_strfreev(info);
		g_strfreev(addr);
	}
	g_free(line);
	g_print("gg_get_server_addr(): hostname, port: %s:%d\n",
		status.hostname, status.port);
}

void gg_connect_resolve(GGWatch * r)
{
	gchar *buf = NULL;

	buf = http_query_read(r->fd);

	close(r->fd);
	watches = g_list_remove(watches, r);
	tcp_watch_destroy(r);

	status.online_timeout =
	    gtk_timeout_add(GG_CONNECT_TIMEOUT,
			    (GtkFunction) gg_timeout_func, NULL);

	gg_get_server_addr(buf);
	gg_second_stage_connect();
}

void gg_connect()
{
	gchar *http_query;
	GGHTTPQuery *q;

	gg_remove_watch(GG_RESOLVE_SERVER);
	if (config.auto_reconnect && status.retry == config.auto_reconnect && 
	     config.server && *(config.server)) {
	    gchar **addr = g_strsplit(config.server, ":", 2);
	    if (addr[0] && *addr[0] && addr[1] && *addr[1]) {
		if (status.hostname) g_free(status.hostname);
		status.hostname = strdup(addr[0]);
		status.port = atoi(addr[1]);
	    }
	    g_print("Using stored server address %s:%s (%s:%d)\n", addr[0], addr[1], config.server, status.port);

	    g_strfreev(addr);
	    gg_second_stage_connect();
	    return;
	}
	
	http_query =
	    g_strdup_printf
	    ("GET %s/appsvc/appmsg.asp?fmnumber=%d HTTP/1.0\r\n" 
	     "Host: "GG_APPMSG_HOST "\r\n" "User-Agent: " USER_AGENT "\r\n"
	     "Pragma: no-cache\r\n\r\n",
	     (config.use_proxy ? "http://" GG_APPMSG_HOST : ""),
	     config.uin);

	q = g_new0(GGHTTPQuery, 1);
	q->type = GG_RESOLVE_SERVER;
	q->http_query = http_query;

	g_print("gg_connect(): %s\n", http_query);
	gg_async_http_query(q, GG_APPMSG_HOST);
}

void gg_delete_userlist_res(GGWatch * r)
{
	gchar line[MAX_BUF];
	gint err = 0;

	memset(line, 0, MAX_BUF);

	http_query_read_head(r->fd);

	if (read(r->fd, line, MAX_BUF - 1) > 0) {
		if (strncmp(line, "put_success:", strlen("put_success:")) != 0) {
			err = 1;
		} else {
		    g_print("gg_export_userlist_res(): %s\n", line);
		}
	} else {
		err = 1;
	}

	close(r->fd);
	watches = g_list_remove(watches, r);
	tcp_watch_destroy(r);

	if (err) {
		show_error_dialog("Bd podczas kasowania listy z serwera!");
	} else {
		show_message_dialog("Lista zostaa skasowana.");
	}
}

void gg_delete_userlist()
{
	gchar *http_query;
	gchar *query;
	GGHTTPQuery *q;

	gg_remove_watch(GG_DELETE_USERLIST);

	query =
	    g_strdup_printf("FmNum=%d&Pass=%s&Delete=1", config.uin, config.password);

	http_query =
	    g_strdup_printf
	    ("POST %s/appsvc/fmcontactsput.asp HTTP/1.0\r\n" 
	     "Host: " GG_PUBDIR_HOST "\r\n" 
	     "User-Agent: " USER_AGENT "\r\n"
   	     "Content-Type: application/x-www-form-urlencoded\r\n"
	     "Content-Length: %d\r\n"
	     "Pragma: no-cache\r\n\r\n"
	     "%s\r\n",
	     (config.use_proxy ? "http://" GG_PUBDIR_HOST : ""), strlen(query), query);

	q = g_new0(GGHTTPQuery, 1);
	q->type = GG_DELETE_USERLIST;
	q->http_query = http_query;

	g_print("gg_delete_userlist(): %s\n", http_query);
	gg_async_http_query(q, GG_PUBDIR_HOST);

	g_free(query);
}

void gg_export_userlist_res(GGWatch * r)
{
	gchar line[MAX_BUF];
	gint err = 0;

	memset(line, 0, MAX_BUF);

	http_query_read_head(r->fd);

	if (read(r->fd, line, MAX_BUF - 1) > 0) {
		if (strncmp(line, "put_success:", strlen("put_success:")) != 0) {
			err = 1;
		} else {
		    g_print("gg_export_userlist_res(): %s\n", line);
		}
	} else {
		err = 1;
	}

	close(r->fd);
	watches = g_list_remove(watches, r);
	tcp_watch_destroy(r);

	if (err) {
		show_error_dialog("Bd podczas eksportu listy na serwer (res)!");
	} else {
		show_message_dialog("Lista zostaa wyeksportowana.");
	}
}


void gg_export_userlist()
{
	gchar *http_query;
	gchar *query;
	GGHTTPQuery *q;
	GList *tmplist;
	gchar *export = NULL, *tmp;

	gg_remove_watch(GG_EXPORT_USERLIST);

	tmplist = kontakty;
	
	while (tmplist) {
	    gchar *buf;	    
	    GGContact *k;

	    k = (GGContact *)tmplist->data;
	    
	    buf = g_strdup_printf("%s;%s;%s;%s;%s;%s;%d\r\n", 
				    k->first_name ? k->first_name : "", 
				    k->last_name ? k->last_name : "",
				    k->nick ? k->nick : "", 
				    k->display ? k->display : "",
				    k->mobile ? k->mobile : "", 
				    k->group ? k->group : "", 
				    k->uin);
	    if (!export) {
		export = g_strconcat(buf, NULL);
	    } else {
		export = g_strconcat(export, buf, NULL);
	    }
	    g_free(buf);
	    tmplist = tmplist->next;
	}
	
	if (!export) {
	    show_error_dialog(_("Twoja lista kontaktw jest pusta!"));
	    return;
	}
	iso_to_cp(export);
	tmp = export;
	export = gg_urlencode(tmp);
	g_free(tmp);
	tmp =
	    g_strdup_printf("FmNum=%d&Pass=%s&Contacts=", config.uin, config.password);

	query = g_strconcat(tmp, export, NULL);
	g_free(export);

	http_query =
	    g_strdup_printf
	    ("POST %s/appsvc/fmcontactsput.asp HTTP/1.0\r\n" 
	     "Host: " GG_PUBDIR_HOST "\r\n" 
	     "User-Agent: " USER_AGENT "\r\n"
   	     "Content-Type: application/x-www-form-urlencoded\r\n"
	     "Content-Length: %d\r\n"
	     "Pragma: no-cache\r\n"
	     "\r\n"
	     "%s\r\n",
	     (config.use_proxy ? "http://" GG_PUBDIR_HOST : ""), strlen(query), query);

	q = g_new0(GGHTTPQuery, 1);
	q->type = GG_EXPORT_USERLIST;
	q->http_query = http_query;
	
	g_print("gg_export_uerlist():\n%s\n", http_query);
	
	gg_async_http_query(q, GG_PUBDIR_HOST);
	g_free(query);
}

void gg_import_userlist_res(GGWatch * r)
{
	gchar line[MAX_BUF], *res = NULL;
	gint err = 0;

	memset(line, 0, MAX_BUF);

	http_query_read_head(r->fd);

	while (read(r->fd, line, MAX_BUF - 1) > 0) {
		gchar *tmp;
	        if (res == NULL) {
	    	    res = g_strdup(line);
		} else {
		    tmp = res;
		    res = g_strconcat(res, line, NULL);
		    g_free(tmp);
		}
	}
	
	cp_to_iso(res);
	g_print("REZ:\n %s\n", res);
	if (strncmp(res, "get_results:", 12) == 0) {
		gchar **all, **tmp1;
		
		all = g_strsplit(res + 12, "\r\n", 1000);
		tmp1 = all;
		while (*all) {	    
		    gchar *first_name, *last_name, *nick, *display, *mobile, *group, *uin;
		    gchar **l, **tmp;
		    GGContact *k;
		    
		    l = g_strsplit(*all++, ";", 7);
		    tmp = l;
		    first_name = *l;
		    l++;
		    last_name = *l;
		    l++;
		    nick = *l;
		    l++;
		    display = *l;
		    l++;
		    mobile = *l;
		    l++;
		    group = *l;
		    l++;
		    uin = *l;
		    
		    if ((!uin || !*uin) && (!mobile || *mobile)) 
			continue;
		    
		    if (!strlen(mobile))
			mobile = NULL;
			
		    k = g_new0(GGContact, 1);
		    if (uin != NULL);
			    k->uin = strtol(uin, NULL, 0);
		    
		    k->first_name = g_strdup(first_name);
		    k->last_name = g_strdup(last_name);
		    k->nick = g_strdup(nick);
		    k->display = g_strdup(display);
		    k->group = g_strdup(group);
		    k->mobile = g_strdup(mobile);
		    k->status = GG_STATUS_OFFLINE;
		    
		    if ((k->uin != 0 && gg_get_contact_by_uin(k->uin) != NULL)) {
			    g_warning("Kontakt z uin=%d istnieje. %s Ignoruje.", k->uin, k->mobile);
			    free_contact(k);
	    	    } else if ((!k->uin && k->mobile && *k->mobile && gg_get_contact_by_mobile(k->mobile) != NULL)) {
			    g_warning("Kontakt z mobile=%s istnieje. Ignoruje.", k->mobile);
			    free_contact(k);
		    } else {
			    kontakty = g_list_append(kontakty, k);
			    add_kontakt_to_list(k);
			    if (status.connected) {
				gg_send_user_add(k->uin);
			    }
		    }
		    g_print("K: %s %s %s %s\n", first_name, last_name, nick, uin);
		    g_strfreev(tmp);
    		}
		g_strfreev(tmp1);
	} else {
		err = 1;
		g_free(res);
	}
	close(r->fd);
	watches = g_list_remove(watches, r);
	tcp_watch_destroy(r);

	if (err) {
		show_error_dialog("Bd podczas importu listy na serwer (res)!");
	} else {
		show_message_dialog("Lista zostaa zaimportowana.");
	}
}

void gg_import_userlist()
{
	gchar *http_query;
	gchar *query;
	GGHTTPQuery *q;
	
	gg_remove_watch(GG_IMPORT_USERLIST);
		
	query =
	    g_strdup_printf("FmNum=%d&Pass=%s", config.uin, config.password);
	
	http_query =
	    g_strdup_printf
	    ("POST %s/appsvc/fmcontactsget.asp HTTP/1.0\r\n" 
	     "Host: "GG_PUBDIR_HOST "\r\n" 
	     "User-Agent: " USER_AGENT "\r\n"
   	     "Content-Type: application/x-www-form-urlencoded\r\n"
	     "Content-Length: %d\r\n"
	     "Pragma: no-cache\r\n\r\n"
	     "%s\r\n",
	     (config.use_proxy ? "http://" GG_PUBDIR_HOST : ""), strlen(query), query);

	q = g_new0(GGHTTPQuery, 1);
	q->type = GG_IMPORT_USERLIST;
	q->http_query = http_query;
    
	g_print("gg_import_remote_list(): %s\n", http_query);
	gg_async_http_query(q, GG_PUBDIR_HOST);
	g_free(query);
}

gint read_pubdir_from_file(struct gg_change_info_request *r)
{
    FILE *fp;
    gchar *path;
    gchar *info[10];
    gint pos = 0;
    path = g_strconcat(config.homedir, "/pubdir", NULL);
        
    fp = fopen(path, "r");
    
    g_free(path);
    
    if (!fp) {
	g_warning("Nie mog zapisa katalogu publicznego do pliku!");
	return -1;	
    }
    
    while (!feof(fp)) {
	gchar line[MAX_BUF];
	g_print("read_pubdir_from_file(): %d\n", pos);	
	if (pos < 10 && fgets(line, MAX_BUF - 1, fp)) {
	    if (line[strlen(line)-1] == '\n') 
		line[strlen(line)-1] = 0;
	    info[pos] = g_strdup(line);
	} else 
	    break;	
	pos++;
    }
    fclose(fp);
    
    for (;pos < 10;pos++) info[pos] = NULL;
    
    g_print("read_pubdir_from_file(): done reading...\n");	
    
    if (info[0] && *info[0])
	r->first_name = g_strdup((info[0] ? info[0] : ""));
    if (info[1] && *info[1])
        r->last_name = g_strdup((info[1] ? info[1] : ""));
    if (info[2] && *info[2])
	r->nick = g_strdup((info[2] ? info[2] : ""));
    if (info[5] && *info[5])
        r->born = atoi(info[5]);
    if (info[6] && *info[6])
	r->gender = atoi(info[6]);
    if (info[7] && *info[7])
        r->city = g_strdup((info[7] ? info[7] : ""));
//    if (info[8] && *info[8])
//	r->familycity = g_strdup((info[8] ? info[8] : ""));
//    if (info[9] && *info[9])
//	r->familyname = g_strdup((info[9] ? info[9] : ""));

    for (pos = 0;pos < 10;pos++) g_free(info[pos]);
    
    return 0;

}

gint write_pubdir_to_file(struct gg_change_info_request *r)
{
    FILE *fp;
    gchar *path;
    
    path = g_strconcat(config.homedir, "/pubdir", NULL);
    
    fp = fopen(path, "w");
    
    g_free(path);
    
    if (!fp) {
	g_warning("Nie mog zapisa katalogu publicznego do pliku!");
	return -1;	
    }
    fprintf(fp, "%s\n", (r->first_name) ? r->first_name : "");
    fprintf(fp, "%s\n", (r->last_name) ? r->last_name : "");
    fprintf(fp, "%s\n", (r->nick) ? r->nick : "");
    fprintf(fp, "\n");     
    fprintf(fp, "\n");   
    fprintf(fp, "%d\n", r->born);
    fprintf(fp, "%d\n", r->gender);
    fprintf(fp, "%s\n", (r->city) ? r->city : "");     
//    fprintf(fp, "%s\n", (r->familycity) ? r->familycity : "");     
//    fprintf(fp, "%s\n", (r->familyname) ? r->familyname : "");     

    fclose(fp);
    
    return 0;
}

void gg_change_pubdir_info(struct gg_change_info_request *req)
{
    gchar *http_query;
    gchar *query;
    gchar *_fn, *_ln, *_n, *_c;
    GGHTTPQuery *q;
    
    gg_remove_watch(GG_CHANGE_PUBDIR_INFO);

    _fn = gg_urlencode(req->first_name);
    _ln = gg_urlencode(req->last_name);
    _n = gg_urlencode(req->nick);
    _c = gg_urlencode(req->city);
//    _fa = gg_urlencode(req->familyname);
//    _fc = gg_urlencode(req->familycity);
    
    query = g_strdup_printf("FmNum=%d&Pass=%s&FirstName=%s&LastName=%s&NickName=%s&BirthYear=%d&Gender=%d&City=%s&", 
			    config.uin, config.password, _fn, _ln, _n, req->born, req->gender, _c);
    
    g_free(_fn); g_free(_ln); g_free(_n); g_free(_c);; 
    
    g_print("%s\n", query);
        
    http_query =
	    g_strdup_printf
	    ("POST %s/appsvc/fmpubreg2.asp HTTP/1.0\r\n" 
	     "Host: "GG_PUBDIR_HOST "\r\n" 
	     "User-Agent: " USER_AGENT "\r\n"
   	     "Content-Type: application/x-www-form-urlencoded\r\n"
	     "Content-Length: %d\r\n"
	     "Pragma: no-cache\r\n\r\n"
	     "%s\r\n",
	     (config.use_proxy ? "http://" GG_PUBDIR_HOST : ""), strlen(query), query);

    q = g_new0(GGHTTPQuery, 1);
    q->type = GG_CHANGE_PUBDIR_INFO;
    q->http_query = http_query;

    g_print("gg_change_pubdir_info(): %s\n", http_query);
    gg_async_http_query(q, GG_PUBDIR_HOST);
    g_free(query);
    
}

void gg_change_pubdir_info_res(GGWatch *r) 
{
	gchar line[MAX_BUF];
	gint err = 0;

	memset(line, 0, MAX_BUF);

	http_query_read_head(r->fd);

	if (read(r->fd, line, MAX_BUF - 1) > 0) {
		if (strncmp (line, GG_REG_SUCCESS,
		     strlen(GG_REG_SUCCESS)) != 0) {
			err = 1;
		}
	} else {
		err = 1;
	}

	close(r->fd);
	watches = g_list_remove(watches, r);
	tcp_watch_destroy(r);

	if (err) {
		show_error_dialog("Bd podczas wysyania danych do katalogu publicznego!!!");
	} else {
		show_message_dialog ("Twoje dane w katalogu publicznym zostay zmodyfikowane.");
	}
}

void free_contact(GGContact *k) 
{
    g_free(k->first_name);
    g_free(k->last_name);
    g_free(k->nick);
    g_free(k->comment);
    g_free(k->mobile);
    g_free(k->group);
    g_free(k);
}
