--------------------------------------------------------------------------------
-- |
-- Module      :  Sound.OpenAL.ALC.Device
-- Copyright   :  (c) Sven Panne 2003
-- License     :  BSD-style (see the file libraries/OpenAL/LICENSE)
-- 
-- Maintainer  :  sven_panne@yahoo.com
-- Stability   :  provisional
-- Portability :  portable
--
-- Managing Devices
--
--------------------------------------------------------------------------------

module Sound.OpenAL.ALC.Device (
   Device, openDevice, closeDevice,
   deviceSpecifier, alcVersion,
   defaultDeviceSpecifier, allDeviceSpecifiers
) where

import Control.Monad ( liftM, liftM2 )
import Foreign.Ptr ( Ptr, nullPtr )
import Graphics.Rendering.OpenGL.GL.StateVar (
   GettableStateVar, makeGettableStateVar, get )
import Sound.OpenAL.AL.BasicTypes ( ALubyte, ALint )
import Sound.OpenAL.ALC.Queries(
   StringQuery(..), alcGetString, alcGetString_,
   IntQuery(..), getInteger,
   alcIsExtensionPresent )
import Sound.OpenAL.ALC.BasicTypes ( Device, isDevice, withALString )
import Sound.OpenAL.ALC.Errors ( checkError )

--------------------------------------------------------------------------------

-- | Opens a device, using an alrc expression to specify attributes for the
-- device. Throws an IOError on failure. The ALC error flag is cleared in any
-- case.

openDevice :: String -> IO Device
openDevice deviceSpec =
   checkError isDevice $
      if deviceSpec == ""
         then alcOpenDevice nullPtr   -- use preferred device
         else withALString deviceSpec alcOpenDevice

foreign import CALLCONV unsafe "alcOpenDevice"
   alcOpenDevice :: Ptr ALubyte -> IO Device

-- | Closes the given device.

foreign import CALLCONV unsafe "alcCloseDevice"
   closeDevice :: Device -> IO () 

--------------------------------------------------------------------------------

deviceSpecifier :: Device -> GettableStateVar String
deviceSpecifier device = makeGettableStateVar $
   alcGetString (Just device) DeviceSpecifier

--------------------------------------------------------------------------------

alcVersion :: Device -> GettableStateVar (ALint, ALint) -- ToDo: Without Device?????
alcVersion device = makeGettableStateVar $
   liftM2 (,) (getInteger device MajorVersion) (getInteger device MinorVersion)

--------------------------------------------------------------------------------

defaultDeviceSpecifier :: GettableStateVar String
defaultDeviceSpecifier = makeGettableStateVar $
   alcGetString Nothing DefaultDeviceSpecifier

allDeviceSpecifiers :: GettableStateVar [String]
allDeviceSpecifiers = makeGettableStateVar $ do
   enumExtPresent <- alcIsExtensionPresent Nothing "ALC_ENUMERATION_EXT"
   if enumExtPresent
      then do
         ptr <- alcGetString_ Nothing DeviceSpecifier
         return []  -- Todo!!!!!
      else liftM (\s -> [s]) $ get defaultDeviceSpecifier
