-- #hide
--------------------------------------------------------------------------------
-- |
-- Module      :  Sound.OpenAL.ALC.Queries
-- Copyright   :  (c) Sven Panne 2003-2004
-- License     :  BSD-style (see the file libraries/OpenAL/LICENSE)
-- 
-- Maintainer  :  sven.panne@aedion.de
-- Stability   :  provisional
-- Portability :  portable
--
--------------------------------------------------------------------------------

module Sound.OpenAL.ALC.Queries (
   StringQuery(..), alcGetString, alcGetString_,
   IntQuery(..), marshalIntQuery, getInteger, getIntegerv,
   alcIsExtensionPresent
) where

import Control.Monad ( liftM, when )
import Foreign
import Sound.OpenAL.AL.BasicTypes ( ALubyte, ALint, ALenum, ALsizei )
import Sound.OpenAL.AL.ALboolean ( ALboolean, unmarshalALboolean )
import Sound.OpenAL.ALC.BasicTypes (
   Device, fromDevice, withALString, peekALString )

--------------------------------------------------------------------------------

#include "HsOpenALConfig.h"

--------------------------------------------------------------------------------

data StringQuery =
     DefaultDeviceSpecifier
   | DeviceSpecifier
   | Extensions

marshalStringQuery :: StringQuery -> ALenum
marshalStringQuery x = case x of
   DefaultDeviceSpecifier -> CONST_ALC_DEFAULT_DEVICE_SPECIFIER
   DeviceSpecifier -> CONST_ALC_DEVICE_SPECIFIER
   Extensions -> CONST_ALC_EXTENSIONS

--------------------------------------------------------------------------------

alcGetString :: Maybe Device -> StringQuery -> IO String
alcGetString device query = alcGetString_ device query >>= peekALString

alcGetString_ :: Maybe Device -> StringQuery -> IO (Ptr ALubyte)
alcGetString_ device = alcGetString__ (fromDevice device) . marshalStringQuery

foreign import CALLCONV unsafe "alcGetString"
   alcGetString__ :: Device -> ALenum -> IO (Ptr ALubyte)

--------------------------------------------------------------------------------

data IntQuery =
     MajorVersion
   | MinorVersion
   | AttributesSize
   | AllAttributes

marshalIntQuery :: IntQuery -> ALenum
marshalIntQuery x = case x of
   MajorVersion -> CONST_ALC_MAJOR_VERSION
   MinorVersion -> CONST_ALC_MINOR_VERSION
   AttributesSize -> CONST_ALC_ATTRIBUTES_SIZE
   AllAttributes -> CONST_ALC_ALL_ATTRIBUTES

--------------------------------------------------------------------------------

getInteger :: Device -> IntQuery -> IO ALint
getInteger device query = liftM head $ getIntegerv device query 1

-- We are extremely careful below to avoid segfaults in case that there is no
-- current context, an invalid device, etc.
getIntegerv :: Device -> IntQuery -> Int -> IO [ALint]
getIntegerv device query numALints =
   withArray (replicate numALints 0) $ \buf -> do
      let numBytes = fromIntegral (numALints * sizeOf (0 :: ALint))
      when (numALints > 0) $
         alcGetIntegerv device (marshalIntQuery query) numBytes buf
      peekArray numALints buf

foreign import CALLCONV unsafe "alcGetIntegerv"
   alcGetIntegerv :: Device -> ALenum -> ALsizei -> Ptr ALint -> IO ()

--------------------------------------------------------------------------------

-- | Returns 'True' if the given ALC extension is present on the given device,
-- 'False' otherwise.

alcIsExtensionPresent :: Maybe Device -> String -> IO Bool
alcIsExtensionPresent device extensionName =
   liftM unmarshalALboolean $
      withALString extensionName (alcIsExtensionPresent_ (fromDevice device))

foreign import CALLCONV unsafe "alcIsExtensionPresent"
   alcIsExtensionPresent_ :: Device -> Ptr ALubyte -> IO ALboolean
