--------------------------------------------------------------------------------
-- |
-- Module      :  Sound.OpenAL.AL.Listener
-- Copyright   :  (c) Sven Panne 2003-2005
-- License     :  BSD-style (see the file libraries/OpenAL/LICENSE)
-- 
-- Maintainer  :  sven.panne@aedion.de
-- Stability   :  provisional
-- Portability :  portable
--
--------------------------------------------------------------------------------

module Sound.OpenAL.AL.Listener (
   listenerGain, listenerPosition, listenerVelocity, orientation
) where

import Foreign.Marshal.Alloc ( alloca )
import Foreign.Marshal.Array ( allocaArray, withArray )
import Foreign.Marshal.Utils ( with )
import Foreign.Ptr ( Ptr )
import Graphics.Rendering.OpenGL.GL.CoordTrans ( Vector3(..) )
import Graphics.Rendering.OpenGL.GL.VertexSpec ( Vertex3(..) )
import Graphics.Rendering.OpenGL.GL.StateVar ( StateVar, makeStateVar )
import Sound.OpenAL.AL.BasicTypes ( ALenum, ALfloat )
import Sound.OpenAL.AL.PeekPoke ( peek1, peek3, peek6 )
import Sound.OpenAL.AL.QueryUtils (
   GetPName(GetGain,GetPosition,GetVelocity,GetOrientation), marshalGetPName )

--------------------------------------------------------------------------------

listenerGain :: StateVar ALfloat
listenerGain =
   makeStateVar
      (do alloca $ \buf -> do
          getListenerfv GetGain buf
          peek1 id buf)
      (listenerf GetGain)

--------------------------------------------------------------------------------

listenerPosition :: StateVar (Vertex3 ALfloat)
listenerPosition =
   makeStateVar
      (do allocaArray 3 $ \buf -> do
          getListenerfv GetPosition buf
          peek3 Vertex3 buf)
      (\pos -> with pos $ listenerVertex3 GetPosition)

--------------------------------------------------------------------------------

listenerVelocity :: StateVar (Vector3 ALfloat)
listenerVelocity =
   makeStateVar
      (do allocaArray 3 $ \buf -> do
          getListenerfv GetVelocity buf
          peek3 Vector3 buf)
      (\vel -> with vel $ listenerVector3 GetVelocity)

--------------------------------------------------------------------------------

orientation :: StateVar (Vector3 ALfloat, Vector3 ALfloat)
orientation =
   makeStateVar
      (do allocaArray 6 $ \buf -> do
          getListenerfv GetOrientation buf
          peek6 Vector3 buf)
      (\(at, up) -> withArray [at, up] $ listenerVector3 GetOrientation)

--------------------------------------------------------------------------------

listenerf :: GetPName -> ALfloat -> IO ()
listenerf = alListenerf . marshalGetPName

foreign import CALLCONV unsafe "alListenerf"
   alListenerf :: ALenum -> ALfloat -> IO ()

listenerVertex3 :: GetPName -> Ptr (Vertex3 ALfloat) -> IO ()
listenerVertex3 = alListenerfv . marshalGetPName

listenerVector3 :: GetPName -> Ptr (Vector3 ALfloat) -> IO ()
listenerVector3 = alListenerfv . marshalGetPName

foreign import CALLCONV unsafe "alListenerfv"
   alListenerfv :: ALenum -> Ptr a -> IO ()

getListenerfv :: GetPName -> Ptr ALfloat -> IO ()
getListenerfv = alGetListenerfv . marshalGetPName

foreign import CALLCONV unsafe "alGetListenerfv"
   alGetListenerfv :: ALenum -> Ptr ALfloat -> IO ()

{-
listenerf:
AL_GAIN_LINEAR_LOKI
AL_GAIN

listenerfv:
AL_POSITION
AL_VELOCITY
AL_ORIENTATION (6: at + up)

getlistenerfv:
all above
-}
