--------------------------------------------------------------------------------
-- |
-- Module      :  Sound.OpenAL.ALC.Errors
-- Copyright   :  (c) Sven Panne 2003-2005
-- License     :  BSD-style (see the file libraries/OpenAL/LICENSE)
-- 
-- Maintainer  :  sven.panne@aedion.de
-- Stability   :  provisional
-- Portability :  portable
--
-- Errors
--
--------------------------------------------------------------------------------

module Sound.OpenAL.ALC.Errors (
   ALCerror(..), alcGetError, checkError
) where

import Sound.OpenAL.ALC.BasicTypes ( ALCenum, Device, fromDevice )

--------------------------------------------------------------------------------

#include "HsOpenALConfig.h"

--------------------------------------------------------------------------------

data ALCerror =
     ALCInvalidDevice
   | ALCInvalidContext
   | ALCInvalidEnum
   | ALCInvalidValue
   | ALCOutOfMemory
   deriving ( Eq, Ord, Show )

unmarshalALCerror :: ALCenum -> ALCerror
unmarshalALCerror x
   | x == CONST_ALC_INVALID_DEVICE = ALCInvalidDevice
   | x == CONST_ALC_INVALID_CONTEXT = ALCInvalidContext
   | x == CONST_ALC_INVALID_ENUM = ALCInvalidEnum
   | x == CONST_ALC_INVALID_VALUE = ALCInvalidValue
   | x == CONST_ALC_OUT_OF_MEMORY = ALCOutOfMemory
   | otherwise = error ("unmarshalALCerror: illegal value " ++ show x)

--------------------------------------------------------------------------------

alcGetError :: Maybe Device -> IO (Maybe ALCerror)  -- ToDo: The device is always unused!
alcGetError device = do
   e <- alcGetError_ (fromDevice device)
   return $ if e == CONST_ALC_NO_ERROR then Nothing else Just (unmarshalALCerror e)

foreign import CALLCONV unsafe "alcGetError"
   alcGetError_ :: Device -> IO ALCenum

--------------------------------------------------------------------------------

-- OpenAL does not always set the error flag, especially for out of memory
-- conditions, so we hack around this fact below.   *sigh*
checkError :: (a -> Bool) -> IO a -> IO a
checkError predicate action = do
   alcGetError Nothing   -- clear error flag
   val <- action
   if predicate val
      then return val
      else do err <- alcGetError Nothing
              ioError . userError . show . maybe ALCOutOfMemory id $ err
