with Ada.Text_IO;
with Types; use Types;
with Std_Names;
with Name_Table;
with Std_Package;
with Back_End;
with Flags;
with Translation;
with Iirs; use Iirs;
with Libraries; use Libraries;
with Sem;
with Errorout; use Errorout;
with GNAT.OS_Lib;
with Canon;
with Disp_Vhdl;
with Disp_Tree;
with Post_Sems;
with Bug;

package body Ortho_Front is
   --  The action to be performed by the compiler.
   type Action_Type is
     (
      --  Normal mode: compile a design file.
      Action_Compile,

      --  Elaborate a design unit.
      Action_Elaborate,

      --  Analyze files and elaborate unit.
      Action_Anaelab,

      --  Generate code for std.package.
      Action_Compile_Std_Package
      );
   Action : Action_Type := Action_Compile;

   --  Name of the entity to elaborate.
   Elab_Entity : String_Acc;
   --  Name of the architecture to elaborate.
   Elab_Architecture : String_Acc;
   --  Filename for the list of files to link.
   Elab_Filelist : String_Acc;

   Flag_Expect_Failure : Boolean;

   Anaelab_Files : Iir_List := Null_Iir_List;

   procedure Finish_Compilation
     (Unit : Iir_Design_Unit; Main : Boolean := False);

   procedure Init is
   begin
      -- Initialize.
      Back_End.Finish_Compilation := Finish_Compilation'Access;
      Std_Names.Std_Names_Initialize;
      Libraries.Init_Pathes;
      Elab_Filelist := null;
      Elab_Entity := null;
      Elab_Architecture := null;
      Flag_Expect_Failure := False;
   end Init;

   function Decode_Elab_Option (Args : String_Acc_Array) return Natural
   is
      Arg : String_Acc;
   begin
      Elab_Architecture := null;
      Elab_Filelist := null;
      --  Entity (+ architecture) to elaborate
      if Args'Last = 0 then
         Error_Msg_Option
           ("entity or configuration name required after --elab");
         return 0;
      end if;
      Arg := Args (1);
      if Arg (Arg.all'Last) = ')' then
         --  Name is ENTITY(ARCH).
         --  Split.
         declare
            P : Natural;
            Len : Natural;
            Is_Ext : Boolean;
         begin
            P := Arg.all'Last - 1;
            Len := P - Arg.all'First + 1;
            --  Must be at least 'e(a)'.
            if Len < 4 then
               Error_Msg_Option ("ill-formed name after --elab");
               return 0;
            end if;
            --  Handle extended name.
            if Arg (P) = '\' then
               P := P - 1;
               Is_Ext := True;
            else
               Is_Ext := False;
            end if;
            loop
               if P = Arg.all'First then
                  Error_Msg_Option ("ill-formed name after --elab");
                  return 0;
               end if;
               exit when Arg (P) = '(' and Is_Ext = False;
               if Arg (P) = '\' then
                  if Arg (P - 1) = '\' then
                     P := P - 2;
                  elsif Arg (P - 1) = '(' then
                     P := P - 1;
                     exit;
                  else
                     Error_Msg_Option ("ill-formed name after --elab");
                     return 0;
                  end if;
               else
                  P := P - 1;
               end if;
            end loop;
            Elab_Architecture := new String'(Arg (P + 1 .. Arg'Last - 1));
            Elab_Entity := new String'(Arg (Arg'First .. P - 1));
         end;
      else
         Elab_Entity := new String'(Arg.all);
      end if;
      return 2;
   end Decode_Elab_Option;

   function Decode_Option (Args : String_Acc_Array) return Natural
   is
   begin
      if Args (0).all = "--compile-standard" then
         Action := Action_Compile_Std_Package;
         Flags.Bootstrap := True;
         return 1;
      elsif Args (0).all = "--elab" then
         if Action /= Action_Compile then
            Error_Msg_Option ("several --elab options");
            return 0;
         end if;
         Action := Action_Elaborate;
         return Decode_Elab_Option (Args);
      elsif Args (0).all = "--anaelab" then
         if Action /= Action_Compile then
            Error_Msg_Option ("several --anaelab options");
            return 0;
         end if;
         Action := Action_Anaelab;
         return Decode_Elab_Option (Args);
      elsif Args (0).all = "-c" then
         if Action /= Action_Anaelab then
            Error_Msg_Option
              ("-c option allowed only after --anaelab options");
            return 0;
         end if;
         if 1 > Args'Last then
            Error_Msg_Option ("filename required after -c");
            return 0;
         end if;
         if Anaelab_Files = Null_Iir_List then
            Anaelab_Files := Create_Iir_List;
         end if;
         declare
            File : Iir;
         begin
            File := Create_Iir (Iir_Kind_Simple_Name);
            Set_Identifier (File, Name_Table.Get_Identifier (Args (1).all));
            Append_Element (Anaelab_Files, File);
         end;
         return 2;
      elsif Args (0).all = "-l" then
         if 1 > Args'Last then
            Error_Msg_Option ("filename required after -l");
         end if;
         if Elab_Filelist /= null then
            Error_Msg_Option ("several -l options");
         else
            Elab_Filelist := new String'(Args (1).all);
         end if;
         return 2;
      elsif Args (0).all = "--help" then
         Flags.Disp_Options_Help;
         return 1;
      elsif Args (0).all = "--expect-failure" then
         Flag_Expect_Failure := True;
         return 1;
      elsif Args (0)'Length > 7 and then Args (0)(1 .. 7) = "--ghdl-" then
         if Flags.Parse_Option (Args (0)(7 .. Args (0)'Last)) then
            return 1;
         else
            return 0;
         end if;
      elsif Flags.Parse_Option (Args (0).all) then
         return 1;
      else
         return 0;
      end if;
   end Decode_Option;


   --  Lighter version of libraries.is_obselete, since DESIGN_UNIT must be in
   --  the currently analyzed design file.
   function Is_Obsolete (Design_Unit : Iir_Design_Unit)
     return Boolean
   is
      List : Iir_List;
      El : Iir;
   begin
      if Get_Date (Design_Unit) = Date_Obsolete then
         return True;
      end if;
      List := Get_Dependence_List (Design_Unit);
      if Is_Null_List (List) then
         return False;
      end if;
      for I in Natural loop
         El := Get_Nth_Element (List, I);
         exit when Is_Null (El);
         --  FIXME: there may be entity_aspect_entity...
         if Get_Kind (El) = Iir_Kind_Design_Unit
           and then Get_Date (El) = Date_Obsolete
         then
            return True;
         end if;
      end loop;
      return False;
   end Is_Obsolete;

   Nbr_Parse : Natural := 0;

   function Parse (Filename : String_Acc) return Boolean
   is
      Res : Iir_Design_File;
      New_Design_File : Iir_Design_File;
      Design : Iir_Design_Unit;
      Next_Design : Iir_Design_Unit;

      --  The vhdl filename to compile.
      Vhdl_File : Name_Id;
   begin
      if Nbr_Parse = 0 then
         --  Initialize only once...
         Std_Package.Create_Std_Standard_Package;

         -- Here, time_base can be set.
         Translation.Initialize;
         Canon.Canon_Flag_Add_Labels := True;

         if Flags.List_All and then Flags.List_Annotate then
            Disp_Vhdl.Disp_Vhdl (Std_Package.Std_Standard_Unit);
         end if;

         if Action = Action_Anaelab and then Anaelab_Files /= Null_Iir_List
         then
            Libraries.Load_Work_Library (True);
         else
            Libraries.Load_Work_Library (False);
         end if;
      end if;
      Nbr_Parse := Nbr_Parse + 1;

      case Action is
         when Action_Elaborate =>
            Flags.Flag_Elaborate := True;
            Flags.Flag_Only_Elab_Warnings := True;
            Translation.Chap12.Elaborate
              (Elab_Entity, Elab_Architecture, Elab_Filelist, False);

            if Errorout.Nbr_Errors > 0 then
               --  This may happen (bad entity for example).
               raise Compilation_Error;
            end if;
         when Action_Anaelab =>
            --  Parse files.
            if Anaelab_Files = Null_Iir_List then
               Flags.Flag_Elaborate_With_Outdated := False;
            else
               Flags.Flag_Elaborate_With_Outdated := True;
               declare
                  File : Iir;
               begin
                  for I in Natural loop
                     File := Get_Nth_Element (Anaelab_Files, I);
                     exit when File = Null_Iir;
                     Res := Libraries.Load_File (Get_Identifier (File));
                     if Errorout.Nbr_Errors > 0 then
                        raise Compilation_Error;
                     end if;

                     --  Put units into library.
                     Design := Get_Design_Unit_Chain (Res);
                     while not Is_Null (Design) loop
                        Next_Design := Get_Chain (Design);
                        Set_Chain (Design, Null_Iir);
                        Libraries.Add_Design_Unit_Into_Library (Design);
                        Design := Next_Design;
                     end loop;
                  end loop;
               end;
            end if;

            Flags.Flag_Elaborate := True;
            Flags.Flag_Only_Elab_Warnings := False;
            Translation.Chap12.Elaborate
              (Elab_Entity, Elab_Architecture, Elab_Filelist, True);

            if Errorout.Nbr_Errors > 0 then
               --  This may happen (bad entity for example).
               raise Compilation_Error;
            end if;
         when Action_Compile_Std_Package =>
            if Filename /= null then
               Error_Msg_Option
                 ("--compile-standard is not compatible with a filename");
               return False;
            end if;
            Translation.Translate_Standard (True);

         when Action_Compile =>
            if Filename = null then
               Error_Msg_Option ("no input file");
               return False;
            end if;
            if Nbr_Parse > 1 then
               Error_Msg_Option ("can compile only one file (file """ &
                                 Filename.all & """ ignored)");
               return False;
            end if;
            Vhdl_File := Name_Table.Get_Identifier (Filename.all);

            Translation.Translate_Standard (False);

            Flags.Flag_Elaborate := False;
            Res := Libraries.Load_File (Vhdl_File);
            if Errorout.Nbr_Errors > 0 then
               raise Compilation_Error;
            end if;

            -- Semantize all design units.
            --  FIXME: outdate the design file?
            New_Design_File := Null_Iir;
            Design := Get_Design_Unit_Chain (Res);
            while not Is_Null (Design) loop
               -- Sem, canon, annotate a design unit.
               Back_End.Finish_Compilation (Design, True);

               Next_Design := Get_Chain (Design);
               if Errorout.Nbr_Errors = 0 then
                  Set_Chain (Design, Null_Iir);
                  Libraries.Add_Design_Unit_Into_Library (Design);
                  New_Design_File := Get_Design_File (Design);
               end if;

               Design := Next_Design;
            end loop;

            if Errorout.Nbr_Errors > 0 then
               raise Compilation_Error;
            end if;

            --  Do late analysis checks.
            Design := Get_Design_Unit_Chain (New_Design_File);
            while not Is_Null (Design) loop
               Sem.Sem_Analysis_Checks_List
                 (Design, Flags.Warn_Delayed_Checks);
               Design := Get_Chain (Design);
            end loop;

            --  Compile only now.
            if not Is_Null (New_Design_File) then
               --  Note: the order of design unit is kept.
               Design := Get_Design_Unit_Chain (New_Design_File);
               while not Is_Null (Design) loop
                  if not Is_Obsolete (Design) then
                     Translation.Translate (Design, True);

                     if Errorout.Nbr_Errors > 0 then
                        --  This can happen (foreign attribute).
                        raise Compilation_Error;
                     end if;
                  end if;

                  Design := Get_Chain (Design);
               end loop;
            end if;

            -- Save the working library.
            Libraries.Save_Work_Library;
      end case;
      if Flag_Expect_Failure then
         return False;
      else
         return True;
      end if;
   exception
      --when File_Error =>
      --   Error_Msg_Option ("cannot open file '" & Filename.all & "'");
      --   return False;
      when Compilation_Error
        | Parse_Error
        | Elaboration_Error =>
         if Flag_Expect_Failure then
            --  Very brutal...
            GNAT.OS_Lib.OS_Exit (0);
         end if;
         return False;
      when Option_Error =>
         return False;
      when E: others =>
         Bug.Disp_Bug_Box (E);
         return False;
   end Parse;

   procedure Finish_Compilation
     (Unit : Iir_Design_Unit; Main : Boolean := False)
   is
      use Ada.Text_IO;
      Lib : Iir;
   begin
      --  No need to semantize during elaboration.
      --if Flags.Will_Elaborate then
      --   return;
      --end if;

      Lib := Get_Library_Unit (Unit);

      if (Main or Flags.Dump_All) and then Flags.Dump_Parse then
         Disp_Tree.Disp_Tree (Unit);
      end if;

      --  Semantic analysis.
      if Flags.Verbose then
         Put_Line ("semantize " & Disp_Node (Lib));
      end if;
      Sem.Semantic (Unit);

      if (Main or Flags.Dump_All) and then Flags.Dump_Sem then
         Disp_Tree.Disp_Tree (Unit);
      end if;

      if Errorout.Nbr_Errors > 0 then
         raise Compilation_Error;
      end if;

      if (Main or Flags.List_All) and then Flags.List_Sem then
         Disp_Vhdl.Disp_Vhdl (Unit);
      end if;

      --  Post checks
      ----------------

      Post_Sems.Post_Sem_Checks (Unit);

      if Errorout.Nbr_Errors > 0 then
         raise Compilation_Error;
      end if;

      --  Canonalisation.
      ------------------
      if Flags.Verbose then
         Put_Line ("canonicalize " & Disp_Node (Lib));
      end if;

      Canon.Canonicalize (Unit);

      if (Main or Flags.Dump_All) and then Flags.Dump_Canon then
         Disp_Tree.Disp_Tree (Unit);
      end if;

      if Errorout.Nbr_Errors > 0 then
         raise Compilation_Error;
      end if;

      if (Main or Flags.List_All) and then Flags.List_Canon then
         Disp_Vhdl.Disp_Vhdl (Unit);
      end if;

      if Flags.Flag_Elaborate then
         if Get_Kind (Lib) = Iir_Kind_Architecture_Declaration then
            declare
               Config : Iir_Design_Unit;
            begin
               Config := Canon.Create_Default_Configuration_Declaration (Lib);
               Set_Default_Configuration_Declaration (Lib, Config);
               if (Main or Flags.Dump_All) and then Flags.Dump_Canon then
                  Disp_Tree.Disp_Tree (Config);
               end if;
               if (Main or Flags.List_All) and then Flags.List_Canon then
                  Disp_Vhdl.Disp_Vhdl (Config);
               end if;
            end;
         end if;
         return;
      end if;

      --  Translation
      ---------------
      if not Main then
         --  Main units (those from the analyzed design file) are translated
         --  directly by ortho_front.
         if Flags.Verbose then
            Put_Line ("translate " & Disp_Node (Lib));
         end if;

         Translation.Translate (Unit, Main);

         if Errorout.Nbr_Errors > 0 then
            raise Compilation_Error;
         end if;
      end if;

   end Finish_Compilation;

end Ortho_Front;
