
import datetime
from xml.sax import saxutils
from gettext import gettext as _

import gobject
import gtk
import pango

from gimmie_base import Topic, Item, ItemSource
from gimmie_util import icon_factory

from gimmie_topicwin import SearchToolItem, ZoomMenuToolItem, HideOnDeleteWindow


#
#  All in one browser window
#

class GimmieBrowser(HideOnDeleteWindow):
    def __init__(self, topics):
        HideOnDeleteWindow.__init__(self)
        self.topics = topics
        self.topic = topics[0]
        self.active_source = None
        self.tooltips = gtk.Tooltips()

        self.set_title(_("Gimmie"))
        self.set_position(gtk.WIN_POS_CENTER)
        self.set_default_size(730, 530)

        self.accel_group = gtk.AccelGroup()
        self.add_accel_group(self.accel_group)

        ### Uncomment to use the topic's color as a border
        #self.modify_bg(gtk.STATE_NORMAL, topic.get_hint_color())

        # Vbox containing the toolbar and content
        self.vbox = gtk.VBox(False, 0)
        self.vbox.show()
        self.add(self.vbox)

        self.toolbar = gtk.Toolbar()
        self.toolbar.set_show_arrow(False)
        self.toolbar.show()
        self.vbox.pack_start(self.toolbar, False, False, 0)

        # Contains the visual frame, giving it some space
        self.content = gtk.HBox(False, 0)
        self.content.set_border_width(12)
        self.content.show()
        self.vbox.add(self.content)

        # Visual frame surrounding content
        frame = gtk.Notebook()
        frame.set_show_border(True)
        frame.set_show_tabs(False)
        frame.show()
        self.content.pack_start(frame, True, True, 0)

        scroll = gtk.ScrolledWindow()
        scroll.set_border_width(12)
        scroll.set_policy(gtk.POLICY_ALWAYS, gtk.POLICY_NEVER)
        scroll.set_shadow_type(gtk.SHADOW_NONE)
        scroll.show()
        frame.prepend_page(scroll)

        # Vbox for the sidebar buttons
        self.content_box = gtk.HBox(True, 4)
        self.content_box.show()
        scroll.add_with_viewport(self.content_box)

        # Add the data
        self.add_toolbar_items()

        self.load(topics)

    def make_view(self):
        model = gtk.ListStore(gobject.TYPE_STRING,
                              gtk.gdk.Pixbuf,
                              gobject.TYPE_PYOBJECT,
                              gobject.TYPE_STRING)
        view = gtk.TreeView(model)
        view.set_headers_visible(False)
        view.set_rules_hint(True)
        view.set_enable_search(False)

        renderer = gtk.CellRendererPixbuf()
        column = gtk.TreeViewColumn("Icon", renderer, pixbuf = 1)
        column.set_max_width(gtk.icon_size_lookup(gtk.ICON_SIZE_LARGE_TOOLBAR)[0] * 2)
        view.append_column(column)
        
        renderer = gtk.CellRendererText()
        renderer.set_property("ellipsize", pango.ELLIPSIZE_END)
        column = gtk.TreeViewColumn("Name", renderer, markup = 0)
        view.append_column(column)

        arrow = gtk.CellRendererPixbuf()
        arrow.set_property("stock-size", gtk.ICON_SIZE_MENU)
        column.pack_end(arrow, False)
        column.add_attribute(arrow, "stock-id", 3)

        return view

    def add_view(self, view):
        scroll = gtk.ScrolledWindow()
        scroll.set_policy(gtk.POLICY_NEVER, gtk.POLICY_AUTOMATIC)
        scroll.set_shadow_type(gtk.SHADOW_IN)
        scroll.add(view)
        scroll.show()
        self.content_box.pack_start(scroll, True, True, 0)

    def get_markup(self, i, include_comment = True):
        markup1 = "<span size='large'>%s</span>" % saxutils.escape(i.get_name())
        if include_comment:
            if hasattr(i, "get_comment") and i.get_comment():
                markup2 = "\n<span foreground='darkgrey'>%s</span>" % \
                          saxutils.escape(i.get_comment())
            elif hasattr(i, "get_command"):
                markup2 = "\n<span size='small' font_family='monospace'>%s</span>" % \
                          saxutils.escape(i.get_command())
        else:
            markup2 = ""
        return markup1 + markup2

    def selection_changed(self, tree_selection):
        model, iter = tree_selection.get_selected()
        thing = model.get(iter, 2)[0]

        if isinstance(thing, Topic):
            view = self.source_view
            model = view.get_model()
            model.clear()

            for source in thing.get_sidebar_source_list():
                if not source:
                    continue # need spacer
                iter = model.append()
                model.set(iter, 0, self.get_markup(source, False))
                model.set(iter, 1, source.get_icon(gtk.ICON_SIZE_LARGE_TOOLBAR))
                model.set(iter, 2, source)
                model.set(iter, 3, gtk.STOCK_GO_FORWARD)

            view.get_selection().select_iter(model.get_iter_first())
            view.columns_autosize()
        elif isinstance(thing, ItemSource):
            view = self.item_view
            model = view.get_model()
            model.clear()

            for item in thing.get_items():
                iter = model.append()
                model.set(iter, 0, self.get_markup(item))
                model.set(iter, 1, item.get_icon(gtk.ICON_SIZE_LARGE_TOOLBAR))
                model.set(iter, 2, item)

            #view.get_selection().select_iter(model.get_iter_first())
            view.columns_autosize()
        elif isinstance(thing, Item):
            print "FOO!"
        else:
            print " XXX unknown thing!", thing

        topic, source, item = None, None, None

        model, iter = self.topic_view.get_selection().get_selected()
        topic = model.get(iter, 2)[0]

        model, iter = self.source_view.get_selection().get_selected()
        if iter:
            source = model.get(iter, 2)[0]

        model, iter = self.item_view.get_selection().get_selected()
        if iter:
            item = model.get(iter, 2)[0]

        self.reload_navbar(topic, source, item)

    def reload_navbar(self, topic, source, item):
        for toolitem in self.toolbar.get_children():
            if isinstance(toolitem, self.FooToolButton):
                self.toolbar.remove(toolitem)

        btn = self.FooToolButton()
        btn.set_label(topic.get_name())
        btn.set_icon_widget(icon_factory.load_image("gnome-fs-directory",
                                                    gtk.ICON_SIZE_SMALL_TOOLBAR))
        btn.set_is_important(True)
        btn.show_all()
        if not source:
            btn.set_active(True)
        self.toolbar.insert(btn, 0)

        if source:
            btn = self.FooToolButton()
            btn.set_label(source.get_name())
            btn.set_icon_widget(
                icon_factory.load_image(source.get_icon(gtk.ICON_SIZE_SMALL_TOOLBAR),
                                        gtk.ICON_SIZE_SMALL_TOOLBAR))
            btn.set_is_important(True)
            if not item:
                btn.set_active(True)
            btn.show_all()
            self.toolbar.insert(btn, 1)

        if item:
            btn = self.FooToolButton()
            btn.set_label(item.get_name())
            btn.set_icon_widget(
                icon_factory.load_image(item.get_icon(gtk.ICON_SIZE_SMALL_TOOLBAR)))
            btn.set_is_important(True)
            btn.set_active(True)
            btn.show_all()
            self.toolbar.insert(btn, 2)

    def load(self, topics):
        self.topic_view = self.make_view()
        self.topic_view.show()
        self.topic_view.get_selection().connect("changed", self.selection_changed)
        self.add_view(self.topic_view)

        self.source_view = self.make_view()
        self.source_view.show()
        self.source_view.get_selection().connect("changed", self.selection_changed)
        self.add_view(self.source_view)

        self.item_view = self.make_view()
        self.item_view.show()
        self.item_view.get_selection().connect("changed", self.selection_changed)
        self.add_view(self.item_view)

        model = self.topic_view.get_model()
        for topic in topics:
            iter = model.append()
            model.set(iter, 0, self.get_markup(topic, False))
            model.set(iter, 1, icon_factory.load_icon("gnome-fs-directory", 32))
            model.set(iter, 2, topic)
            model.set(iter, 3, gtk.STOCK_GO_FORWARD)
        #iter = model.append()
        #model.set(iter, 0, "<span size='large'>%s</span>" % "Places")
        #model.set(iter, 1, icon_factory.load_icon("gnome-fs-directory",  32))
        #model.set(iter, 3, gtk.STOCK_GO_FORWARD)

        iter = model.append()
        model.set(iter, 0, "<span size='large'>%s</span>" % "Timeline")
        model.set(iter, 1, icon_factory.load_icon("stock_timer",  32))
        model.set(iter, 3, gtk.STOCK_GO_FORWARD)

        iter = model.append()
        model.set(iter, 0, "<span size='large'>%s</span>" % "Tags")
        model.set(iter, 1, icon_factory.load_icon("stock_bookmark",  32))
        model.set(iter, 3, gtk.STOCK_GO_FORWARD)

        self.topic_view.get_selection().select_iter(model.get_iter_first())

    def set_active_source(self, source):
        self.active_source = source

        if source:
            try:
                # Setting the days filter causes a reload, so avoid infinite loops
                if source.get_days_filter() != self.zoom_menu.get_zoom_level():
                    source.set_days_filter(self.zoom_menu.get_zoom_level())
                self.zoom_menu.set_sensitive(True)
            except:
                self.zoom_menu.set_sensitive(False)

            # FIXME: This loses the focused item
            self.load_items(source.get_items())
        else:
            self.zoom_menu.set_sensitive(False)
            self.store.clear()

    def _reload_source(self, source, sidebar_btn):
        if sidebar_btn.get_active():
            self.set_active_source(source)

    def _toggle_first_button(self):
        # Activates the first sidebar button which is not insensitive
        for btn in self.btn_box:
            if isinstance(btn, gtk.ToggleButton) and btn.get_property("sensitive"):
                btn.set_active(True)
                break

    def _untoggle_other_buttons(self, active_btn):
        if active_btn.get_active(): 
            for btn in self.btn_box:
                if isinstance(btn, gtk.ToggleButton) and \
                       btn.get_property("sensitive") and \
                       btn != active_btn:
                    btn.set_active(False)

    def _untoggle_all_buttons(self):
        for btn in self.btn_box:
            if isinstance(btn, gtk.ToggleButton):
                btn.set_active(False)

    def _ignore_untoggle(self, btn, ev):
        if btn.get_active():
            return True

    def _open_item(self, view, path):
        model = view.get_model()
        model.get_value(model.get_iter(path), 2).open()
	self.iconify()

    def _deactivate_item_popup(self, menu, view, old_selected):
        view.unselect_all()
        print " *** Restoring previous selection"
        for path in old_selected:
            view.select_path(path)

    def _show_item_popup(self, view, ev):
        if ev.button == 3:
            path = view.get_path_at_pos(int(ev.x), int(ev.y))
            if path:
                model = view.get_model()
                item = model.get_value(model.get_iter(path), 2)
                if item:
                    old_selected = view.get_selected_items()
                    
                    view.unselect_all()
                    view.select_path(path)
                    
                    menu = gtk.Menu()
                    menu.attach_to_widget(view, None)
                    menu.connect("deactivate", self._deactivate_item_popup, view, old_selected)

                    print " *** Showing item popup"
                    
                    item.populate_popup(menu)
                    menu.popup(None, None, None, ev.button, ev.time)
                    return True

    def _item_drag_data_get(self, view, drag_context, selection_data, info, timestamp):
        # FIXME: Prefer ACTION_LINK if available
        if info == 100: # text/uri-list
            selected = view.get_selected_items()
            if not selected:
                return

            model = view.get_model()
            uris = []
            for path in selected:
                item = model.get_value(model.get_iter(path), 2)
                if not item:
                    continue
                uris.append(item.get_uri())

            print " *** Dropping URIs:", uris
            selection_data.set_uris(uris)

    # Not used
    def _add_new_sidebar_source_btn(self):
        img = gtk.image_new_from_stock(gtk.STOCK_ADD, gtk.ICON_SIZE_MENU)
        img.show()
        btn = gtk.Button()
        btn.set_property("can-focus", False)
        btn.set_relief(gtk.RELIEF_HALF)
        
        btn.add(img)
        btn.show()

        align = gtk.Alignment(0.0)
        align.add(btn)
        align.show()
        self.btn_box.pack_end(align, False, False, 0)

    def add_sources(self):
        btn_list = []
        
        for source in self.topic.get_sidebar_source_list():
            if not source:
                btn = gtk.Label("") # spacer
                btn.show()
            else:
                img = gtk.Image()
                img.set_from_pixbuf(source.get_icon(gtk.ICON_SIZE_LARGE_TOOLBAR))
                img.show()

                ### Uncomment to show the number of items in label
                #name = "%s (%d)" % (source.get_name(), len(source.get_items()))
                
                label = gtk.Label(source.get_name())
                label.show()
                
                hbox = gtk.HBox(False, 4)
                hbox.pack_start(img, False, False, 0)
                hbox.pack_start(label, False, False, 0)
                hbox.show()
                
                btn = gtk.ToggleButton()
                btn.add(hbox)
                btn.set_focus_on_click(False)
                btn.set_active(False)
                btn.show()
                
                btn.connect("button-press-event", self._ignore_untoggle)
                btn.connect("toggled", self._untoggle_other_buttons)
                btn.connect("toggled", lambda w, s: self._reload_source(s, w), source)
                btn_list.append(btn)

                source.connect_after("reload", self._reload_source, btn)

            self.btn_box.pack_start(btn, False, False, 0)

        ### Uncomment to use OSX-style Add buttons at the end of the sidebar
        #self._add_new_sidebar_source_btn()

        self._toggle_first_button()

    def load_items(self, items):
        self.store.clear()

        cnt = len(items)
        today = datetime.date.today()
        
        for i in items:
            markup1 = "<span size='large'>%s</span>" % saxutils.escape(i.get_name())
            if i.get_comment():
                markup2 = "\n<span foreground='darkgrey'>%s</span>" % \
                          saxutils.escape(i.get_comment())
            elif hasattr(i, "get_command"):
                markup2 = "\n<span size='small' font_family='monospace'>%s</span>" % \
                          saxutils.escape(i.get_command())

            ### FIXME: Need to work out sizing/ellipsizing the text to fit in cramped space
            #layout = self.create_pango_layout("")
            #layout.set_markup(markup2)
            #layout.set_width(100)
            #layout.set_wrap(pango.WRAP_WORD)
            #layout.set_ellipsize(pango.ELLIPSIZE_MIDDLE)

            iter = self.store.append()
            icon_size = 32
            
            if cnt > 30:
                self.store.set(iter, 0, markup1)
                if cnt > 60:
                    icon_size = 16
                else:
                    icon_size = 32
            else:
                self.store.set(iter, 0, markup1 + markup2)

            if datetime.date.fromtimestamp(i.get_timestamp()) == today:
                icon_size = icon_size + 24

            self.store.set(iter, 1, i.get_icon(icon_size), 2, i)

    class FooToolButton(gtk.ToggleToolButton):
        __gsignals__ = {
            'toolbar_reconfigured' : 'override'
            }

        def __init__(self):
            gtk.ToggleToolButton.__init__(self)

        def do_toolbar_reconfigured(self):
            print "XXXXXXXXX on_toolbar_reconfigured"
            gtk.ToggleToolButton.do_toolbar_reconfigured(self)
            self.child.set_relief(gtk.RELIEF_NORMAL)

    def add_toolbar_items(self):
        btn = self.FooToolButton()
        btn.set_label("Applications")
        btn.set_is_important(True)
        btn.show_all()
        self.toolbar.insert(btn, -1)

        btn = self.FooToolButton()
        btn.set_label("Recently Used")
        btn.set_is_important(True)
        btn.set_active(True)
        btn.show_all()
        self.toolbar.insert(btn, -1)

        #for i in self.topic.get_toolbar_items(self.tooltips):
        #    if not i:
        #        i = gtk.SeparatorToolItem()
        #    i.show_all()
        #    self.toolbar.insert(i, -1)

        # Right-align the zoom and search tool items
        sep = gtk.SeparatorToolItem()
        sep.set_draw(False)
        sep.set_expand(True)
        sep.show()
        self.toolbar.insert(sep, -1)

        self.add_zoom_menu()
        self.add_search()

    def _zoom_changed(self, zoom_menu, num_days):
        if self.active_source:
            try:
                print " *** Resetting zoom level: %d days" % num_days
                self.active_source.set_days_filter(num_days)
                zoom_menu.set_sensitive(True)
            except:
                zoom_menu.set_sensitive(False)

            # FIXME: This loses the focused item
            self.load_items(self.active_source.get_items())

    def add_zoom_menu(self):
        self.zoom_menu = ZoomMenuToolItem()
        self.zoom_menu.set_tooltip(self.tooltips, _("Set the zoom level"))
        self.zoom_menu.set_is_important(True)
        self.zoom_menu.connect("zoom-changed", self._zoom_changed)
        self.toolbar.insert(self.zoom_menu, -1)

    def _search(self, w, text):
        self.zoom_menu.set_sensitive(False)
        self._untoggle_all_buttons()

        # clear the current items while we search
        self.set_active_source(None)
        
        self.load_items(self.topic.find_items(text))
        
    def _search_clear(self, w):
        self.zoom_menu.set_sensitive(True)
        self._toggle_first_button()
        
    def add_search(self):
        item = SearchToolItem(self.accel_group)
        item.set_tooltip(self.tooltips, _("Search"))
        item.connect("search", self._search)
        item.connect("clear", self._search_clear)
        self.toolbar.insert(item, -1)
