
import datetime
import os
import pwd
from gettext import gettext as _

import gobject
import gtk
import gnomevfs

try:
    import gnomecups
except ImportError:
    gnomecups = None

import gdmclient

from gimmie_globals import gimmie_is_panel_applet
from gimmie_base import Item, ItemSource, DisabledItemSource, Topic, IOrientationAware
from gimmie_file import FileItem
from gimmie_logout import LogoutDialog
from gimmie_recent import RecentlyUsedOfMimeType
from gimmie_util import *

# FIXME: Move these to another file?
from gimmie_applications import MenuTree, RecentSettingsLaunchers, LauncherItem
from gimmie_trash import TrashItem


class RecentComputerItems(RecentlyUsedOfMimeType):
    '''
    This RecentlyUsedOfMimeType subclass maps recently used Items onto Items
    with matching URIs from a list of other ItemSources.
    '''
    default_mime_type_filter = ("x-directory/normal",
                                "x-gimmie/printer",
                                "application/octet-stream")

    def __init__(self,
                 sources,
                 mimetype_list = default_mime_type_filter):
        RecentlyUsedOfMimeType.__init__(self,
                                        name=_("Recently Used"),
                                        icon="stock_calendar",
                                        mimetype_list=mimetype_list)

        # Sources provide the real items we will display
        self.sources = sources
        for source in sources:
            self._listen_to_source(source)

        # Throw settings launchers in too
        self.settings_launchers = RecentSettingsLaunchers(None, None)
        self._listen_to_source(self.settings_launchers)

    def _listen_to_source(self, source):
        source.connect_after("reload", lambda x: self.emit("reload"))

    def do_reload(self):
        RecentlyUsedOfMimeType.do_reload(self) # Reload our items
        items = []

        # Include recent settings
        items += self.settings_launchers.get_items()

        # Find items matching recent uris
        recent_uris = [x.get_uri() for x in self.get_items()]
        for source in self.sources:
            items += [x for x in source.get_items() if x.get_uri() in recent_uris]

        self.set_items(items) # Override our items


class FavoritesSource(ItemSource):
    '''
    Item source that lists all favorite items.
    '''
    def __init__(self, items = None, uri = None):
        ItemSource.__init__(self,
                            name=_("All Favorites"),
                            icon="gnome-favorites",
                            uri=uri,
                            filter_by_date=False)
        bookmarks.connect("reload", lambda x: self.emit("reload"))

    def do_reload(self):
        items = []
        for uri, itemclass in bookmarks.get_bookmarks_and_class():
            try:
                mod, cls = itemclass.rsplit(".", 1)
                dynmod = __import__(mod, None, None, cls)
                dyncls = getattr(dynmod, cls)
                dynobj = dyncls(uri=uri)
                items.append(dynobj)
            except (ValueError, TypeError, ImportError, AttributeError), err:
                # ValueError - thrown by Item constructor, or strange itemclass
                # TypeError - thrown by Item not accepting uri arg
                # ImportError - error importing mod
                # AttributeError - mod doesn't contain cls
                print "Error creating %s for URI \"%s\": %s" % (itemclass, uri, err)

        self.set_items(items)


class PlacesSource(ItemSource):
    '''
    Use the PlacesManager to list out the user\'s places, and watch for updates.
    '''
    
    def __init__(self):
        ItemSource.__init__(self, _("Places"), icon="gnome-fs-directory", filter_by_date=False)

        # Listen for changes in the ~/.gtk-bookmarks file
        places.connect("reload", lambda x: self.emit("reload"))

    def do_reload(self):
        items = []
        for uri, name, mime, icon in places.get_places():
            items.append(FileItem(uri=uri, icon=icon))

        # Add a trashcan so people can toggle it's pinned setting on the gimmie bar.
        items.append(TrashItem())

        self.set_items(items)


class DriveItem(Item):
    def __init__(self, drive):
        Item.__init__(self,
                      uri=drive.get_activation_uri(),
                      timestamp=0,
                      mimetype="x-gimmie/drive",
                      icon=drive.get_icon())
        self.drive = drive

    def get_name(self):
        return self.drive.get_display_name()

    def get_comment(self):
        if not self.get_is_mounted():
            # FIXME: Check if drive is set to auto-mount, and if not show "Not Mounted"
            return _("No disk inserted")
        else:
            comment = ""

            volumes = self.drive.get_mounted_volumes()
            if len(volumes) == 1:
                vol_name = volumes[0].get_display_name()
                if vol_name != self.get_name():
                    comment += volumes[0].get_display_name()

            # FIXME: If read-only drive, show allocated size instead
            try:
                space = gnomevfs.get_volume_free_space(gnomevfs.URI(self.get_uri()))
            except (TypeError, gnomevfs.NotSupportedError):
                # When the URI or free space is unknown
                space = 0

            if space:
                if comment: comment += "\n"
                comment += _("Free space: %s") % gnomevfs.format_file_size_for_display(space)

            return comment

    def get_icon(self, icon_size):
        volumes = self.drive.get_mounted_volumes()
        if len(volumes) == 1:
            return icon_factory.load_icon(volumes[0].get_icon(), icon_size)
        else:
            icon = icon_factory.load_icon(self.drive.get_icon(), icon_size)
            if not volumes:
                # Not mounted, shade the icon
                return icon_factory.transparentize(icon, 70)
            return icon

    def get_is_mounted(self):
        return self.drive.is_mounted()

    def populate_popup(self, menu):
        Item.populate_popup(self, menu)

        # FIXME: Add mount/unmount toggle?  Need to track nautilus.

        eject = gtk.MenuItem(_("_Eject"), use_underline=True)
        ### FIXME: PyGTK Bindings not bound yet.
        #eject.set_sensitive(self.drive.needs_eject())
        eject.connect("activate", lambda w: self.drive.eject(lambda x, y, z: None))
        eject.show()
        menu.append(eject)


class DevicesSource(ItemSource):
    '''
    Use the gnome-vfs VolumeMonitor to list the currently connected devices.
    '''
    def __init__(self):
        ItemSource.__init__(self,
                            name=_("Devices & Media"),
                            icon="gnome-dev-removable-usb",
                            uri="source://Devices",
                            filter_by_date=False)
        self.add_bluetooth = Item(name=_("Add Bluetooth..."),
                                  comment=_("Access a wireless device"),
                                  icon="stock_bluetooth",
                                  special=True)
        self.add_bluetooth.do_open = lambda: self._add_bluetooth()

        self.cd_burner = Item(name=_("Create CD/DVD..."),
                              comment=_("Burn files onto a CD or DVD disk"),
                              icon="gnome-dev-cdrom",
                              special=True)
        self.cd_burner.do_open = lambda: launcher.launch_command("nautilus --no-desktop burn:///")

        self.vol_monitor = gnomevfs.VolumeMonitor()
        self.vol_monitor.connect("drive_connected", lambda v, d: self.emit("reload"))
        self.vol_monitor.connect("drive_disconnected", lambda v, d: self.emit("reload"))
        self.vol_monitor.connect("volume_mounted", lambda v, d: self.emit("reload"))
        self.vol_monitor.connect("volume_unmounted", lambda v, d: self.emit("reload"))

    def _add_bluetooth(self):
        ImplementMe()

    def do_reload(self):
        items = []
        items.append(self.add_bluetooth)
        items.append(self.cd_burner)
        for drive in self.vol_monitor.get_connected_drives():
            items.append(DriveItem(drive))
        self.set_items(items)


class PrinterItem(Item):
    '''
    Wraps a GnomeCupsPrinter.
    '''
    def __init__(self, printer):
        Item.__init__(self, mimetype="x-gimmie/printer")
        self.printer = printer

        printer.connect("is_default_changed", lambda p: self.emit("reload"))
        printer.connect("attributes_changed", lambda p: self.emit("reload"))

    def do_reload(self):
        print " *** Reloading Printer: %s (state: %s, jobs: %s, location: %s, " \
              "make/model: %s, info: %s)" \
              % (self.get_name(),
                 self.printer.get_state_name(),
                 self.printer.get_job_count(),
                 self.printer.get_location(),
                 self.printer.get_make_and_model(),
                 self.printer.get_info())

    def get_uri(self):
        if self.printer.get_attributes_initialized():
            return self.printer.get_uri()
        return None

    def get_name(self):
        return self.printer.get_name()

    def get_comment(self):
        jobcnt = self.printer.get_job_count()
        state = self.printer.get_state_name()
        loc = self.printer.get_location().strip()

        comment = ""
        if loc and loc.lower() != "location unknown" and loc != self.printer.get_name():
            comment = loc + "\n"

        if jobcnt:
            #return "<span color=\"#337F33\">%s - %d Jobs</span>" % (state, jobcnt)
            return comment + "%s - %d Jobs" % (state, jobcnt)
        else:
            #return "<span color=\"#33337F\">%s</span>" % state
            return comment + state

    def get_timestamp(self):
        uri = self.get_uri()
        if uri:
            try:
                return recent_model.get_item(uri).get_timestamp()
            except KeyError:
                pass
        return 0

    def do_open(self):
        launcher.launch_command("gnome-cups-manager --view \"%s\"" % self.printer.get_name())
        
        recent_item = launcher.make_recent_item(self.get_uri(), "x-gimmie/printer")
        recent_item.add_group("Printer")
        recent_model.add_full(recent_item)

    def _open_properties(self):
        launcher.launch_command("gnome-cups-manager --properties \"%s\"" % self.printer.get_name())

    def get_icon(self, icon_size):
        icon_name, emblems = self.printer.get_icon()
        icon = icon_factory.load_icon(icon_name, icon_size) or \
               icon_factory.load_icon(gtk.STOCK_PRINT, icon_size) # Backup
        if icon_size >= 32:
            for emblem_name in emblems:
                emblem = icon_factory.load_icon(emblem_name, icon_size)
                if emblem:
                    icon = icon.copy() # NOTE: The composite can crash if we don't copy first.
                                       # gnome-cups-manager does this too.
                    emblem.composite(icon,
                                     0, 0,
                                     icon.get_width(), icon.get_height(),
                                     0, 0,
                                     1.0, 1.0,
                                     gtk.gdk.INTERP_NEAREST,
                                     255)
                else:
                    print " !!! Unable to load printer emblem '%s'" % emblem_name
        return icon

    def populate_popup(self, menu):
        Item.populate_popup(self, menu)
        
        props = gtk.ImageMenuItem (gtk.STOCK_PROPERTIES)
        props.connect("activate", lambda w: self._open_properties())
        props.show()
        menu.append(props)


class PrinterSource(ItemSource):
    '''
    Use libgnomecups to list out printers.
    '''
    def __init__(self):
        ItemSource.__init__(self,
                            _("Printers"),
                            icon=gtk.STOCK_PRINT,
                            uri="source://Printers",
                            filter_by_date=False)

        self.add_printer = Item(name=_("Add Printer..."),
                                comment=_("Setup attached or networked printer"),
                                icon="gnome-dev-printer-new",
                                special=True)
        self.add_printer.do_open = lambda: launcher.launch_command("gnome-cups-add")

        gnomecups.new_printer_notify_add(lambda name: self.emit("reload"))
        gnomecups.printer_removed_notify_add(lambda name: self.emit("reload"))

    def do_reload(self):
        items = [self.add_printer]

        for printer_name in gnomecups.get_printers():
            printer = gnomecups.printer_get(printer_name)
            items.append(PrinterItem(printer))

        self.set_items(items)


class RemoteMountItem(Item):
    def __init__(self, volume):
        Item.__init__(self,
                      name=volume.get_display_name(),
                      uri=volume.get_activation_uri(),
                      icon=volume.get_icon())

    def get_timestamp(self):
        try:
            return recent_model.get_item(self.get_uri()).get_timestamp()
        except KeyError:
            return 0

    def get_comment(self):
        seen = self.get_timestamp()
        if seen:
            return self.pretty_print_time_since(seen)
        return None

    def do_open(self):
        launcher.launch_command("nautilus %s" % self.get_uri())

        recent_item = launcher.make_recent_item(self.get_uri(), "application/octet-stream")
        recent_model.add_full(recent_item)


class BonjourSource(ItemSource):
    '''
    Use Avahi to list out computers on the network.  Show a dialog on opening a
    computer listing the available services each with a button to open them in
    an appropriate way: web, ftp, smb, nfs, etc.

    This doesn\'t do anything yet.
    '''
    def __init__(self):
        ItemSource.__init__(self,
                            name=_("Nearby Computers"),
                            icon=gtk.STOCK_NETWORK,
                            uri="source://Network",
                            filter_by_date=False)

        self.connect_to = Item(name=_("Connect to..."),
                               comment=_("Access a computer on the network"),
                               icon=gtk.STOCK_CONNECT,
                               special=True)
        self.connect_to.do_open = lambda: self._connect_to()

        self.smb = Item(name=_("Windows Network"),
                        comment=_("Browse nearby Windows computers"),
                        icon=gtk.STOCK_NETWORK,
                        special=True)
        self.smb.do_open = lambda: launcher.launch_command("nautilus smb://")

        self.vol_monitor = gnomevfs.VolumeMonitor()
        self.vol_monitor.connect("volume_mounted", lambda v, d: self.emit("reload"))
        self.vol_monitor.connect("volume_unmounted", lambda v, d: self.emit("reload"))

    def do_reload(self):
        items = [self.connect_to, self.smb]

        for vol in self.vol_monitor.get_mounted_volumes():
            if vol.get_volume_type() == gnomevfs.VOLUME_TYPE_CONNECTED_SERVER:
                items.append(RemoteMountItem(vol))

        # FIXME: List avahi-discovered hosts

        self.set_items(items)

    def _connect_to(self):
        launcher.launch_command("nautilus-connect-server")


class ComputerTopic(Topic):
    '''
    Lists a heterogeneous set of browsable system items, such as devices/drives,
    printers, and nearby computers (using Avahi).  Also lists control panel
    launchers in a flat list.
    '''
    def __init__(self):
        import platform
        system_alias_name = platform.system_alias(platform.system(),
                                                  platform.release(),
                                                  platform.version())[0] or _("Computer")
        Topic.__init__(self, system_alias_name, uri="topic://Computer")

        # Get the user's real name from /etc/passwd
        pwent = pwd.getpwuid(os.getuid())
        self.username = pwent.pw_gecos.split(",")[0] or pwent.pw_name

        from gimmie_running import ComputerRunningSource
        self.set_running_source_factory(lambda: ComputerRunningSource())

        self.update_time_timeout_id = None
        self.current_time = 0

        self.settings_menu_tree = None
        try:
            # Gnome 2.17.91+ changed the file name arbitrarily.  Yay.
            self.settings_menu_tree = MenuTree("gnomecc.menu")
        except ValueError:
            try:
                # This is the file panel menu-based gnomecc uses.  It generally
                # includes preferences.menu and system-settings.menu.
                self.settings_menu_tree = MenuTree("settings.menu")
            except ValueError:
                try:
                    # Some distros rename settings.menu.
                    self.settings_menu_tree = MenuTree("gnome-settings.menu")
                except ValueError:
                    pass

        self.face_monitor = FileMonitor(os.path.expanduser("~/.face"))
        self.face_monitor.open()

    def do_reload(self):
        places = PlacesSource()
        devices = DevicesSource()
        network = BonjourSource()

        if gnomecups:
            printers = PrinterSource()
        else:
            printers = DisabledItemSource(_("Printers"), gtk.STOCK_PRINT, uri="source://Printers")

        if gimmie_is_panel_applet():
            ### Show the Favorites source if in applet mode.  This is because
            ### applet mode doesn't have a dock for favorites.
            source_list = [FavoritesSource(),
                           "---"] # separator
        else:
            source_list = []

        source_list += [RecentComputerItems((devices, places, printers, network)),
                        None,
                        places,
                        devices,
                        network,
                        printers,
                        None]

        if self.settings_menu_tree:
            settings_src = self.settings_menu_tree.get_toplevel_flat_source()
            settings_src.get_name = lambda: _("Settings")
            settings_src.get_uri = lambda: "source://Settings"
            settings_src.always_show_descriptions = lambda: True
            source_list.append(settings_src)

            ### Uncomment to list settings inside their toplevel folders in the sidebar
            #source_list += self.settings_menu_tree.get_toplevel_source_list()
            #source_list.append(None)

        if os.environ.has_key("GIMMIE_TIMELINE"):
            source_list += [None,
                            DisabledItemSource(_("Timeline"), "stock_timer",
                                               uri="source://Timeline")]

        self.set_sidebar_source_list(source_list)

    def accept_drops(self):
        return False
    
    def get_hint_color(self):
        return None

    def get_button_content(self, edge_gravity):
        if edge_gravity in (gtk.gdk.GRAVITY_EAST, gtk.gdk.GRAVITY_WEST):
            box = gtk.VBox(False, 0)
        else:
            box = gtk.HBox(False, 0)

        time_btn = Topic.get_button_content(self, edge_gravity)
        time_btn.show()
        box.pack_start(time_btn, True, True, 0)

        ### Uncomment to show Find button in corner
        #sep = gtk.VSeparator()
        #find = icon_factory.load_image(gtk.STOCK_FIND, 12)
        #find.show_all()
        #box.pack_start(sep, False, False, 0)
        #box.pack_start(find, False, False, 6)

        gconf_bridge.connect("changed::clockapplet",
                             lambda gb: self._clockapplet_changed(time_btn))
        self._clockapplet_changed(time_btn)

        return box

    def _clockapplet_changed(self, time_btn):
        if self.update_time_timeout_id:
            gobject.source_remove(self.update_time_timeout_id)
            self.update_time_timeout_id = None

        if gconf_bridge.get('clockapplet'):
            # Using the clock applet, just show the topic name
            time_btn.set_text(self.get_name())
        else:
            # Not using the clock applet, show the time in the topic button.
            # FIXME: Is updating ~every second acceptable?
            self.update_time_timeout_id = gobject.timeout_add(1000, self._update_time, time_btn)
            self._update_time(time_btn)

    def _update_time(self, time_btn):
        now_str = datetime.datetime.now().strftime(_("Computer, %l:%M %p"))
        if time_btn.get_text() != now_str:
            time_btn.set_text(now_str)
        return True

    def _dot_face_changed(self, monitor, uri, img):
        face = os.path.expanduser("~/.face")
        if os.path.exists(face):
            img.set_from_pixbuf(icon_factory.load_icon(face, gtk.ICON_SIZE_LARGE_TOOLBAR))
        else:
            img.set_from_pixbuf(icon_factory.load_icon("stock_person",
                                                       gtk.ICON_SIZE_LARGE_TOOLBAR))

    def get_toolbar_items(self, tooltips):
        tools = []

        img = gtk.Image()
        self.face_monitor.connect("created", self._dot_face_changed, img)
        self.face_monitor.connect("deleted", self._dot_face_changed, img)
        self.face_monitor.connect("changed", self._dot_face_changed, img)
        self._dot_face_changed(None, None, img)
        
        btn = ToolMenuButton(img, self.username)
        btn.set_tooltip(tooltips, _("Switch to another user"))
        btn.set_is_important(True)
        menu = gtk.Menu()
        self._add_user_items(menu)
        btn.set_menu(menu)
        tools.append(btn)

        '''
        img = icon_factory.load_image(gtk.STOCK_NETWORK, gtk.ICON_SIZE_LARGE_TOOLBAR)
        btn = ToolMenuButton(img, "")
        btn.set_tooltip(tooltips, _("Change the network location"))
        btn.set_is_important(True)
        menu = gtk.Menu()
        mi = gtk.MenuItem(_("Not Implemented"))
        mi.set_sensitive(False)
        mi.show()
        menu.append(mi)
        btn.set_menu(menu)
        tools.append(btn)
        '''

        if not gimmie_is_panel_applet():
            img = icon_factory.load_image("gnome-audio", gtk.ICON_SIZE_LARGE_TOOLBAR)
            btn = ToolMenuButton(img, "")
            btn.set_tooltip(tooltips, _("Volume: 75%"))
            btn.set_is_important(True)
            menu = gtk.Menu()
            mi = gtk.MenuItem(_("Not Implemented"))
            mi.set_sensitive(False)
            mi.show()
            menu.append(mi)
            btn.set_menu(menu)
            tools.append(btn)

        tools.append(None)

        img = icon_factory.load_image("gnome-shutdown", gtk.ICON_SIZE_LARGE_TOOLBAR)
        btn = gtk.ToolButton(img, _("Shutdown..."))
        btn.set_tooltip(tooltips, _("Shutdown or suspend this computer"))
        btn.set_is_important(True)
        btn.connect("clicked", self.logout, LogoutDialog.SHUTDOWN)
        tools.append(btn)

        tools.append(None)

        img = icon_factory.load_image(gtk.STOCK_HELP, gtk.ICON_SIZE_LARGE_TOOLBAR)
        btn = ToolMenuButton(img, _("Help"))
        btn.set_is_important(True)
        menu = gtk.Menu()
        about = gtk.ImageMenuItem(_("About Gnome"))
        about.set_image(icon_factory.load_image(gtk.STOCK_ABOUT, gtk.ICON_SIZE_LARGE_TOOLBAR))
        about.connect("activate", lambda w: launcher.launch_command("gnome-about"))
        about.show()
        menu.append(about)
        btn.set_menu(menu)
        tools.append(btn)

        return tools

    def _add_user_items(self, menu):
        btn = gtk.ImageMenuItem(_("About Me"))
        btn.set_image(icon_factory.load_image("user-info", gtk.ICON_SIZE_LARGE_TOOLBAR))
        btn.connect("activate", lambda b: launcher.launch_command("gnome-about-me"))
        btn.show()
        menu.append(btn)

        btn = gtk.ImageMenuItem(_("Switch User"))
        btn.set_image(icon_factory.load_image("gnome-lockscreen", gtk.ICON_SIZE_LARGE_TOOLBAR))
        btn.connect("activate", self.switch_user)
        btn.show()
        menu.append(btn)

        btn = gtk.ImageMenuItem(_("Log Out..."))
        btn.set_image(icon_factory.load_image("gnome-logout", gtk.ICON_SIZE_LARGE_TOOLBAR))
        btn.connect("activate", self.logout, LogoutDialog.LOGOUT)
        btn.show()
        menu.append(btn)

    def _make_settings_menu(self):
        column = 0
        row_max = 14

        menu = gtk.Menu()
        for source in self.settings_menu_tree.get_toplevel_source_list():
            menu_items = []

            for item in source.get_items():
                img = gtk.Image()
                img.set_from_pixbuf(item.get_icon(gtk.ICON_SIZE_LARGE_TOOLBAR))
                menu_item = gtk.ImageMenuItem(item.get_name())
                menu_item.set_image(img)
                menu_item.connect("activate", lambda mi, i: i.open(), item)
                menu_item.show()
                menu_items.append(menu_item)

            for idx in range(len(menu_items)):
                mi_column = column + int(idx / row_max)
                mi_row = idx % row_max

                # Offset 2 rows for header and separator menu item
                menu.attach(menu_items[idx],
                            mi_column, mi_column + 1,
                            mi_row + 2, mi_row + 3)

            col_cnt = idx / row_max
            if idx % row_max > 0:
                col_cnt = col_cnt + 1

            img = gtk.Image()
            img.set_from_pixbuf(source.get_icon(gtk.ICON_SIZE_LARGE_TOOLBAR))
            heading_item = gtk.ImageMenuItem(source.get_name())
            heading_item.set_image(img)
            heading_item.set_sensitive(False)
            heading_item.show()
            menu.attach(heading_item, column, column + col_cnt, 0, 1);

            sep = gtk.SeparatorMenuItem()
            sep.show()
            menu.attach(sep, column, column + col_cnt, 1, 2);

            column = column + col_cnt
        return menu

    def get_context_menu_items(self):
        items = []

        if self.settings_menu_tree:
            img = icon_factory.load_image(gtk.STOCK_PREFERENCES, gtk.ICON_SIZE_LARGE_TOOLBAR)
            btn = gtk.ImageMenuItem(_("Settings"))
            btn.set_image(img)
            btn.set_submenu(self._make_settings_menu())
            btn.show()
            items.append(btn)

        img = icon_factory.load_image("stock_person", gtk.ICON_SIZE_LARGE_TOOLBAR)
        btn = gtk.ImageMenuItem(self.username)
        btn.set_image(img)
        menu = gtk.Menu()
        self._add_user_items(menu)
        btn.set_submenu(menu)
        btn.show()
        items.append(btn)

        img = icon_factory.load_image(gtk.STOCK_NETWORK, gtk.ICON_SIZE_LARGE_TOOLBAR)
        btn = gtk.ImageMenuItem(_("Network"))
        btn.set_image(img)
        menu = gtk.Menu()
        btn.set_submenu(menu)
        btn.show()
        items.append(btn)

        img = icon_factory.load_image("gnome-shutdown", gtk.ICON_SIZE_LARGE_TOOLBAR)
        btn = gtk.ImageMenuItem(_("Shutdown..."))
        btn.set_image(img)
        btn.show()
        btn.connect("activate", self.logout, LogoutDialog.SHUTDOWN)
        items.append(btn)

        return items

    def logout(self, button, method):
        dialog = LogoutDialog(method)
        dialog.show()

    def switch_user(self, button):
        gdmclient.new_login()
