/*
 * Copyright (C) 2008 Gustavo Noronha Silva
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.  You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */

#include <gksu-environment.h>
#include <gee/hashmap.h>
#include <gee/map.h>
#include <gee/set.h>


#define GKSU_TYPE_VARIABLE (gksu_variable_get_type ())
#define GKSU_VARIABLE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GKSU_TYPE_VARIABLE, GksuVariable))
#define GKSU_VARIABLE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), GKSU_TYPE_VARIABLE, GksuVariableClass))
#define GKSU_IS_VARIABLE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GKSU_TYPE_VARIABLE))
#define GKSU_IS_VARIABLE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GKSU_TYPE_VARIABLE))
#define GKSU_VARIABLE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GKSU_TYPE_VARIABLE, GksuVariableClass))

typedef struct _GksuVariable GksuVariable;
typedef struct _GksuVariableClass GksuVariableClass;
typedef struct _GksuVariablePrivate GksuVariablePrivate;

struct _GksuVariable {
	GObject parent_instance;
	GksuVariablePrivate * priv;
	char* name;
	char* regex;
};

struct _GksuVariableClass {
	GObjectClass parent_class;
};



enum  {
	GKSU_VARIABLE_DUMMY_PROPERTY
};
static GksuVariable* gksu_variable_new (void);
static gpointer gksu_variable_parent_class = NULL;
static void gksu_variable_dispose (GObject * obj);
static GType gksu_variable_get_type (void);
struct _GksuEnvironmentPrivate {
	GeeHashMap* variables;
};

#define GKSU_ENVIRONMENT_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), GKSU_TYPE_ENVIRONMENT, GksuEnvironmentPrivate))
enum  {
	GKSU_ENVIRONMENT_DUMMY_PROPERTY
};
static void gksu_environment_read_variables_from_path (GksuEnvironment* self, const char* path);
static void gksu_environment_read_variables_from_file (GksuEnvironment* self, const char* path);
static GObject * gksu_environment_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties);
static gpointer gksu_environment_parent_class = NULL;
static void gksu_environment_dispose (GObject * obj);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);
static int _vala_strcmp0 (const char * str1, const char * str2);



static GksuVariable* gksu_variable_new (void) {
	GksuVariable * self;
	self = g_object_newv (GKSU_TYPE_VARIABLE, 0, NULL);
	return self;
}


static void gksu_variable_class_init (GksuVariableClass * klass) {
	gksu_variable_parent_class = g_type_class_peek_parent (klass);
	G_OBJECT_CLASS (klass)->dispose = gksu_variable_dispose;
}


static void gksu_variable_instance_init (GksuVariable * self) {
}


static void gksu_variable_dispose (GObject * obj) {
	GksuVariable * self;
	self = GKSU_VARIABLE (obj);
	self->name = (g_free (self->name), NULL);
	self->regex = (g_free (self->regex), NULL);
	G_OBJECT_CLASS (gksu_variable_parent_class)->dispose (obj);
}


static GType gksu_variable_get_type (void) {
	static GType gksu_variable_type_id = 0;
	if (G_UNLIKELY (gksu_variable_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (GksuVariableClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) gksu_variable_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (GksuVariable), 0, (GInstanceInitFunc) gksu_variable_instance_init };
		gksu_variable_type_id = g_type_register_static (G_TYPE_OBJECT, "GksuVariable", &g_define_type_info, 0);
	}
	return gksu_variable_type_id;
}


GHashTable* gksu_environment_get_variables (GksuEnvironment* self) {
	GeeSet* keysset;
	GHashTable* envpairs;
	GHashTable* _tmp3;
	g_return_val_if_fail (GKSU_IS_ENVIRONMENT (self), NULL);
	keysset = gee_map_get_keys (GEE_MAP (self->priv->variables));
	envpairs = g_hash_table_new (g_str_hash, g_str_equal);
	{
		GeeSet* variable_collection;
		GeeIterator* variable_it;
		variable_collection = keysset;
		variable_it = gee_iterable_iterator (GEE_ITERABLE (variable_collection));
		while (gee_iterator_next (variable_it)) {
			char* variable;
			variable = ((char*) (gee_iterator_get (variable_it)));
			{
				const char* _tmp0;
				char* value;
				const char* _tmp2;
				const char* _tmp1;
				_tmp0 = NULL;
				value = (_tmp0 = g_getenv (variable), (_tmp0 == NULL ? NULL : g_strdup (_tmp0)));
				_tmp2 = NULL;
				_tmp1 = NULL;
				g_hash_table_insert (envpairs, (_tmp1 = variable, (_tmp1 == NULL ? NULL : g_strdup (_tmp1))), (_tmp2 = value, (_tmp2 == NULL ? NULL : g_strdup (_tmp2))));
				variable = (g_free (variable), NULL);
				value = (g_free (value), NULL);
			}
		}
		(variable_it == NULL ? NULL : (variable_it = (g_object_unref (variable_it), NULL)));
	}
	_tmp3 = NULL;
	return (_tmp3 = envpairs, (keysset == NULL ? NULL : (keysset = (g_object_unref (keysset), NULL))), _tmp3);
}


gboolean gksu_environment_validate_hash_table (GksuEnvironment* self, GHashTable* hash_table) {
	GList* varnames;
	gboolean _tmp2;
	g_return_val_if_fail (GKSU_IS_ENVIRONMENT (self), FALSE);
	g_return_val_if_fail (hash_table != NULL, FALSE);
	varnames = g_hash_table_get_keys (hash_table);
	{
		GList* name_collection;
		GList* name_it;
		name_collection = varnames;
		for (name_it = name_collection; name_it != NULL; name_it = name_it->next) {
			const char* _tmp1;
			char* name;
			_tmp1 = NULL;
			name = (_tmp1 = ((char*) (name_it->data)), (_tmp1 == NULL ? NULL : g_strdup (_tmp1)));
			{
				const char* value;
				value = ((const char*) (g_hash_table_lookup (hash_table, name)));
				if (!gksu_environment_is_variable_valid (self, name, value)) {
					gboolean _tmp0;
					return (_tmp0 = FALSE, (name = (g_free (name), NULL)), (varnames == NULL ? NULL : (varnames = (g_list_free (varnames), NULL))), _tmp0);
				}
				name = (g_free (name), NULL);
			}
		}
	}
	return (_tmp2 = TRUE, (varnames == NULL ? NULL : (varnames = (g_list_free (varnames), NULL))), _tmp2);
}


/* this verifies that the variable should be passed through,
 * and that it contains a valid value
 */
gboolean gksu_environment_is_variable_valid (GksuEnvironment* self, const char* name, const char* value) {
	GError * inner_error;
	GksuVariable* _tmp0;
	gboolean _tmp1;
	GksuVariable* variable;
	gboolean _tmp5;
	g_return_val_if_fail (GKSU_IS_ENVIRONMENT (self), FALSE);
	g_return_val_if_fail (name != NULL, FALSE);
	g_return_val_if_fail (value != NULL, FALSE);
	inner_error = NULL;
	/* first we verify that the variable is specified */
	_tmp0 = NULL;
	if ((_tmp1 = (_tmp0 = ((GksuVariable*) (gee_map_get (GEE_MAP (self->priv->variables), name)))) == NULL, (_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL))), _tmp1)) {
		return FALSE;
	}
	/* then we verify that the variable regular expression matches */
	variable = ((GksuVariable*) (gee_map_get (GEE_MAP (self->priv->variables), name)));
	if ((variable->regex != NULL) && (_vala_strcmp0 (variable->regex, "") != 0)) {
		{
			GRegex* regex;
			gboolean _tmp3;
			regex = g_regex_new (variable->regex, 0, 0, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == G_REGEX_ERROR) {
					goto __catch0_g_regex_error;
				}
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
			}
			return (_tmp3 = g_regex_match (regex, value, 0, NULL), (regex == NULL ? NULL : (regex = (g_regex_unref (regex), NULL))), (variable == NULL ? NULL : (variable = (g_object_unref (variable), NULL))), _tmp3);
		}
		goto __finally0;
		__catch0_g_regex_error:
		{
			GError * error;
			error = inner_error;
			inner_error = NULL;
			{
				gboolean _tmp4;
				g_warning ("gksu-environment.vala:78: bad regular expression for variable %s", name);
				return (_tmp4 = FALSE, (error == NULL ? NULL : (error = (g_error_free (error), NULL))), (variable == NULL ? NULL : (variable = (g_object_unref (variable), NULL))), _tmp4);
			}
		}
		__finally0:
		;
	}
	/* the variable looks OK */
	return (_tmp5 = TRUE, (variable == NULL ? NULL : (variable = (g_object_unref (variable), NULL))), _tmp5);
}


static void gksu_environment_read_variables_from_path (GksuEnvironment* self, const char* path) {
	GError * inner_error;
	GDir* directory;
	const char* entry;
	g_return_if_fail (GKSU_IS_ENVIRONMENT (self));
	g_return_if_fail (path != NULL);
	inner_error = NULL;
	directory = NULL;
	{
		GDir* _tmp0;
		_tmp0 = NULL;
		directory = (_tmp0 = g_dir_open (path, ((guint) (0)), &inner_error), (directory == NULL ? NULL : (directory = (g_dir_close (directory), NULL))), _tmp0);
		if (inner_error != NULL) {
			if (inner_error->domain == G_FILE_ERROR) {
				goto __catch1_g_file_error;
			}
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
		}
	}
	goto __finally1;
	__catch1_g_file_error:
	{
		GError * error;
		error = inner_error;
		inner_error = NULL;
		{
			(error == NULL ? NULL : (error = (g_error_free (error), NULL)));
			(directory == NULL ? NULL : (directory = (g_dir_close (directory), NULL)));
			return;
		}
	}
	__finally1:
	;
	entry = NULL;
	while ((entry = g_dir_read_name (directory)) != NULL) {
		if (g_str_has_suffix (entry, ".variables")) {
			char* full_path;
			full_path = g_strconcat (path, entry, NULL);
			gksu_environment_read_variables_from_file (self, full_path);
			full_path = (g_free (full_path), NULL);
		}
	}
	(directory == NULL ? NULL : (directory = (g_dir_close (directory), NULL)));
}


static void gksu_environment_read_variables_from_file (GksuEnvironment* self, const char* path) {
	GError * inner_error;
	GKeyFile* file;
	gint variable_names_length1;
	char** variable_names;
	char** _tmp1;
	gint _tmp0;
	g_return_if_fail (GKSU_IS_ENVIRONMENT (self));
	g_return_if_fail (path != NULL);
	inner_error = NULL;
	file = g_key_file_new ();
	variable_names = (variable_names_length1 = 0, NULL);
	{
		g_key_file_load_from_file (file, path, 0, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == G_FILE_ERROR) {
				goto __catch2_g_file_error;
			}
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
		}
	}
	goto __finally2;
	__catch2_g_file_error:
	{
		GError * error;
		error = inner_error;
		inner_error = NULL;
		{
			g_warning ("gksu-environment.vala:112: %s", error->message);
			(error == NULL ? NULL : (error = (g_error_free (error), NULL)));
			(file == NULL ? NULL : (file = (g_key_file_free (file), NULL)));
			variable_names = (_vala_array_free (variable_names, variable_names_length1, ((GDestroyNotify) (g_free))), NULL);
			return;
		}
	}
	__finally2:
	;
	_tmp1 = NULL;
	variable_names = (_tmp1 = g_key_file_get_groups (file, &_tmp0), (variable_names = (_vala_array_free (variable_names, variable_names_length1, ((GDestroyNotify) (g_free))), NULL)), variable_names_length1 = _tmp0, _tmp1);
	{
		char** name_collection;
		int name_collection_length1;
		int name_it;
		name_collection = variable_names;
		name_collection_length1 = variable_names_length1;
		for (name_it = 0; (variable_names_length1 != -1 && name_it < variable_names_length1) || (variable_names_length1 == -1 && name_collection[name_it] != NULL); name_it = name_it + 1) {
			const char* _tmp6;
			char* name;
			_tmp6 = NULL;
			name = (_tmp6 = name_collection[name_it], (_tmp6 == NULL ? NULL : g_strdup (_tmp6)));
			{
				char* policy;
				GksuVariable* variable;
				char* _tmp4;
				const char* _tmp3;
				policy = NULL;
				{
					char* _tmp2;
					_tmp2 = NULL;
					policy = (_tmp2 = g_key_file_get_value (file, name, "Policy", &inner_error), (policy = (g_free (policy), NULL)), _tmp2);
					if (inner_error != NULL) {
						if (inner_error->domain == G_KEY_FILE_ERROR) {
							goto __catch3_g_key_file_error;
						}
						g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
						g_clear_error (&inner_error);
					}
				}
				goto __finally3;
				__catch3_g_key_file_error:
				{
					GError * error;
					error = inner_error;
					inner_error = NULL;
					{
						(error == NULL ? NULL : (error = (g_error_free (error), NULL)));
					}
				}
				__finally3:
				;
				if (_vala_strcmp0 (policy, "send") != 0) {
					name = (g_free (name), NULL);
					policy = (g_free (policy), NULL);
					continue;
				}
				variable = gksu_variable_new ();
				_tmp4 = NULL;
				_tmp3 = NULL;
				variable->name = (_tmp4 = (_tmp3 = name, (_tmp3 == NULL ? NULL : g_strdup (_tmp3))), (variable->name = (g_free (variable->name), NULL)), _tmp4);
				{
					char* _tmp5;
					_tmp5 = NULL;
					variable->regex = (_tmp5 = g_key_file_get_value (file, name, "Regex", &inner_error), (variable->regex = (g_free (variable->regex), NULL)), _tmp5);
					if (inner_error != NULL) {
						if (inner_error->domain == G_KEY_FILE_ERROR) {
							goto __catch4_g_key_file_error;
						}
						g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
						g_clear_error (&inner_error);
					}
				}
				goto __finally4;
				__catch4_g_key_file_error:
				{
					GError * error;
					error = inner_error;
					inner_error = NULL;
					{
						(error == NULL ? NULL : (error = (g_error_free (error), NULL)));
					}
				}
				__finally4:
				;
				gee_map_set (GEE_MAP (self->priv->variables), name, variable);
				name = (g_free (name), NULL);
				policy = (g_free (policy), NULL);
				(variable == NULL ? NULL : (variable = (g_object_unref (variable), NULL)));
			}
		}
	}
	(file == NULL ? NULL : (file = (g_key_file_free (file), NULL)));
	variable_names = (_vala_array_free (variable_names, variable_names_length1, ((GDestroyNotify) (g_free))), NULL);
}


GksuEnvironment* gksu_environment_new (void) {
	GksuEnvironment * self;
	self = g_object_newv (GKSU_TYPE_ENVIRONMENT, 0, NULL);
	return self;
}


static GObject * gksu_environment_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties) {
	GObject * obj;
	GksuEnvironmentClass * klass;
	GObjectClass * parent_class;
	GksuEnvironment * self;
	klass = GKSU_ENVIRONMENT_CLASS (g_type_class_peek (GKSU_TYPE_ENVIRONMENT));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	obj = parent_class->constructor (type, n_construct_properties, construct_properties);
	self = GKSU_ENVIRONMENT (obj);
	{
		char** _tmp0;
		gint search_path_length1;
		char** search_path;
		GeeHashMap* _tmp1;
		_tmp0 = NULL;
		search_path = (_tmp0 = g_get_system_data_dirs (), search_path_length1 = -1, _tmp0);
		_tmp1 = NULL;
		self->priv->variables = (_tmp1 = gee_hash_map_new (G_TYPE_STRING, ((GBoxedCopyFunc) (g_strdup)), g_free, GKSU_TYPE_VARIABLE, ((GBoxedCopyFunc) (g_object_ref)), g_object_unref, g_str_hash, g_str_equal, g_direct_equal), (self->priv->variables == NULL ? NULL : (self->priv->variables = (g_object_unref (self->priv->variables), NULL))), _tmp1);
		{
			char** path_collection;
			int path_collection_length1;
			int path_it;
			path_collection = search_path;
			path_collection_length1 = search_path_length1;
			for (path_it = 0; (search_path_length1 != -1 && path_it < search_path_length1) || (search_path_length1 == -1 && path_collection[path_it] != NULL); path_it = path_it + 1) {
				const char* _tmp2;
				char* path;
				_tmp2 = NULL;
				path = (_tmp2 = path_collection[path_it], (_tmp2 == NULL ? NULL : g_strdup (_tmp2)));
				{
					char* full_path;
					full_path = g_strconcat (path, "gksu-polkit-1/environment/", NULL);
					gksu_environment_read_variables_from_path (self, full_path);
					path = (g_free (path), NULL);
					full_path = (g_free (full_path), NULL);
				}
			}
		}
	}
	return obj;
}


static void gksu_environment_class_init (GksuEnvironmentClass * klass) {
	gksu_environment_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (GksuEnvironmentPrivate));
	G_OBJECT_CLASS (klass)->constructor = gksu_environment_constructor;
	G_OBJECT_CLASS (klass)->dispose = gksu_environment_dispose;
}


static void gksu_environment_instance_init (GksuEnvironment * self) {
	self->priv = GKSU_ENVIRONMENT_GET_PRIVATE (self);
}


static void gksu_environment_dispose (GObject * obj) {
	GksuEnvironment * self;
	self = GKSU_ENVIRONMENT (obj);
	(self->priv->variables == NULL ? NULL : (self->priv->variables = (g_object_unref (self->priv->variables), NULL)));
	G_OBJECT_CLASS (gksu_environment_parent_class)->dispose (obj);
}


GType gksu_environment_get_type (void) {
	static GType gksu_environment_type_id = 0;
	if (G_UNLIKELY (gksu_environment_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (GksuEnvironmentClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) gksu_environment_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (GksuEnvironment), 0, (GInstanceInitFunc) gksu_environment_instance_init };
		gksu_environment_type_id = g_type_register_static (G_TYPE_OBJECT, "GksuEnvironment", &g_define_type_info, 0);
	}
	return gksu_environment_type_id;
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if (array != NULL && destroy_func != NULL) {
		int i;
		if (array_length >= 0)
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) (array))[i] != NULL)
			destroy_func (((gpointer*) (array))[i]);
		}
		else
		for (i = 0; ((gpointer*) (array))[i] != NULL; i = i + 1) {
			destroy_func (((gpointer*) (array))[i]);
		}
	}
	g_free (array);
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return (str1 != str2);
	}
	return strcmp (str1, str2);
}




