/*
 * Copyright 1999-2006 University of Chicago
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 * commlink.c
 *
 * Communication Link Interface Routines
 */

#include "internal.h"

#define GLOBUS_L_NEXUS_STARTPOINT_VERSION		0

typedef struct _sp_copy_monitor_t
{
    nexus_mutex_t mutex;
    nexus_cond_t cond;
    nexus_bool_t done;

    /* data portion */
    nexus_startpoint_t *dest_sp;
} sp_copy_monitor_t;

#define GLOBUS_L_NEXUS_ENDPOINT_TABLE_SIZE 1023
static globus_hashtable_t		globus_l_nexus_endpoint_table;
static nexus_mutex_t			globus_l_nexus_endpoint_table_lock;

/* start copy_locally == NEXUS_FALSE */
static nexus_endpointattr_t StartpointCopyEpAttr;
void _nx_startpoint_copy_handler(nexus_endpoint_t *endpoint,
				nexus_buffer_t *buffer,
				nexus_bool_t is_non_threaded_handler);
static void startpoint_copy_reply_handler(nexus_endpoint_t *endpoint,
					nexus_buffer_t *buffer,
					nexus_bool_t is_non_threaded_handler);
static nexus_handler_t startpoint_copy_handlers[] =
{
    {NEXUS_HANDLER_TYPE_THREADED,
     (nexus_handler_func_t) startpoint_copy_reply_handler}
};
#define STARTPOINT_COPY_REPLY_HANDLER_ID 0
/* end copy_locally == NEXUS_FALSE */

/* start destroy_locally == NEXUS_FALSE */
static nexus_endpointattr_t StartpointDestroyandNotifyEpAttr;
void _nx_startpoint_destroy_and_notify_handler(nexus_endpoint_t *endpoint,
				nexus_buffer_t *buffer,
				nexus_bool_t is_non_threaded_handler);
static void startpoint_destroy_and_notify_reply_handler(
					nexus_endpoint_t *endpoint,
					nexus_buffer_t *buffer,
					nexus_bool_t is_non_threaded_handler);
static nexus_handler_t startpoint_destroy_and_notify_handlers[] =
{
    {NEXUS_HANDLER_TYPE_NON_THREADED,
     (nexus_handler_func_t) startpoint_destroy_and_notify_reply_handler}
};
#define STARTPOINT_DESTROY_AND_NOTIFY_REPLY_HANDLER_ID 0
/* end destroy_locally == NEXUS_FALSE */

static int same_startpoint(nexus_startpoint_t *sp1,
			   nexus_startpoint_t *sp2,
			   nexus_bool_t compare_addresses);

/*********************************************************************
 * Standard Nexus Functions
 *********************************************************************/
 
/*
 * _nx_commlink_new_process_params()
 */
int _nx_commlink_new_process_params(char *buf, int size)
{
    return(0);
} /* _nx_commlink_new_process_params() */


/*
 * _nx_commlink_init()
 *
 * Initialize the handler table for the passed 'context'.
 */
void _nx_commlink_init(void)
{

    int i;

    /* initializing endpoint hash table */
    if (LibaSizeof() <= sizeof(void *))
    {
	globus_hashtable_init(&globus_l_nexus_endpoint_table,
			      GLOBUS_L_NEXUS_ENDPOINT_TABLE_SIZE,
			      globus_hashtable_voidp_hash,
			      globus_hashtable_voidp_keyeq);
			      
    }
    else
    {
	nexus_fatal("Internal Error: unsupported LIBA size.  "
		    "Contact authors.");
	
    }

    nexus_mutex_init(&globus_l_nexus_endpoint_table_lock,
		     (nexus_mutexattr_t *) NULL);
    
    /* for copy_locally == NEXUS_FALSE */
    nexus_endpointattr_init(&StartpointCopyEpAttr);
    nexus_endpointattr_set_handler_table(&StartpointCopyEpAttr, 
					startpoint_copy_handlers, 
		    sizeof(startpoint_copy_handlers)/sizeof(nexus_handler_t));

    /* for destroy_locally == NEXUS_FALSE */
    nexus_endpointattr_init(&StartpointDestroyandNotifyEpAttr);
    nexus_endpointattr_set_handler_table(&StartpointDestroyandNotifyEpAttr, 
					startpoint_destroy_and_notify_handlers, 
	sizeof(startpoint_destroy_and_notify_handlers)/sizeof(nexus_handler_t));

} /* _nx_commlink_init() */


/*********************************************************************
 * Endpoint Attribute Functions
 *********************************************************************/

/*
 * nexus_endpointattr_init()
 */
int nexus_endpointattr_init(nexus_endpointattr_t *epattr)
{
    NexusAssert2((epattr), ("nexus_endpointattr_init(): rcvd NULL gpattr\n"));

    epattr->handler_table = (nexus_handler_t *) NULL;
    epattr->handler_table_size = 0;
    epattr->unknown_handler = NULL;
    epattr->unknown_handler_type = NEXUS_HANDLER_TYPE_NON_THREADED;
    epattr->transform_id = NEXUS_TRANSFORM_NONE;
    epattr->transform_attr = (nexus_transformattr_t *) NULL;
    epattr->proto_type = NEXUS_PROTO_TYPE_ALL;
    epattr->proto_info = NULL;
    _nx_context(&(epattr->context));

    return(0);
} /* nexus_endpointattr_init() */


/*
 * nexus_endpointattr_destroy()
 */
int nexus_endpointattr_destroy(nexus_endpointattr_t *epattr)
{
    NexusAssert2((epattr),
		 ("nexus_endpointattr_destroy(): rcvd NULL epattr\n"));

    if (   epattr->transform_id != NEXUS_TRANSFORM_NONE
	&& epattr->transform_attr)
    {
	nexus_transformattr_destroy(epattr->transform_id,
				    epattr->transform_attr);
    }

    if (epattr->proto_info != NULL)
    {
	NexusFree(epattr->proto_info);
    }

    return(0);
} /* nexus_endpointattr_destroy() */


/*
 * nexus_endpointattr_set_handler_table()
 */
int nexus_endpointattr_set_handler_table(nexus_endpointattr_t *epattr,
					 nexus_handler_t *handler_table,
					 int handler_table_size)
{
    if (!epattr || !handler_table || handler_table_size < 0)
    {
        return(-1);
    }
    
    epattr->handler_table = handler_table;
    epattr->handler_table_size = handler_table_size;

    return(0);
} /* nexus_endpointattr_set_handler_table() */


/*
 * nexus_endpointattr_get_handler_table()
 */
int nexus_endpointattr_get_handler_table(nexus_endpointattr_t *epattr, 
					 nexus_handler_t **handler_table,
					 int *handler_table_size)
{
    if (!epattr)
    {
        return(-1);
    }

    *handler_table = epattr->handler_table;
    *handler_table_size = epattr->handler_table_size;

    return(0);
} /* nexus_endpointattr_get_handler_table() */


/*
 * nexus_endpointattr_set_unknown_handler()
 */
int nexus_endpointattr_set_unknown_handler(nexus_endpointattr_t *epattr,
					   nexus_unknown_handler_func_t func,
					   nexus_handler_type_t type)
{
    epattr->unknown_handler = func;
    epattr->unknown_handler_type = type;

    return(0);
} /* nexus_endpointattr_set_unknown_handler() */


/*
 * nexus_endpointattr_get_unknown_handler()
 */
int nexus_endpointattr_get_unknown_handler(nexus_endpointattr_t *epattr,
					   nexus_unknown_handler_func_t *func,
					   nexus_handler_type_t *type)
{
    *func = epattr->unknown_handler;
    *type = epattr->unknown_handler_type;

    return(0);
} /* nexus_endpointattr_get_unknown_handler() */


/*
 * nexus_endpointattr_set_transform()
 */
int nexus_endpointattr_set_transform(nexus_endpointattr_t *epattr,
				     int transform_id,
				     void *transform_info)
{
    epattr->transform_id = transform_id;
    nexus_transformattr_init(transform_id,
			     transform_info,
			     &(epattr->transform_attr));
    return(0);
} /* nexus_endpointattr_set_transform() */


/*
 * nexus_endpoinattr_get_transform()
 *
 * *transform_info should be freed using nexus_free()
 */
int nexus_endpointattr_get_transform(nexus_endpointattr_t *epattr,
				     int *transform_id,
				     void **transform_info)
{
    *transform_id = epattr->transform_id;
    nexus_transformattr_get_info(epattr->transform_id,
				 epattr->transform_attr,
				 transform_info);
    return(0);
} /* nexus_endpointattr_get_transform() */


int nexus_startpointstate_get_transform(nexus_startpoint_t *sp,
					void *info)
{
	int id;

	id = sp->transform_id;
	if (id != NEXUS_TRANSFORM_NONE)
	{ 
	    nexus_startpoint_info(id,
				sp->transform_state,
				info);
	    return(0);
	} else 
	    return(-1);
}


int nexus_endpointstate_get_transform(nexus_endpoint_t *ep,
					void *info)
{
	int id;

	id = ep->transform_id;
	if (id != NEXUS_TRANSFORM_NONE)
	{
	    nexus_endpoint_info(id,
				ep->transform_state,
				info);
	    return(0);
	} else
	    return(-1);
}


/*
 * nexus_endpointattr_set_protocol()
 */
int nexus_endpointattr_set_protocol(nexus_endpointattr_t *epattr,
				    nexus_proto_type_t proto_type,
				    void *proto_info,
				    int proto_info_length)
{
    int rc;
    if (_nx_proto_check_type(proto_type))
    {
	epattr->proto_type = proto_type;
	if (proto_info)
	{
	    if (epattr->proto_info != NULL)
	    {
		NexusFree(epattr->proto_info);
	    }

	    NexusMalloc(nexus_endpointattr_set_protocol(),
			epattr->proto_info,
			void *,
			proto_info_length);
	    memcpy(epattr->proto_info, proto_info, proto_info_length);
	}
	else
	{
	    epattr->proto_info = NULL;
	}
	rc = 0;
    }
    else
    {
	rc = 1;
    }
    return(rc);
} /* nexus_endpointattr_set_protocol() */


/*
 * nexus_endpointattr_get_protocol()
 */
int nexus_endpointattr_get_protocol(nexus_endpointattr_t *epattr,
				    nexus_proto_type_t *proto_type,
				    void **proto_info)
{
    *proto_type = epattr->proto_type;
    *proto_info = epattr->proto_info;
    return(0);
} /* nexus_endpointattr_get_protocol() */


/*********************************************************************
 * Endpoint Functions
 *********************************************************************/

/*
 * nexus_endpoint_init()
 *
 * this is called only for user-created endpoints ... NOT for
 * the endpoints that are created as a result of a copy.
 * 
 * NOTE: - only signed & nonspoofable endpoints are placed into the hashtable
 *       - removed from hashtable once destroyed.  in other 
 *         words, sits in hashtable iff bound to a gp
 */
int nexus_endpoint_init(nexus_endpoint_t *endpoint,
			nexus_endpointattr_t *epattr)
{
    int rc;
    
    if (!endpoint || !epattr)
    {
        return(GLOBUS_NEXUS_ERROR_INVALID_PARAMETER);
    }

    endpoint->handler_table = epattr->handler_table;
    endpoint->handler_table_size = epattr->handler_table_size;
    endpoint->unknown_handler = epattr->unknown_handler;
    endpoint->unknown_handler_type = epattr->unknown_handler_type;
    endpoint->transform_id = epattr->transform_id;
    if (epattr->transform_id != NEXUS_TRANSFORM_NONE)
    {
	nexus_transformstate_init_on_endpoint(epattr->transform_id,
					      epattr->transform_attr,
					      &(endpoint->transform_state));
    }

    endpoint->proto_type = epattr->proto_type;
    if (epattr->proto_type == NEXUS_PROTO_TYPE_ALL)
    {
	endpoint->mi_proto = _nx_my_mi_proto;
    }
    else
    {
	rc = _nx_mi_proto_create_for_proto_type(epattr->proto_type,
						epattr->proto_info,
						endpoint,
						&(endpoint->mi_proto));

	if (rc != GLOBUS_SUCCESS)
	{
	    if (endpoint->transform_id != NEXUS_TRANSFORM_NONE
		&& endpoint->transform_state)
	    {
		nexus_transformstate_destroy_on_endpoint(
		    endpoint->transform_id,
		    endpoint->transform_state);
	    }

	    return(rc);
	}
    }

    endpoint->id = globus_i_nexus_endpoint_table_add(endpoint);
    endpoint->context = epattr->context;
    endpoint->user_pointer = (void *) NULL;

    return(GLOBUS_SUCCESS);
} /* nexus_endpoint_init() */


/*
 * nexus_endpoint_destroy()
 *
 * must destroy this one and ALL within the bind_or_copy chain
 * in which this one appears.
 *
 * this function renders the endpoint completely useless.
 * in order to use this endpoint again must call nexus_endpoint_init()
 * again
 */
int nexus_endpoint_destroy(nexus_endpoint_t *endpoint)
{
    int return_code = GLOBUS_SUCCESS;
    
    if (!endpoint)
    {
        return(GLOBUS_NEXUS_ERROR_INVALID_PARAMETER);
    }

    globus_i_nexus_endpoint_table_remove(endpoint);
    
    if (   endpoint->transform_id != NEXUS_TRANSFORM_NONE
	&& endpoint->transform_state)
    {
	nexus_transformstate_destroy_on_endpoint(endpoint->transform_id,
						 endpoint->transform_state);
    }

    if (endpoint->proto_type != NEXUS_PROTO_TYPE_ALL)
    {
	return_code = _nx_mi_proto_destroy_for_proto_type(endpoint,
							  endpoint->mi_proto);
    }
    
    return(return_code);
} /* nexus_endpoint_destroy() */


/*********************************************************************
 * Startpoint Management Functions
 *********************************************************************/

/*
 * nexus_startpoint_bind()
 */
int nexus_startpoint_bind(nexus_startpoint_t *startpoint,
			  nexus_endpoint_t *endpoint)
{
    nexus_bool_t copy_sp_locally;
    nexus_bool_t destroy_sp_locally;
    
    if (!endpoint || !startpoint)
    {
        return(GLOBUS_NEXUS_ERROR_INVALID_PARAMETER);
    }

    startpoint->mi_proto = endpoint->mi_proto;
    
    startpoint->endpoint_id = endpoint->id;
    
    /* Setup the transform stuff on this startpoint */
    startpoint->transform_id = endpoint->transform_id;
    if (startpoint->transform_id != NEXUS_TRANSFORM_NONE)
    {
	nexus_transformstate_init_on_startpoint(endpoint->transform_id,
						endpoint->transform_state,
						&(startpoint->transform_state),
						&copy_sp_locally,
						&destroy_sp_locally);
	startpoint->copy_locally = (copy_sp_locally ? 1 : 0);
	startpoint->destroy_locally = (destroy_sp_locally ? 1 : 0);
    }
    else
    {
	startpoint->transform_state = (nexus_transformstate_t *) NULL;
	startpoint->copy_locally = 1;
	startpoint->destroy_locally = 1;
    }

    /* Create a liba */
    /* TODO: Need to add signature to liba */
    startpoint->liba_size = LibaSizeof();
    if (startpoint->liba_size <= NEXUS_DEFAULT_LIBA_SIZE)
    {
	startpoint->liba_is_inline = 1;
	LibaPack(startpoint->liba.array, endpoint->id);
    }
    else
    {
	startpoint->liba_is_inline = 0;
	NexusMalloc(bind_startpoint_to_endpoint(),
		    startpoint->liba.pointer,
		    nexus_byte_t *,
		    startpoint->liba_size);
	LibaPack(startpoint->liba.pointer, endpoint->id);
    }

#ifdef BUILD_PROFILE
    _nx_node_id(&(startpoint->node_id));
    _nx_context_id(&(startpoint->context_id));
#endif /* BUILD_PROFILE */

    return(GLOBUS_SUCCESS);

} /* nexus_startpoint_bind() */


/*
 * nexus_startpoint_copy()
 *
 * Copy the startpoint, 'src_sp', to 'dest_sp'.  
 *
 * If this sp has changing transform state, then the endpoint must be
 * contacted to create a new endpoint and sp.  This may be an
 * expensive operation.  This is signalled by src_sp->copy_locally==0.
 *
 * If this sp has static transform state, then a simple copy can be
 * made locally without contacting the endpoint.
 * This is signalled by src_sp->copy_locally==0.
 */
int nexus_startpoint_copy(nexus_startpoint_t *dest_sp, 
			  nexus_startpoint_t *source_sp)
{

    nexus_buffer_t buffer;
    sp_copy_monitor_t monitor;
    nexus_endpoint_t reply_done_ep;
    nexus_startpoint_t reply_done_sp;

    if (!dest_sp || !source_sp)
    {
        return(GLOBUS_NEXUS_ERROR_INVALID_PARAMETER);
    }

    if (source_sp->copy_locally)
    {
	/*
	 * The transform state can be copied locally
	 */
	dest_sp->endpoint_id = source_sp->endpoint_id;
	dest_sp->copy_locally = source_sp->copy_locally;
	dest_sp->destroy_locally = source_sp->destroy_locally;

	/* transform stuff */
	dest_sp->transform_id = source_sp->transform_id;
	/* transform state */
	if (source_sp->transform_id != NEXUS_TRANSFORM_NONE)
	{
	    nexus_transformstate_copy(source_sp->transform_id,
				      source_sp->transform_state,
				      &(dest_sp->transform_state));
	}
	else
	{
	    dest_sp->transform_state = (nexus_transformstate_t *) NULL;
	}
	/* liba */
	dest_sp->liba_is_inline = source_sp->liba_is_inline;
	dest_sp->liba_size = source_sp->liba_size;
	if (source_sp->liba_is_inline)
	{
	    memcpy(dest_sp->liba.array,
		   source_sp->liba.array, 
		   source_sp->liba_size);
	}
	else
	{
	    NexusMalloc(nexus_startpoint_copy(),
			dest_sp->liba.pointer,
			nexus_byte_t *,
			source_sp->liba_size);
	    memcpy(dest_sp->liba.pointer,
		   source_sp->liba.pointer, 
		   source_sp->liba_size);
	}

#ifdef NEXUS_PROFILE
	dest_sp->node_id = source_sp->node_id;
	dest_sp->context_id = source_sp->context_id;
#endif /* NEXUS_PROFILE */

	/* mi_proto */
	dest_sp->mi_proto = source_sp->mi_proto;
	if (dest_sp->mi_proto->proto->funcs->increment_reference_count)
	{
	    (dest_sp->mi_proto->proto->funcs->increment_reference_count)
						(dest_sp->mi_proto->proto);
	}
    }
    else
    {
	/*
	* Copying this startpoint requires a round-trip to the server.
	* TODO: Fillin this section.
	*/

	nexus_mutex_init(&(monitor.mutex), (nexus_mutexattr_t *) NULL);
	nexus_cond_init(&(monitor.cond), (nexus_condattr_t *) NULL);

	nexus_mutex_lock(&(monitor.mutex));
	monitor.done = NEXUS_FALSE;
	monitor.dest_sp = dest_sp;
	nexus_mutex_unlock(&(monitor.mutex));

	nexus_endpoint_init(&reply_done_ep, &StartpointCopyEpAttr);
	nexus_endpoint_set_user_pointer(&reply_done_ep, (void *) (&monitor));

	nexus_startpoint_bind(&reply_done_sp, &reply_done_ep);

	nexus_buffer_init(&buffer,
			    nexus_sizeof_startpoint(&reply_done_sp, 1),
			    0);
	nexus_put_startpoint_transfer(&buffer, (&reply_done_sp), 1);
	nexus_send_rsr(&buffer,
		    source_sp,
		    GLOBUS_NEXUS_STARTPOINT_COPY_HANDLER_ID,
		    NEXUS_TRUE,
		    NEXUS_FALSE);

	nexus_mutex_lock(&(monitor.mutex));
	while (!(monitor.done))
	{
	    nexus_cond_wait(&(monitor.cond), &(monitor.mutex));
	} /* endwhile */
	nexus_mutex_unlock(&(monitor.mutex));

	nexus_mutex_destroy(&(monitor.mutex));
	nexus_cond_destroy(&(monitor.cond));

	nexus_endpoint_destroy(&reply_done_ep);
    }

    return(GLOBUS_SUCCESS);
} /* nexus_startpoint_copy() */

/*
 * _nx_startpoint_copy_handler()
 */
void _nx_startpoint_copy_handler(nexus_endpoint_t *endpoint,
				    nexus_buffer_t *buffer,
				    nexus_bool_t is_non_threaded_handler)
{

    nexus_startpoint_t reply_done_sp;
    nexus_startpoint_t new_sp;
    nexus_buffer_t reply_buffer;

    nexus_get_startpoint(buffer, &reply_done_sp, 1);

    nexus_startpoint_bind(&new_sp, endpoint);

    nexus_buffer_init(&reply_buffer,  nexus_sizeof_startpoint(&new_sp, 1), 0);
    nexus_put_startpoint_transfer(&reply_buffer, &new_sp, 1);
    nexus_send_rsr(&reply_buffer,
                    &reply_done_sp,
                    STARTPOINT_COPY_REPLY_HANDLER_ID,
                    NEXUS_TRUE,
                    is_non_threaded_handler);

    /************/
    /* clean-up */
    /************/

    nexus_buffer_destroy(buffer);
    nexus_startpoint_destroy(&reply_done_sp);

} /* end _nx_startpoint_copy_handler() */

/*
 * startpoint_copy_reply_handler()
 */
static void startpoint_copy_reply_handler(nexus_endpoint_t *endpoint,
					nexus_buffer_t *buffer,
					nexus_bool_t is_non_threaded_handler)
{
    sp_copy_monitor_t *mp;

    mp = (sp_copy_monitor_t *) nexus_endpoint_get_user_pointer(endpoint);

    nexus_mutex_lock(&(mp->mutex));
    nexus_get_startpoint(buffer, mp->dest_sp, 1);
    nexus_mutex_unlock(&(mp->mutex));

    nexus_buffer_destroy(buffer);

    nexus_mutex_lock(&(mp->mutex));
    mp->done = NEXUS_TRUE;
    nexus_cond_signal(&(mp->cond));
    nexus_mutex_unlock(&(mp->mutex));

} /* end startpoint_copy_reply_handler() */

/*
 * nexus_startpoint_destroy()
 *
 * Destroy the startpoint, 'sp', and set it NULL.
 * NOTE: do NOT notify endpoint ... use nexus_startpoint_destroy_and_notify() 
 *       for that kind of sp destruction
 */
int nexus_startpoint_destroy(nexus_startpoint_t *sp)
{
    if (!sp)
    {
        return(GLOBUS_NEXUS_ERROR_INVALID_PARAMETER);
    }

    if (!nexus_startpoint_is_null(sp))
    {
	if (sp->transform_id != NEXUS_TRANSFORM_NONE)
	{
	    nexus_transformstate_destroy_on_startpoint(sp->transform_id,
						       sp->transform_state);
	}
	_nx_mi_proto_destroy(sp->mi_proto);
	if (!(sp->liba_is_inline))
	{
	    NexusFree(sp->liba.pointer);
	}
	nexus_startpoint_set_null(sp);
    }

    return(GLOBUS_SUCCESS);
} /* nexus_startpoint_destroy() */

static void startpoint_destroy_and_notify_reply_handler(
					nexus_endpoint_t *endpoint,
					nexus_buffer_t *buffer,
					nexus_bool_t is_non_threaded_handler)
{}

/*
 * nexus_startpoint_destroy_and_notify()
 *
 * Destroy the startpoint, 'sp', set it to NULL. 
 *
 * If this sp has per-sp state, contact the endpoint to free up any
 * resources associated with this sp ... wait for confirmation.
 *
 */
int nexus_startpoint_destroy_and_notify(nexus_startpoint_t *sp)
{
    if (!sp)
    {
        return(GLOBUS_NEXUS_ERROR_INVALID_PARAMETER);
    }

    if (!sp->destroy_locally)
    {
#if TODO
	/*
	 * Do a round-trip to the context pointed to by the startpoint,
	 * to give the endpoint an opportunity to free up the
	 * transformstate for this startpoint.
	 */
	nexus_buffer_t buffer;
	nexus_endpoint_t reply_ep;
	nexus_startpoint_t reply_sp;
	nexus_barrier_t barrier;

	/* Initialize the barrier one which we will wait for a reply */
	nexus_mutex_init(&(barrier.mutex), (nexus_mutexattr_t *) NULL);
	nexus_cond_init(&(barrier.cond), (nexus_condattr_t *) NULL);
	barrier.count = 1;

	/* Create and endpoint and startpoint for the reply */
	nexus_endpoint_init(&reply_ep, &destroy_and_notify_reply_epattr);
	nexus_endpoint_set_user_pointer(&reply_dp, &barrier);
	nexus_startpoint_bind(&reply_sp, &reply_ep);

	/* Send message to destination node */
	nexus_buffer_init(&buffer,
			  (nexus_sizeof_startpoint(&reply_sp, 1)
			   + nexus_sizeof_u_long(1) /* endpoint_id */
			   + nexus_transformstate_sizeof(sp->transform_id,
							 sp->transform_state)),
			  0);
	nexus_put_startpoint_transfer(&buffer, &reply_sp, 1);
	nexus_put_u_long(&buffer, &(sp->endpoint_id), 1);
	nexus_transformstate_put(sp->transform_id,
				 &(buffer->current_base_segment->current),
				 sp->transform_state);
	nexus_send_rsr(&buffer,
		       sp,
		       GLOBUS_NEXUS_STARTPOINT_DESTROY_AND_NOTIFY_HANDLER_ID,
		       NEXUS_TRUE,
		       NEXUS_FALSE);

	/*
	 * TODO
	 *
	 * Write nexus_startpoint_destroy_and_notify_handler().
	 * This should call
	 * nexus_transformstate_update_endpoint_with_startpoint_destroy().
	 *
	 * Write startpoint_destroy_and_notify_reply_handler().
	 * This should wakeup the barrier.
	 */
	
	nexus_mutex_lock(&(barrier.mutex));
	while (barrier.count > 0)
	{
	    nexus_cond_wait(&(barrier.cond), &(barrier.mutex));
	}
	nexus_mutex_unlock(&(barrier.mutex));

	nexus_mutex_destroy(&(barrier.mutex));
	nexus_cond_destroy(&(barrier.cond));
	nexus_endpoint_destroy(&reply_ep);
#endif
    }
    
    /* now we can destroy the sp locally */
    nexus_startpoint_destroy(sp);

    return(GLOBUS_SUCCESS);
} /* nexus_startpoint_destroy_and_notify() */


/*
 * destroy_and_notify_reply_handler()
 */
static void destroy_and_notify_reply_handler(nexus_endpoint_t *ep,
				nexus_buffer_t *buffer,
				nexus_bool_t called_from_non_threaded_handler)
{
} /* destroy_and_notify_reply_handler() */

/*********************************************************************
 * Startpoint sizeof/put/get Functions
 *********************************************************************/

/*
 * nexus_sizeof_startpoint()
 *
 * Return the size in bytes needed to encode 'count' startpoints
 * that start at 'sp_array'.
 */
int nexus_sizeof_startpoint(nexus_startpoint_t *sp_array, int count)
{
    nexus_startpoint_t *sp;
    int c;
    int size = 0;

    if (!sp_array || (count < 0))
    {
	return(0);
    }
    
    for (c = 0, sp = sp_array; c < count; c++, sp++)
    {
        /* sizeof version, is_null */
        size += 2 * nexus_sizeof_byte(1);

        if (!nexus_startpoint_is_null(sp))
        {
#ifdef BUILD_PROFILE
            /* sizeof the node_id, context_id */
            size += nexus_sizeof_int(1);
            size += nexus_sizeof_int(1);
#endif /* BUILD_PROFILE */    

	    /* sizeof endpoint_id */
	    size += nexus_sizeof_u_long(1);
	    
	    /* sizeof copy_locally, destroy_locally, transform_id */
	    size += 3 * nexus_sizeof_byte(1);
	    
            /* sizeof liba */
            size += nexus_sizeof_int(1); /* liba_size */
            size += nexus_sizeof_byte(sp->liba_size); /* LIBA */
            
            /* sizeof mi_proto */
            size += nexus_sizeof_int(1);
            size += nexus_sizeof_byte(sp->mi_proto->size);

	    /* sizeof transform_state */
	    if (sp->transform_id != NEXUS_TRANSFORM_NONE)
	    {
		size += nexus_transformstate_sizeof(sp->transform_id,
						    sp->transform_state);
	    }
        } /* endif */
    } /* endfor */
    
    return(size);
} /* nexus_sizeof_startpoint() */


/*
 * nexus_user_put_startpoint_transfer()
 */
int nexus_user_put_startpoint_transfer(nexus_byte_t **buffer, 
				       nexus_startpoint_t *sp_array,
				       unsigned long count)
{
    unsigned long i;
    int tmp_int;
    nexus_byte_t tmp_byte;
    nexus_startpoint_t *sp;

    if (!buffer || !sp_array)
    {
        return(GLOBUS_NEXUS_ERROR_INVALID_PARAMETER);
    }

    for (i = 0, sp = sp_array; i < count; i++, sp++)
    {
	/* put version */
	tmp_byte = GLOBUS_L_NEXUS_STARTPOINT_VERSION;
	nexus_user_put_byte(buffer, &tmp_byte, 1);

	/* put is_null */
	tmp_byte = (nexus_startpoint_is_null(sp) ? 1 : 0);
	nexus_user_put_byte(buffer, &tmp_byte, 1);

	if (!nexus_startpoint_is_null(sp))
	{
            unsigned long tmp_ulong;

#ifdef BUILD_PROFILE
	    /* put the node_id, context_id */
	    nexus_user_put_int(buffer, &(sp->node_id), 1);
	    nexus_user_put_int(buffer, &(sp->context_id), 1);
#endif /* BUILD_PROFILE */    
	
	    /* put endpoint_id */
            tmp_ulong = (unsigned long) sp->endpoint_id;

	    nexus_user_put_u_long(buffer,
                                  &tmp_ulong,
                                  1);
	    
	    /* put copy_locally, destroy_locally, transform_id */
	    tmp_byte = sp->copy_locally;
	    nexus_user_put_byte(buffer, &tmp_byte, 1);
	    tmp_byte = sp->destroy_locally;
	    nexus_user_put_byte(buffer, &tmp_byte, 1);
	    tmp_byte = sp->transform_id;
	    nexus_user_put_byte(buffer, &tmp_byte, 1);
	    
	    /* put the liba */
	    tmp_int = sp->liba_size;
	    nexus_user_put_int(buffer, &tmp_int, 1);
	    nexus_user_put_byte(buffer,
				(sp->liba_is_inline 
				 ? (void *) sp->liba.array 
				 : (void *) sp->liba.pointer),
				sp->liba_size);
	    
	    /* put the mi_proto */
	    nexus_user_put_int(buffer, &(sp->mi_proto->size), 1);
	    nexus_user_put_byte(buffer,
				sp->mi_proto->array,
				sp->mi_proto->size);

	    /* put transform_state */
	    if (sp->transform_id != NEXUS_TRANSFORM_NONE)
	    {
		nexus_transformstate_put(sp->transform_id,
					 buffer,
					 sp->transform_state);
	    }
	    
	    nexus_startpoint_destroy(sp);
	} /* endif */
    } /* endfor */

    return(GLOBUS_SUCCESS);
} /* nexus_user_put_startpoint_transfer() */


/*
 * nexus_user_get_startpoint()
 *
 * Copy 'count' startpoints from message buffer into sp's starting at 'sp' 
 */
int nexus_user_get_startpoint(nexus_byte_t **buffer,
			      nexus_startpoint_t *sp_array, 
			      unsigned long count,
			      int format)
{
    nexus_startpoint_t *		sp;
    unsigned long			i;
    int					tmp_int;
    nexus_byte_t			tmp_byte;
    int					mi_proto_size;
    nexus_mi_proto_t *			mi_proto;
    int					rc;

    rc = 0;

    if (!buffer || !sp_array)
    {
        rc = GLOBUS_NEXUS_ERROR_INVALID_PARAMETER;
	goto fn_exit;
    }

    for (i = 0, sp = sp_array; i < count; i++, sp++)
    {
        /* get version */
        nexus_user_get_byte(buffer, &tmp_byte, 1, format);
	if (tmp_byte != GLOBUS_L_NEXUS_STARTPOINT_VERSION)
	{
	    /* null out the remaining startpoints */
	    for ( ; i < count; i++, sp++)
	    {
		nexus_startpoint_set_null(sp);
	    }
	    
	    /* return a version mismatch error */
	    rc = GLOBUS_NEXUS_ERROR_VERSION_MISMATCH;
	    goto fn_exit;
	}

	/* get is_null */
        nexus_user_get_byte(buffer, &tmp_byte, 1, format);

        if (tmp_byte)
	{
            nexus_startpoint_set_null(sp);
	}
        else
        {
            unsigned long tmp_ulong;

#ifdef BUILD_PROFILE
            /* get the node_id, context_id */
            nexus_user_get_int(buffer, &(sp->node_id), 1, format);
            nexus_user_get_int(buffer, &(sp->context_id), 1, format);
#endif /* BUILD_PROFILE */    

	    /* put endpoint_id */
	    nexus_user_get_u_long(buffer, &tmp_ulong, 1, format);
            sp->endpoint_id = (globus_i_nexus_endpoint_id_t) tmp_ulong;
	    
	    /* get copy_locally, destroy_locally, transform_id */
	    nexus_user_get_byte(buffer, &tmp_byte, 1, format);
	    sp->copy_locally = (tmp_byte ? 1 : 0);
	    nexus_user_get_byte(buffer, &tmp_byte, 1, format);
	    sp->destroy_locally = (tmp_byte ? 1 : 0);
	    nexus_user_get_byte(buffer, &tmp_byte, 1, format);
	    sp->transform_id = (unsigned int) tmp_byte;
	    
            /* get the liba */
            nexus_user_get_int(buffer, &tmp_int, 1, format);
            sp->liba_size = tmp_int;
            if (sp->liba_size <= NEXUS_DEFAULT_LIBA_SIZE)
            {
                sp->liba_is_inline = 1;
                nexus_user_get_byte(buffer,
				    (nexus_byte_t *) sp->liba.array,
				    sp->liba_size,
				    format);
            }
            else
            {
                sp->liba_is_inline = 0;
                NexusMalloc(nexus_user_get_global_pointer(),
                            sp->liba.pointer,
                            nexus_byte_t *,
                            sp->liba_size);
                nexus_user_get_byte(buffer,
				    (nexus_byte_t *) sp->liba.pointer,
				    sp->liba_size,
				    format);
            } /* endif */
            
            /* get the mi_proto */
            nexus_user_get_int(buffer, &mi_proto_size, 1, format);
            mi_proto = _nx_mi_proto_create(mi_proto_size,
					   (nexus_byte_t *) NULL,
					   (nexus_proto_t *) NULL);
            nexus_user_get_byte(buffer,
				mi_proto->array,
				mi_proto_size,
				format);
            rc = _nx_mi_proto_table_insert(mi_proto,
					   &(sp->mi_proto));
	    if (rc != GLOBUS_SUCCESS)
	    {
		goto fn_exit;
	    }

	    /* get transform_state */
	    if (sp->transform_id != NEXUS_TRANSFORM_NONE)
	    {
		nexus_transformstate_get(sp->transform_id,
					 buffer,
					 format,
					 &(sp->transform_state));
	    }
	    
        } /* endif */
    } /* endfor */

  fn_exit:
    return(rc);
} /* nexus_user_get_startpoint() */


/*
 * nexus_put_startpoint_transfer()
 */
int nexus_put_startpoint_transfer(nexus_buffer_t *buffer, 
				  nexus_startpoint_t *sp_array,
				  unsigned long count)
{
    return(nexus_user_put_startpoint_transfer(
				&((*(buffer))->current_base_segment->current),
				sp_array,
				count));
} /* nexus_put_startpoint_transfer() */


/*
 * nexus_get_startpoint()
 */
int nexus_get_startpoint(nexus_buffer_t *buffer,
			 nexus_startpoint_t *sp_array, 
			 unsigned long count)
{
    return(nexus_user_get_startpoint(
				&((*(buffer))->current_base_segment->current),
				sp_array,
				count,
				(*(buffer))->format));
} /* nexus_get_startpoint() */


/*      
 * _nx_write_startpoint()
 *      
 * Write the startpoint to the passed file descriptor.
 *        
 * Return:  
 *      0 on success
 *      1 if the filedesc is unexpectedly closed
 */ 
int _nx_write_startpoint(globus_io_handle_t * handle, 
			 nexus_startpoint_t *sp)
{ 
    nexus_byte_t tmp_byte;
    int sp_size;
    int total_size;
    nexus_byte_t *buf, *b;

    if (!sp)
    {
        return(GLOBUS_NEXUS_ERROR_INVALID_PARAMETER);
    }

    sp_size = nexus_sizeof_startpoint(sp, 1);
    total_size = (nexus_sizeof_byte(1)
		  + nexus_sizeof_int(1)
		  + sp_size);
    NexusMalloc(_nx_write_startpoint(),
		buf,
		nexus_byte_t *,
		total_size);
    b = buf;
    tmp_byte = GLOBUS_DC_FORMAT_LOCAL;
    nexus_user_put_byte(&b, &tmp_byte, 1);		/* write the format */
    nexus_user_put_int(&b, &sp_size, 1);		/* write the sp size */
    nexus_user_put_startpoint_transfer(&b, sp, 1);	/* write the sp */

    if (_nx_write_blocking(handle, buf, total_size) != 0)
    {
	return(1);
    }

    NexusFree(buf);
    
    return(GLOBUS_SUCCESS);
} /* _nx_write_startpoint() */


/*
 * _nx_read_startpoint()
 *  
 * Read a startpoint from the passed file descriptor and fill
 * it into 'sp'.
 *      
 * Return:  
 *      0 on success
 *      1 if the filedesc is unexpectedly closed
 */  
int _nx_read_startpoint(globus_io_handle_t * handle, nexus_startpoint_t *sp)
{       
    nexus_byte_t format;
    int int_size;
    int sp_size;
    nexus_byte_t *buf, *b;

    /* Read the format */
    if (_nx_read_blocking(handle, &format, 1) != 0)
    {
	return(1);
    }

    /* Read the sp size */
    int_size = nexus_dc_sizeof_remote_int(1, format);
    NexusMalloc(_nx_read_startpoint(),
		buf,
		nexus_byte_t *,
		int_size);
    if (_nx_read_blocking(handle, buf, int_size) != 0)
    {
	return(1);
    }
    b = buf;
    nexus_user_get_int(&b, &sp_size, 1, format);
    NexusFree(buf);

    /* Read the sp */
    NexusMalloc(_nx_read_startpoint(),
		buf,
		nexus_byte_t *,
		sp_size);
    if (_nx_read_blocking(handle, buf, sp_size) != 0)
    {
	return(1);
    }
    b = buf;
    nexus_user_get_startpoint(&b, sp, 1, format);
    NexusFree(buf);
    
    return(GLOBUS_SUCCESS);
} /* _nx_read_startpoint() */


/*
 * nexus_startpoint_equal()
 *
 * Return non-zero if the two startpoints point to the same address
 * in the same context on the same node, or zero otherwise.
 */
int nexus_startpoint_equal(nexus_startpoint_t *sp1, nexus_startpoint_t *sp2)
{
    if (nexus_startpoint_equal_context(sp1, sp2)
	&& sp1 && sp2 && (sp1->endpoint_id == sp2->endpoint_id) )
    {
	return(NEXUS_TRUE);
    }
    else
    {
	return(NEXUS_FALSE);
    }
} /* nexus_startpoint_equal() */


/*
 * nexus_startpoint_equal_context()
 *
 * Return non-zero if the two startpoints point to the same context
 * on the same node, or zero otherwise.
 */
int nexus_startpoint_equal_context(nexus_startpoint_t *sp1, 
				   nexus_startpoint_t *sp2)
{
    nexus_byte_t *a1, *a2;
    
    if (nexus_startpoint_is_null(sp1) && nexus_startpoint_is_null(sp2))
    {
        /* Both are NULL global pointers */
        return(NEXUS_TRUE);
    }
    if (nexus_startpoint_is_null(sp1) || nexus_startpoint_is_null(sp2))
    {
        /* One is a NULL global pointer, the other isn't */
        return(NEXUS_FALSE);
    }

    /* Compare their context differentiators */
    /*
     * TODO: Context differentiator may need to be generalized to handle
     * multiple contexts within the same process.
     */
    a1 = sp1->mi_proto->array;
    a2 = sp2->mi_proto->array;
    if (memcmp(a1, a2, 4) != 0)
    {
        return(NEXUS_FALSE);
    }

    /* Compare their hostnames */
    if (strcmp((char *) (a1+4), (char *) (a2+4)) != 0)
    {
        return(NEXUS_FALSE);
    }
    
    /* They are the same context */
    return(NEXUS_TRUE);
} /* nexus_startpoint_equal_context() */


/*
 * nexus_startpoint_to_current_context()
 *
 * Return non-zero if the startpoint points to an address in the
 * context of the calling thread. 
 */
int nexus_startpoint_to_current_context(nexus_startpoint_t *sp)
{
    nexus_byte_t *a1, *a2;
    
    if (nexus_startpoint_is_null(sp))
    {
        /* It is a NULL global pointer */
        return(NEXUS_TRUE);
    }

    /* Compare their context differentiators */
    /*
     * TODO: Context differentiator may need to be generalized to handle
     * multiple contexts within the same process.
     */
    a1 = sp->mi_proto->array;
    a2 = _nx_my_mi_proto->array;
    if (memcmp(a1, a2, 4) != 0)
    {
        return(NEXUS_FALSE);
    }

    /* Compare their hostnames */
    if (strcmp((char *) (a1+4), (char *) (a2+4)) != 0)
    {
        return(NEXUS_FALSE);
    }
    
    /* They are the same context */
    return(NEXUS_TRUE);
} /* nexus_startpoint_to_current_context() */


/*
 * nexus_startpoint_get_endpoint()
 *
 * Assume the startpoint is local.
 */
int nexus_startpoint_get_endpoint(nexus_startpoint_t *sp,
				  nexus_endpoint_t **ep)
{
    nexus_byte_t *			liba;

    if (!sp || !ep)
    {
	return(GLOBUS_NEXUS_ERROR_INVALID_PARAMETER);
    }
    
    if (sp->liba_is_inline)
    {
	liba = sp->liba.array;
    }
    else
    {
	liba = sp->liba.pointer;
    }

    if (globus_i_nexus_endpoint_table_lookup(liba, ep)
	!= GLOBUS_SUCCESS)
    {
	return(GLOBUS_FAILURE);
    }

    return(GLOBUS_SUCCESS);
} /* nexus_startpoint_get_endpoint() */


/*
 * nexus_startpoint_string()
 *
 * Return a string representation of the startpoint.
 */
int nexus_startpoint_string(nexus_startpoint_t *sp, char *string, int length)
{
    if (!sp || !string || length <0)
    {
        return(GLOBUS_NEXUS_ERROR_INVALID_PARAMETER);
    }

    string[0] = '\0';

#ifdef BUILD_PROFILE
    if (length < 50 /* some free space, but not much */)
    {
        return(GLOBUS_NEXUS_ERROR_UNKNOWN);
    }
    sprintf(string, "[STARTPOINT:%08lx:%08lx:n%d:c%d]",
            (unsigned long) sp,
            (unsigned long) sp->endpoint_id,
            sp->node_id,
            sp->context_id);
#else  /* BUILD_PROFILE */
    if (length < 30 /* exact */)
    {
        return(GLOBUS_NEXUS_ERROR_UNKNOWN);
    }
    sprintf(string, "[STARTPOINT:%08lx:%08lx]",
            (unsigned long) sp,
            (unsigned long) sp->endpoint_id);
#endif /* BUILD_PROFILE */    
              
    return(GLOBUS_SUCCESS);
} /* nexus_startpoint_string() */


/*
 * nexus_endpoint_string()
 */
int nexus_endpoint_string(nexus_endpoint_t *ep, char *string, int length)
{
    if (!ep || !string)
    {
        return(GLOBUS_NEXUS_ERROR_INVALID_PARAMETER);
    }

    /* do something nice here */
    if (length < 0)
    {
        return(GLOBUS_NEXUS_ERROR_UNKNOWN);
    }

    return(GLOBUS_NEXUS_ERROR_UNIMPLEMENTED);
} /* nexus_endpoint_string() */


/*********************************************************************
 * Local Functions
 *********************************************************************/

/*
 * same_startpoint()
 *
 * Compare two startpoints to see if they are in the same context.
 * If compare_addresses==NEXUS_TRUE, then also compare if
 * they point to the same address.
 *
 * NOTE: this unconditionally checks the liba_flag
 *
 * Return NEXUS_TRUE if they are same, or NEXUS_FALSE if they are different.
 */
static int same_startpoint(nexus_startpoint_t *sp1, 
			   nexus_startpoint_t *sp2,
			   nexus_bool_t compare_addresses)
{
    nexus_byte_t *a1, *a2;
    
    NexusAssert2((sp1), ("same_startpoint(): rcvd NULL sp1\n"));
    NexusAssert2((sp2), ("same_startpoint(): rcvd NULL sp2\n"));

    if (nexus_startpoint_is_null(sp1) && nexus_startpoint_is_null(sp2))
    {
        /* Both are NULL global pointers */
        return(NEXUS_TRUE);
    }
    if (nexus_startpoint_is_null(sp1) || nexus_startpoint_is_null(sp2))
    {
        /* One is a NULL global pointer, the other isn't */
        return(NEXUS_FALSE);
    }

    /* Compare their liba's */
    a1 = (sp1->liba_is_inline ? sp1->liba.array : sp1->liba.pointer);
    a2 = (sp2->liba_is_inline ? sp2->liba.array : sp2->liba.pointer);

    /* Compare their node numbers and context differentiators */
    /*
     * TODO: Need to add a second context differentiator to handle
     * multiple contexts within the same process.
     */
    a1 = sp1->mi_proto->array;
    a2 = sp2->mi_proto->array;
    if (memcmp(a1, a2, 8) != 0)
    {
        return(NEXUS_FALSE);
    }

    /* Compare their node names */
    if (strcmp((char *) (a1+4), (char *) (a2+4)) != 0)
    {
        return(NEXUS_FALSE);
    }
    
    /* They are the same context */
    return(NEXUS_TRUE);
} /* same_startpoint() */



globus_i_nexus_endpoint_id_t
globus_i_nexus_endpoint_table_add(
    globus_nexus_endpoint_t *		endpoint)
{
    static
	globus_i_nexus_endpoint_id_t	next_id = 0;
    globus_i_nexus_endpoint_id_t	id;
	
    globus_mutex_lock(&globus_l_nexus_endpoint_table_lock);
    {
	while (globus_hashtable_lookup(&globus_l_nexus_endpoint_table,
				       (void *) next_id) != GLOBUS_NULL)
	{
	    next_id++;
	}

	id = next_id;
	globus_hashtable_insert(&globus_l_nexus_endpoint_table,
				(void *) id,
				(void *) endpoint);

	next_id++;
    }
    globus_mutex_unlock(&globus_l_nexus_endpoint_table_lock);

    return id;
    
}
/* globus_nexus_endpoint_table_add() */


int
globus_i_nexus_endpoint_table_remove(
    globus_nexus_endpoint_t *		endpoint)
{
    globus_nexus_endpoint_t *		ep;

    globus_mutex_lock(&globus_l_nexus_endpoint_table_lock);
    {
	ep = globus_hashtable_remove(&globus_l_nexus_endpoint_table,
				     (void *) endpoint->id);
    }
    globus_mutex_unlock(&globus_l_nexus_endpoint_table_lock);

	
    if (ep != GLOBUS_NULL)
    {
	return GLOBUS_SUCCESS;
    }
    else
    {
	return GLOBUS_FAILURE;
    }
}
/* globus_nexus_endpoint_table_remove() */

/*
 * globus_nexus_endpoint_table_lookup()
 */
int
globus_i_nexus_endpoint_table_lookup(
    globus_byte_t *			liba,
    globus_nexus_endpoint_t **		endpoint)
{
    globus_nexus_endpoint_t *		ep;
    globus_i_nexus_endpoint_id_t	id;

    LibaUnpack(liba, id);
    
    globus_mutex_lock(&globus_l_nexus_endpoint_table_lock);
    {
	ep = globus_hashtable_lookup(&globus_l_nexus_endpoint_table,
				     (void *) id);
    }
    globus_mutex_unlock(&globus_l_nexus_endpoint_table_lock);

    if (ep != GLOBUS_NULL)
    {
	*endpoint = ep;
	return GLOBUS_SUCCESS;
    }
    else
    {
	return GLOBUS_FAILURE;
    }
}
/* globus_nexus_endpoint_table_lookup() */

