/*
 * Copyright 1999-2006 University of Chicago
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**************************************************************************

 Transform        Example 
 Module:          transform_example.c

 Written at:      High-Performance Computing Laboratory
                  Department of Computer Science
                  Northern Illinois University
                  DeKalb, IL

 Original Author: Doug Sale 
                  July 1997

 Description:	  This module employs a mechanism for tracking messages
				  and messages lost.  Each transformed message is
				  assigned a sequence number by the transform module
				  from the startpoint state.  On receiving the message
				  the transform checks the sequence number against that
				  kept by the endpoint state.  This number should be one
				  greater than that of the last message sent.  In this
				  way the endpoint tracks the number of messages sent
				  and received.

				  The startpoint state contains a sequence number and the
				  address of its endpoint sequence node.  Everytime a
				  message is sent using said startpoint, its sequence
				  number is incremented and the sequence number and node
				  address are sent with the message as transform_info.

				  The endpoint state is a linked list of sequence nodes,
				  one for each startpoint bound to the endpoint.  The
				  sequence node contains a sequence number and the number
				  of messages lost.

				  This module also performs rudimentary data
				  transformation, adding 1 to each byte of the user data
				  during the transform and subtracting 1 from each byte
				  during the untransform.

**************************************************************************/

/**************************************************************************

Altering Nexus to Employ Transform Modules

 	When writing new transform modules for nexus, there are three 
 files that must be updated besides the transform module,
 transform_xxxxx.c.  These three files are nexus.h, initwrap.c, and 
 Makefile.in, all of which can be found in the nexus subdirectory of
 your original nexus installation.

	In nexus.h do a search for "TRANSFORM" and you should be looking
 at the section where transform modules' unique id's are defined.  The
 format is 

		#define NEXUS_TRANSFORM_XXXXX Y

 Add a new entry for your transform module.

	In initwrap.c there is a table that defines which module's function
 tables are to be loaded at initialization.  The function tables are 
 defined in the transform modules and accessed by calling the function
 _nx_transform_xxxxx_info which is provided by the transform module. In
 initwrap.c you must provide an external reference to
 _nx_transform_xxxxx_info and update the nexus_module_list_t to ensure
 your module is initialized and can be called by Nexus.  The external
 reference should look like

		extern void *_nx_transform_xxxxx_info(void);

 The line you add to the nexus_module_list_t should be of the form

		{"xxxxx", "transform", _nx_transform_xxxxx_info},

 and should come before the line

		{NULL, NULL, NULL}, 

	Finally, you must update Makefile.in so that your transform
 module will be included when you make Nexus.  In Makefile.in, do a search
 for transform_iface.  You should be looking at a line that reads

		transform_iface.$(OFILE) \

 After this line insert a line that reads

		transform_xxxxx.$(OFILE) \

 Now do another search for transform_iface.  You should now be looking at
 the make rule 

		transform_iface.$(OFILE): $(SRC_DIR_NEXUS)/transform_iface.c $(HDRS)
        		$(CC) $(NEXUS_CFLAGS) -c $(SRC_DIR_NEXUS)/transform_iface.c
 
 You will want to add a similar entry for your transform module.

		transform_xxxxx.$(OFILE): $(SRC_DIR_NEXUS)/transform_xxxxx.c $(HDRS)
        		$(CC) $(NEXUS_CFLAGS) -c $(SRC_DIR_NEXUS)/transform_xxxxx.c

 	Now you should be ready to compile and test your transform module.
 In the directory where you have built Nexus (where your object files
 reside) run ./config.status.  This will update your Makefile using
 Makefile.in.
	Once you have successfully made Nexus with your transform
 module, you will need to have an application to test it.  Any of the 
 example programs in the nexus_examples sub-directory of your Nexus
 installation should work.  Once an endpointattr is initialized using
      
 	nexus_endpointattr_init(nexus_endpointattr_t *endpointattr)

 a transform may be specified for that endpoint attribute using

  	nexus_endpointattr_set_transform(nexus_endpointattr_t *endpointattr,
									 int transform_type,
	  								 void *non-ep-specific_transform_info);
 
 Now any endpoints created from that endpoint attribute using

	nexus_endpoint_init(nexus_endpoint_t *endpoint,
                        nexus_endpointattr_t *endpointattr)

 and consequently any startpoints bound to said endpoints using 

	nexus_startpoint_bind(nexus_startpoint_t *startpoint,
                          nexus_endpoint_t *endpoint)

 will use the specified transform/untransform module when sending messages.
 An endpoint attribute's transform properties may also be altered by using

    nexus_endpointattr_set_transform(nexus_endpointattr_t *epa, int tid,
    		                         void *transform_info)
**************************************************************************/

#include "internal.h"
#include "nexus_dc.h"
#include <nexus.h>

#define TRANSFORM_BLOCKSIZE	8
#define LARGEST_POSSIBLE_SIZEOF_ULONG 8

/********************************/
/* Externally Visible Functions */
/********************************/

void *_nx_transform_example_info(void);

/******************************/
/* Transform Table Functions */
/*****************************/

static int transform_example_transform_id();
static void transform_example_init(
			    nexus_bool_t   *modifies_data,
			    unsigned long  *header_size,
			    unsigned long  *trailer_size);
static void transform_example_init_endpoint_state(nexus_transformattr_t *attr, 
					    nexus_transformstate_t **ep_state);
static void transform_example_destroy_endpoint_state(
					    nexus_transformstate_t *ep_state);
static void transform_example_init_startpoint_state(
					    nexus_transformstate_t *ep_state,
					    nexus_transformstate_t **sp_state,
					    nexus_bool_t  *copy_sp_locally,
					    nexus_bool_t  *destroy_sp_locally);
static void transform_example_destroy_startpoint_state(
					    nexus_transformstate_t *sp_state);
static void transform_example_copy_startpoint_state(
					nexus_transformstate_t *sp_state,
					nexus_transformstate_t **sp_state_copy);
static int transform_example_sizeof_startpoint_state(nexus_transformstate_t *s);
static void transform_example_put_startpoint_state(nexus_byte_t **buffer, 
						nexus_transformstate_t *state);
static void transform_example_get_startpoint_state(nexus_byte_t **buffer, 
						int format, 
						nexus_transformstate_t **state);
static int transform_example_transform(nexus_transformstate_t *startpoint_state,
                                        nexus_byte_t *storage_start,
                                        unsigned long storage_size,
                                        nexus_byte_t *data_start,
                                        unsigned long *data_size,
                                        nexus_bool_t must_alloc_new_buffer,
                                        nexus_byte_t *transform_info_start,
                                        nexus_byte_t **out_storage_start,
                                        unsigned long *out_storage_size,
                                        nexus_byte_t **out_data_start,
                                        unsigned long *out_data_size);

static int transform_example_untransform(nexus_transformstate_t *endpoint_state,
					 nexus_byte_t *data_start,
					 unsigned long *data_size,
					 nexus_byte_t *transform_info_start,
					 int format,
					 nexus_byte_t **destination_start,
					 unsigned long *destination_size);
 
/*******************/
/* Data Structures */
/*******************/

typedef struct _sequence_node_t
{
    struct _sequence_node_t *next;
    
    /* number of the message last received */           
    unsigned long sequence;          
    
    /* count of messages lost */                           
    unsigned long msg_lost;                           
} sequence_node_t;

typedef struct _endpoint_transform_state_t
{
    /* singly linked list of sequence nodes, one for each startpoint bound to me  */
    sequence_node_t *sequence_hdr;

} endpoint_transform_state_t; 

typedef struct _startpoint_transform_state_t
{
    /* address of my sequence node on endpoint side */
    nexus_byte_t ep_sequence_liba[LARGEST_POSSIBLE_SIZEOF_ULONG];

    /* number of the message last sent */           
    unsigned long sequence;          
} startpoint_transform_state_t;

/**************************/
/* Local Global Variables */
/**************************/

static nexus_transform_funcs_t transform_example_funcs =
{
    transform_example_transform_id,
    transform_example_init,
    NULL, /* transform_example_shutdown */
    NULL, /* transform_example_transformattr_init, */
    NULL, /* transform_example_transformattr_destroy, */
    NULL, /* transform_example_transformattr_get_info */
    transform_example_init_endpoint_state,
    transform_example_destroy_endpoint_state,
    NULL, /* transform_example_update_endpoint_state */
    transform_example_init_startpoint_state,
    NULL, /* transform_example_copy_startpoint_state, */
	  /* do not provide copy_sp routine when copy_locally = NEXUS_FALSE */
    transform_example_destroy_startpoint_state,
    transform_example_sizeof_startpoint_state,
    transform_example_put_startpoint_state,
    transform_example_get_startpoint_state,
    transform_example_transform,
    transform_example_untransform
};

/********************************/
/* Externally Visible Functions */
/********************************/

/*
 * _nx_transform_example_info()
 *
 * Return the function table for this module.
 */
void *_nx_transform_example_info(void)
{
    return ((void *) (&transform_example_funcs));
}

/*******************/
/* Local Functions */
/*******************/

/*
 * transform_example_transform_id()
 *
 * Return the #define of this transform module 
 */
static int transform_example_transform_id()
{
    return(NEXUS_TRANSFORM_EXAMPLE);
} /* transform_example_transform_id() */

/*
 * transform_example_init()
 *
 *  Written if the transform module needs to perform initialization code or needs
 *  anything from the command line. 
 * 
 *  modifies_data - does the transform alter user data?
 *  header_size - the size of the transform_info (key, encryption init vector, etc.)
 *  trailer_size - a *SUGGESTION* of how much padding would be useful at the end 
 *                 of the buffer to avoid allocating a new buffer, NOT GUARANTEED
 */
static void transform_example_init(
                                nexus_bool_t   *modifies_data,
                                unsigned long  *header_size,   
                                unsigned long  *trailer_size)
{
    NexusAssert2((modifies_data), 
	("transform_example_init(): rcvd NULL modifies_data\n"));
    NexusAssert2((header_size), 
	("transform_example_init(): rcvd NULL header_size\n"));
    NexusAssert2((trailer_size), 
	("transform_example_init(): rcvd NULL trailer_size\n"));

    NexusAssert2((sizeof(unsigned long) <= LARGEST_POSSIBLE_SIZEOF_ULONG), 
("transform_example_init(): detected sizeof(unsigned long) > LARGEST_POSSIBLE_SIZEOF_ULONG\n"));

    /* adds 1 to every byte of msg data */
    *modifies_data = NEXUS_TRUE;

    /* ep_seq_liba and sequence */
    *header_size = 2 * LARGEST_POSSIBLE_SIZEOF_ULONG;

    /* would like to have enough extra bytes to pad to TRANSFORM_BLOCKSIZE-byte 
    boundary without allocating a new buffer */
     *trailer_size = TRANSFORM_BLOCKSIZE - 1;

} /* transform_example_init() */

/*
 * void (*shutdown)(void);
 *
 * Written if transform module needs to do anything during normal termination,
 * e.g., notify any servers that it might have registered with.  In most 
 * modules this function will not be written.
 */

/*
 * transform_example_init_endpoint_state()
 *
 * This gets called when an endpoint is being initialized.  It's single arg
 * is the transform data from the endpointattr this endpoint is being 
 * initialized with.  
 * This function must nexus_malloc enough space in a single block to
 * hold ALL the transform information for this endpoint.  It then copies
 * the attr stuff from endpointattr, if appropriate, and fills the other
 * fields.
 */
static void transform_example_init_endpoint_state(nexus_transformattr_t *attr,
					      nexus_transformstate_t **ep_state)
{
nexus_printf("transform_example_init_endpoint_state\n");   

    /* NOTE ignoring attr for EXAMPLE */
    NexusAssert2((ep_state), 
	("transform_example_init_endpoint_state(): rcvd NULL ep_state\n"));

    *ep_state = (endpoint_transform_state_t *) 
		    nexus_malloc(sizeof(endpoint_transform_state_t));

    ((endpoint_transform_state_t *) (*ep_state))->sequence_hdr = 
							(sequence_node_t *) NULL;
} /* transform_example_init_endpoint_state() */

/*
 * transform_example_destroy_endpoint_state()
 *
 * Called when destroying an endpoint.  ep_state is a pointer to the block that
 * was nexus_malloc'd in init_endpoint_state.  This function needs to 
 * nexus_free any memory in state (if complex structure with nexus_malloc'd 
 * fields) and then nexus_free(state).
 */
static void transform_example_destroy_endpoint_state(
					    nexus_transformstate_t *ep_state)
{
    sequence_node_t *seq_node;

    nexus_printf("transform_example_destroy_endpoint_state\n");

    NexusAssert2((ep_state), 
	("transform_example_destroy_endpoint_state(): rcvd NULL ep_state\n"));

    while (((endpoint_transform_state_t *) (ep_state))->sequence_hdr)
    {
		seq_node = ((endpoint_transform_state_t *) (ep_state))->sequence_hdr;
		((endpoint_transform_state_t *) (ep_state))->sequence_hdr = seq_node->next;
		nexus_free(seq_node);
    } /* endwhile */

    nexus_free(ep_state);

} /* transform_example_destroy_endpoint_state() */

/*
 * transform_example_init_startpoint_state()
 *
 * This gets called when an startpoint is bound to an endpoint.  The single
 * arg is the transform state info from the endpoint.  This function needs
 * to nexus_malloc enough memory for a startpoint to perform transformation
 * and copy info from state as necessary.
 */
static void transform_example_init_startpoint_state(
					    nexus_transformstate_t *ep_state,
					    nexus_transformstate_t **sp_state,
					    nexus_bool_t  *copy_sp_locally,
					    nexus_bool_t  *destroy_sp_locally)
{
    sequence_node_t *seq_node;
    unsigned long temp_ulong;

    nexus_printf("transform_example_init_startpoint_state\n");

    NexusAssert2((ep_state), 
	("transform_example_init_startpoint_state(): rcvd NULL ep_state\n"));
    NexusAssert2((sp_state), 
	("transform_example_init_startpoint_state(): rcvd NULL sp_state\n"));
    NexusAssert2((copy_sp_locally), 
	("transform_example_init_startpoint_state(): rcvd NULL copy_sp_locally\n"));
    NexusAssert2((destroy_sp_locally), 
    ("transform_example_init_startpoint_state(): rcvd NULL destroy_sp_locally\n"));

    /* instantiate endpoint's sequence node for this startpoint state */
    seq_node = (sequence_node_t *) nexus_malloc(sizeof(sequence_node_t));
    temp_ulong = (unsigned long) seq_node;

    /* instantiate startpoint state */
    *sp_state = (startpoint_transform_state_t *) 
		nexus_malloc(sizeof(startpoint_transform_state_t));

    /* bind startpoint to endpoint */
    memcpy((void *)((startpoint_transform_state_t *) (*sp_state))->ep_sequence_liba,
	    (void *) &temp_ulong,
	    sizeof(unsigned long));

    /* initialize endpoint's sequence node and this startpoint state */
    seq_node->sequence = 0;
    seq_node->msg_lost = 0;
    ((startpoint_transform_state_t *)(*sp_state))->sequence = 0;

    /* inserting new sequence node into ep linked list */
    seq_node->next = ((endpoint_transform_state_t *) ep_state)->sequence_hdr;
    ((endpoint_transform_state_t *) ep_state)->sequence_hdr = seq_node;

    *copy_sp_locally = NEXUS_FALSE;
    *destroy_sp_locally = NEXUS_FALSE;
} /* transform_example_init_startpoint_state() */

/*
 * transform_example_destroy_startpoint_state()
 *
 * This gets called when a startpoint is destroyed.  It is passed the transform
 * data that was nexus_malloc'd in init_startpoint_state.  It needs to 
 * nexus_free() this stuff.
 */
static void transform_example_destroy_startpoint_state(
					    nexus_transformstate_t *sp_state)
{
    nexus_printf("transform_example_destroy_startpoint_state\n");

    NexusAssert2((sp_state), 
	("transform_example_destroy_startpoint_state(): rcvd NULL sp_state\n"));

    nexus_free(sp_state); 
} /* transform_example_destroy_startpoint_state() */

/*
 * void *(*copy_startpoint_state) (void *state);
 *
 * This wants a copy of a startpoints transformation data.  It simply 
 * nexus_malloc's an appropriate block, copies from state, and returns
 * new block as a void *.
 */

/*
 * transform_example_sizeof_startpoint_state()
 *
 * This wants to know the number of bytes a startpoint's transformation
 * state requires.  It returns that value as an integer.  
 * In general, a transformation state will be made up from simple data
 * fields.  The value should be calculated by calling nexus_sizeof_XXX
 * for each field.
 * For example, the transform module using compression only requires a single 
 * int.  This function in that module simply returns nexus_sizeof_int(1).
 * Transform modules requiring 8-byte keys return nexus_sizeof_byte(8).
 * Those transform modules using compression and encryption should
 * return nexus_sizeof_int(1)+nexus_sizeof_byte(8).  
 * The nexus_sizeof_XXX() macros for all fundamental data types is in
 * nexus.h.
 */
static int transform_example_sizeof_startpoint_state(nexus_transformstate_t *s)
{
    NexusAssert2((s), 
	("transform_example_sizeof_startpoint_state(): rcvd NULL s\n"));
    
    /* ep_sequence_liba and sequence */
    return (2 * nexus_sizeof_byte(LARGEST_POSSIBLE_SIZEOF_ULONG));
} /* transform_example_sizeof_startpoint_state() */

/*
 * transform_example_put_startpoint_state()
 *
 * The user wants to put a startpoint's transformation information into
 * a buffer.  Just like sizeof_state above, this is simply a putting
 * of the individual fields of the state. 
 *
 * All puts should use the fundametal nexus_dc_put_xxx() macros found
 * in nexus.h.  For example, a transform module using compression (one int)
 * and encryption (8 bytes) should call:
 *
 * nexus_dc_put_int(buffer, &(state->compression_level), 1);
 * nexus_dc_put_byte(buffer, &(state->key), 8);
 */
static void transform_example_put_startpoint_state(nexus_byte_t **buffer, 
						nexus_transformstate_t *state)
{
    nexus_printf("transform_example_put_startpoint_state\n");

    NexusAssert2((buffer), 
	("transform_example_put_startpoint_state(): rcvd NULL buffer\n"));
    NexusAssert2((state), 
	("transform_example_put_startpoint_state(): rcvd NULL state\n"));

    /* need to place ep_sequence_liba and sequence into buffer */
    nexus_dc_put_byte(buffer, 
		    ((startpoint_transform_state_t *) state)->ep_sequence_liba, 
		    LARGEST_POSSIBLE_SIZEOF_ULONG);
    nexus_dc_put_byte(buffer, 
		    &(((startpoint_transform_state_t *) state)->sequence), 
		    LARGEST_POSSIBLE_SIZEOF_ULONG);

} /* transform_example_put_startpoint_state() */

/*
 * transform_example_get_startpoint_state()
 *
 * This is the exact same thing as put_state, instead we are extracting
 * information from a buffer and placing it into a startpoint.  Two
 * things should be pointed out here:
 *
 * 1. use nexus_dc_get_XXXX found in nexus.h
 *	   and
 * 2. the information MUST be extracted in the same order that it was 
 *    placed in put_state().
 */
static void transform_example_get_startpoint_state(nexus_byte_t **buffer, 
						int format, 
						nexus_transformstate_t **state)
{
    nexus_printf("transform_example_get_startpoint_state\n");

    NexusAssert2((buffer), 
	("transform_example_get_startpoint_state(): rcvd NULL buffer\n"));
    NexusAssert2((state), 
	("transform_example_get_startpoint_state(): rcvd NULL state\n"));

    *state = (startpoint_transform_state_t *) 
		nexus_malloc(sizeof(startpoint_transform_state_t));

    nexus_dc_get_byte(buffer, 
	(unsigned char *) ((startpoint_transform_state_t *) (*state))->ep_sequence_liba,
			LARGEST_POSSIBLE_SIZEOF_ULONG, 
			format); 
    nexus_dc_get_byte(buffer, 
	(unsigned char *) &(((startpoint_transform_state_t *) (*state))->sequence),
			LARGEST_POSSIBLE_SIZEOF_ULONG, 
			format); 

} /* transform_example_get_startpoint_state() */

/*
 * transform_example_transform()
 */
static int transform_example_transform(nexus_transformstate_t *startpoint_state,
                                        nexus_byte_t *storage_start,
                                        unsigned long storage_size,
                                        nexus_byte_t *data_start,
                                        unsigned long *data_size,
                                        nexus_bool_t must_alloc_new_buffer,
                                        nexus_byte_t *transform_info_start,
                                        nexus_byte_t **out_storage_start,
                                        unsigned long *out_storage_size,
                                        nexus_byte_t **out_data_start,
                                        unsigned long *out_data_size)
/**************************************************************************

     storage_start
     |
     |           data_start
     |                |
     v                v
     +----------------+-----------------------+----------------+
     |  unused space  |       user data       |  unused space  |
     +----------------+-----------------------+----------------+

		      |<----- data_size ----->|

     |<--------------------- storage_size -------------------->|

Transform decides whether or not it needs to allocate a new buffer
in the same way it did before, that is, it allocates a new buffer
if must_alloc_new_buffer == NEXUS_TRUE or if there's not enough space
to transform in place.  

If it doesn't need to allocate a new buffer, then simply transform in
place, and for MD5 place the MAC into transform_info_start. 

If it DOES need to allocate a new buffer, then you set all the
out_* args to draw the following picture:

     *out_storage_start
     |
     |           *out_data_start
     |                |
     v                v
     +----------------+-----------------------+
     |  unused space  |       user data       |
     +----------------+-----------------------+

		      |<-- *out_data_size --->|

     |<-------- *out_storage_size ----------->|

You must make sure that the leading "unused space" in the newly allocated
buffer is exactly the same size as it is in the original buffer, that is,
data_start - storage_start = *out_data_start - *out_storage_start.

Note that you never allocate new space for transform_info, even if you
allocate space for a new buffer.

If you allocated a new buffer, then simply tranform into that buffer.

**************************************************************************/
{
    int i,
        transform_size,
  		needed_unused_size,
		current_unused_size;
    char *source,
		 *dest;

/*    nexus_printf("transform_example_transform()\n"); */

    NexusAssert2((startpoint_state),
        ("transform_ecb_transform(): rcvd NULL startpoint_state\n"));   
    NexusAssert2((storage_start), ("transform_ecb_transform(): rcvd NULL storage_start\n"));
    NexusAssert2((data_start),
        ("transform_ecb_transform(): rcvd NULL data_start\n"));
    NexusAssert2((data_size),
        ("transform_ecb_transform(): rcvd NULL data_size\n"));
    NexusAssert2((transform_info_start),
        ("transform_ecb_transform(): rcvd NULL transform_info_start\n"));
    NexusAssert2((out_storage_start),
        ("transform_ecb_transform(): rcvd NULL out_storage_start\n"));
    NexusAssert2((out_storage_size),
        ("transform_ecb_transform(): rcvd NULL out_storage_size\n"));
    NexusAssert2((out_data_start),
        ("transform_ecb_transform(): rcvd NULL out_data_start\n"));
    NexusAssert2((out_data_size),
        ("transform_ecb_transform(): rcvd NULL out_data_size\n"));

    /* increment sequence */
    ((startpoint_transform_state_t *)(startpoint_state))->sequence += 1;
/*
nexus_printf("transform_example_transform(): sp_sequence = %d\n", 
	((startpoint_transform_state_t *)(startpoint_state))->sequence);
*/

    /* placing ep_sequence_liba and sequence into transform_info */
    memcpy((void *) transform_info_start,
	(void *) ((startpoint_transform_state_t *) (startpoint_state))->ep_sequence_liba,
	LARGEST_POSSIBLE_SIZEOF_ULONG);
	memcpy((void *) (transform_info_start + LARGEST_POSSIBLE_SIZEOF_ULONG),
	(void *) &(((startpoint_transform_state_t *) (startpoint_state))->sequence),
	LARGEST_POSSIBLE_SIZEOF_ULONG);

    /* make transform_size a multiple of TRANSFORM_BLOCKSIZE */
    transform_size = *data_size;
    if ((needed_unused_size = TRANSFORM_BLOCKSIZE - (transform_size % TRANSFORM_BLOCKSIZE))
								== TRANSFORM_BLOCKSIZE) 
    {
		needed_unused_size = 0;
    }
    transform_size += needed_unused_size;
    current_unused_size = storage_size - (data_start - storage_start + *data_size);

    /* determine if a new buffer is needed */
    if (must_alloc_new_buffer || (needed_unused_size > current_unused_size))
    {
/* nexus_printf("transform_example_transform(): allocating new buffer\n"); */

		/* must allocate new buffer */
		*out_storage_size = data_start - storage_start + transform_size;
		*out_storage_start = (nexus_byte_t *) nexus_malloc(*out_storage_size);
		*out_data_start = *out_storage_start + (data_start - storage_start);
		*out_data_size = transform_size;

		/* transform from old buffer to new */
		source = (char *) data_start;
		dest = (char *) *out_data_start;
    }
    else
    {
/* nexus_printf("transform_example_transform(): NOT allocating new buffer\n"); */

		/* no need to allocate new buffer */
		*out_storage_start = NULL;
		*data_size = transform_size;

		/* transform in place */
		source = dest = (char *) data_start;

    } /* endif */

    /* transform user-data ... perhaps in place */
    for (i = 0; i < transform_size; i++, source++, dest++)
    {
		*dest = *source + 1;
    } 

/* nexus_printf("exit transform_example_transform()\n"); */

    return(NEXUS_SUCCESS);

} /* transform_example_transform() */

/*
 * transform_example_untransform()
 */
static int transform_example_untransform(nexus_transformstate_t *endpoint_state,
					 nexus_byte_t *data_start,
					 unsigned long *data_size,
					 nexus_byte_t *transform_info_start,
					 int format,
					 nexus_byte_t **destination_start,
					 unsigned long *destination_size)
/**************************************************************************

     data_start
	  |
	  v
	  +-----------------------+
	  |       user data       |
	  +-----------------------+

	  |<----- data_size ----->|


The untranform process works like this:

    if (*dest_start == NULL)
    {
		if (enough room to untransform in place)
		{
	   		untransform in place
		}
		else
		{
	    	*destination_start = nexus_malloc(enough to untransform)
	    	*destination_size = amount just nexus_malloc'd
	    	untransform into newly nexus_malloc'd memory
		} 
    }
    else
    {
		if (*destination_size == EXACT amount needed to untransform)
		{
	    	untransform into *destination_size
		}
		else
		{
	    	DO NOTHING ... return non-zero value
		} 
    } 
**************************************************************************/
{
    int 			i,
        			data_format;
    unsigned long 	temp_ulong,
      		  	  	sp_sequence;
    sequence_node_t *seq_node;
    char 			*source,
	 				*dest;

  
/*    nexus_printf("transform_example_untransform()\n"); */

    NexusAssert2((endpoint_state),
        ("transform_ecb_untransform(): rcvd NULL endpoint_state\n"));
    NexusAssert2((data_start),
        ("transform_ecb_untransform(): rcvd NULL data_start\n"));
    NexusAssert2((data_size),
        ("transform_ecb_untransform(): rcvd NULL data_size\n"));
    NexusAssert2((transform_info_start),
        ("transform_ecb_untransform(): rcvd NULL transform_info_start\n"));
    NexusAssert2((destination_start),
        ("transform_ecb_untransform(): rcvd NULL destination_start\n"));
    NexusAssert2((destination_size),
        ("transform_ecb_untransform(): rcvd NULL destination_size\n"));

    /* extracting ep_sequence_liba and sequence from transform_info */
    memcpy((void *) &temp_ulong, 
	    (void *) transform_info_start, 
	    sizeof(unsigned long));
    seq_node = (sequence_node_t *) temp_ulong;
    memcpy((void *) &sp_sequence, 
	    (void *) (transform_info_start + LARGEST_POSSIBLE_SIZEOF_ULONG), 
	    sizeof(unsigned long));

/* nexus_printf("transform_example_untransform(): received sp sequence = %d\n", sp_sequence); */

    NexusAssert2((seq_node), 
    ("transform_example_untransform(): extracted NULL sequence node pointer from buffer\n"));

#ifdef BUILD_DEBUG
{
    /* verifying that seq_node is still in this endpoints sequence_node list */

    sequence_node_t *temp_seq_node;

    if (NexusDebug(1))
    {
        nexus_printf("transform_example_untransform(): SEARCHING verifying sequence %x is in endpoint's sequence list\n", seq_node);
    } /* endif */

    for (temp_seq_node = ((endpoint_transform_state_t *) endpoint_state)->sequence_hdr;
	temp_seq_node && temp_seq_node != seq_node;
	    temp_seq_node = temp_seq_node->next)
	/* do nothing */ ;

    if (!temp_seq_node)
    {
    	globus_fatal("transform_example_untransform() could not find sequence node in ep's list\n");
    } 
    else if (NexusDebug(1))
    {
		nexus_printf("transform_example_untransform(): FOUND sequence node %x in endpoint's list\n", seq_node);
    } /* endif */
}
#endif

/* nexus_printf("transform_example_untransform(): endpoint sequence node's sequence = %d\n",
			 seq_node->sequence); */

    /* take care of sequence and lost message information */
    if (sp_sequence == (seq_node->sequence + 1))
    {
		/* received the message number we expected */
		seq_node->sequence = sp_sequence;
    }
    else if (sp_sequence > (seq_node->sequence + 1))
    {
		/* received a message number greater than we expected -- some messages lost */
		seq_node->msg_lost = sp_sequence - (seq_node-> sequence + 1);
		seq_node->sequence = sp_sequence;
    }
    else
    {
		/* received a message we earlier counted as lost */
		seq_node->msg_lost -= 1;
    }

/*nexus_printf("transform_example_untransform(): endpoint sequence node's msg_lost = %d\n",
			 seq_node->msg_lost);*/

    /* making sure data to transform is a multiple of TRANSFORM_BLOCKSIZE  */
    if (*data_size % TRANSFORM_BLOCKSIZE != 0) 
    {
		globus_fatal("transform_example_untransform(): data_size not multiple of TRANSFORM_BLOCKSIZE\n");
    }

    /* determining if we can decrypt in place */
    if (!(*destination_start))
    {
/* nexus_printf("transform_example_untransform(): decrypting in place\n"); */

		/* dec in place */
		source = dest = (char *) data_start;
    }
    else
    {
		if (*destination_size == *data_size)
 		{
/* nexus_printf("transform_example_untransform(): decrypting to destination_start\n"); */

	    	/* dec to new buffer */
	    	source = (char *) data_start;
	    	dest = (char *) *destination_start;
		}
		else
		{
/* nexus_printf("transform_example_untransform(): not decrypting\n"); */
/* nexus_printf("exit transform_example_untransform()\n"); */

	    	/* do nothing */
	    	return (!NEXUS_SUCCESS);
		}
    } /* endif */

    /* untransform user-data ... perhaps in place */
    for (i = 0; i < *data_size; i++, source++, dest++)
    {
		*dest = *source - 1;
    } 

/* nexus_printf("exit transform_example_untransform()\n"); */

    return(NEXUS_SUCCESS);

} /* transform_example_untransform */

