/*
 * Copyright 1999-2006 University of Chicago
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <globus_io.h>
#include "globus_rls_client.h"
#include "globus_rls_rpc.h"
#include "misc.h"
#include "lock.h"
#include "db.h"
#include "event.h"
#include "lrc.h"
#include "bloom.h"
#include "update.h"
#include <syslog.h>
#include <string.h>
#include <fnmatch.h>

#define METHOD_RLI_UPDATE	"rli_update"
#define METHOD_RLI_UPDATE_LEN	sizeof(METHOD_RLI_UPDATE)

#define METHOD_RLI_UPDATEBF	"rli_updatebf"
#define METHOD_RLI_UPDATEBF_LEN	sizeof(METHOD_RLI_UPDATEBF)

/* Minimum size of bloom filter	*/
#define MINBFSIZE		10000

extern int		lrc_bloomfilter_numhash;
extern int		lrc_bloomfilter_ratio;
extern int		loglevel;
extern char		*lrc_dbname;
extern char		*db_user;
extern char		*db_pwd;
extern char		*myurl;
extern int		myurllen;
extern int		update_buftime;
extern int		update_immediate;
extern int		update_bf_int;
extern int		update_ll_int;
extern int		update_factor;
static int		bloomfiltercb(void *a, char *lfn);
static void		*lrc_llupdates(void *a);
static void		*lrc_bfupdates(void *a);
static void		*lrc_bfiupdates(void *a);
static int		lrc_update_ll_cb(void *a, char *lfn);
extern TABLE		table[];
bloomfilter_t		bloomfilter;
RLI			*lrc_rlilist;
LOCK			lrc_rlilistlock;

#ifdef COUNTUPDATE
extern int		updatebytes;
#endif

int
lrc_init(char *dbname, char *db_user, char *db_pwd)

{
  int	rc;
  void	*lrch;
  RLI	*rli;
  char	errbuf[MAXERRMSG];

  if (loglevel > 1)
    logit(LOG_DEBUG, "lrc_init: %s %s", dbname, db_user);

  if ((rc = rls_lock_init(&lrc_rlilistlock)) != GLOBUS_RLS_SUCCESS)
    return rc;
  lrc_rlilist = NULL;

  bf_init(&bloomfilter, 0, 0, 0);
  globus_mutex_init(&bloomfilter.mtx, GLOBUS_NULL);

  /*
   * Get list of RLIs we update from db.
   */
  if ((rc = db_open(dbname, db_user, db_pwd, 1, &lrch,
		    errbuf)) != GLOBUS_RLS_SUCCESS) {
    logit(LOG_WARNING, "lrc_init(%s): %s", dbname, errbuf);
    return rc;
  }
  update_readrli(lrch, &lrc_rlilistlock, &lrc_rlilist);
  rls_lock_get(&lrc_rlilistlock, readlock);
  for (rli = lrc_rlilist; rli; rli = rli->nxt)
    if (rli->flags & FR_BLOOMFILTER) {
      lrc_getbf(lrch);
      break;
    }
  rls_lock_release(&lrc_rlilistlock, readlock);
  db_close(lrch);

  /*
   * Queue periodic events (lfn list and bloom filter updates).
   */
  event_queue(lrc_llupdates, &update_ll_int);
  event_queue(lrc_bfupdates, &update_bf_int);
  if (update_immediate)
    event_queue(lrc_bfiupdates, &update_buftime);

  if (loglevel > 1)
    logit(LOG_DEBUG, "lrc_init succeeded");
  return GLOBUS_RLS_SUCCESS;
}

void
lrc_end()

{
  if (event_exists(lrc_llupdates))
    event_cancel(lrc_llupdates);
  if (event_exists(lrc_bfupdates))
    event_cancel(lrc_bfupdates);
  if (event_exists(lrc_bfiupdates))
    event_cancel(lrc_bfiupdates);
  if (lrc_rlilist) {
    rls_lock_get(&lrc_rlilistlock, writelock);
    update_freelist(&lrc_rlilist);
    rls_lock_release(&lrc_rlilistlock, writelock);
  }
  globus_mutex_destroy(&bloomfilter.mtx);
  bf_free(&bloomfilter);
  rls_lock_destroy(&lrc_rlilistlock);
}

void
lrc_getbf(void *lrch)

{
  int		bfsize;
  time_t	now;
  int		rc;
  char		buf[MAXERRMSG];
  char		errbuf[MAXERRMSG];

  if (loglevel)
    logit(LOG_DEBUG, "lrc_getbf:");
  bloomfilter.startlfns = table[T_LRCLFN].count;
  bfsize = bloomfilter.startlfns * lrc_bloomfilter_ratio;
  if (bfsize < MINBFSIZE)
    bfsize = MINBFSIZE;
  if (bf_init(&bloomfilter, bfsize, lrc_bloomfilter_numhash,
	      1) != GLOBUS_RLS_SUCCESS) {
    logit(LOG_WARNING, "lrc_getbf: bf_init(%d): No memory", bfsize);
    return;
  }
  if (loglevel)
    now = time(0);
  globus_mutex_lock(&bloomfilter.mtx);
  if ((rc = db_lrc_alllfn(lrch, bloomfiltercb, NULL,
			  errbuf)) != GLOBUS_RLS_SUCCESS)
    logit(LOG_WARNING, "lrc_getbf: db_lrc_alllfn: %s",
	  globus_rls_errmsg(rc, errbuf, buf, MAXERRMSG));
  else if (loglevel)
    logit(LOG_DEBUG, "lrc_getbf: Computing bloom filter took %d seconds",
	  time(0) - now);
  globus_mutex_unlock(&bloomfilter.mtx);
}

static int
bloomfiltercb(void *a, char *lfn)

{
  bf_addlfn(&bloomfilter, lfn);
  return GLOBUS_TRUE;
}

/*
 * Requeue update events to run immediately.
 */
void
lrc_updatenow()

{
  event_now(lrc_llupdates);
  event_now(lrc_bfupdates);
}

static void *
lrc_bfupdates(void *a)

{
  EVENTCONTROL	*ec = (EVENTCONTROL *) a;
  RLI		*rli;
  int		ui;

  while (1) {
    if (loglevel)
      logit(LOG_DEBUG, "lrc_bfupdates:");

    rls_lock_get(&lrc_rlilistlock, readlock);
    for (rli = lrc_rlilist; rli; rli = rli->nxt) {
      if (!(rli->flags & FR_BLOOMFILTER))
	continue;
      /*
       * Softstatestart and lastupdate will be the same if
       * last update succeeded, hence we update only every
       * update_factor multiples of update_bf_int.
       */
      if (update_immediate && rli->softstatestart == rli->lastupdate)
	ui = update_bf_int * update_factor;
      else
	ui = update_bf_int;
      if (rli->softstatestart && time(0) - rli->lastupdate < ui)
	continue;
      update_sendbf(rli, &bloomfilter, NULL);
    }

    rls_lock_release(&lrc_rlilistlock, readlock);

    globus_mutex_lock(&bloomfilter.mtx);
    bloomfilter.flags &= ~BF_NEEDUPDATE;
    globus_mutex_unlock(&bloomfilter.mtx);

    globus_mutex_lock(&ec->mtx);
    while (!(ec->flags & (EF_EXIT|EF_RUN)))
      globus_cond_wait(&ec->cond, &ec->mtx);

    if (ec->flags & EF_EXIT) {
      ec->flags &= ~EF_RUNNING;
      globus_cond_signal(&ec->cond);
      globus_mutex_unlock(&ec->mtx);
      pthread_exit(0);
    } else {
      ec->flags &= ~EF_RUN;
      globus_mutex_unlock(&ec->mtx);
    }

    /*
     * Immediate mode may have been enabled since we started, queue
     * immediate bloomfilter updates if it's not running.
     */
    if (update_immediate && !event_exists(lrc_bfiupdates))
      event_queue(lrc_bfiupdates, &update_buftime);
  }
}

/*
 * Client (LRC) side bloom filter update.  Called every update_buftime seconds
 * when immediate mode is enabled to check if the bloom filter has been
 * modified, if it is it is sent to RLIs.  Note this differs from immediate
 * mode when using LFN lists, in that case only the changed LFNs are
 * sent, here we send entire bloom filter.
 */
static void *
lrc_bfiupdates(void *a)

{
  EVENTCONTROL	*ec = (EVENTCONTROL *) a;
  RLI		*rli;

  while (1) {
    if (loglevel)
      logit(LOG_DEBUG, "lrc_bfiupdates: %X", bloomfilter.flags);

    if (bloomfilter.flags & BF_NEEDUPDATE) {
      rls_lock_get(&lrc_rlilistlock, readlock);
      for (rli = lrc_rlilist; rli; rli = rli->nxt)
	if (rli->flags & FR_BLOOMFILTER)
	  update_sendbf(rli, &bloomfilter, NULL);
      rls_lock_release(&lrc_rlilistlock, readlock);

      globus_mutex_lock(&bloomfilter.mtx);
      bloomfilter.flags &= ~BF_NEEDUPDATE;
      globus_mutex_unlock(&bloomfilter.mtx);
    }

    globus_mutex_lock(&ec->mtx);
    if (!update_immediate)
      ec->flags |= EF_EXIT;
    while (!(ec->flags & (EF_EXIT|EF_RUN)))
      globus_cond_wait(&ec->cond, &ec->mtx);

    if (ec->flags & EF_EXIT) {
      ec->flags &= ~EF_RUNNING;
      globus_cond_signal(&ec->cond);
      globus_mutex_unlock(&ec->mtx);
      pthread_exit(0);
    } else {
      ec->flags &= ~EF_RUN;
      globus_mutex_unlock(&ec->mtx);
    }
  }
}

/*
 * Read entire lfn table and queue updates to appropriate RLI servers.
 * An RLI is sent the softstate update if we're not in immediate mode,
 * or we're out of sync with the RLI server (cause a previous update failed),
 * or it's been update * update_factor seconds since the last
 * full update.
 */
static void *
lrc_llupdates(void *a)

{
  EVENTCONTROL	*ec = (EVENTCONTROL *) a;
  int		rc;
  void		*lrch = NULL;
  char		errbuf[MAXERRMSG];
  char		errbuf2[MAXERRMSG];
  RLI		*rli;
  time_t	now;
  int		unsynced;
  int		ui;

  while (1) {
    if (loglevel)
      logit(LOG_DEBUG, "lrc_llupdates: db %s", lrch ? "open" : "closed");

    if (!lrch) {
      if ((rc = db_open(lrc_dbname, db_user, db_pwd, 1, &lrch,
			errbuf)) != GLOBUS_RLS_SUCCESS) {
	logit(LOG_WARNING, "lrc_llupdates(%s): %s", lrc_dbname, errbuf);
	lrch = NULL;
	goto done1;
      }
    }

    rls_lock_get(&lrc_rlilistlock, readlock);

    /*
     * Decide which RLIs will get updated now.
     */
    now = time(0);
    unsynced = 0;
    for (rli = lrc_rlilist; rli; rli = rli->nxt) {
      if (rli->flags & FR_BLOOMFILTER)
	continue;
      globus_mutex_lock(&rli->mtx);
      if (update_immediate && (rli->flags & FR_SYNCED))
	ui = update_ll_int * update_factor;
      else
	ui = update_ll_int;
      if (now - rli->softstatestart >= ui) {
	rli->softstatestart = now;
	/*
	 * Since we're going to do a full update now flush queue and any
	 * pending output.  Avoids memory leak if RLI is down.
	 */
	rli->flags |= (FR_DOUPDATE|FR_SYNCED|FR_FLUSHPENDING);
	update_flushqueue(rli);
	unsynced++;
      } else
	rli->flags &= ~FR_DOUPDATE;
      globus_mutex_unlock(&rli->mtx);
    }
    if (!unsynced) {
      if (loglevel)
	logit(LOG_WARNING, "lrc_llupdates: No unsynced RLI servers");
      goto done;
    }

    if ((rc = db_lrc_alllfn(lrch, lrc_update_ll_cb, rliop_add,
			    errbuf)) != GLOBUS_RLS_SUCCESS) {
      logit(LOG_WARNING, "lrc_llupdates: alllfn: %s",
	    globus_rls_errmsg(rc, errbuf, errbuf2, MAXERRMSG));
      goto done;
    }

    /*
     * Queue magic record so queue processor knows when a softstate update
     * is finished and can log the time it took.
     */
    for (rli = lrc_rlilist; rli; rli = rli->nxt)
      if (rli->flags & FR_DOUPDATE)
	update_queue(rli, RU_SOFTSTATEEND);

   done:
    rls_lock_release(&lrc_rlilistlock, readlock);

   done1:
    globus_mutex_lock(&ec->mtx);
    while (!(ec->flags & (EF_EXIT|EF_RUN)))
      globus_cond_wait(&ec->cond, &ec->mtx);

    if (ec->flags & EF_EXIT) {
      ec->flags &= ~EF_RUNNING;
      globus_cond_signal(&ec->cond);
      globus_mutex_unlock(&ec->mtx);
      if (lrch)
	db_close(lrch);
      pthread_exit(0);
    } else {
      ec->flags &= ~EF_RUN;
      globus_mutex_unlock(&ec->mtx);
    }
  }
}

/*
 * Queue lfn to RLIs that need update.  lrc_rlilist should be read locked by
 * caller.
 */
static int
lrc_update_ll_cb(void *a, char *lfn)

{
  UPDATE	*u;
  RLI		*rli;

  if (loglevel > 4)
    logit(LOG_DEBUG, "lrc_update_ll_cb: %s", lfn);

  /* Delay allocating UPDATE until we know we need to send to an RLI	*/
  u = NULL;

  for (rli = lrc_rlilist; rli; rli = rli->nxt) {
    if (!(rli->flags & FR_DOUPDATE))
      continue;
    if (update_patternmatch(rli, lfn)) {
      if (!u)		/* Allocate UPDATE if haven't already	*/
	if ((u = update_new(lfn, NULL, (RLIOP) a)) == NULL)
	  return GLOBUS_FALSE;
      update_queue(rli, u);
    }
  }
  return GLOBUS_TRUE;
}
